/**
 **
 ** IntersceneMatchParameters - Class for handling multiple configuration parameters
 ** related to the interscene match 
 **
 ** Copyright (C) 2022 Elphel, Inc.
 **
 ** -----------------------------------------------------------------------------**
 **
 **  IntersceneMatchParameters.java is free software: you can redistribute it and/or modify
 **  it under the terms of the GNU General Public License as published by
 **  the Free Software Foundation, either version 3 of the License, or
 **  (at your option) any later version.
 **
 **  This program is distributed in the hope that it will be useful,
 **  but WITHOUT ANY WARRANTY; without even the implied warranty of
 **  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 **  GNU General Public License for more details.
 **
 **  You should have received a copy of the GNU General Public License
 **  along with this program.  If not, see <http://www.gnu.org/licenses/>.
 ** -----------------------------------------------------------------------------**
 **
 */
package com.elphel.imagej.tileprocessor;
import java.awt.Color;
import java.io.IOException;
import java.util.Properties;
import java.util.StringTokenizer;

import com.elphel.imagej.common.GenericJTabbedDialog;

public class IntersceneMatchParameters {
	public static String [] MODES3D = {"RAW", "INF", "FG", "BG"}; // RAW:-1
	public static String [] MODES_AVI = {"RAW", "JPEG", "PNG"};
	// Maybe add parameters to make sure there is enough data? Enough in each zone? Enough spread?
	public  boolean ims_use =            true;   // use IMS data
	public  boolean ims_rebuild =        false;  // recalculate INS data (after lag change)
	public  double  ims_offset =         0.0;    // IMS data lead relative to images (seconds) 
	public  double  gmt_plus =           3.0;    // (hrs) filename timestamp is this lower than in IMU log
	public  double  [] ims_ortho =       {0.5, -0.5, 0.5, -0.5}; // approximate (90-deg) IMS to camera
	public  double  [] ims_mount_atr =   {0, 0, 0}; // IMS mount fine correction (A,T,R around camera axes)
	public  double  [] ims_mount_xyz =   {0, 0, 0}; // IMS center in camera coordinates
	public  double  [] ims_scale_xyz =   {1.1, 1.1, 1.1};
	public  double  [] ims_scale_atr =   {1.1, 1.1, 1.1};
		
	
	public  boolean sfm_use =            true;   // use SfM to improve depth map
	public  double  sfm_min_base =       2.0;    // use SfM if baseline exceeds this
	public  double  sfm_min_gain =       5.0;    // Minimal SfM gain to apply SfM to the depth map
	public  double  sfm_min_frac =       0.5;    // Minimal fraction of defined tiles to have SfM correction
	
    public  int     sfm_num_pairs =     16;      // desired number of SfM pairs to average
    public  double  sfp_tolerance =      0.05;   // average SfM pairs if their baselines differ less
	public  int     sfm_readjust =       5;      // number of SfM readjustment cycles 
	public  double  sfm_prev_frac =      0.6;    // update if new sfm gain > this fraction of the old one
	public  double  sfm_same_weight =    0.8;    // correction weight when new SfM gain is the same as the old one
	public  int     sfm_shrink =         2;      // shrink sfm gain area before applying sfm_fade_sigma
    public  double  sfm_fade_sigma =     3.0;    // fade SfM gains at the edges 
	public  double  sfm_min_str1 =       0.2;    // update if correction strength exceeds
	public  double  sfm_min_str16 =      0.4;    // update if correction strength exceeds (for >=16 pairs)
	public  boolean sfm_use_neibs =      true;   // use neighbors if individual corr is too weak
	public  double  sfm_neib_too_str1=   0.3;    // do not count neighbors stronger than that
	public  double  sfm_neib_too_str16=  0.4;    // do not count neighbors stronger than that  (for >=16 pairs)
	public  double  sfm_neib_str1=       0.2;    // update if no-individual and neibs correction strength exceeds 
	public  double  sfm_neib_str16=      0.5;    // update if no-individual and neibs correction strength exceeds  (for >=16 pairs) 
	public  int     sfm_extrap_steps =   5;      // extrapolate disparity this number over undefined by SfM area
	public  int     sfm_extrap_radius =  5;      // find tilt for extrapolation using this radius circle around the new tile
	
	public boolean  sfm_average_neibs = false;   // Average neighbors disparity among them  
	public boolean  sfm_fill_weak =     false;   // Fill too weak tiles from nearest neighbors
	public boolean  sfm_extrapolate =   false;   // Extrapolate tiles with no SfM data (near edges)
	
	public double [] getImsMountATR() {
		return new double [] {
				ims_mount_atr[0] * Math.PI/180,
				ims_mount_atr[1] * Math.PI/180,
				ims_mount_atr[2] * Math.PI/180};
	}
	public  boolean center_reference =   false;
	public  boolean reset_photometric =  true;  // reset photometric calibration - once for each new series
	public  boolean force_ref_dsi =      false; // true;
	public  boolean force_orientations = false;
	public  boolean run_ly =             false; // will return just after LY adjustments, skipping all output generation
	public  int min_num_orient =         2; // make from parameters, should be >= 1
	public  int min_num_interscene =     2; // make from parameters, should be >= 1

	public  boolean calc_quat_corr =     true; // calculate camera orientation correction from predicted by IMS 
    public  boolean apply_quat_corr =    true; // apply camera orientation correction from predicted by IMS
	
	public  boolean generate_egomotion = true; // generate egomotion table (image-based and ims)
	public  boolean generate_mapped =    true; // generate any of the sequences - Tiff, video, mono or stereo
	public  boolean reuse_video =        false; // dry-run video generation, just reassemble video fragments (active if !generate_mapped)
	public  boolean save_mapped_color =  false; // may be not needed when using AVI
	public  boolean save_mapped_mono =   true;  // may be not needed when using AVI
	public  boolean generate_raw =       false;
	public  boolean generate_inf =       false;
	public  boolean generate_fg =        true;
	public  boolean generate_bg =        false;

	public  boolean generate_stereo =    false; // applies to all stereobases, including 0
	
	public  boolean export_images =      true;  // pseudo-color 16-slice images (same disparity, COMBO_DSN_INDX_DISP_FG and COMBO_DSN_INDX_DISP_BG_ALL,
	public  boolean show_images =        false; // color, infinity
	public  boolean show_images_bgfg =   false; // bg and fg
	public  boolean show_images_mono =   false; // float, monochrome 16-slice images (same disparity, COMBO_DSN_INDX_DISP_FG and COMBO_DSN_INDX_DISP_BG_ALL,
	public  boolean show_color_nan =     true;  // use NAN background for color images (sharp, but distinct black)
	public  boolean show_mono_nan =      false; // use NAN background for monochrome images (sharp, but distinct black)
	
	public  double [][] stereo_views = {  // base, up, back
			{   0.0,   0.0,    0.0},
			{ 200.0,   0.0,    0.0},
			{ 500.0,   0.0, 2000.0},
			{1000.0, 500.0, 3000.0}};
	
	public  boolean [] generate_stereo_var = new boolean[stereo_views.length];

	// Other parameters
	public  double  scale_extrap_atr =    1.0; // scale angular extrapolation
	public  double  scale_extrap_xyz =    0.0; // scale linear extrapolation (1.0 for drones)
	public  int     avg_len =             1;   // average pose over number of previous scenes
	public  int     min_num_scenes =     10;   // abandon series if there are less than this number of scenes in it 
	public  int     max_num_scenes =    100;   // abandon series if longer than 
	public  double  blur_egomotion =      2.0;
	
	
	public  boolean export_ranges =      true;
	public  boolean debug_ranges =       false;
	public  boolean show_ranges =        true;
	
	public  boolean export_ml_files =    false; // export files for DNN training
	
	public  double  range_disparity_offset =   -0.08;
	public  double  range_min_strength = 0.5;
	public  double  range_max =       5000.0;
	// Export 3D model
	public  boolean export3d =           false; // true;
	
	
	
	// Debug and visualization
	public  boolean scene_is_ref_test=              false; // correlate ref-2-ref for testing
	private boolean render_ref =                    true;  // render reference scene
	private boolean render_scene =                  true;  // render scene to be correlated with ref
	public  boolean toRGB =                         true;  // render scenes in pseudo-colors DOES NOT WORK - use ColorProcParameters
	private boolean show_2d_correlations =          true;  // show raw 2D correlations (individual and combined)
	private boolean show_motion_vectors =           true;  // show calculated motion vectors
	public  int     debug_level =                     -1;  // all renders are disable for debug_level < 0, scene "renders" for for debug_level < 1

	// Pairwise ERS testing
	public boolean  test_ers =                     false;
	public  int     test_ers0 =                       -1; // try adjusting a pair of scenes with ERS. Reference scene index
	public  int     test_ers1 =                       -1; // try adjusting a pair of scenes with ERS. Other scene index

	// Other parameters for filtering depth maps	
	public  int         num_bottom =                      3; // 6; // average this number of lowest disparity neighbors (of 8)
	public  int         num_passes =                    100;
	public  double      max_change =                      1.0e-3;
	public  double      min_disparity =                  -0.15; // 0.2;
	public  double      max_sym_disparity =               0.1; // 0.2;
	// 2 next are wrong currently - minimal strength is ~0.25
	public  double      min_strength_lma =                0.0; // 0.3;  // no real filtering
	public  double      min_strength_replace =            0.05; ///  0.14; /// Before /// - LWIR, after - RGB
	public  double      min_strength_blur =               0.06; ///  0.2;
	public  double      sigma =                           2; /// 5;
	public  int         num_blur =                        1; // 3;
	public  double      disparity_corr =                  0.0;
	public  int         outliers_nth_fromextrem =         1; // second from min/max - removes dual-tile max/mins
	public  double      outliers_tolerance_absolute =     0.2;
	public  double      outliers_tolerance_relative =     0.02;
	public  int         outliers_max_iter =             100;
	public  double      outliers_max_strength2 =          1.0; // 0.5; any
	public  int         outliers_nth_fromextrem2 =        0; // second from min/max - removes dual-tile max/mins
	public  double      outliers_tolerance_absolute2 =    0.5;
	public  double      outliers_tolerance_relative2 =    0.1;
	public  double      outliers_lma_max_strength =       0.4; // 0.5;
	public  double      outliers_max_strength =           0.1; ///  0.25;
	public  double      outliers_from_lma_max_strength =  0.8;
	public  int         search_radius =                   3;  // Search farther if no LMA neighbor is found closer. Original value - 1 (8 neighbors)
	public  boolean     remove_no_lma_neib =              true;
	public  double      diff_from_lma_pos =             100.0;
	public  double      diff_from_lma_neg =               2.0;
	public  int         outliers_lma_nth_fromextrem =     0; // 1; 
	public  int         filter_margin =                  -8; // 8; // pixels

	public  double      weak_tolerance_absolute=          0.25;
	public  double      weak_tolerance_relative=          0.025;
	public  int         weak_min_neibs =                  5;
	public  double      strong_strength=                  0.5;
	public  double      weak_strength=                    0.2; // none is below 
	//------------	
	
	// Sky detection/filtering
	public  boolean sky_extract =                    true;  // do not try to extract blue sky
	public  boolean sky_recalc =                     false; // force blue sky recalculation even if it exists
	public  double  sky_highest_min =                -50; // lowest absolute value should not be higher (requires photometric) 
	public  double  cold_frac =                        0.005; // this and lower will scale fom by  cold_scale
	public  double  hot_frac =                         0.9;    // this and above will scale fom by 1.0
	public  double  cold_scale =                       0.2;  // <=1.0. 1.0 - disables temperature dependence
	
	public  double  sky_seed =                         5.0;  // 5.0 start with product of strength by diff_second below this
	public  double  lma_seed =                         2.0;  // seed - disparity_lma limit		double sky_lim  =      15.0; // then expand to product of strength by diff_second below this
	public  double  seed_temp =                        0.5;  // seed colder that this point between min and max temp
	
	public  int     sky_shrink =                       2;
	public  int     seed_rows =                        5; // sky should appear in this top rows 
	public  double  max_disparity =                    0.3;  // maximal strong sky disparity 
	public  double  max_disparity_strength =           0.2;  // maximal strength of high disparity sky
	public  double  sky_lim =                         15.0; // then expand to product of strength by diff_second below this
	public  double  lim_temp =                         0.5;  // sky colder that this point between min and max temp
	public  int     sky_expand_extra =                 0; // 1?
	public  int     sky_bottleneck =                   5;
	public  int     sky_reexpand_extra =               9;//  re-expand after bottleneck in addition to how it was shrank	
	public  double  min_strength =                     0.08;
	public  int     lowest_sky_row =                  50; // appears that low - invalid, remove completely
	public  double  sky_bottom_override =           -300; // maximal average sky value to override lowest_sky_row test
	public  int     sky_override_shrink =             6; // shrink detected sky before finding hottest tile there
	
	
	// processing clouds in the sky
	public boolean  clouds_en =                       true;  // enable clouds in the sky detection / processing
	public  double  clouds_fom =                      30.0;  // maximal FOM for clouds (must be <=)         
	public  double  clouds_spread =                   60.0;  // maximal spread for clouds (must be <=)         
	public  double  clouds_disparity =                 0.1;  // maximal disparity for strong clouds         
	public  double  clouds_weak =                      0.18; // maximal strength for near definite clouds         
	public  double  clouds_strength =                  0.25; // minimal strength for far strong clouds (definitely cloud)
	public  double  clouds_not_strength =              0.4;  // maximal strength for near maybe clouds (if it has strong cloud neighbor)
	public boolean  clouds_strong =                   true;  // allow weak cloud if it has strong (1.5x) cloud neib
	
	// process smooth walls mistaken for sky (disable for natural environments)
	public boolean  wall_en =                         true;  // enable smooth walls detection/processing
	public boolean  wall_dflt =                       false; // default (undetected) is wall (false - sky)
	public  double  wall_str =                        0.1;   // minimal strength of the far object (small - just non-NaN disparity)         
	public  double  wall_far =                        0.2;   // maximal disparity to consider cluster to be sky         
	public  double  wall_near =                       1.0;   // minimal disparity to consider cluster to be wall         

	// processing far treeline that may be confused with clouds. Only behind far objects such as far horizontal surface.
	// Maybe add temperature (treeline warmer than sky)?
	public boolean  treeline_en =                     true;  // enable treeline processing
	public boolean  treeline_wide =                   true;  // look not only under, but diagonal too.
	public int      treeline_height =                  8;    // maximal height of the treeline (tiles)
	public int      treeline_width =                   3;    // minimal horizontal width of the treeline (tiles)
	public boolean  treeline_lim_high =               false; // limit too high treeline (false - delete completely)
	public double   treeline_str =                     0.3;  // treeline minimal strength 
	public double   treeline_far =                     0.03; // treeline min disparity (pix)
	public double   treeline_near =                    0.4;  // treeline max disparity (pix)
	public double   treeline_fg_str =                  0.4;  // pre-treeline FG objects (such as flat ground) minimal strength 
	public double   treeline_fg_far =                  0.2;  // pre-treeline FG objects  min disparity (pix)
	public double   treeline_fg_near =                 0.6;  // pre-treeline FG objects  max disparity (pix)
	
	// suspecting indoors (disabling sky)
	public boolean  indoors_en =                      true; // allow weak cloud if it has strong (1.5x) cloud neib
	public  double  indoors_str =                      0.5; // minimal strength of the far object         
	public  double  indoors_disp =                     0.8; // maximal minimal outdoor strong disparity         
	public  int     indoors_min_out  =                10;   // minimal strong far tiles to deny indoors
	
	// Increase fom if there are enough LMA-defined similar tiles around
	public  double  disp_boost_min  =                  0.5;
	public  double  disp_boost_diff  =                 0.35;
	public  int     disp_boost_neibs  =                2;
	public  double  disp_boost_amount  =               2.0;
	public  double  scale_combo_strength  =            0.4; // reduce strength when it comes from combo, not DSI-MAIN
	public  boolean save_debug_images =               true;  // Save blue sky debug images in model directory
	/// IMS intergration
	
	
	
	// Some "AGC" to adjust how much to discard
	public  int     margin =                       1;      // do not use tiles if their centers are closer to the image edge
	public  int     sensor_mask_inter =           -1;      // bitmask of the sensors to use (-1 - all)
	public  boolean use_partial =                   true;  // find motion vectors for individual pairs, false - for sum only
	public  boolean run_poly =                      false; // not yet implemented
	public  double  centroid_radius =               4.0;   // 
	public  int     n_recenter =                    2;     // when cosine window, re-center window this many times
	// filtering motion vectors
	// TD accumulation of the inter-scene correlations  demonstrated artifacts (horizontally offset by 8 pixels
	// false maximum that is sharper than the real one. Still not understood - maybe float precision related. 
	public  double  td_weight =                     0.25; // 0.75;   // mix correlations accumulated in TD with 
	public  double  td_neib_weight =                0.6; // 0.75;   // mix correlations accumulated in TD (neibs) 
	public  double  pd_weight =                     0.15; // 0.25;   // correlations (post) accumulated in PD
	public  boolean td_nopd_only =                  false; // true;  // only use TD accumulated data if no safe PD is available for the tile.
	public  boolean neib_notd_only =                false; // true;  // use neighbors only if individual TD is too weak

	// averaging with neighbors for interscene matching
	public  boolean use_neibs =                     true;  // false; // true;
	public  boolean neibs_nofpn_only =              false; // consolidate neighbors for non-fpn tiles only!
	public  boolean neibs_nofpn_init =              true;  // Non-fpn only for initial adjustment
	public  boolean redo_both =                     true;  // use average of neighbors for both pd,td if any of the center tile tests (td, pd) fails
	public  int min_num_neibs =                     4;     // plus center, total number >= (min_num_neibs+1)
	public  double scale_neibs_pd =                 0.5;   // scale threshold for the pixel-domain average maximums
	public  double scale_neibs_td =                 0.5;   // scale threshold for the transform-domain average maximums
	public  double scale_avg_weight =               0.5;   // reduce influence of the averaged correlations compared to the single-tile ones
	
	public  double  min_str_fpn =                   0.2;   // 0.25; // minimal correlation strength for all but TD-accumulated layer
	public  double  min_str_sum_fpn =               0.42;  // 0.5; // 0.8;  // minimal correlation strength for TD-accumulated layer
	public  double  min_str_neib_fpn =              0.5;   // 0.8;  // minimal correlation strength for TD-accumulated neighbors layer

	public  double  min_str =                       0.15;  // 12; //18;  // tiles w/o FPN: minimal correlation strength for all but TD-accumulated layer
	public  double  min_str_sum =                   0.3;   //  0.2;  // 0.33; // tiles w/o FPN: minimal correlation strength for TD-accumulated layer
	public  double  min_str_neib =                  0.3;   // 0.33; // tiles w/o FPN: minimal correlation strength for TD-accumulated layer

	public  int     min_neibs =                     2;	   // minimal number of strong neighbors (> min_str)
	public  double  weight_zero_neibs =             0.2;   // Reduce weight for no-neib (1.0 for all 8)
	public  double  half_disparity =                5.0;   // Reduce weight twice for this disparity
	public  double  half_avg_diff =                 0.2;   // when L2 of x,y difference from average of neibs - reduce twice
	
	// Detect initial match
	public boolean  use_combo_relaible =           true;  // use combo dsi if available for relaible tiles
	public boolean  ref_need_lma =                 true;  // need LMA output for reliable tiles (no combo available)
	public boolean  ref_need_lma_combo =           true;  // need LMA output for reliable tiles (when combo is available)
	public  double  min_ref_str =                   0.33;  // 0.22;  // For orientations: use only tiles of the reference scene DSI_MAIN is stronger  
	public  double  min_ref_frac =                  0.2;   // 0.22;  if fraction number of reliable tiles is less than this, use best possible  
	public  int     pix_step =                      4;     // Azimuth/tilt search step in pixels
	public  int     search_rad =                   10;     // Search radius in steps
	public  double  maybe_sum =                     1.0;   // minimal sum of strengths (will search for the best)
	public  double  sure_sum =                     50.0;   // definitely good sum of strengths (no more search) too high, will be FOM-defined
	public  double  maybe_avg =                     0.01;  // maybe average strength
	public  double  sure_avg =                      0.15;  // 0.015; // sure average strength disabling (was 0.03)
	public  double  max_search_rms =                1.5;   // good - 0.34, so-so - 0.999
	public  double  maybe_fom =                     1.0;   // good - 38, second good - 4.5
	public  double  sure_fom =                     12.0;   // good - 38, second good - 4.5
	public  boolean treat_serch_fpn =              false;  // use FPN (higher) thresholds during search (even if offset is not small)
	
	
	// Reference scene disparity 
	public  boolean use_combo_dsi =                 true;  // use interscene DSI if available (instead of the single-scene)
	public  boolean use_lma_dsi =                   true;  // only use reference DSI tiles that have LMA (strong) disparity
	
	// Remove correlation caused by FPN
	public  boolean fpn_remove =                    true;  // only use reference DSI tiles that have LMA (strong) disparity
	public  double  fpn_max_offset =                8.0;   // pix - ignore larger FPN offsets
	public  double  fpn_radius =                    0.75;  // pix - zero around center 
	public  boolean fpn_ignore_border =             false; // only if fpn_mask != null - ignore tile if maximum touches fpn_mask
	
	public  double  min_offset =                    1.5;   // pixels - minimal average pixel offset between images to consider interscene matching
	public  double  max_rel_offset =                0.5;   // maximal interscene offset as a fraction of image width
	public  double  max_roll_deg =                  5.0;   // maximal interscene roll to consider matching
	public  double  max_zoom_diff =                 0;     // for down-views when changing altitude (0 - ignore)
	public  boolean fpn_skip =                      true;   // skip too close scenes (false - abort, previous behavior)
	public  boolean fpn_rematch =                   true;   // match fpn-failed scenes to later scenes with larger difference
	
	// Remove moving objects (goal is not to detect slightest movement, but to improve pose matching
	public  boolean mov_en =                        true;  // enable detection/removal of the moving objects during pose matching
	public  double  mov_sigma =                     1.5;   // pix - weighted-blur offsets before detection
	// next two to prevent motion detection while errors are too big
	public  double  mov_max_std =                   1.2;   // pix
	public  double  mov_thresh_rel =                3.5;   // .0;   // exceed average error
	public  double  mov_thresh_abs=                 0.5;   // sqrt(dx^2+dy^2) in moving areas 
	public  double  mov_clust_max =                 1.5;   // cluster maximum should exceed threshold this times
	public  int     mov_grow =                      4;     // grow detected moving area
	public  int     mov_max_len =                   0;     // (0 - no limit) do not remove moving objects if they do not fit into the square   
	public  boolean mov_show =                      true;  // show debug images for movement detection
	public  int     mov_debug_level =               1;     // >0 verbose
	//LMA parameters
	public  boolean [] adjust_atr = new boolean [] {true,true,true};
	public  boolean [] adjust_xyz = new boolean [] {true,true,true};
	public  double  exit_change_atr =               1.0E-5;//rad,  L2 norm for difference to ext LMA 
	public  double  exit_change_xyz =               1.0E-3;//meters, L2 norm for difference to ext LMA 
	public  int     max_cycles =                   10;     // hard limit on full correlation/LMA cycles
	public  int     max_LMA =                      25;     // hard limit on LMA iterations
	public  double  max_rms =                       2.0;   // maximal RMS to consider adjustment to be a failure
	
// equalization of the inter-scene correlations	
	public  boolean eq_en =                 true;// equalize "important" FG tiles for better camera XYZ fitting
	public  int     eq_stride_hor =         8;   // half of a equalization supertile width
	public  int     eq_stride_vert =        8;   // half of a equalization supertile height
	// Supertile qualification
	public  double  eq_min_stile_weight =   0.2; // minimal total weight of the tiles in a supertile (lower will not be modified)
	public  int     eq_min_stile_number =   10;  // minimal number of defined tiles in a supertile
	public  double  eq_min_stile_fraction = 0.02;// minimal total tile strength compared to the average one
	// Individual tiles qualification
	public  double  eq_min_disparity =      5; // minimal disparity of tiles to consider (applies after filtering / boosting)
	public  double  eq_max_disparity =   1000; // maximal disparity of tiles to consider (applies after filtering / boosting)
	// Boost amount 
	public  double  eq_weight_add =         0.03; // calculate from min-strengths
	public  double  eq_weight_scale =      20.0;   // maximal boost ratio 
	public  double  eq_level =              0.9; // equalization level (0.0 - leave as is, 1.0 - boost to have the same supertile strength as average)
	
	public  boolean mb_en =                 true;
	public  double  mb_tau =                0.008; // time constant, sec
	public  double  mb_max_gain =           5.0;   // motion blur maximal gain (if more - move second point more than a pixel
	public  double  mb_max_gain_inter =     2.0;   // same for interscene correlation for pose adjustment
	
	
	public  boolean stereo_merge =       true;
	public  int     stereo_gap =         32;    // pixels between right and left frames
	public  double  stereo_intereye =    63.5;  // mm 
	public  double  stereo_phone_width = 143.0; // mm if 0 - no padding. Will be padded only when merging

	public  int     extra_hor_tile =     15;
	public  int     extra_vert_tile =    10;
	public  boolean crop_3d =            true; // do not show extra of FG/BG views (currently they only ref scene has disparity)
	public  int     sensor_mask =        1; // -1 - all
	public  boolean merge_all =          true; // merge all 16 channels for 3D modes 

	public  boolean show_mapped_color =  true;
	public  boolean show_mapped_mono =   true;

	public  boolean gen_avi_color =      true; // will use save_mapped_color
	public  boolean gen_avi_mono =       true; // will use save_mapped_mono
	public  double  video_fps =          15; // 4x slowmo
	public  double  sensor_fps =         60; // sensor fps
	public  int     mode_avi =           0;  // 0 - raw, 1 - JPEG, 2 - PNG
	public  int     avi_JPEG_quality =  90;
	public  boolean run_ffmpeg =         true; // only after AVI
	public  String  video_ext =          ".webm";
	public  String  video_codec =        "vp8";
	public  int     video_crf =         40;    // lower - better, larger file size
	public  boolean remove_avi =         true; // remove avi after conversion to webm
	public  String  video_codec_combo =  "vp8"; // applies when combining videos
	public  int     video_crf_combo =   40;    // lower - better, larger file size applies when combining videos
	public  boolean um_mono =            true; // applies to both TIFF and AVI 
	public  double  um_sigma =          10;
	public  double  um_weight =          0.97; //
	public  boolean mono_fixed =         true; // normalize to fixed range when converting to 8 bits 
	public  double  mono_range =       500.0;  // monochrome full-scale range (+/- half)
	public  boolean anaglyth_en =        true; // applies to both TIFF and AVI 
	public static Color   anaglyph_left_default = new Color (255, 0, 0); // red
	public static Color   anaglyph_right_default =new Color (0, 255, 255); // cyan
	public Color   anaglyph_left =    anaglyph_left_default;
	public Color   anaglyph_right =   anaglyph_right_default; // cyan
	
	public  boolean annotate_color =     true; // annotate pseudo-color video frames with timestamps 
	public  boolean annotate_mono =      true; // annotate monochrome video frames with timestamps 
	public Color   annotate_color_color =new Color( 255, 255, 255); // greenish over "fire"
	public Color annotate_color_mono =   new Color( 255, 180,  50); // reddish over grey
	public boolean annotate_transparent_mono =  false; // // black if not transparent
	
	
	public boolean renderRef()            {return renderRef             (debug_level);}
	public boolean renderScene()          {return renderScene           (debug_level);}
	public boolean show2dCorrelations()   {return show2dCorrelations    (debug_level);}
	public boolean showMotionVectors()    {return showMotionVectors     (debug_level);}
	public boolean showCorrMotion()       {return showCorrMotion        (debug_level);}
	public boolean showMovementDetection(){return showMovementDetection (debug_level);}
	public int movDebugLevel()            {return movDebugLevel         (debug_level);}

	public boolean renderRef(int dlev)            {return (dlev>1) && render_ref;}
	public boolean renderScene(int dlev)          {return (dlev>1) && render_scene;}
	public boolean show2dCorrelations(int dlev)   {return (dlev>1) && show_2d_correlations;}
	public boolean showMotionVectors(int dlev)    {return (dlev>1) && show_motion_vectors;}
	public boolean showCorrMotion(int dlev)       {return (dlev>0) && show_motion_vectors;}
	public boolean showMovementDetection(int dlev){return (dlev>0) && mov_show;}
	public int movDebugLevel(int dlev)            {return (dlev > -1) ? mov_debug_level : 0;}
	
	
	public IntersceneMatchParameters() {
		
	}

	public void dialogQuestions(GenericJTabbedDialog gd) {
		//		gd.addMessage  ("Scene parameters selection");
//		gd.addTab         ("Inter-Match", "Parameters for full-resolution (no decimation/macrotiles) scene matching");
		gd.addTab         ("IMS", "IMS Integration for interscene matching");
		gd.addCheckbox ("Use IMS data",                              this.ims_use,
				"Use IMS data for egomotion.");
		gd.addCheckbox ("Recalculate IMS",                           this.ims_rebuild,
				"Recalculate scene INS data.");
		gd.addNumericField("IMS offset",                             this.ims_offset, 5,8,"s",
				"Positive if the IMS data has lower lag than images.");
		gd.addNumericField("GMT plus",                               this.gmt_plus, 5,8,"hrs",
				"Timestamp filename is lower by this hours than in the IMU log.");
		
		gd.addStringField ("IMS ortho mount quaterion", IntersceneMatchParameters.doublesToString(ims_ortho), 80,
				"Approximate (90-deg) IMS to camera {0.5, -0.5, 0.5, -0.5}.");
		gd.addStringField ("IMS mount ATR correction (deg)",  IntersceneMatchParameters.doublesToString(ims_mount_atr), 80,
				"IMS mount fine correction (A,T,R around camera axes).");
		gd.addStringField ("IMS mount XYZ correction (m)",  IntersceneMatchParameters.doublesToString(ims_mount_xyz), 80,
				"MS center (X,Y,Z m) in camera coordinates.");

		gd.addStringField ("scale XYZ from IMS", IntersceneMatchParameters.doublesToString(ims_scale_xyz), 80,
				"Scales for X, Y, and Z velocities when converting from IMS data, such as '1.1 1.1 1.1'.");
		gd.addStringField ("scale ATR from IMS", IntersceneMatchParameters.doublesToString(ims_scale_atr), 80,
				"Scales for Azimuth, Tilt, and Roll velocities when converting from IMS data, such as '1.1 1.1 1.1'.");
		
		gd.addTab         ("SfM", "Structure from Motion to improve depth map for the lateral views");
		gd.addCheckbox ("Use SfM",                                   this.sfm_use,
				"Use SfM for the depth map enhancement for laterally moving camera.");
		gd.addNumericField("Minimal SfM baseline",                   this.sfm_min_base, 5,8,"m",
				"Use SfM only if the baseline (lateral offset between scenes in a series) exceeds this value.");
		gd.addNumericField("Minimal SfM gain",                       this.sfm_min_gain, 5,8,"",
				"Minimal SfM gain to apply SfM to the depth map.");
		gd.addNumericField("Minimal SfM fraction",                   this.sfm_min_frac, 5,8,"",
				"Minimal fraction of defined tiles to have SfM correction.");
		gd.addNumericField("Number of SfM pairs to average",         this.sfm_num_pairs, 0,3,"",
				"Desired number of SfM pairs to average, will be reduced if not enough or tolerance (below) is violated.");
		gd.addNumericField("SfP inter-pair baseline tolerance",      this.sfp_tolerance, 5,8,"",
				"Average SfM pairs if their baselines differ less.");
		gd.addNumericField("Readjust SfM cycles",                    this.sfm_readjust, 0,3,"",
				"Number of SfM readjustment cycles.");
		gd.addNumericField("Minimal SfM gain from previous",         this.sfm_prev_frac, 5,8,"",
				"Update if new sfm gain > this fraction of the old one.");
		gd.addNumericField("Correction scale for equal gains",         this.sfm_same_weight, 5,8,"",
				"Correction scale when new SfM gain is the same as the old one.");
		gd.addNumericField("Shrink SfM area",                        this.sfm_shrink, 0,3,"",
				"Shrink sfm gain area before applying sfm_fade_sigmas.");
		gd.addNumericField("Fade SfM area sigma",                    this.sfm_fade_sigma, 5,8,"",
				"Fade SfM gains at the edges.");
		gd.addNumericField("Minimal correlation strength",           this.sfm_min_str1, 5,8,"",
				"Update if correction strength of individual tiles exceeds.");
		gd.addNumericField("Minimal correlation strength >= 16 pairs", this.sfm_min_str16, 5,8,"",
				"Update if correction strength of individual tiles exceeds (for >=16 pairs).");
		gd.addCheckbox ("Use neighbors",                             this.sfm_use_neibs,
				"Use neighbors if individual corr is too weak.");
		gd.addNumericField("Too strong neighbors >= 16 pairs",       this.sfm_neib_too_str1, 5,8,"",
				"Do not accumulate neighbors stronger than that.");
		gd.addNumericField("Too strong neighbors",                   this.sfm_neib_too_str16, 5,8,"",
				"Do not accumulate neighbors stronger than that (for >=16 pairs).");
		gd.addNumericField("Minimal neighbors strength",             this.sfm_neib_str1, 5,8,"",
				"Update if no-individual and neighbors correction strength exceeds this.");
		gd.addNumericField("Minimal neighbors strength >=16 pairs",  this.sfm_neib_str16, 5,8,"",
				"Update if no-individual and neighbors correction strength exceeds this (for >=16 pairs).");
		gd.addNumericField("Extrapolation steps",                    this.sfm_extrap_steps, 0,3,"",
				"Extrapolate disparity this number of times over the undefined by SfM area.");
		gd.addNumericField("Extrapolation radius",                   this.sfm_extrap_radius, 0,3,"",
				"Find tilt for extrapolation using this radius circle around the new tile.");
		gd.addCheckbox ("Average neighbors",                         this.sfm_average_neibs,
				"Average neighbors disparity among them.");
		gd.addCheckbox ("Fill too weak tiles",                       this.sfm_fill_weak,
				"Fill too weak tiles from nearest neighbors.");
		gd.addCheckbox ("Extrapolate no SfM data",                   this.sfm_extrapolate,
				"Extrapolate tiles with no SfM data (near edges).");
		
		
		gd.addTab         ("Scene Series", "Processing series of scenes and multi-series sets");
		gd.addMessage  ("Build series options");
		
		gd.addCheckbox ("Use center scene as a reference",           this.center_reference,
				"True for mapping with post-processing. If false, use the last scene as a reference (driving).");
		gd.addCheckbox ("Reset photometric calibration",             this.reset_photometric,
				"Reset photometric calibration, will use basic one before re-calibrating.");
		gd.addCheckbox ("Force reference scene DSI calculation",     this.force_ref_dsi,
				"Calculate reference scene DSI even if the file exists.");
		gd.addCheckbox ("Force egomotion calculation",               this.force_orientations,
				"Calculate relative poses of each scene camera relative to the reference scene even if the data exists.");
		gd.addCheckbox ("Run LY adjustments",                        this.run_ly,
				"Adjust cameras orientations (after building DSI) and exit without generatin result files.");
		gd.addNumericField("Minimal number of egomotion calculations",this.min_num_orient, 0,3,"",
				"Minimal number of fitting scenes cycles, should be >=1. First cycle includes spiral search for the first scene");
		gd.addNumericField("Minimal number of interscene accumulations", this.min_num_interscene, 0,3,"",
				"Minimal required number of re-calculations of the interscene-accumulated DSI.");
		gd.addCheckbox ("Calculate IMS orientation correction",      this.calc_quat_corr,
				"Calculate camera orientation correction from predicted by IMS .");
		gd.addCheckbox ("Apply IMS orientation correction",          this.apply_quat_corr,
				"Apply camera orientation correction from predicted by IMS.");
		
		gd.addMessage  ("Generate/show scene sequences");
		gd.addCheckbox ("Generate egomotion table",                  this.generate_egomotion,
				"Generate table with image-based and IMS egomotion data.");
		gd.addCheckbox ("Generate mapped scene sequence",            this.generate_mapped,
				"Generate scene sequence mapped to the reference scene.");
		gd.addCheckbox ("Reuse existing video fragments, disable other options",this.reuse_video,
				"Dry-run video generation, just re-assemble earlier generated video fragments. Disables all other file generation.");
		gd.addCheckbox ("Save scene sequences in (pseudo)colors as TIFF",this.save_mapped_color,
				"Save generated scene sequences in (pseudo)color mode as a multi-slice TIFF.");
		gd.addCheckbox ("Save scene sequences in monochrome as TIFF",    this.save_mapped_mono,
				"Show generated scene sequences in monochrome mode as a multi-slice TIFF. May use Unsharp Mask.");
		gd.addCheckbox ("Generate color video",                      this.gen_avi_color,
				"Generate video for color scene sequences.");
		gd.addCheckbox ("Generate monochrome video",                 this.gen_avi_mono,
				"Generate video for monochrome scene sequences.");
		gd.addCheckbox ("Show scene sequences in (pseudo) colors",    this.show_mapped_color,
				"Show generated scene sequences in (pseudo) color mode. Disabled in batch mode");
		gd.addCheckbox ("Show scene sequences in monochrome",        this.show_mapped_mono,
				"Show generated scene sequences in monochrome mode. May use Unsharp Mask. Disabled in batch mode.");
		
		
		gd.addCheckbox ("Generate RAW images",                       this.generate_raw,
				"Raw images from single (top) camera using original view - just after aberration correction and aligning sensors .");
		gd.addCheckbox ("Generate INF images",                       this.generate_inf,
				"Single-camera images aligned at infinity - 2D correction functionally similar to camera stabilization.");
		gd.addCheckbox ("Generate 3D-corrected FG images",           this.generate_fg,
				"Correct in 3D scene images (from all 16 sensors) matching reference (last in sequence) scene with foreground (FG) priority.");
		gd.addCheckbox ("Generate 3D-corrected BG images",           this.generate_bg,
				"Correct in 3D scene images (from all 16 sensors) matching reference (last in sequence) scene with background (BG) priority.");
		gd.addCheckbox ("Generate binocular stereo pairs",           this.generate_stereo,
				"Generate stereo-pairs for 3D-corrected videos (FG,BG). Ebables specific modes (including 0-baseline / mono).");

		for (int i = 0; i < stereo_views.length; i++) {
			double base = stereo_views[i][0]; // stereo_bases[i];
			String ub = String.format("(%.0fmm up, %.0fmm back) ",stereo_views[i][1],stereo_views[i][2]);
			if ((stereo_views[i][1]==0) && (stereo_views[i][2]==0)){
				ub="";
			}
			String title = (base == 0.0)?
					"Generate mono (single camera) scene sequences"+ub:
						"Generate "+base+"mm-baseline stereo scene sequences"+ub;
			String tooltip = (base == 0.0)?
					"Generate mono (single camera) scene sequences "+ub+"as Tiff and/or video.":
						"Generate "+base+"mm-baseline stereo scene sequences "+ub+"as Tiff and/or video.";
			gd.addCheckbox (title,                                   this.generate_stereo_var[i], tooltip);
		}
		
		
		
		
		gd.addMessage  ("Generate/save reference (last) scene images");
		gd.addCheckbox ("Export all-sensor images",                  this.export_images,
				"Export multi-slice images: with constant disparity, with foreground disparity, and with background disparity");
		gd.addCheckbox ("Show exported images (same disparity)",     this.show_images,
				"Display generated/saved image set, pseudocolors");
		gd.addCheckbox ("Show exported FG/BG 3d-corrected",          this.show_images_bgfg,
				"Show foreground and background exported images");
		gd.addCheckbox ("Show floating-point monochrome images",     this.show_images_mono,
				"Display generated/saved monochrome images (in addition to color) ");
		gd.addCheckbox ("Generate metric depth map",                 this.export_ranges,
				"Calculate strength, distance, X, and Y in meters");
		gd.addCheckbox ("Debug metric depth map",                    this.debug_ranges,
				"Debug metric depth maps");
		gd.addCheckbox ("Show metric depth map",                     this.show_ranges,
				"Display calculated depth map in meters");
		gd.addCheckbox ("Export files for DNN training/testing",     this.export_ml_files,
				"Display calculated depth map in meters");
		
		gd.addMessage  ("Additional parameters");
		gd.addCheckbox ("Color NaN background",                      this.show_color_nan,
				"Use NaN for undefined tiles (false - 0.0f). NaN produces sharp distinct result, 0.0f - blended");
		gd.addCheckbox ("Mono NaN background",                       this.show_mono_nan,
				"Use NaN for undefined tiles (false - 0.0f). NaN produces sharp distinct result, 0.0f - blended");

		gd.addNumericField("Scale angular pose extrapolation",       this.scale_extrap_atr, 3,5,"",
				"Scale angular extrapolation relative to the previous pair");
		gd.addNumericField("Scale linear pose extrapolation",        this.scale_extrap_xyz, 3,5,"",
				"Scale linear extrapolation relative to the previous pair (use 1.0 for a drone)");
		gd.addNumericField("Number of previous scenes to extrapolate",this.avg_len, 0,3,"",
				"Average this number of previous scenes movement for next scene prediction");

		gd.addNumericField("Minimal number of scenes to keep series",this.min_num_scenes, 0,3,"",
				"Scrap all seriest if less numer of scenes can be matched to the reference scene (including reference itself)");
		gd.addNumericField("Cut longer series",                      this.max_num_scenes, 0,3,"",
				"Cut longer series");
		
		gd.addNumericField("LPF egomotion sigma",                    this.blur_egomotion, 3,5,"scenes",
				"LPF egomotion components with this sigma before using as ERS (not implemented).");

		gd.addMessage  ("Metric distance map generation");
		gd.addNumericField("Disparity at infinity",                  this.range_disparity_offset, 5,7,"pix",
				"Disparity at infinity - subtract from measured disparity when converting to ranges.");
		gd.addNumericField("Minimal strength for range calculation", this.range_min_strength, 5,7,"",
				"Disregard weaker results when measuring range.");
		gd.addNumericField("Maximal displayed range",                this.range_max, 5,7,"m",
				"Do not display extremely far objects.");

		gd.addMessage  ("3D model generation");
		gd.addCheckbox ("Generate 3D model",                         this.export3d,
				"Generate textures and model.");

		gd.addMessage  ("Debug and visialization parameters");
		gd.addCheckbox ("Replace scene with reference scene",        this.scene_is_ref_test,
				"Correlate reference scene with itself for testing (may want to manually change scene_atr and scene_xyz in debug mode)");
		gd.addCheckbox    ("Render reference scene",                 this.render_ref,
				"Render reference scene as a multi-layer (per-port) image");
		gd.addCheckbox    ("Render compared scene",                  this.render_scene,
				"Render reference compared as a multi-layer (per-port) image");
		gd.addCheckbox    ("Pseudo-color render (does not work)",    this.toRGB,
				"Use pseudo-color palette for scene render  - DOES NOT WORK - use ColorProcParameters");
		gd.addCheckbox    ("Show 2D correlations",                   this.show_2d_correlations,
				"Show raw 2D correlations (per channel and combined)");
		gd.addCheckbox    ("Show motion vectors",                    this.show_motion_vectors,
				"Show motion vectors and strengths per channel and combines, NaN-s are when strengths corresponding are below thresholds");
		gd.addNumericField("Debug Level for interscene match",       this.debug_level, 0,3,"",
				"Debug Level for the above parameters: renders and raw correlations need >1,  motion vectors > 0");

		gd.addMessage  ("Pairwise ERS testing");
		gd.addCheckbox ("Test pairwise matching with ERS",           this.test_ers,
				"Test pairwise dispaly and pose correction to test ERS compensation");
		gd.addNumericField("Test scene reference index",             this.test_ers0, 0,3,"",
				"Reference scene index in a scene sequence");
		gd.addNumericField("Test scene other scene index",           this.test_ers1, 0,3,"",
				"Other scene index in a scene sequence (should have a very different angular/linear velocity component)");
		
		
		gd.addTab("Fiter DSI","Filter DSI before use (interscene fitting, interscene accumulation, ML generation");
//		gd.addMessage  ("Depth map filtering parameters");
		gd.addNumericField("Average lowest disparity neighbors",     this.num_bottom, 0,3,"",
				"Average this number of lowest disparity neighbors (of 8)");
		gd.addNumericField("Number of filter iterations",            this.num_passes, 0,3,"",
				"Number of filter iterations");
		gd.addNumericField("Maximal change to exit",                 this.max_change, 5,7,"pix",
				"Maximal change to exit iterations");
		gd.addNumericField("Minimal disparity",                      this.min_disparity, 5,7,"pix",
				"Minimal acceptable disparity");
		gd.addNumericField("Disparity range for symmetrical pull",   this.max_sym_disparity, 5,7,"pix",
				"For larger disparities undefined tiles are pull predominantly down (to average of lowest), "+
		        "but for small disparities (sky) pull is to average of all neighbors.");
		gd.addNumericField("Minimal strength LMA",                   this.min_strength_lma, 5,7,"",
				"Lower strength for LMA tiles - treat as non-LMA");
		gd.addNumericField("Minimal strength to replace",            this.min_strength_replace, 5,7,"",
				"Minimal strength to replace (now not used)");
		gd.addNumericField("Minimal strength to blur",               this.min_strength_blur, 5,7,"",
				"Minimal strength to blur  (now not used)");
		gd.addNumericField("Blur weak sigma",                        this.sigma, 5,7,"",
				"Blur weak sigma");
		gd.addNumericField("Number of blur passes",                  this.num_blur, 0,3,"",
				"Number of blur passes");
		gd.addNumericField("*** Disparity offset (used 04/07/2023) ***",     this.disparity_corr, 5,7,"",
				"for setInterTasks() - used 04/07/2023.");
		gd.addNumericField("Outlier N-th from extreme",              this.outliers_nth_fromextrem, 0,3,"",
				"0 - use min/max, 1 - use second min/max, ... second from min/max - removes dual-tile clusters max/mins");
		gd.addNumericField("Outliers tolerance absolute",            this.outliers_tolerance_absolute, 5,7,"pix",
				"Absolute difference from extreme neighbor");
		gd.addNumericField("Outliers tolerance realtive",            this.outliers_tolerance_relative, 5,7,"pix/pix",
				"Add to tolerance as a fraction of absolute tile disparity");
		gd.addNumericField("Maximal number of outlier iterations",   this.outliers_max_iter, 0,3,"",
				"Maximal number of iterations for removing outliers");
		gd.addNumericField("Maximal outlier strength 2",             this.outliers_max_strength2, 5,7,"",
				"Maximal outlier strength for second pass filtering");
		gd.addNumericField("Outlier N-th from extreme 2",            this.outliers_nth_fromextrem2, 0,3,"",
				"Second filter: 0 - use min/max, 1 - use second min/max, ... second from min/max - removes dual-tile clusters max/mins");
		gd.addNumericField("Outliers tolerance absolute 2",          this.outliers_tolerance_absolute2, 5,7,"",
				"Absolute difference from extreme neighbor, second filter");
		gd.addNumericField("Outliers tolerance realtive 2",          this.outliers_tolerance_relative2, 5,7,"",
				"Add to tolerance as a fraction of absolute tile disparity, second filter");
		gd.addNumericField("Outliers LMA maximal strength",          this.outliers_lma_max_strength, 5,7,"",
				"Maximal LMA strength for an outlier");
		gd.addNumericField("Outliers non-LMA maximal strength",      this.outliers_max_strength, 5,7,"",
				"Outliers non-LMA (centroid) maximal strength");
		gd.addNumericField("Outliers from LMA max strength",         this.outliers_from_lma_max_strength, 5,7,"",
				"Non-LMA outliers from LMA neighbors maximal strength");
		
		gd.addNumericField("Search for LMA radius",                  this.search_radius, 0,3,"tile",
				"Search farther if no LMA tiles are found around this non-LMA tile");
		gd.addCheckbox ("Remove non-LMA tile if no LMA near",        this.remove_no_lma_neib,
				"Remove non-LMA tile if no LMA one is found within specified radius");
		
		gd.addNumericField("LMA difference positive",                this.diff_from_lma_pos, 5,7,"",
				"Maximal non-LMA tile positive difference from the LMA neighbor");
		gd.addNumericField("LMA difference negative",                this.diff_from_lma_neg, 5,7,"",
				"Maximal non-LMA tile negative difference from the LMA neighbor");
		gd.addNumericField("LMA outliers N-th from extreme",         this.outliers_lma_nth_fromextrem, 0,3,"",
				"0 - use min/max, 1 - use second min/max, ... ");
		gd.addNumericField("Margin from the sensor FOV edge ",       this.filter_margin, 0,3,"pix",
				"Disregard tiles with centers closer to the sensor FoV in pixels");
		
		gd.addMessage  ("Removing weak tiles with insufficient matching neighbors");
		gd.addNumericField("Neighbor disparity tolerance (absolute)",this.weak_tolerance_absolute, 5,7,"pix",
				"Absolute disparity difference to be considered a neighbor");
		gd.addNumericField("Neighbor disparity tolerance (relative)", this.weak_tolerance_relative, 5,7,"",
				"Relative disparity difference to be considered a neighbor");
		gd.addNumericField("Minimal matching neighbors",              this.weak_min_neibs, 0,3,"",
				"Minimal number of weak matching neighbors (at least one is needed if there are strong non-neighbors).");
		gd.addNumericField("Minimal strength to be strong",           this.strong_strength, 5,7,"",
				"Minimal strength to be considered strong(not weak). LMA are also considered strong.");
		gd.addNumericField("Minimal weak strength",                   this.weak_strength, 5,7,"",
				"Weaker are removed unconditionally (not used now).");
		
		gd.addTab("Sky","Featureless sky areas detection and filtering (LWIR only)");
		gd.addMessage  ("Temperature-related filtering (requires photometric calibration)");
		gd.addCheckbox ("Try blue sky calculation/use",              this.sky_extract,
				"Disable if no sky will appear in the image (such as for down-pointed camera).");
		gd.addCheckbox ("Force blue sky recalculation",              this.sky_recalc,
				"Recalculate blue sky even if it already exists.");
		gd.addNumericField("High limit of the scene coldest tile",   this.sky_highest_min, 5,7,"",
				"Sky is normally cold. If the (globally offset) minimal scene absolutre pixel value is above - no sky in the scene (debug images still show it)");
		gd.addNumericField("Cold level fraction",                    this.cold_frac, 5,7,"",
				"All tiles below this percentile are considered \"cold\".");
		gd.addNumericField("Hot level fraction",                     this.hot_frac, 5,7,"",
				"All tiles above this percentile are considered \"hot\".");
		gd.addNumericField("Cold FOM scale",                         this.cold_scale, 5,7,"",
				"Multiply (strength*spread) by this value (reduce), and linearly scale up to 1.0 for hot ones.");
		gd.addMessage  ("Generate seed tiles for the potential sky area");
		gd.addNumericField("Seed FOM level",                         this.sky_seed, 5,7,"",
				"Maximal FOM (strength*spread*temp_scale) level to seed sky area, temp_scale is generated from average pixel value "+
		        "at infinity and 3 above parameters");
		gd.addNumericField("Seed maximal disparity",                 this.lma_seed, 5,7,"pix",
				"Seed tiles should not have LMA-found disparity above this");
		gd.addNumericField("Seed maximal relative temperature",      this.seed_temp, 5,7,"",
				"Seed tiles should be  colder than this point between min and max scene temperature.");
		gd.addNumericField("Shrink sky seed",                        this.sky_shrink, 0,3,"",
				"Shrink sky seed tiles to eliminate too small (false) clusters. One shrinks in hor/vert, 2 - with diagonals, ...");
		gd.addNumericField("Seed rows",                              this.seed_rows, 0,3,"",
				"Sky is above and normally should appear in the top image rows. Applies after shrinking.");
		gd.addMessage  ("Expand seed tiles");
		gd.addNumericField("Maximal strong sky disparity",           this.max_disparity, 5,7,"",
				"Maximal disparity of the sky tiles if they are strong enough.");
		gd.addNumericField("Strength of the high disparity sky",     this.max_disparity_strength, 5,7,"",
				"Required tile strenghth to be considered strong to test against maximal disparity.");
		gd.addNumericField("Expand FOM limit",                       this.sky_lim, 5,7,"",
				"Expand while FOM is below this value (usually to a high-contrast skyline).");
		gd.addNumericField("Sky maximal relative temperature",      this.lim_temp, 5,7,"",
				"Sky tiles should be  colder than this point between min and max scene temperature.");
		gd.addNumericField("Expand extra",                           this.sky_expand_extra, 0,3,"",
				"Additionally expand sky area after reaching threshold in the previous step.");
		gd.addNumericField("Bottleneck width",                       this.sky_bottleneck, 0,3,"",
				"Shrink/reexpand from the seed detected sky to prevent \"leaks\" through narrow gaps in the high-contrast sky limit.");
		gd.addNumericField("Re-expand extra after bottleneck",       this.sky_reexpand_extra, 0,3,"",
				"Additionally expand sky area after bottleneck shrink in excess of re-expanding by the same amount as shrank.");
		gd.addNumericField("Modify strength to be at least this",    this.min_strength, 5,7,"",
				"Input strength has some with zero values resulting in zero FOM. Make them at least this.");
		gd.addNumericField("Lowest sky row",                         this.lowest_sky_row, 0,3,"",
				"Last defense - if the detected sky area reaches near-bottom of the page - it is invalid, remove it (but keep in debug images).");
		gd.addNumericField("Hottest sky tile to override lowest row",this.sky_bottom_override, 5,7,"",
				"If the detected sky is all cold enough, bypass lowest row test, allow to raise camera.");
		gd.addNumericField("Shrink before finding hottest sky",      this.sky_override_shrink, 0,3,"",
				"Shrink detected sky before looking for the hottest sky tile (blurred skyline in wet atmosphere).");
		
		gd.addMessage  ("Detect/process clouds in the sky");
		gd.addCheckbox ("Enable clouds",                             this.clouds_en,
				"Enable clouds in the sky detection / processing.");
		gd.addNumericField("Clouds maximal FOM",                     this.clouds_fom, 5,7,"",
				"Maximal FOM for clouds (must be <=).");
		gd.addNumericField("Clouds maximal spread",                  this.clouds_spread, 5,7,"",
				"Maximal spread for clouds (must be <=).");
		gd.addNumericField("Clouds disparity",                       this.clouds_disparity, 5,7,"pix",
				"Maximal disparity for strong clouds.");
		gd.addNumericField("Clouds weak",                            this.clouds_weak, 5,7,"",
				"Maximal strength for near definite clouds.");
		gd.addNumericField("Clouds sure strength",                   this.clouds_strength, 5,7,"",
				"Minimal strength for far strong clouds (definitely cloud).");
		gd.addNumericField("Non-clouds strength",                    this.clouds_not_strength, 5,7,"",
				"Maximal strength for near maybe clouds (if it has strong cloud neighbor). Stronger near - definitely not clouds.");
		gd.addCheckbox ("Clouds by strong neighbor",                 this.clouds_strong,
				"Treat weak non-cloud as cloud if it has strong (1.5x strength) cloud neighbor.");

		gd.addMessage  ("Detect smooth walls mistaken for sky. Disable for natural environments");
		gd.addCheckbox ("Detect smooth walls",                       this.wall_en,
				"Suspect isolated sky clusters to be smooth walls.");
		gd.addCheckbox ("Walls default",                             this.wall_dflt,
				"If unsure (both or none) - treat as a wall (false - as sky).");
		gd.addNumericField("Wall strength",                          this.wall_str, 5,7,"",
				"Minimal strength to consider cluster to be a wall or sky. Use very small to use any non-NAN disparity.");
		gd.addNumericField("Wall far",                               this.wall_far, 5,7,"pix",
				"Maximal disparity to designate cluster as sky.");
		gd.addNumericField("Wall near",                              this.wall_near, 5,7,"pix",
				"Minimal disparity to designate cluster as a wall.");
		
		gd.addMessage  ("Processing far treeline/mountains that may be confused with clouds. Only behind far objects such as far horizontal surface");
		gd.addCheckbox ("Enable treeline processing",                this.treeline_en,
				"Enable treeline processing (reduces sky/clouds).");
		gd.addCheckbox ("Look not only under, but diagonal too",     this.treeline_wide,
				"Look not only under, but diagonal too.");
		gd.addNumericField("Maximal treeline height",                this.treeline_height, 0,3,"tiles",
				"Maximal height of the treeline. Increase for mountain ridges.");
		gd.addNumericField("Minimal treeline width",                 this.treeline_width, 0,3,"tiles",
				"Minimal horizontal width of the treeline.");
		gd.addCheckbox ("Limit treeline height (not delete it)",     this.treeline_lim_high,
				"Limit too high treeline (false - delete completely).");
		gd.addNumericField("Treeline min strength",                  this.treeline_str, 5,7,"",
				"Treeline min strength.");
		gd.addNumericField("Treeline min disparity",                 this.treeline_far, 5,7,"pix",
				"Treeline min (far) disparity.");
		gd.addNumericField("Treeline max disparity",                 this.treeline_near, 5,7,"pix",
				"Treeline max (near) disparity.");
		gd.addNumericField("Pre-treeline FG min strength",           this.treeline_fg_str, 5,7,"",
				"Pre-treeline FG objects (such as flat ground) minimal strength.");
		gd.addNumericField("Pre-treeline FG min disparity",          this.treeline_fg_far, 5,7,"pix",
				"Pre-treeline FG min (far) disparity.");
		gd.addNumericField("Pre-treeline FG max disparity",          this.treeline_fg_near, 5,7,"pix",
				"Pre-treeline FG max (near) disparity.");
		
		gd.addMessage  ("Detect indoors (and disable sky)");
		gd.addCheckbox ("Enable indoors",                             this.indoors_en,
				"Enable indoors detection and disable sky (if detected).");
		gd.addNumericField("Outdoors objects minimal strength",       this.indoors_str, 5,7,"",
				"Minimal strength of outdoors (far) objects.");
		gd.addNumericField("Outdoors disparity",                      this.indoors_disp, 5,7,"pix",
				"Maximal disparity for strong outdoors objects.");
		gd.addNumericField("Minimal outdoor tiles",                  this.indoors_min_out, 0,3,"tiles",
				"Minimal number of far strong tiles to deny indoor mode.");
		
		gd.addMessage  ("Boost FOM if there are enough LMA-defined good neighbors around (Thin wires over the sky)");
		gd.addNumericField("Minimal disparity",                      this.disp_boost_min, 5,7,"pix",
				"Minimal LMA-defined disparity to boost FOM.");
		gd.addNumericField("Disparity difference",                   this.disp_boost_diff, 5,7,"pix",
				"Maximal disparity difference to neighbor to count.");
		gd.addNumericField("Number of good neighbors",               this.disp_boost_neibs, 0,3,"",
				"Number of neighbors (of 8) to have small disparity difference to boost FOM.");
		gd.addNumericField("Boost amount",                           this.disp_boost_amount, 5,7,"x",
				"Multiply FOM by this value if number of neighbors is exactly minimal. Scale proportional to the total number of neighbors.");
		gd.addNumericField("Scale if strength is combo, not main",   this.scale_combo_strength, 5,7,"x",
				"Reduce strength when it comes from combo, not DSI-MAIN.");
		gd.addCheckbox ("Save debug images",                          this.save_debug_images,
				"Save debug images for Blue Sky generation in model directory.");

		gd.addTab         ("Inter-Match", "Parameters for full-resolution scene matching");
		gd.addMessage  ("Interscene match parameters");
		gd.addNumericField("Image margin",                           this.margin, 0,5,"pix",
				"Do not use tiles if their centers are closer to the virtual image edge");
		gd.addNumericField("Used sensors mask",                      this.sensor_mask_inter, 0,5,"",
				"Bitmask of the sensors to use, -1 - use all");
		gd.addCheckbox ("Preserve partial",                          this.use_partial,
				"Preserve motion vectors for each of the selected sensors, fallse - use only combined (in transform domain and in pixel domain)");
		gd.addCheckbox ("Poly maximums",                             this.run_poly,
				"Use polinomial interpolationn to determin correlation maximums instead of centroids (not yet implemented, ignored)");
		gd.addNumericField("Centroid radius",                        this.centroid_radius, 5,7,"pix",
				"Calculate centroids after multiplication by a half-cosine window. All correlation data farther than this value from the center is ignored");
		gd.addNumericField("Refine centroids",                       this.n_recenter, 0,5,"",
				"Repeat centroids after moving the window center to the new centroid location this many times (0 - calculate once)");
		gd.addMessage  ("Mixing TD and PD accumulation of 2d correlations");
		gd.addNumericField("TD-accumulated weight",                  this.td_weight, 5,7,"",
				"Mix argmax from TD-accumulated correlation.");
		gd.addNumericField("TD-accumulated neighbors weight",       this.td_neib_weight, 5,7,"",
				"Mix argmax from TD-accumulated neighbors (center plus 8-neighbors) correlation.");
		gd.addNumericField("PD-accumulated weight",                  this.pd_weight, 5,7,"",
				"Mix argmax from PD-accumulated correlation.");
		gd.addCheckbox ("TD when no PD only",                        this.td_nopd_only,
				"Use argmax from TD only if PD data is not available for this tile.");
		gd.addCheckbox ("Use neighbors only if no individual",       this.neib_notd_only,
				"Use argmax from averaged neighbors TD only if this tile own maximum is not strong enough.");
		
		gd.addMessage  ("Averaging inter-scene 2D correlations with 8 immediate neighbors");
		gd.addCheckbox ("Use averaging neighbors",              this.use_neibs,
				"Average tile 2D correlation with 8 immediate neighbors if single-tile correlation is not strong enough.");
		// TODO: Consider/Implement pure TD averaging
		gd.addCheckbox ("No-FPN averaging only (all passes)",   this.neibs_nofpn_only,
				"Use averaging with neighbors only if tile offset is large enough not to have FPN interference (for all passes).");
		gd.addCheckbox ("No-FPN averaging only during init",    this.neibs_nofpn_init,
				"Additionally disable FPN-prone tiles during initial scene pose adjustment.");
		gd.addCheckbox ("Use averaging for both PD and TD",     this.redo_both,
				"Recalculate tile average for both TD and PD if at least one of them is not strong enough.");
		gd.addNumericField("Minimal neighbors to average",      this.min_num_neibs, 0,3,"",
				"Minimal number of defined (non-null) neighbors 2D correlations (of 8) to use averaging.");
		gd.addNumericField("Scale PD thershold",                this.scale_neibs_pd, 5,7,"",
				"Use averaging for tiles that are stronger than scaled threshold for a single-tile motion vector for pixel-domain values.");
		gd.addNumericField("Scale TD threshold",                this.scale_neibs_td, 5,7,"",
				"Use averaging for tiles that are stronger than scaled threshold for a single-tile motion vector for transform-domain values.");
		gd.addNumericField("Scale averaged strengths",          this.scale_avg_weight, 5,7,"",
				"Scale averaged motion vector strengths (for each of the TD and PD).");
		
		gd.addMessage  ("Filtering motion vectors");
		gd.addNumericField("Minimal correlation strength (non-sum)",       this.min_str, 5,7,"",
				"Minimal correlation strength for individual correlation and for pixel-domain averaged one. Weeker tiles results are removed.");
		gd.addNumericField("Minimal correlation strength (non-sum) w/FPN", this.min_str_fpn, 5,7,"",
				"Similar to above, but for small offsets where FPN correlation may be present");
		
		gd.addNumericField("Minimal correlation strength (sum only)",      this.min_str_sum, 5,7,"",
				"Minimal correlation strength for transform-domain averaging among sensors. Weeker tiles results are removed.");
		gd.addNumericField("Minimal correlation strength (sum only) w/FPN",this.min_str_sum_fpn, 5,7,"",
				"Similar to above, but for small offsets where FPN correlation may be present");
		
		gd.addNumericField("Minimal correlation neighbors strength",       this.min_str_neib, 5,7,"",
				"Minimal correlation strength for transform-domain averaging among sensors and neighbors. Weeker tiles results are removed.");
		gd.addNumericField("Minimal correlation neighbors strength w/FPN",this.min_str_neib_fpn, 5,7,"",
				"Similar to above, but for small offsets where FPN correlation may be present");
		
		gd.addNumericField("Minimal number of neighbors (of 8)",     this.min_neibs, 0,3,"",
				"Remove motion vectors with less than this number of defined (passing min_str) neighbors.");
		gd.addNumericField("No-neighbors weight (<1.0)",             this.weight_zero_neibs, 5,7,"",
				"Punish for reduced neighbors - weigh for no-neighbors), weight of 8 neighbors = 1.0.");
		gd.addNumericField("Disparity to reduce weight twice from infinity", this.half_disparity, 5,7,"",
				"Weight at this disparity is 0.5, at infinity - 1.0.");
		gd.addNumericField("Difference from neighbors average ",     this.half_avg_diff, 5,7,"",
				"Reduce twice for high difference from neighbors average.");
		
		gd.addMessage  ("Initial search for the inter-scene match");
		
		gd.addCheckbox ("Use combo DSI (if available) for reliable", this.use_combo_relaible,
				"Use interscene DSI if available (instead of the single-scene) for selecting reliable tiles");
		gd.addCheckbox ("Need LMA",                                  this.ref_need_lma,
				"Require LMA result for reliable reference");
		gd.addCheckbox ("Need LMA if combo",                         this.ref_need_lma_combo,
				"Require LMA result for reliable reference");
		
		gd.addNumericField("DSI_MAIN minimal strength",              this.min_ref_str, 5,7,"",
					"Match only tiles where DSI_MAIN is stronger than that (and has LMA).");
		gd.addNumericField("DSI_MAIN minimal fraction",              this.min_ref_frac, 5,7,"",
				"If relative number of the reliable tiles is less than this - use this best fraction.");
		
		gd.addNumericField("Azimuth/tilt step",                      this.pix_step, 0,3,"pix",
				"Search in a spiral starting with no-shift with this step between probes, in approximate pixels");
		gd.addNumericField("Search spiral radius",                   this.search_rad, 0,3,"steps",
				"Maximal search radius in steps");
		gd.addNumericField("\"Maybe\" sum of strengths",             this.maybe_sum, 5,7,"",
				"Minimal acceptable sum of defined tiles strengths (will look for the best among matching)");
		gd.addNumericField("\"Sure\" sum of strengths",              this.sure_sum, 5,7,"",
				"Definitely sufficient sum of defined tiles strengths (will not continue looking for better).");
		gd.addNumericField("\"Maybe\" average of strengths",         this.maybe_avg, 5,7,"",
				"Minimal acceptable average of defined tiles strengths (will look for the best among matching)");
		gd.addNumericField("\"Sure\" average of strengths",          this.sure_avg, 5,7,"",
				"Definitely sufficient average of defined tiles strengths (will not continue looking for better).");
		
		gd.addNumericField("Maximal offset standard deviation",      this.max_search_rms, 5,7,"pix",
				"Standard deviation for X,Y offsets.");
		gd.addNumericField("\"Maybe\" FOM",                          this.maybe_fom, 5,7,"",
				"Minimal acceptable Figure of Merit (semi-total strength divided by standard deviation of offsets), will look for the best among matching.");
		gd.addNumericField("\"Sure\" FOM",                           this.sure_fom, 5,7,"",
				"Definitely sufficient FOM (semi-total strength divided by standard deviation of offsets), will non continue looking for better.");
		gd.addCheckbox ("Treat search all FPN",                      this.treat_serch_fpn,
				"Use FPN (higher) thresholds during search (even if offset is not small).");
		
		
		gd.addMessage  ("Reference scene disparity");
		gd.addCheckbox ("Use combo DSI (if available)",              this.use_combo_dsi,
				"Use interscene DSI if available (instead of the single-scene)");
		gd.addCheckbox ("LMA tiles only",                            this.use_lma_dsi,
				"Use only strong correlation tiles (with LMA available) for interscene correlation (pose matching)");

		gd.addMessage  ("Supress FPN in interscene correlations");
		gd.addCheckbox ("Enable FPN suppression",                    this.fpn_remove,
				"Zero-out integrated inter-scene correlation around zero-shift offset");
		gd.addNumericField("Maximal FPN offset to consider",         this.fpn_max_offset, 6,7,"pix",
				"Maximal offset from the zero shift to consider (normally just 8 pix)");
		gd.addNumericField("FPN suppression radius",                 this.fpn_radius, 6,7,"pix",
				"Blank correlation pixels closer than this distance from the FPN offset");
		gd.addCheckbox ("Ignore maximums \"touching\" FPN",          this.fpn_ignore_border,
				"Discard TD integrated tiles where local maximum is a neighbor (including diagonal) to blanked FPN correlation pixels");

		gd.addMessage  ("Limit series, handle FPN-related problems");
		gd.addNumericField("Minimal inter-scene offset",             this.min_offset, 6,7,"pix",
				"Minimal average pixel offset between images to consider interscene matching");
		gd.addNumericField("Maximal interscene offset fraction of width", this.max_rel_offset, 6,7,"x width",
				"Maximal interscene offset as a fraction of image width to handle low overlap");
		gd.addNumericField("Maximal interscene roll",                this.max_roll_deg, 6,7,"degrees",
				"Maximal interscene roll to consider matching");
		gd.addNumericField("Maximal interscene relative zoom difference", this.max_zoom_diff, 6,7,"",
				"Applicable for the down views from a drone. Saet to 0 to ignore.");
		gd.addCheckbox ("Skip too close to reference scenes",        this.fpn_skip,
				"Skip too close to reference scenes (false - abort, previous behavior)");
		gd.addCheckbox ("Match FPN-failed with other scenes",        this.fpn_rematch,
				"Match fpn-failed scenes to later scenes with larger difference");
		
		gd.addMessage  ("Detect and remove moving objects from pose matching");
		gd.addCheckbox ("Enable movement detection/elimination",     this.mov_en,
				"Detect and mask areas with detected movement to improve pose matching");
		gd.addNumericField("Detection blur sigma",                   this.mov_sigma, 6,7,"pix",
				"Blur squared difference (dx^2+dy^2) befopre thresholding for movement detection");
		gd.addNumericField("Max weighted mismatch std",              this.mov_max_std, 6,7,"pix",
				"Do not try to detect moving objects until approximate pose match is achieved (standard deviation)");
		gd.addNumericField("Relative threshold",                     this.mov_thresh_rel, 6,7,"x",
				"Moving areas over standard deviation. Both relative and absolute should be exceeded.");
		gd.addNumericField("Absolute threshold",                     this.mov_thresh_abs, 6,7,"pix",
				"Moving areas sqrt(dx*dx+dy*dy). Both relative and absolute should be exceeded.");
		gd.addNumericField("Cluster max over threshold",             this.mov_clust_max, 6,7,"",
				"Moving cluster should contain tile with this exceed over thresholds");
		gd.addNumericField("Moving cluster grow",                    this.mov_grow, 0,3,"",
				"Standard grow values - 1 - ortho, 2 - diagonal, 3 - twice orto, 4 - twice diagonal.");
		gd.addNumericField("Maximal movement size",                    this.mov_max_len, 0,3,"tiles",
				"Do not remove moving objects if they do not fit into the square with this side.");
		
		gd.addCheckbox ("Show movement debug images",                this.mov_show,
				"Disabled if 'Debug Level for interscene match' < 1");
		gd.addNumericField("Debug level for movement detection (0/1)", this.mov_debug_level, 0,3,"",
				"Disabled if 'Debug Level for interscene match' < 0");
		
		gd.addMessage  ("LMA parameters");
		gd.addCheckbox ("Azimuth",                                   this.adjust_atr[0],
				"Adjust scene camera azimuth with LMA");
		gd.addCheckbox ("Tilt",                                      this.adjust_atr[1],
				"Adjust scene camera tilt with LMA");
		gd.addCheckbox ("Roll",                                      this.adjust_atr[2],
				"Adjust scene camera roll with LMA");
		gd.addCheckbox ("X",                                         this.adjust_xyz[0],
				"Adjust scene camera X with LMA");
		gd.addCheckbox ("Y",                                         this.adjust_xyz[1],
				"Adjust scene camera Y with LMA");
		gd.addCheckbox ("Z",                                         this.adjust_xyz[2],
				"Adjust scene camera Z with LMA");

		gd.addNumericField("Exit ATR change",                        this.exit_change_atr, 6,7,"pix",
				"L2 of azimuth, tilt, roll change (in pixels at infinity) to exit fitting");
		gd.addNumericField("Exit XYZ change",                        this.exit_change_xyz, 5,7,"m",
				"L2 of linear scene camera position change (in meters) to exit fitting");


		gd.addNumericField("Max full iterations",                    this.max_cycles, 0,3,"",
				"Hard limit on interscene correlations plus LMA cycles");
		gd.addNumericField("Max LMA iterations",                     this.max_LMA, 0,3,"",
				"Hard limit on LMA iterations");
		gd.addNumericField("Maximal RMS to fail",                    this.max_rms, 5,7,"pix",
				"Maximal fitting RMS to consider fitting failure");
		
		
		gd.addMessage  ("Interscene Equalization (Equalization of the interscene correlation confidence to improve camera X,Y,Z matching)");
		gd.addCheckbox ("Enable equalization",                       this.eq_en,
				"Enable boosting of the weak but important interscene correlation tiles strength by equalizing average strength of the \"supertiles\"");
		gd.addMessage  ("Equalization supertiles dimensions");
		gd.addNumericField("Supertile horizontal stride",            this.eq_stride_hor, 0,3,"tiles",
				"Half of a equalization supertile width - supertiles have a 50% overlap in each direction.");
		gd.addNumericField("Supertile vertical stride",              this.eq_stride_hor, 0,3,"tiles",
				"Half of a equalization supertile height - supertiles have a 50% overlap in each direction.");
		gd.addMessage  ("Supertile qualification (not qualified will keep the original strength value)");
		gd.addNumericField("Minimal supertile total weight",         this.eq_min_stile_weight, 5,7,"",
				"Minimal total weight of the tiles in a supertile (lower will not be modified).");
		gd.addNumericField("Minimal number of defined tiles",        this.eq_min_stile_number, 0,3,"tiles",
				"Minimal number of defined tiles in a supertile.");
		gd.addNumericField("Minimal weight fraction of average",     this.eq_min_stile_fraction, 5,7,"",
				"Minimal total supertile strength RELATIVE to the average one.");
		gd.addMessage  ("Individual tiles qualification (applied after calculating the amount of boost).");
		gd.addNumericField("Minimal equalized tile disparity",       this.eq_min_disparity, 5,7,"pix",
				"Allows to disqualify infinity and far tiles (do not contribute to camera X,Y,Z) freom boosting.");
		gd.addNumericField("Maximal equalized tile disparity",       this.eq_max_disparity, 5,7,"pix",
				"Unlikely to have meaning, introduced for symmetry. Can be set to a large number.");
		gd.addMessage  ("Boost amount. Each defined tile will be increased by approximately previous subtracted offset and then scaled");
		gd.addNumericField("Add to strength before scaling",         this.eq_weight_add, 5,7,"",
				"Add to strength, then scale.");
		gd.addNumericField("Maximal boost ratio",                    this.eq_weight_scale, 5,7,"",
				"Strength scale. If the new supertile strength exceed the target value, each tile will be scaled down to match.");
		gd.addNumericField("Equalization level",                     this.eq_level, 5,7,"",
				"Target supertile strength will be set to: 0 - original strength (no modification), 1.0 - average supertile strength.");
		
		gd.addTab("MB","Motion Blur");
		gd.addCheckbox ("Compensate motion blur",                    this.mb_en,
				"Ebable motion blur correction.");
		gd.addNumericField("Sensor time constant",                   this.mb_tau, 5,7,"s",
				"Sensor exponential decay in seconds.");
		gd.addNumericField("Maximal gain",                           this.mb_max_gain, 5,7,"x",
				"Maximal gain for motion blur correction (if needed more for 1 pixel, increase offset). Will be forced fro the last adjustment");
		gd.addNumericField("Maximal gain pose",                      this.mb_max_gain_inter, 5,7,"x",
				"Maximal gain for motion blur correction during interscene correlation. Will be used for all but the last adjustment.");
		
		gd.addTab("Stereo/Video","Parameters for stereo video generation");
		gd.addMessage  ("Stereo");
		if (stereo_views.length > 0) {
			String [] stereo_choices = new String [stereo_views.length + 1];
			stereo_choices[0] = "--none--";
			for (int i = 0; i < stereo_views.length; i++) {
				stereo_choices[i+1] = doublesToString(stereo_views[i],"%.0f")+" mm";	
			}
			gd. addChoice("Remove stereo-view (base, up, back)",      stereo_choices, stereo_choices[0], 
					"Remove selected stereo-view, consisting of streo-base, viewpoint above camera, viewpoint behing camera - all in mm");
		}
		gd.addStringField("Add another stereo view (baseline, above, behind)",                 "", 40,
				"Add another stereo view by providing baseline, above camera, behind camera (mm).");
		gd.addCheckbox ("Stereo merge right, left",                  this.stereo_merge,
				"Combine stereo pair in a single (wide) frame. Unchecked - generate separate videos.");
		gd.addNumericField("Stereo gap",                             this.stereo_gap, 0,3,"pix",
				"Distance (pixels) between right and left sub-images in a stereo frame.");
		gd.addNumericField("VR headset stereo base",                 this.stereo_intereye, 5,7,"mm",
				"Inter-lens distance of the VR phone adapter (Google Cardboard - 63.5)");
		gd.addNumericField("Phone screen width",                     this.stereo_phone_width, 5,7,"mm",
				"Phone screen width in full screen video mode.");
		gd.addNumericField("Scene sequence horizontal extra",        this.extra_hor_tile, 0,3,"tiles",
				"Enlarge reference scene window horizontally in each direction to accommodate other scenes in a sequence");
		gd.addNumericField("Scene sequence vertical extra",          this.extra_vert_tile, 0,3,"tiles",
				"Enlarge reference scene window vertically in each direction to accommodate other scenes in a sequence");
		gd.addCheckbox ("Crop 3D",                                   this.crop_3d,
				"Do not enlarge reference scene windows fro 3D views (FG, BG)");
		
		gd.addNumericField("Sensor mask (bitmask, -1 - all sensors)",this.sensor_mask, 0,3,"",
				"Select which sensors to be included in each scene of the sequence");
		gd.addCheckbox ("Merge all channels in 3D modes",                                   this.merge_all,
				"Ignore sensor mask, use all channels and merge them into one in 3D modes (FG and BG)");
		
		gd.addMessage  ("Generate video for scene sequences");
		gd.addNumericField("Video output frame rate",                this.video_fps, 5,7,"fps",
				"Video frames per second (original is 60fps).");
		gd.addNumericField("Sensor frame rate (60 for Boson)",       this.sensor_fps, 5,7,"fps",
				"Sensor (real) frame rate to be used in realtime videos after slowmo.");
		gd. addChoice("AVI mode",                                     MODES_AVI, MODES_AVI[this.mode_avi], 
				"AVI generation mode");
		gd.addNumericField("AVI JPEG Quality",                       this.avi_JPEG_quality, 0,3,"",
				"Only used in JPEG compression mode");		
		
		gd.addCheckbox ("Convert AVI to WEBM",                       this.run_ffmpeg,
				"Run ffmpeg on AVI video to convert to smaller modern video.");
		gd.addStringField ("Result video extension (e.g. \".webm\")",this.video_ext, 60,
				"Converted video extension, starting with dot.");
		gd.addStringField ("Video encoder",                          this.video_codec, 60,
				"FFMPEG video encoder, such as \"VP8\" or \"VP9\".");
		gd.addNumericField("Video CRF",                              this.video_crf, 0,3,"",
				"Quality - the lower the better. 40 - OK");		

		gd.addCheckbox ("Remove AVI",                                this.remove_avi,
				"Remove large AVI files after (and only) conversion with ffmpeg.");
		
		gd.addStringField ("Video encoder for combining",            this.video_codec_combo, 60,
				"FFMPEG video encoder, such as \"VP8\" or \"VP9\". Applies when merging segments.");
		gd.addNumericField("Video CRF for combining",                this.video_crf_combo, 0,3,"",
				"Quality - the lower the better. 40 - OK.  Applies when merging segments.");		
		
		gd.addCheckbox ("Apply unsharp mask to mono",                this.um_mono,
				"Apply unsharp mask to monochrome image sequences/video.  Applies to TIFF generatiojn too");
		gd.addNumericField("Unsharp mask sigma (radius)",            this.um_sigma, 5,7,"pix",
				"Unsharp mask Gaussian sigma.");
		gd.addNumericField("Unsharp mask weight",                    this.um_weight, 5,7,"",
				"Unsharp mask weightt (multiply blurred version before subtraction from the original).");
		
		gd.addCheckbox ("Fixed monochrome range",                    this.mono_fixed,
				"Normalize monochrome (after UM) to a fixed range when converting to 8 bit RGB.");
		gd.addNumericField("Monochrome full range",                  this.mono_range, 5,7,"",
				"Monochrome full range to convert to 0..255.");
		

		gd.addCheckbox ("Generate anaglyph stereo",                  this.anaglyth_en,
				"Apply unsharp mask to monochrome image sequences/video.  Applies to TIFF generatiojn too");

		{String scolor = String.format("%08x", getLongColor(this.anaglyph_left));
		gd.addStringField ("Anaglyph color left",scolor, 8, "Any invalid hex number sets default red");}
		
		{String scolor = String.format("%08x", getLongColor(this.anaglyph_right));
		gd.addStringField ("Anaglyph color right",scolor, 8, "Any invalid hex number sets default cyan");}
		gd.addCheckbox ("Timestamp color videos",                    this.annotate_color,
				"Annotate pseudo-color video frames with timestamps.");
		gd.addCheckbox ("Timestamp monochrome videos",               this.annotate_mono,
				"Annotate monochrome video frames with timestamps.");
		String scolor = (this.annotate_color_color==null)?"none":String.format("%08x", getLongColor(this.annotate_color_color));
		gd.addStringField ("Timestamp color for pseudocolor frames",scolor, 8, "Any invalid hex number disables annotation");
		scolor = (this.annotate_color_mono==null)?"none":String.format("%08x", getLongColor(this.annotate_color_mono));
		
		gd.addStringField ("Timestamp color for monochrome frames",scolor, 8, "Any invalid hex number disables annotation");
		gd.addCheckbox ("Transparent timestamp background (monochrome)",this.annotate_transparent_mono,
				"Put monochrome timestamp over image (unchecked - use black background). Color - always black.");
		
	}

	public void dialogAnswers(GenericJTabbedDialog gd) {
		this.ims_use =                        gd.getNextBoolean();
		this.ims_rebuild =                    gd.getNextBoolean();
		this.ims_offset =                     gd.getNextNumber();
		this.gmt_plus =                       gd.getNextNumber();
		this.ims_ortho =     IntersceneMatchParameters. StringToDoubles(gd.getNextString(), 4);
		this.ims_mount_atr = IntersceneMatchParameters. StringToDoubles(gd.getNextString(), 3);
		this.ims_mount_xyz = IntersceneMatchParameters. StringToDoubles(gd.getNextString(), 3);
		this.ims_scale_xyz = IntersceneMatchParameters. StringToDoubles(gd.getNextString(), 3);
		this.ims_scale_atr = IntersceneMatchParameters. StringToDoubles(gd.getNextString(), 3);
		this.sfm_use =                        gd.getNextBoolean();
		this.sfm_min_base =                   gd.getNextNumber();
		this.sfm_min_gain =                   gd.getNextNumber();
		this.sfm_min_frac =                   gd.getNextNumber();
        this.sfm_num_pairs =            (int) gd.getNextNumber(); 
        this.sfp_tolerance =                  gd.getNextNumber();  
        this.sfm_readjust =             (int) gd.getNextNumber(); 
        this.sfm_prev_frac =                  gd.getNextNumber();
        this.sfm_same_weight =                gd.getNextNumber();
        this.sfm_shrink =               (int) gd.getNextNumber();
        this.sfm_fade_sigma =                 gd.getNextNumber(); 
        this.sfm_min_str1 =                   gd.getNextNumber(); 
        this.sfm_min_str16 =                  gd.getNextNumber(); 
        this.sfm_use_neibs =                  gd.getNextBoolean(); 
        this.sfm_neib_too_str1 =              gd.getNextNumber(); 
        this.sfm_neib_too_str16 =             gd.getNextNumber(); 
        this.sfm_neib_str1 =	              gd.getNextNumber(); 
        this.sfm_neib_str16 =	              gd.getNextNumber(); 
        this.sfm_extrap_steps =         (int) gd.getNextNumber();
        this.sfm_extrap_radius =        (int) gd.getNextNumber();
		this.sfm_average_neibs =              gd.getNextBoolean();
		this.sfm_fill_weak =                  gd.getNextBoolean();
		this.sfm_extrapolate =                gd.getNextBoolean();
        
		this.center_reference =               gd.getNextBoolean();
		this.reset_photometric =              gd.getNextBoolean();
		this.force_ref_dsi =                  gd.getNextBoolean();
		this.force_orientations =             gd.getNextBoolean();
		this.run_ly =                         gd.getNextBoolean();
		this.min_num_orient =           (int) gd.getNextNumber(); if (min_num_orient < 1) min_num_orient = 1;
		this.min_num_interscene =       (int) gd.getNextNumber(); if (min_num_interscene < 1) min_num_interscene = 1;
		this.calc_quat_corr =                 gd.getNextBoolean();
		this.apply_quat_corr =                gd.getNextBoolean();
		this.generate_egomotion =             gd.getNextBoolean();
		this.generate_mapped =                gd.getNextBoolean();
		this.reuse_video =                    gd.getNextBoolean();
		this.save_mapped_color =              gd.getNextBoolean(); // as tiff
		this.save_mapped_mono =               gd.getNextBoolean(); // as tiff
		this.gen_avi_color =                  gd.getNextBoolean(); // as video
		this.gen_avi_mono =                   gd.getNextBoolean(); // as video
		this.show_mapped_color =              gd.getNextBoolean();
		this.show_mapped_mono =               gd.getNextBoolean();
		this.generate_raw =                   gd.getNextBoolean();
		this.generate_inf =                   gd.getNextBoolean();
		this.generate_fg =                    gd.getNextBoolean();
		this.generate_bg =                    gd.getNextBoolean();
		this.generate_stereo =                gd.getNextBoolean();
		for (int i = 0; i < stereo_views.length; i++) {
			this.generate_stereo_var[i] =     gd.getNextBoolean();
		}
		
		this.export_images =                  gd.getNextBoolean();
		this.show_images =                    gd.getNextBoolean();
		this.show_images_bgfg =               gd.getNextBoolean();
		this.show_images_mono =               gd.getNextBoolean();
		this.export_ranges =                  gd.getNextBoolean();
		this.debug_ranges =                   gd.getNextBoolean();
		this.show_ranges =                    gd.getNextBoolean();
		this.export_ml_files =                gd.getNextBoolean();

		// additional parameters		
		this.show_color_nan =                 gd.getNextBoolean();
		this.show_mono_nan =                  gd.getNextBoolean();
		
		this.scale_extrap_atr =               gd.getNextNumber();
		this.scale_extrap_xyz =               gd.getNextNumber();
		this.avg_len =                  (int) gd.getNextNumber();
		this.min_num_scenes =           (int) gd.getNextNumber();
		this.max_num_scenes =           (int) gd.getNextNumber();
		this.blur_egomotion =                 gd.getNextNumber();
		this.range_disparity_offset =         gd.getNextNumber();
		this.range_min_strength =             gd.getNextNumber();
		this.range_max =                      gd.getNextNumber();
		
		this.export3d =                       gd.getNextBoolean();
		
		this.scene_is_ref_test =        gd.getNextBoolean();
		this.render_ref =               gd.getNextBoolean();
		this.render_scene =             gd.getNextBoolean();
		this.toRGB =                    gd.getNextBoolean();
		this.show_2d_correlations =     gd.getNextBoolean();
		this.show_motion_vectors =      gd.getNextBoolean();
		this.debug_level =        (int) gd.getNextNumber();

		this.test_ers =                 gd.getNextBoolean();
		this.test_ers0 =          (int) gd.getNextNumber();
		this.test_ers1 =          (int) gd.getNextNumber();

		this.num_bottom =               (int) gd.getNextNumber();
		this.num_passes =               (int) gd.getNextNumber();
		this.max_change =                     gd.getNextNumber();
		this.min_disparity =                  gd.getNextNumber();
		this.max_sym_disparity =              gd.getNextNumber();
		this.min_strength_lma =               gd.getNextNumber();
		this.min_strength_replace =           gd.getNextNumber();
		this.min_strength_blur =              gd.getNextNumber();
		this.sigma =                          gd.getNextNumber();
		this.num_blur =                 (int) gd.getNextNumber();
		this.disparity_corr =                 gd.getNextNumber();
		this.outliers_nth_fromextrem =  (int) gd.getNextNumber();
		this.outliers_tolerance_absolute =    gd.getNextNumber();
		this.outliers_tolerance_relative =    gd.getNextNumber();
		this.outliers_max_iter =        (int) gd.getNextNumber();
		this.outliers_max_strength2 =         gd.getNextNumber();
		this.outliers_nth_fromextrem2 = (int) gd.getNextNumber();
		this.outliers_tolerance_absolute2 =   gd.getNextNumber();
		this.outliers_tolerance_relative2 =   gd.getNextNumber();
		this.outliers_lma_max_strength =      gd.getNextNumber();
		this.outliers_max_strength =          gd.getNextNumber();
		this.outliers_from_lma_max_strength = gd.getNextNumber();
		this.search_radius =            (int) gd.getNextNumber();
		this.remove_no_lma_neib =             gd.getNextBoolean();
		this.diff_from_lma_pos =              gd.getNextNumber();
		this.diff_from_lma_neg =              gd.getNextNumber();
		this.outliers_lma_nth_fromextrem=(int)gd.getNextNumber();
		this.filter_margin =            (int) gd.getNextNumber();
		
		this.weak_tolerance_absolute =        gd.getNextNumber();
		this.weak_tolerance_relative =        gd.getNextNumber();
		this.weak_min_neibs =           (int) gd.getNextNumber();
		this.strong_strength =                gd.getNextNumber();
		this.weak_strength =                  gd.getNextNumber();
		
		this.sky_extract =              gd.getNextBoolean();
		this.sky_recalc =               gd.getNextBoolean();
		this.sky_highest_min =          gd.getNextNumber(); 	
		this.cold_frac =                gd.getNextNumber();       	
		this.hot_frac =                 gd.getNextNumber();        	
		this.cold_scale =               gd.getNextNumber();      	
		this.sky_seed =                 gd.getNextNumber();        	
		this.lma_seed =                 gd.getNextNumber();
		this.seed_temp =                gd.getNextNumber();
		this.sky_shrink =         (int) gd.getNextNumber();      	
		this.seed_rows =          (int) gd.getNextNumber();
		this.max_disparity =            gd.getNextNumber();         	
		this.max_disparity_strength =   gd.getNextNumber();         	
		this.sky_lim =                  gd.getNextNumber(); 
		this.lim_temp =                 gd.getNextNumber(); 
		this.sky_expand_extra =   (int) gd.getNextNumber();	
		this.sky_bottleneck =     (int) gd.getNextNumber();	
		this.sky_reexpand_extra = (int) gd.getNextNumber();	
		this.min_strength =             gd.getNextNumber();    
		this.lowest_sky_row =     (int) gd.getNextNumber();
		this.sky_bottom_override =      gd.getNextNumber();    
		this.sky_override_shrink =(int) gd.getNextNumber();
		
		this.clouds_en =                gd.getNextBoolean();
		this.clouds_fom =               gd.getNextNumber();    
		this.clouds_spread =            gd.getNextNumber();    
		this.clouds_disparity =         gd.getNextNumber();    
		this.clouds_weak =              gd.getNextNumber();    
		this.clouds_strength =          gd.getNextNumber();    
		this.clouds_not_strength =      gd.getNextNumber();    
		this.clouds_strong =            gd.getNextBoolean();

		this.wall_en =                  gd.getNextBoolean();
		this.wall_dflt =                gd.getNextBoolean();
		this.wall_str =                 gd.getNextNumber();    
		this.wall_far =                 gd.getNextNumber();    
		this.wall_near =                gd.getNextNumber();    
		
		this.treeline_en =              gd.getNextBoolean();
		this.treeline_wide =            gd.getNextBoolean();
		this.treeline_height =    (int) gd.getNextNumber();
		this.treeline_width =     (int) gd.getNextNumber();
		this.treeline_lim_high =        gd.getNextBoolean();
		this.treeline_str =             gd.getNextNumber();
		this.treeline_far =             gd.getNextNumber();
		this.treeline_near =            gd.getNextNumber();
		this.treeline_fg_str =          gd.getNextNumber();
		this.treeline_fg_far =          gd.getNextNumber();
		this.treeline_fg_near =         gd.getNextNumber();
		
		this.indoors_en =               gd.getNextBoolean();
		this.indoors_str =              gd.getNextNumber();    
		this.indoors_disp =             gd.getNextNumber();    
		this.indoors_min_out =    (int) gd.getNextNumber();

		this.disp_boost_min =           gd.getNextNumber();    
		this.disp_boost_diff =          gd.getNextNumber();    
		this.disp_boost_neibs =   (int) gd.getNextNumber();
		this.disp_boost_amount =        gd.getNextNumber();    
		this.scale_combo_strength =     gd.getNextNumber();
		this.save_debug_images =        gd.getNextBoolean();
		
		this.margin =             (int) gd.getNextNumber();
		this.sensor_mask_inter=   (int) gd.getNextNumber();
		this.use_partial =              gd.getNextBoolean();
		this.run_poly =                 gd.getNextBoolean();
		this.centroid_radius =          gd.getNextNumber();
		this.n_recenter =         (int) gd.getNextNumber();
		this.td_weight =                gd.getNextNumber();
		this.td_neib_weight =           gd.getNextNumber();
		this.pd_weight =                gd.getNextNumber();
		this.td_nopd_only =             gd.getNextBoolean();
		this.neib_notd_only =           gd.getNextBoolean();
		
		this.use_neibs =                gd.getNextBoolean();
		this.neibs_nofpn_only =         gd.getNextBoolean();
		this.neibs_nofpn_init =         gd.getNextBoolean();
		this.redo_both =                gd.getNextBoolean();
		this.min_num_neibs =      (int) gd.getNextNumber();
		this.scale_neibs_pd =           gd.getNextNumber();
		this.scale_neibs_td =           gd.getNextNumber();
		this.scale_avg_weight =         gd.getNextNumber();
		
		this.min_str =                  gd.getNextNumber();
		this.min_str_fpn =              gd.getNextNumber();
		this.min_str_sum =              gd.getNextNumber();
		this.min_str_sum_fpn =          gd.getNextNumber();
		this.min_str_neib =             gd.getNextNumber();
		this.min_str_neib_fpn =         gd.getNextNumber();
		this.min_neibs =          (int) gd.getNextNumber();
		this.weight_zero_neibs =        gd.getNextNumber();
		this.half_disparity =           gd.getNextNumber();
		this.half_avg_diff =            gd.getNextNumber();
		this.use_combo_relaible =       gd.getNextBoolean();
		this.ref_need_lma =             gd.getNextBoolean();
		this.ref_need_lma_combo =       gd.getNextBoolean();
		this.min_ref_str =              gd.getNextNumber();
		this.min_ref_frac =             gd.getNextNumber();
		this.pix_step =           (int) gd.getNextNumber();
		this.search_rad =         (int) gd.getNextNumber();
		this.maybe_sum =                gd.getNextNumber();
		this.sure_sum =                 gd.getNextNumber();
		this.maybe_avg =                gd.getNextNumber();
		this.sure_avg =                 gd.getNextNumber();
		this.max_search_rms =           gd.getNextNumber();
		this.maybe_fom =                gd.getNextNumber();
		this.sure_fom =                 gd.getNextNumber();
		this.treat_serch_fpn =          gd.getNextBoolean();
		this.use_combo_dsi =            gd.getNextBoolean();
		this.use_lma_dsi =              gd.getNextBoolean();
		this.fpn_remove =               gd.getNextBoolean();
		this.fpn_max_offset =           gd.getNextNumber();
		this.fpn_radius =               gd.getNextNumber();
		this.fpn_ignore_border =        gd.getNextBoolean();

		this.min_offset =               gd.getNextNumber();
		this.max_rel_offset =           gd.getNextNumber();
		this.max_roll_deg =             gd.getNextNumber();
		this.max_zoom_diff =            gd.getNextNumber();
		
		this.fpn_skip =                 gd.getNextBoolean();
		this.fpn_rematch =              gd.getNextBoolean();
		
		this.mov_en =                   gd.getNextBoolean();
		this.mov_sigma =                gd.getNextNumber();
		this.mov_max_std =              gd.getNextNumber();
		this.mov_thresh_rel =           gd.getNextNumber();
		this.mov_thresh_abs =           gd.getNextNumber();
		this.mov_clust_max =            gd.getNextNumber();
		this.mov_grow =           (int) gd.getNextNumber();
		this.mov_max_len =        (int) gd.getNextNumber();
		
		this.mov_show =                 gd.getNextBoolean();
		this.mov_debug_level =    (int) gd.getNextNumber();
		this.adjust_atr[0] =            gd.getNextBoolean();
		this.adjust_atr[1] =            gd.getNextBoolean();
		this.adjust_atr[2] =            gd.getNextBoolean();
		this.adjust_xyz[0] =            gd.getNextBoolean();
		this.adjust_xyz[1] =            gd.getNextBoolean();
		this.adjust_xyz[2] =            gd.getNextBoolean();
		this.exit_change_atr =          gd.getNextNumber();
		this.exit_change_xyz =          gd.getNextNumber();
		this.max_cycles =         (int) gd.getNextNumber();
		this.max_LMA =            (int) gd.getNextNumber();
		this.max_rms =                  gd.getNextNumber();
		
		
		// equalization of the inter-scene correlations	
		this.eq_en =                    gd.getNextBoolean();
		this.eq_stride_hor =      (int) gd.getNextNumber();
		this.eq_stride_vert =     (int) gd.getNextNumber();
		this.eq_min_stile_weight =      gd.getNextNumber();    
		this.eq_min_stile_number =(int) gd.getNextNumber();
		this.eq_min_stile_fraction =    gd.getNextNumber();    
		this.eq_min_disparity =         gd.getNextNumber();    
		this.eq_max_disparity =         gd.getNextNumber();    
		this.eq_weight_add =            gd.getNextNumber();    
		this.eq_weight_scale =          gd.getNextNumber();    
		this.eq_level =                 gd.getNextNumber();
		
		this.mb_en =                    gd.getNextBoolean();
		this.mb_tau =                   gd.getNextNumber();    
		this.mb_max_gain =              gd.getNextNumber();    
		this.mb_max_gain_inter =        gd.getNextNumber();    
		if (stereo_views.length > 0) {
			int i =                           gd.getNextChoiceIndex();
			if (i > 0) {
				removeStereoView(i-1);
			}
		}
		String s =                            gd.getNextString();
		addStereoView(s, true);
		this.stereo_merge =             gd.getNextBoolean();
		this.stereo_gap =         (int) gd.getNextNumber();
		this.stereo_intereye =          gd.getNextNumber();
		this.stereo_phone_width =       gd.getNextNumber();
		this.extra_hor_tile =     (int) gd.getNextNumber();
		this.extra_vert_tile =    (int) gd.getNextNumber();
		this.crop_3d =                  gd.getNextBoolean();
		this.sensor_mask =        (int) gd.getNextNumber();
		this.merge_all =                gd.getNextBoolean();
		this.video_fps =                gd.getNextNumber();
		this.sensor_fps =               gd.getNextNumber();
		this.mode_avi =                 gd.getNextChoiceIndex();
		this.avi_JPEG_quality =   (int) gd.getNextNumber();
		this.run_ffmpeg =               gd.getNextBoolean();
		this.video_ext=                 gd.getNextString();	
		this.video_codec=               gd.getNextString();	
		this.video_crf =          (int) gd.getNextNumber();
		this.remove_avi =               gd.getNextBoolean();
		this.video_codec_combo=         gd.getNextString();	
		this.video_crf_combo =    (int) gd.getNextNumber();
		this.um_mono =                  gd.getNextBoolean();
		this.um_sigma =                 gd.getNextNumber();
		this.um_weight =                gd.getNextNumber();
		this.mono_fixed =               gd.getNextBoolean();
		this.mono_range =               gd.getNextNumber();
		
		this.anaglyth_en =              gd.getNextBoolean();
		{
			String scolor =             gd.getNextString();
			long lcolor = -1;
			try {
				lcolor = Long.parseLong(scolor,16);
				this.anaglyph_left = setLongColor(lcolor);
			} catch(NumberFormatException e){
				this.anaglyph_left = anaglyph_left_default;
			}
		}

		{
			String scolor =             gd.getNextString();
			long lcolor = -1;
			try {
				lcolor = Long.parseLong(scolor,16);
				this.anaglyph_right = setLongColor(lcolor);
			} catch(NumberFormatException e){
				this.anaglyph_right = anaglyph_right_default;
			}
		}
		
		this.annotate_color =           gd.getNextBoolean();
		this.annotate_mono =            gd.getNextBoolean();
		{
			String scolor =             gd.getNextString();
			long lcolor = -1;
			try {
				lcolor = Long.parseLong(scolor,16);
				this.annotate_color_color = setLongColor(lcolor);
			} catch(NumberFormatException e){
				this.annotate_color_color = null;
			}
		}
		{
			String scolor =             gd.getNextString();
			long lcolor = -1;
			try {
				lcolor = Long.parseLong(scolor,16);
				this.annotate_color_mono = setLongColor(lcolor);
			} catch(NumberFormatException e){
				this.annotate_color_mono = null;
			}
		}
		this.annotate_transparent_mono= gd.getNextBoolean();
		
			
		
		
		if (this.weight_zero_neibs > 1.0) this.weight_zero_neibs = 1.0;
	}
	
	public void setProperties(String prefix,Properties properties){
		properties.setProperty(prefix+"ims_use",                       this.ims_use + "");                     // boolean
		properties.setProperty(prefix+"ims_rebuild",                   this.ims_rebuild + "");                 // boolean
		properties.setProperty(prefix+"ims_offset",                    this.ims_offset+"");                    // double
		properties.setProperty(prefix+"gmt_plus",                      this.gmt_plus+"");                      // double
		properties.setProperty(prefix+"ims_ortho",                     IntersceneMatchParameters.doublesToString(this.ims_ortho));		
		properties.setProperty(prefix+"ims_mount_atr",                 IntersceneMatchParameters.doublesToString(this.ims_mount_atr));		
		properties.setProperty(prefix+"ims_mount_xyz",                 IntersceneMatchParameters.doublesToString(this.ims_mount_xyz));
		properties.setProperty(prefix+"ims_scale_xyz",                 IntersceneMatchParameters.doublesToString(this.ims_scale_xyz));
		properties.setProperty(prefix+"ims_scale_atr",                 IntersceneMatchParameters.doublesToString(this.ims_scale_atr));		
		properties.setProperty(prefix+"sfm_use",                       this.sfm_use + "");                     // boolean
		properties.setProperty(prefix+"sfm_min_base",                  this.sfm_min_base+"");                  // double
		properties.setProperty(prefix+"sfm_min_gain",                  this.sfm_min_gain+"");                  // double
		properties.setProperty(prefix+"sfm_min_frac",                  this.sfm_min_frac+"");                  // double
		properties.setProperty(prefix+"sfm_num_pairs",                 this.sfm_num_pairs+"");                 // int
		properties.setProperty(prefix+"sfp_tolerance",                 this.sfp_tolerance+"");                 // double
		properties.setProperty(prefix+"sfm_readjust",                  this.sfm_readjust+"");                  // int
		properties.setProperty(prefix+"sfm_prev_frac",                 this.sfm_prev_frac+"");                 // double
		properties.setProperty(prefix+"sfm_same_weight",               this.sfm_same_weight+"");               // double
		properties.setProperty(prefix+"sfm_shrink",                    this.sfm_shrink+"");                    // int
		properties.setProperty(prefix+"sfm_fade_sigma",                this.sfm_fade_sigma+"");                // double
		properties.setProperty(prefix+"sfm_min_str1",                  this.sfm_min_str1+"");                  // double
		properties.setProperty(prefix+"sfm_min_str16",                 this.sfm_min_str16+"");                 // double
		properties.setProperty(prefix+"sfm_use_neibs",                 this.sfm_use_neibs+"");                 // boolean
		properties.setProperty(prefix+"sfm_neib_too_str1",             this.sfm_neib_too_str1+"");             // double
		properties.setProperty(prefix+"sfm_neib_too_str16",            this.sfm_neib_too_str16+"");            // double
		properties.setProperty(prefix+"sfm_neib_str1",                 this.sfm_neib_str1 +"");                // double
		properties.setProperty(prefix+"sfm_neib_str16",                this.sfm_neib_str16 +"");               // double
		properties.setProperty(prefix+"sfm_extrap_steps",              this.sfm_extrap_steps+"");              // int
		properties.setProperty(prefix+"sfm_extrap_radius",             this.sfm_extrap_radius+"");             // int
		properties.setProperty(prefix+"sfm_average_neibs",             this.sfm_average_neibs + "");           // boolean
		properties.setProperty(prefix+"sfm_fill_weak",                 this.sfm_fill_weak + "");               // boolean
		properties.setProperty(prefix+"sfm_extrapolate",               this.sfm_extrapolate + "");             // boolean

		properties.setProperty(prefix+"center_reference",              this.center_reference + "");            // boolean
		properties.setProperty(prefix+"reset_photometric",             this.reset_photometric + "");           // boolean
		properties.setProperty(prefix+"force_ref_dsi",                 this.force_ref_dsi + "");               // boolean
		properties.setProperty(prefix+"force_orientations",            this.force_orientations + "");          // boolean
		properties.setProperty(prefix+"run_ly",                        this.run_ly + "");                      // boolean
		properties.setProperty(prefix+"min_num_orient",                this.min_num_orient+"");                // int
		properties.setProperty(prefix+"min_num_interscene",            this.min_num_interscene+"");            // int
		properties.setProperty(prefix+"calc_quat_corr",                this.calc_quat_corr+"");                // boolean
		properties.setProperty(prefix+"apply_quat_corr",               this.apply_quat_corr+"");               // boolean
		properties.setProperty(prefix+"generate_egomotion",            this.generate_egomotion+"");            // boolean
		properties.setProperty(prefix+"generate_mapped",               this.generate_mapped+"");               // boolean
		properties.setProperty(prefix+"reuse_video",                   this.reuse_video+"");                   // boolean
		properties.setProperty(prefix+"save_mapped_color",             this.save_mapped_color+"");             // boolean
		properties.setProperty(prefix+"save_mapped_mono",              this.save_mapped_mono+"");              // boolean
		properties.setProperty(prefix+"gen_avi_color",                 this.gen_avi_color+"");       // boolean
		properties.setProperty(prefix+"gen_avi_mono",                  this.gen_avi_mono+"");        // boolean
		properties.setProperty(prefix+"show_mapped_color",    this.show_mapped_color+"");   // boolean
		properties.setProperty(prefix+"show_mapped_mono",     this.show_mapped_mono+"");    // boolean
		
		properties.setProperty(prefix+"generate_raw",                  this.generate_raw+"");                  // boolean
		properties.setProperty(prefix+"generate_inf",                  this.generate_inf+"");                  // boolean
		properties.setProperty(prefix+"generate_fg",                   this.generate_fg+"");                   // boolean
		properties.setProperty(prefix+"generate_bg",                   this.generate_bg+"");                   // boolean
		properties.setProperty(prefix+"generate_stereo",               this.generate_stereo+"");               // boolean
		/*
		properties.setProperty(prefix+"stereo_bases_num",              this.stereo_bases.length+"");           // int
		for (int i = 0; i < this.stereo_bases.length; i++) {
			properties.setProperty(prefix+"stereo_bases_"+i,           this.stereo_bases[i]+"");               // double
			properties.setProperty(prefix+"generate_stereo_var_"+i,    this.generate_stereo_var[i]+"");        // boolean
		}
	   */
		properties.setProperty(prefix+"stereo_views_num",                  this.stereo_views.length+"");           // int
		for (int i = 0; i < this.stereo_views.length; i++) {
			properties.setProperty(prefix+"stereo_views_"+i,           doublesToString(this.stereo_views[i],"%.0f")); // String
			properties.setProperty(prefix+"generate_stereo_var_"+i,    this.generate_stereo_var[i]+"");        // boolean
		}
		
		
		
		properties.setProperty(prefix+"export_images",                 this.export_images + "");               // boolean
		properties.setProperty(prefix+"show_images",                   this.show_images + "");                 // boolean
		properties.setProperty(prefix+"show_images_bgfg",              this.show_images_bgfg + "");            // boolean
		properties.setProperty(prefix+"show_images_mono",              this.show_images_mono + "");            // boolean
		properties.setProperty(prefix+"export_ranges",                 this.export_ranges + "");               // boolean
		properties.setProperty(prefix+"debug_ranges",                  this.debug_ranges + "");                // boolean
		properties.setProperty(prefix+"show_ranges",                   this.show_ranges + "");                 // boolean
		properties.setProperty(prefix+"export_ml_files",               this.export_ml_files + "");             // boolean
		
		properties.setProperty(prefix+"show_color_nan",                this.show_color_nan + "");              // boolean
		properties.setProperty(prefix+"show_mono_nan",                 this.show_mono_nan + "");               // boolean
		properties.setProperty(prefix+"scale_extrap_atr",              this.scale_extrap_atr+"");              // double
		properties.setProperty(prefix+"scale_extrap_xyz",              this.scale_extrap_xyz+"");              // double
		properties.setProperty(prefix+"avg_len",                       this.avg_len+"");                       // int
		properties.setProperty(prefix+"min_num_scenes",                this.min_num_scenes+"");                // int
		properties.setProperty(prefix+"max_num_scenes",                this.max_num_scenes+"");                // int
		properties.setProperty(prefix+"blur_egomotion",                this.blur_egomotion+"");                // double
		properties.setProperty(prefix+"range_disparity_offset",        this.range_disparity_offset+"");        // double
		properties.setProperty(prefix+"range_min_strength",            this.range_min_strength+"");            // double
		properties.setProperty(prefix+"range_max",                     this.range_max+"");                     // double
		properties.setProperty(prefix+"export3d",                      this.export3d+"");                      // boolean
		
		properties.setProperty(prefix+"scene_is_ref_test",    this.scene_is_ref_test+"");   // boolean
		properties.setProperty(prefix+"render_ref",           this.render_ref+"");          // boolean
		properties.setProperty(prefix+"render_scene",         this.render_scene+"");        // boolean
		properties.setProperty(prefix+"toRGB",                this.toRGB+"");               // boolean
		properties.setProperty(prefix+"show_2d_correlations", this.show_2d_correlations+"");// boolean
		properties.setProperty(prefix+"show_motion_vectors",  this.show_motion_vectors+""); // boolean
		properties.setProperty(prefix+"debug_level",          this.debug_level+"");         // int
		properties.setProperty(prefix+"test_ers",             this.test_ers+"");            // boolean
		properties.setProperty(prefix+"test_ers0",            this.test_ers0+"");           // int
		properties.setProperty(prefix+"test_ers1",            this.test_ers1+"");           // int
	
		properties.setProperty(prefix+"num_bottom",                    this.num_bottom+"");                    // int
		properties.setProperty(prefix+"num_passes",                    this.num_passes+"");                    // int
		properties.setProperty(prefix+"max_change",                    this.max_change+"");                    // double
		properties.setProperty(prefix+"min_disparity",                 this.min_disparity+"");                 // double
		properties.setProperty(prefix+"max_sym_disparity",             this.max_sym_disparity+"");             // double
		properties.setProperty(prefix+"min_strength_lma",              this.min_strength_lma+"");          // double
		properties.setProperty(prefix+"min_strength_replace",          this.min_strength_replace+"");          // double
		properties.setProperty(prefix+"min_strength_blur",             this.min_strength_blur+"");             // double
		properties.setProperty(prefix+"sigma",                         this.sigma+"");                         // double
		properties.setProperty(prefix+"num_blur",                      this.num_blur+"");                      // int
		properties.setProperty(prefix+"disparity_corr",                this.disparity_corr+"");                // double
		properties.setProperty(prefix+"outliers_nth_fromextrem",       this.outliers_nth_fromextrem+"");       // int
		properties.setProperty(prefix+"outliers_tolerance_absolute",   this.outliers_tolerance_absolute+"");   // double
		properties.setProperty(prefix+"outliers_tolerance_relative",   this.outliers_tolerance_relative+"");   // double
		properties.setProperty(prefix+"outliers_max_iter",             this.outliers_max_iter+"");             // int
		properties.setProperty(prefix+"outliers_max_strength2",        this.outliers_max_strength2+"");        // double
		properties.setProperty(prefix+"outliers_nth_fromextrem2",      this.outliers_nth_fromextrem2+"");      // int
		properties.setProperty(prefix+"outliers_tolerance_absolute2",  this.outliers_tolerance_absolute2+"");  // double
		properties.setProperty(prefix+"outliers_tolerance_relative2",  this.outliers_tolerance_relative2+"");  // double
		properties.setProperty(prefix+"outliers_lma_max_strength",     this.outliers_lma_max_strength+"");     // double
		properties.setProperty(prefix+"outliers_max_strength",         this.outliers_max_strength+"");         // double
		properties.setProperty(prefix+"outliers_from_lma_max_strength",this.outliers_from_lma_max_strength+"");// double
		properties.setProperty(prefix+"search_radius",                 this.search_radius+"");                 // int
		properties.setProperty(prefix+"remove_no_lma_neib",            this.remove_no_lma_neib+"");            // boolean
		properties.setProperty(prefix+"diff_from_lma_pos",             this.diff_from_lma_pos+"");             // double
		properties.setProperty(prefix+"diff_from_lma_neg",             this.diff_from_lma_neg+"");             // double
		properties.setProperty(prefix+"outliers_lma_nth_fromextrem",   this.outliers_lma_nth_fromextrem+"");   // int
		properties.setProperty(prefix+"filter_margin",                 this.filter_margin+"");                 // int
		properties.setProperty(prefix+"weak_tolerance_absolute",       this.weak_tolerance_absolute+"");       // double
		properties.setProperty(prefix+"weak_tolerance_relative",       this.weak_tolerance_relative+"");       // double
		properties.setProperty(prefix+"weak_min_neibs",                this.weak_min_neibs+"");                // int
		properties.setProperty(prefix+"strong_strength",               this.strong_strength+"");               // double
		properties.setProperty(prefix+"weak_strength",                 this.weak_strength+"");                 // double

		properties.setProperty(prefix+"sky_extract",          this.sky_extract+"");         // boolean
		properties.setProperty(prefix+"sky_recalc",           this.sky_recalc+"");          // boolean
		properties.setProperty(prefix+"sky_highest_min",      this.sky_highest_min+"");     // double
		properties.setProperty(prefix+"cold_frac",            this.cold_frac+"");           // double
		properties.setProperty(prefix+"hot_frac",             this.hot_frac+"");            // double
		properties.setProperty(prefix+"cold_scale",           this.cold_scale+"");          // double
		properties.setProperty(prefix+"sky_seed",             this.sky_seed+"");            // double
		properties.setProperty(prefix+"lma_seed",             this.lma_seed+"");            // double
		properties.setProperty(prefix+"seed_temp",            this.seed_temp+"");           // double
		properties.setProperty(prefix+"sky_shrink",           this.sky_shrink+"");          // int
		properties.setProperty(prefix+"seed_rows",            this.seed_rows+"");           // int
		properties.setProperty(prefix+"max_disparity",        this.max_disparity+"");       // double
		properties.setProperty(prefix+"max_disparity_strength",this.max_disparity_strength+"");// double
		properties.setProperty(prefix+"sky_lim",              this.sky_lim+"");             // double
		properties.setProperty(prefix+"lim_temp",             this.lim_temp+"");            // double
		properties.setProperty(prefix+"sky_expand_extra",     this.sky_expand_extra+"");    // int
		properties.setProperty(prefix+"sky_bottleneck",       this.sky_bottleneck+"");      // int
		properties.setProperty(prefix+"sky_reexpand_extra",   this.sky_reexpand_extra+"");  // int
		properties.setProperty(prefix+"min_strength",         this.min_strength+"");        // double
		properties.setProperty(prefix+"lowest_sky_row",       this.lowest_sky_row+"");      // int
		properties.setProperty(prefix+"sky_bottom_override",  this.sky_bottom_override+""); // double
		properties.setProperty(prefix+"sky_override_shrink",  this.sky_override_shrink+""); // int
		
		properties.setProperty(prefix+"clouds_en",            this.clouds_en+"");           // boolean
		properties.setProperty(prefix+"clouds_fom",           this.clouds_fom+"");          // double
		properties.setProperty(prefix+"clouds_spread",        this.clouds_spread+"");       // double
		properties.setProperty(prefix+"clouds_disparity",     this.clouds_disparity+"");    // double
		properties.setProperty(prefix+"clouds_weak",          this.clouds_weak+"");         // double
		properties.setProperty(prefix+"clouds_strength",      this.clouds_strength+"");     // double
		properties.setProperty(prefix+"clouds_not_strength",  this.clouds_not_strength+""); // double
		properties.setProperty(prefix+"clouds_strong",        this.clouds_strong+"");       // boolean
		
		properties.setProperty(prefix+"wall_en",              this.wall_en+"");             // boolean
		properties.setProperty(prefix+"wall_dflt",            this.wall_dflt+"");           // boolean
		properties.setProperty(prefix+"wall_str",             this.wall_str+"");            // double
		properties.setProperty(prefix+"wall_far",             this.wall_far+"");            // double
		properties.setProperty(prefix+"wall_near",            this.wall_near+"");           // double

		properties.setProperty(prefix+"treeline_en",          this.treeline_en+"");         // boolean
		properties.setProperty(prefix+"treeline_wide",        this.treeline_wide+"");       // boolean
		properties.setProperty(prefix+"treeline_height",      this.treeline_height+"");     // int
		properties.setProperty(prefix+"treeline_width",       this.treeline_width+"");      // int
		properties.setProperty(prefix+"treeline_lim_high",    this.treeline_lim_high+"");   // boolean
		properties.setProperty(prefix+"treeline_str",         this.treeline_str+"");        // double
		properties.setProperty(prefix+"treeline_far",         this.treeline_far+"");        // double
		properties.setProperty(prefix+"treeline_near",        this.treeline_near+"");       // double
		properties.setProperty(prefix+"treeline_fg_str",      this.treeline_fg_str+"");     // double
		properties.setProperty(prefix+"treeline_fg_far",      this.treeline_fg_far+"");     // double
		properties.setProperty(prefix+"treeline_fg_near",     this.treeline_fg_near+"");    // double
		
		properties.setProperty(prefix+"indoors_en",           this.indoors_en+"");          // boolean
		properties.setProperty(prefix+"indoors_str",          this.indoors_str+"");         // double
		properties.setProperty(prefix+"indoors_disp",         this.indoors_disp+"");        // double
		properties.setProperty(prefix+"indoors_min_out",      this.indoors_min_out+"");     // int
		
		properties.setProperty(prefix+"disp_boost_min",       this.disp_boost_min+"");      // double
		properties.setProperty(prefix+"disp_boost_diff",      this.disp_boost_diff+"");     // double
		properties.setProperty(prefix+"disp_boost_neibs",     this.disp_boost_neibs+"");    // int
		properties.setProperty(prefix+"disp_boost_amount",    this.disp_boost_amount+"");   // double
		properties.setProperty(prefix+"scale_combo_strength", this.scale_combo_strength+"");// double
		properties.setProperty(prefix+"save_debug_images",    this.save_debug_images+"");   // boolean
		
		properties.setProperty(prefix+"margin",               this.margin+"");              // int
		properties.setProperty(prefix+"sensor_mask_inter",    this.sensor_mask_inter+"");   // int
		properties.setProperty(prefix+"use_partial",          this.use_partial+"");         // boolean
		properties.setProperty(prefix+"run_poly",             this.run_poly+"");            // boolean
		properties.setProperty(prefix+"centroid_radius",      this.centroid_radius+"");     // double
		properties.setProperty(prefix+"n_recenter",           this.n_recenter+"");          // int
		properties.setProperty(prefix+"min_ref_str",          this.min_ref_str+"");         // double
		properties.setProperty(prefix+"min_ref_frac",         this.min_ref_frac+"");        // double
		properties.setProperty(prefix+"td_weight",            this.td_weight+"");           // double
		properties.setProperty(prefix+"td_neib_weight",       this.td_neib_weight+"");      // double
		properties.setProperty(prefix+"pd_weight",            this.pd_weight+"");           // double
		properties.setProperty(prefix+"td_nopd_only",         this.td_nopd_only+"");        // boolean
		properties.setProperty(prefix+"neib_notd_only",       this.neib_notd_only+"");      // boolean
		
		properties.setProperty(prefix+"use_neibs",            this.use_neibs+"");           // boolean
		properties.setProperty(prefix+"neibs_nofpn_only",     this.neibs_nofpn_only+"");    // boolean
		properties.setProperty(prefix+"neibs_nofpn_init",     this.neibs_nofpn_init+"");    // boolean
		properties.setProperty(prefix+"redo_both",            this.redo_both+"");           // boolean
		properties.setProperty(prefix+"min_num_neibs",        this.min_num_neibs+"");       // int
        properties.setProperty(prefix+"scale_neibs_pd",       this.scale_neibs_pd+"");      // double
		properties.setProperty(prefix+"scale_neibs_td",       this.scale_neibs_td+"");      // double
		properties.setProperty(prefix+"scale_avg_weight",     this.scale_avg_weight+"");    // double
		
		properties.setProperty(prefix+"min_str",              this.min_str+"");             // double
		properties.setProperty(prefix+"min_str_fpn",          this.min_str_fpn+"");         // double
		properties.setProperty(prefix+"min_str_sum",          this.min_str_sum+"");         // double
		properties.setProperty(prefix+"min_str_sum_fpn",      this.min_str_sum_fpn+"");     // double
		properties.setProperty(prefix+"min_str_neib",         this.min_str_neib+"");        // double
		properties.setProperty(prefix+"min_str_neib_fpn",     this.min_str_neib_fpn+"");    // double
		
		properties.setProperty(prefix+"min_neibs",            this.min_neibs+"");           // int
		properties.setProperty(prefix+"weight_zero_neibs",    this.weight_zero_neibs+"");   // double
		properties.setProperty(prefix+"half_disparity",       this.half_disparity+"");      // double
		properties.setProperty(prefix+"half_avg_diff",        this.half_avg_diff+"");       // double
		properties.setProperty(prefix+"use_combo_relaible",   this.use_combo_relaible+"");  // boolean
		properties.setProperty(prefix+"ref_need_lma",         this.ref_need_lma+"");        // boolean
		properties.setProperty(prefix+"ref_need_lma_combo",   this.ref_need_lma_combo+"");  // boolean
		properties.setProperty(prefix+"pix_step",             this.pix_step+"");            // int
		properties.setProperty(prefix+"search_rad",           this.search_rad+"");          // int
		properties.setProperty(prefix+"maybe_sum",            this.maybe_sum+"");           // double
		properties.setProperty(prefix+"sure_sum",             this.sure_sum+"");            // double
		properties.setProperty(prefix+"maybe_avg",            this.maybe_avg+"");           // double
		properties.setProperty(prefix+"sure_avg",             this.sure_avg+"");            // double
		properties.setProperty(prefix+"max_search_rms",       this.max_search_rms+"");      // double
		properties.setProperty(prefix+"maybe_fom",            this.maybe_fom+"");           // double
		properties.setProperty(prefix+"sure_fom",             this.sure_fom+"");            // double
		properties.setProperty(prefix+"treat_serch_fpn",        this.treat_serch_fpn+"");   // boolean
		properties.setProperty(prefix+"use_combo_dsi",        this.use_combo_dsi+"");       // boolean
		properties.setProperty(prefix+"use_lma_dsi",          this.use_lma_dsi+"");         // boolean
		properties.setProperty(prefix+"fpn_remove",           this.fpn_remove+"");          // boolean
		properties.setProperty(prefix+"fpn_max_offset",       this.fpn_max_offset+"");      // double
		properties.setProperty(prefix+"fpn_radius",           this.fpn_radius+"");          // double
		properties.setProperty(prefix+"fpn_ignore_border",    this.fpn_ignore_border+"");   // boolean
		
		properties.setProperty(prefix+"min_offset",           this.min_offset+"");          // double
		properties.setProperty(prefix+"max_rel_offset",       this.max_rel_offset+"");      // double
		properties.setProperty(prefix+"max_roll_deg",         this.max_roll_deg+"");        // double
		properties.setProperty(prefix+"max_zoom_diff",        this.max_zoom_diff+"");       // double
		properties.setProperty(prefix+"fpn_skip",             this.fpn_skip+"");            // boolean
		properties.setProperty(prefix+"fpn_rematch",          this.fpn_rematch+"");         // boolean

		properties.setProperty(prefix+"mov_en",               this.mov_en+"");              // boolean
		properties.setProperty(prefix+"mov_sigma",            this.mov_sigma+"");           // double
		properties.setProperty(prefix+"mov_max_std",          this.mov_max_std+"");         // double
		properties.setProperty(prefix+"mov_thresh_rel",       this.mov_thresh_rel+"");      // double
		properties.setProperty(prefix+"mov_thresh_abs",       this.mov_thresh_abs+"");      // double
		properties.setProperty(prefix+"mov_clust_max",        this.mov_clust_max+"");       // double
		properties.setProperty(prefix+"mov_grow",             this.mov_grow+"");            // int
		properties.setProperty(prefix+"mov_max_len",          this.mov_max_len+"");         // int
		
		properties.setProperty(prefix+"mov_show",             this.mov_show+"");            // boolean
		properties.setProperty(prefix+"mov_debug_level",      this.mov_debug_level+"");     // int
		properties.setProperty(prefix+"adjust_atr_0",         this.adjust_atr[0]+"");       // boolean
		properties.setProperty(prefix+"adjust_atr_1",         this.adjust_atr[1]+"");       // boolean
		properties.setProperty(prefix+"adjust_atr_2",         this.adjust_atr[2]+"");       // boolean
		properties.setProperty(prefix+"adjust_xyz_0",         this.adjust_xyz[0]+"");       // boolean
		properties.setProperty(prefix+"adjust_xyz_1",         this.adjust_xyz[1]+"");       // boolean
		properties.setProperty(prefix+"adjust_xyz_2",         this.adjust_xyz[2]+"");       // boolean
		properties.setProperty(prefix+"exit_change_atr",      this.exit_change_atr+"");     // double
		properties.setProperty(prefix+"exit_change_xyz",      this.exit_change_xyz+"");     // double
		properties.setProperty(prefix+"max_cycles",           this.max_cycles+"");          // int
		properties.setProperty(prefix+"max_LMA",              this.max_LMA+"");             // int
		properties.setProperty(prefix+"max_rms",              this.max_rms+"");             // double
	
		properties.setProperty(prefix+"eq_en",                this.eq_en+"");               // boolean
		properties.setProperty(prefix+"eq_stride_hor",        this.eq_stride_hor+"");       // int
		properties.setProperty(prefix+"eq_stride_vert",       this.eq_stride_vert+"");      // int
		properties.setProperty(prefix+"eq_min_stile_weight",  this.eq_min_stile_weight+""); // double
		properties.setProperty(prefix+"eq_min_stile_number",  this.eq_min_stile_number+""); // int
		properties.setProperty(prefix+"eq_min_stile_fraction",this.eq_min_stile_fraction+"");// double
		properties.setProperty(prefix+"eq_min_disparity",     this.eq_min_disparity+"");    // double
		properties.setProperty(prefix+"eq_max_disparity",     this.eq_max_disparity+"");    // double
		properties.setProperty(prefix+"eq_weight_add",        this.eq_weight_add+"");       // double
		properties.setProperty(prefix+"eq_weight_scale",      this.eq_weight_scale+"");     // double
		properties.setProperty(prefix+"eq_level",             this.eq_level+"");            // double

		properties.setProperty(prefix+"mb_en",                this.mb_en+"");               // boolean
		properties.setProperty(prefix+"mb_tau",               this.mb_tau+"");              // double
		properties.setProperty(prefix+"mb_max_gain",          this.mb_max_gain+"");         // double
		properties.setProperty(prefix+"mb_max_gain_inter",    this.mb_max_gain_inter+"");   // double
		properties.setProperty(prefix+"stereo_merge",         this.stereo_merge+"");        // boolean
		properties.setProperty(prefix+"stereo_gap",           this.stereo_gap+"");          // int
		properties.setProperty(prefix+"stereo_intereye",      this.stereo_intereye+"");     // double
		properties.setProperty(prefix+"stereo_phone_width",   this.stereo_phone_width+"");  // double
		properties.setProperty(prefix+"extra_hor_tile",       this.extra_hor_tile+"");      // int
		properties.setProperty(prefix+"extra_vert_tile",      this.extra_vert_tile+"");     // int
		properties.setProperty(prefix+"crop_3d",              this.crop_3d+"");             // boolean
		properties.setProperty(prefix+"sensor_mask",          this.sensor_mask+"");         // int
		properties.setProperty(prefix+"merge_all",            this.merge_all+"");             // boolean
		properties.setProperty(prefix+"video_fps",            this.video_fps+"");           // double
		properties.setProperty(prefix+"sensor_fps",           this.sensor_fps+"");          // double
		properties.setProperty(prefix+"mode_avi",             this.mode_avi+"");            // int
		properties.setProperty(prefix+"avi_JPEG_quality",     this.avi_JPEG_quality+"");    // int
		properties.setProperty(prefix+"run_ffmpeg",           this.run_ffmpeg+"");          // boolean
		properties.setProperty(prefix+"video_ext",            this.video_ext+"");           // String
		properties.setProperty(prefix+"video_codec",          this.video_codec+"");         // String
		properties.setProperty(prefix+"video_crf",            this.video_crf+"");           // int
		properties.setProperty(prefix+"remove_avi",           this.remove_avi+"");          // boolean
		properties.setProperty(prefix+"video_codec_combo",    this.video_codec_combo+"");         // String
		properties.setProperty(prefix+"video_crf_combo",      this.video_crf_combo+"");           // int
		properties.setProperty(prefix+"um_mono",              this.um_mono+"");             // boolean
		properties.setProperty(prefix+"um_sigma",             this.um_sigma+"");            // double
		properties.setProperty(prefix+"um_weight",            this.um_weight+"");           // double
		properties.setProperty(prefix+"mono_fixed",           this.mono_fixed+"");          // boolean
		properties.setProperty(prefix+"mono_range",           this.mono_range+"");          // double
		properties.setProperty(prefix+"anaglyth_en",          this.anaglyth_en+"");         // boolean
		properties.setProperty(prefix+"anaglyph_left",        getLongColor(anaglyph_left)+""); // Color
		properties.setProperty(prefix+"anaglyph_right",       getLongColor(anaglyph_right)+""); // Color
		properties.setProperty(prefix+"annotate_color",       this.annotate_color+"");      // boolean
		properties.setProperty(prefix+"annotate_mono",        this.annotate_mono+"");       // boolean
		{
			long lcolor_annotate =        (annotate_color_color == null) ? -1 : getLongColor(annotate_color_color);
			properties.setProperty(prefix+"annotate_color_color",              lcolor_annotate+"");
		}
		{
			long lcolor_annotate =        (annotate_color_mono == null) ? -1 : getLongColor(annotate_color_mono);
			properties.setProperty(prefix+"annotate_color_mono",              lcolor_annotate+"");
		}
		properties.setProperty(prefix+"annotate_transparent_mono",this.annotate_transparent_mono+"");// boolean
		
	}
	
	public void getProperties(String prefix,Properties properties){
		if (properties.getProperty(prefix+"ims_use")!=null)                       this.ims_use=Boolean.parseBoolean(properties.getProperty(prefix+"ims_use"));		
		if (properties.getProperty(prefix+"ims_rebuild")!=null)                   this.ims_rebuild=Boolean.parseBoolean(properties.getProperty(prefix+"ims_rebuild"));		
		if (properties.getProperty(prefix+"ims_offset")!=null)                    this.ims_offset=Double.parseDouble(properties.getProperty(prefix+"ims_offset"));
		if (properties.getProperty(prefix+"gmt_plus")!=null)                      this.gmt_plus=Double.parseDouble(properties.getProperty(prefix+"gmt_plus"));
		if (properties.getProperty(prefix+"ims_ortho")!=null) 	       this.ims_ortho= IntersceneMatchParameters.StringToDoubles(properties.getProperty(prefix+"ims_ortho"),4);
		if (properties.getProperty(prefix+"ims_mount_atr")!=null) 	   this.ims_mount_atr= IntersceneMatchParameters.StringToDoubles(properties.getProperty(prefix+"ims_mount_atr"),3);
		if (properties.getProperty(prefix+"ims_mount_xyz")!=null) 	   this.ims_mount_xyz= IntersceneMatchParameters.StringToDoubles(properties.getProperty(prefix+"ims_mount_xyz"),3);
		if (properties.getProperty(prefix+"ims_scale_xyz")!=null) 	   this.ims_scale_xyz= IntersceneMatchParameters.StringToDoubles(properties.getProperty(prefix+"ims_scale_xyz"),3);
		if (properties.getProperty(prefix+"ims_scale_atr")!=null) 	   this.ims_scale_atr= IntersceneMatchParameters.StringToDoubles(properties.getProperty(prefix+"ims_scale_atr"),3);
		if (properties.getProperty(prefix+"sfm_use")!=null)            this.sfm_use=Boolean.parseBoolean(properties.getProperty(prefix+"sfm_use"));		
		if (properties.getProperty(prefix+"sfm_min_base")!=null)       this.sfm_min_base=Double.parseDouble(properties.getProperty(prefix+"sfm_min_base"));
		if (properties.getProperty(prefix+"sfm_min_gain")!=null)       this.sfm_min_gain=Double.parseDouble(properties.getProperty(prefix+"sfm_min_gain"));
		if (properties.getProperty(prefix+"sfm_min_frac")!=null)       this.sfm_min_frac=Double.parseDouble(properties.getProperty(prefix+"sfm_min_frac"));
		if (properties.getProperty(prefix+"sfm_num_pairs")!=null)      this.sfm_num_pairs=Integer.parseInt(properties.getProperty(prefix+"sfm_num_pairs"));
		if (properties.getProperty(prefix+"sfp_tolerance")!=null)      this.sfp_tolerance=Double.parseDouble(properties.getProperty(prefix+"sfp_tolerance"));
		if (properties.getProperty(prefix+"sfm_readjust")!=null)       this.sfm_readjust=Integer.parseInt(properties.getProperty(prefix+"sfm_readjust"));
		if (properties.getProperty(prefix+"sfm_prev_frac")!=null)      this.sfm_prev_frac=Double.parseDouble(properties.getProperty(prefix+"sfm_prev_frac"));
		if (properties.getProperty(prefix+"sfm_same_weight")!=null)    this.sfm_same_weight=Double.parseDouble(properties.getProperty(prefix+"sfm_same_weight"));
		if (properties.getProperty(prefix+"sfm_shrink")!=null)         this.sfm_shrink=Integer.parseInt(properties.getProperty(prefix+"sfm_shrink"));
		if (properties.getProperty(prefix+"sfm_fade_sigma")!=null)     this.sfm_fade_sigma=Double.parseDouble(properties.getProperty(prefix+"sfm_fade_sigma"));
		if (properties.getProperty(prefix+"sfm_min_str1")!=null)       this.sfm_min_str1=Double.parseDouble(properties.getProperty(prefix+"sfm_min_str1"));
		if (properties.getProperty(prefix+"sfm_min_str16")!=null)      this.sfm_min_str16=Double.parseDouble(properties.getProperty(prefix+"sfm_min_str16"));
		if (properties.getProperty(prefix+"sfm_use_neibs")!=null)      this.sfm_use_neibs=Boolean.parseBoolean(properties.getProperty(prefix+"sfm_use_neibs"));
		if (properties.getProperty(prefix+"sfm_neib_too_str1")!=null)  this.sfm_neib_too_str1=Double.parseDouble(properties.getProperty(prefix+"sfm_neib_too_str1"));
		if (properties.getProperty(prefix+"sfm_neib_too_str16")!=null) this.sfm_neib_too_str16=Double.parseDouble(properties.getProperty(prefix+"sfm_neib_too_str16"));
		if (properties.getProperty(prefix+"sfm_neib_str1")!=null)      this.sfm_neib_str1=Double.parseDouble(properties.getProperty(prefix+"sfm_neib_str1"));
		if (properties.getProperty(prefix+"sfm_neib_str16")!=null)     this.sfm_neib_str16=Double.parseDouble(properties.getProperty(prefix+"sfm_neib_str16"));
		if (properties.getProperty(prefix+"sfm_extrap_steps")!=null)   this.sfm_extrap_steps=Integer.parseInt(properties.getProperty(prefix+"sfm_extrap_steps"));
		if (properties.getProperty(prefix+"sfm_extrap_radius")!=null)  this.sfm_extrap_radius=Integer.parseInt(properties.getProperty(prefix+"sfm_extrap_radius"));
		if (properties.getProperty(prefix+"sfm_average_neibs")!=null)  this.sfm_average_neibs=Boolean.parseBoolean(properties.getProperty(prefix+"sfm_average_neibs"));		
		if (properties.getProperty(prefix+"sfm_fill_weak")!=null)      this.sfm_fill_weak=Boolean.parseBoolean(properties.getProperty(prefix+"sfm_fill_weak"));		
		if (properties.getProperty(prefix+"sfm_extrapolate")!=null)    this.sfm_extrapolate=Boolean.parseBoolean(properties.getProperty(prefix+"sfm_extrapolate"));		
		
		if (properties.getProperty(prefix+"center_reference")!=null)              this.center_reference=Boolean.parseBoolean(properties.getProperty(prefix+"center_reference"));		
		if (properties.getProperty(prefix+"reset_photometric")!=null)             this.reset_photometric=Boolean.parseBoolean(properties.getProperty(prefix+"reset_photometric"));		
		if (properties.getProperty(prefix+"force_ref_dsi")!=null)                 this.force_ref_dsi=Boolean.parseBoolean(properties.getProperty(prefix+"force_ref_dsi"));		
		if (properties.getProperty(prefix+"force_orientations")!=null)            this.force_orientations=Boolean.parseBoolean(properties.getProperty(prefix+"force_orientations"));		
		if (properties.getProperty(prefix+"run_ly")!=null)                        this.run_ly=Boolean.parseBoolean(properties.getProperty(prefix+"run_ly"));		
		if (properties.getProperty(prefix+"min_num_orient")!=null)                this.min_num_orient=Integer.parseInt(properties.getProperty(prefix+"min_num_orient"));
		if (properties.getProperty(prefix+"min_num_interscene")!=null)            this.min_num_interscene=Integer.parseInt(properties.getProperty(prefix+"min_num_interscene"));
		if (properties.getProperty(prefix+"calc_quat_corr")!=null)                this.calc_quat_corr=Boolean.parseBoolean(properties.getProperty(prefix+"calc_quat_corr"));
		if (properties.getProperty(prefix+"apply_quat_corr")!=null)               this.apply_quat_corr=Boolean.parseBoolean(properties.getProperty(prefix+"apply_quat_corr"));
		if (properties.getProperty(prefix+"generate_egomotion")!=null)            this.generate_egomotion=Boolean.parseBoolean(properties.getProperty(prefix+"generate_egomotion"));
		if (properties.getProperty(prefix+"generate_mapped")!=null)               this.generate_mapped=Boolean.parseBoolean(properties.getProperty(prefix+"generate_mapped"));
		if (properties.getProperty(prefix+"reuse_video")!=null)                   this.reuse_video=Boolean.parseBoolean(properties.getProperty(prefix+"reuse_video"));
		
		if (properties.getProperty(prefix+"save_mapped_color")!=null)             this.save_mapped_color=Boolean.parseBoolean(properties.getProperty(prefix+"save_mapped_color"));
		if (properties.getProperty(prefix+"save_mapped_mono")!=null)              this.save_mapped_mono=Boolean.parseBoolean(properties.getProperty(prefix+"save_mapped_mono"));
		if (properties.getProperty(prefix+"gen_avi_color")!=null)                 this.gen_avi_color=Boolean.parseBoolean(properties.getProperty(prefix+"gen_avi_color"));
		if (properties.getProperty(prefix+"gen_avi_mono")!=null)                  this.gen_avi_mono=Boolean.parseBoolean(properties.getProperty(prefix+"gen_avi_mono"));
		if (properties.getProperty(prefix+"show_mapped_color")!=null)    this.show_mapped_color=Boolean.parseBoolean(properties.getProperty(prefix+"show_mapped_color"));
		if (properties.getProperty(prefix+"show_mapped_mono")!=null)     this.show_mapped_mono=Boolean.parseBoolean(properties.getProperty(prefix+"show_mapped_mono"));
		
		if (properties.getProperty(prefix+"generate_raw")!=null)                  this.generate_raw=Boolean.parseBoolean(properties.getProperty(prefix+"generate_raw"));
		if (properties.getProperty(prefix+"generate_inf")!=null)                  this.generate_inf=Boolean.parseBoolean(properties.getProperty(prefix+"generate_inf"));
		if (properties.getProperty(prefix+"generate_fg")!=null)                   this.generate_fg=Boolean.parseBoolean(properties.getProperty(prefix+"generate_fg"));
		if (properties.getProperty(prefix+"generate_bg")!=null)                   this.generate_bg=Boolean.parseBoolean(properties.getProperty(prefix+"generate_bg"));
		
		if (properties.getProperty(prefix+"generate_stereo")!=null)               this.generate_stereo=Boolean.parseBoolean(properties.getProperty(prefix+"generate_stereo"));
		if (properties.getProperty(prefix+"stereo_views_num")!=null) {
			 int stereo_views_num=Integer.parseInt(properties.getProperty(prefix+"stereo_views_num"));
			 this.stereo_views =        new double[stereo_views_num][];
			 this.generate_stereo_var = new boolean[stereo_views_num];
			 for (int i = 0; i < stereo_views_num; i++) {
				 if (properties.getProperty(prefix+"stereo_views_"+i)!=null) {
					 this.stereo_views[i]=StringToDoubles(properties.getProperty(prefix+"stereo_views_"+i),3);
				 }
				 if (properties.getProperty(prefix+"generate_stereo_var_"+i)!=null) {
					 this.generate_stereo_var[i]=Boolean.parseBoolean(properties.getProperty(prefix+"generate_stereo_var_"+i));
				 }
			 }
			 orderStereoViews(); 
		}
		
		if (properties.getProperty(prefix+"export_images")!=null)        this.export_images=Boolean.parseBoolean(properties.getProperty(prefix+"export_images"));		
		if (properties.getProperty(prefix+"show_images")!=null)          this.show_images=Boolean.parseBoolean(properties.getProperty(prefix+"show_images"));		
		if (properties.getProperty(prefix+"show_images_bgfg")!=null)     this.show_images_bgfg=Boolean.parseBoolean(properties.getProperty(prefix+"show_images_bgfg"));		
		if (properties.getProperty(prefix+"show_images_mono")!=null)     this.show_images_mono=Boolean.parseBoolean(properties.getProperty(prefix+"show_images_mono"));		
		if (properties.getProperty(prefix+"export_ranges")!=null)        this.export_ranges=Boolean.parseBoolean(properties.getProperty(prefix+"export_ranges"));
		if (properties.getProperty(prefix+"debug_ranges")!=null)         this.debug_ranges=Boolean.parseBoolean(properties.getProperty(prefix+"debug_ranges"));
		if (properties.getProperty(prefix+"show_ranges")!=null)          this.show_ranges=Boolean.parseBoolean(properties.getProperty(prefix+"show_ranges"));
		if (properties.getProperty(prefix+"export_ml_files")!=null)      this.export_ml_files=Boolean.parseBoolean(properties.getProperty(prefix+"export_ml_files"));

		if (properties.getProperty(prefix+"show_color_nan")!=null)       this.show_color_nan=Boolean.parseBoolean(properties.getProperty(prefix+"show_color_nan"));		
		if (properties.getProperty(prefix+"show_mono_nan")!=null)        this.show_mono_nan=Boolean.parseBoolean(properties.getProperty(prefix+"show_mono_nan"));

		if (properties.getProperty(prefix+"scale_extrap_atr")!=null)     this.scale_extrap_atr=Double.parseDouble(properties.getProperty(prefix+"scale_extrap_atr"));
		if (properties.getProperty(prefix+"scale_extrap_xyz")!=null)     this.scale_extrap_xyz=Double.parseDouble(properties.getProperty(prefix+"scale_extrap_xyz"));
		
		if (properties.getProperty(prefix+"avg_len")!=null)              this.avg_len=Integer.parseInt(properties.getProperty(prefix+"avg_len"));
		if (properties.getProperty(prefix+"min_num_scenes")!=null)       this.min_num_scenes=Integer.parseInt(properties.getProperty(prefix+"min_num_scenes"));
		if (properties.getProperty(prefix+"max_num_scenes")!=null)       this.max_num_scenes=Integer.parseInt(properties.getProperty(prefix+"max_num_scenes"));
		if (properties.getProperty(prefix+"blur_egomotion")!=null)       this.blur_egomotion=Double.parseDouble(properties.getProperty(prefix+"blur_egomotion"));
		if (properties.getProperty(prefix+"range_disparity_offset")!=null)this.range_disparity_offset=Double.parseDouble(properties.getProperty(prefix+"range_disparity_offset"));
		if (properties.getProperty(prefix+"range_min_strength")!=null)   this.range_min_strength=Double.parseDouble(properties.getProperty(prefix+"range_min_strength"));
		if (properties.getProperty(prefix+"range_max")!=null)            this.range_max=Double.parseDouble(properties.getProperty(prefix+"range_max"));

		if (properties.getProperty(prefix+"export3d")!=null)             this.export3d=Boolean.parseBoolean(properties.getProperty(prefix+"export3d"));		
		
		
		if (properties.getProperty(prefix+"scene_is_ref_test")!=null)    this.scene_is_ref_test=Boolean.parseBoolean(properties.getProperty(prefix+"scene_is_ref_test"));		
		if (properties.getProperty(prefix+"render_ref")!=null)           this.render_ref=Boolean.parseBoolean(properties.getProperty(prefix+"render_ref"));		
		if (properties.getProperty(prefix+"render_scene")!=null)         this.render_scene=Boolean.parseBoolean(properties.getProperty(prefix+"render_scene"));		
		if (properties.getProperty(prefix+"toRGB")!=null)                this.toRGB=Boolean.parseBoolean(properties.getProperty(prefix+"toRGB"));		
		if (properties.getProperty(prefix+"show_2d_correlations")!=null) this.show_2d_correlations=Boolean.parseBoolean(properties.getProperty(prefix+"show_2d_correlations"));		
		if (properties.getProperty(prefix+"show_motion_vectors")!=null)  this.show_motion_vectors=Boolean.parseBoolean(properties.getProperty(prefix+"show_motion_vectors"));		
		if (properties.getProperty(prefix+"debug_level")!=null)          this.debug_level=Integer.parseInt(properties.getProperty(prefix+"debug_level"));
		if (properties.getProperty(prefix+"test_ers")!=null)             this.test_ers=Boolean.parseBoolean(properties.getProperty(prefix+"test_ers"));		
		if (properties.getProperty(prefix+"test_ers0")!=null)            this.test_ers0=Integer.parseInt(properties.getProperty(prefix+"test_ers0"));
		if (properties.getProperty(prefix+"test_ers1")!=null)            this.test_ers1=Integer.parseInt(properties.getProperty(prefix+"test_ers1"));
		if (properties.getProperty(prefix+"num_bottom")!=null)                    this.num_bottom=Integer.parseInt(properties.getProperty(prefix+"num_bottom"));
		if (properties.getProperty(prefix+"num_passes")!=null)                    this.num_passes=Integer.parseInt(properties.getProperty(prefix+"num_passes"));
		if (properties.getProperty(prefix+"max_change")!=null)                    this.max_change=Double.parseDouble(properties.getProperty(prefix+"max_change"));
		if (properties.getProperty(prefix+"min_disparity")!=null)                 this.min_disparity=Double.parseDouble(properties.getProperty(prefix+"min_disparity"));
		if (properties.getProperty(prefix+"max_sym_disparity")!=null)             this.max_sym_disparity=Double.parseDouble(properties.getProperty(prefix+"max_sym_disparity"));
		if (properties.getProperty(prefix+"min_strength_lma")!=null)              this.min_strength_lma=Double.parseDouble(properties.getProperty(prefix+"min_strength_lma"));
		if (properties.getProperty(prefix+"min_strength_replace")!=null)          this.min_strength_replace=Double.parseDouble(properties.getProperty(prefix+"min_strength_replace"));
		if (properties.getProperty(prefix+"min_strength_blur")!=null)             this.min_strength_blur=Double.parseDouble(properties.getProperty(prefix+"min_strength_blur"));
		if (properties.getProperty(prefix+"sigma")!=null)                         this.sigma=Double.parseDouble(properties.getProperty(prefix+"sigma"));
		if (properties.getProperty(prefix+"num_blur")!=null)                      this.num_blur=Integer.parseInt(properties.getProperty(prefix+"num_blur"));
		if (properties.getProperty(prefix+"disparity_corr")!=null)                this.disparity_corr=Double.parseDouble(properties.getProperty(prefix+"disparity_corr"));
		if (properties.getProperty(prefix+"outliers_nth_fromextrem")!=null)       this.outliers_nth_fromextrem=Integer.parseInt(properties.getProperty(prefix+"outliers_nth_fromextrem"));
		if (properties.getProperty(prefix+"outliers_tolerance_absolute")!=null)   this.outliers_tolerance_absolute=Double.parseDouble(properties.getProperty(prefix+"outliers_tolerance_absolute"));
		if (properties.getProperty(prefix+"outliers_tolerance_relative")!=null)   this.outliers_tolerance_relative=Double.parseDouble(properties.getProperty(prefix+"outliers_tolerance_relative"));
		if (properties.getProperty(prefix+"outliers_max_iter")!=null)             this.outliers_max_iter=Integer.parseInt(properties.getProperty(prefix+"outliers_max_iter"));
		if (properties.getProperty(prefix+"outliers_max_strength2")!=null)        this.outliers_max_strength2=Double.parseDouble(properties.getProperty(prefix+"outliers_max_strength2"));
		if (properties.getProperty(prefix+"outliers_nth_fromextrem2")!=null)      this.outliers_nth_fromextrem2=Integer.parseInt(properties.getProperty(prefix+"outliers_nth_fromextrem2"));
		if (properties.getProperty(prefix+"outliers_tolerance_absolute2")!=null)  this.outliers_tolerance_absolute2=Double.parseDouble(properties.getProperty(prefix+"outliers_tolerance_absolute2"));
		if (properties.getProperty(prefix+"outliers_tolerance_relative2")!=null)  this.outliers_tolerance_relative2=Double.parseDouble(properties.getProperty(prefix+"outliers_tolerance_relative2"));
		if (properties.getProperty(prefix+"outliers_lma_max_strength")!=null)     this.outliers_lma_max_strength=Double.parseDouble(properties.getProperty(prefix+"outliers_lma_max_strength"));
		if (properties.getProperty(prefix+"outliers_max_strength")!=null)         this.outliers_max_strength=Double.parseDouble(properties.getProperty(prefix+"outliers_max_strength"));
		if (properties.getProperty(prefix+"outliers_from_lma_max_strength")!=null)this.outliers_from_lma_max_strength=Double.parseDouble(properties.getProperty(prefix+"outliers_from_lma_max_strength"));
		if (properties.getProperty(prefix+"search_radius")!=null)                 this.search_radius=Integer.parseInt(properties.getProperty(prefix+"search_radius"));
		if (properties.getProperty(prefix+"remove_no_lma_neib")!=null)            this.remove_no_lma_neib=Boolean.parseBoolean(properties.getProperty(prefix+"remove_no_lma_neib"));
		if (properties.getProperty(prefix+"diff_from_lma_pos")!=null)             this.diff_from_lma_pos=Double.parseDouble(properties.getProperty(prefix+"diff_from_lma_pos"));
		if (properties.getProperty(prefix+"diff_from_lma_neg")!=null)             this.diff_from_lma_neg=Double.parseDouble(properties.getProperty(prefix+"diff_from_lma_neg"));
		if (properties.getProperty(prefix+"outliers_lma_nth_fromextrem")!=null)   this.outliers_lma_nth_fromextrem=Integer.parseInt(properties.getProperty(prefix+"outliers_lma_nth_fromextrem"));
		if (properties.getProperty(prefix+"filter_margin")!=null)                 this.filter_margin=Integer.parseInt(properties.getProperty(prefix+"filter_margin"));
		if (properties.getProperty(prefix+"weak_tolerance_absolute")!=null)       this.weak_tolerance_absolute=Double.parseDouble(properties.getProperty(prefix+"weak_tolerance_absolute"));
		if (properties.getProperty(prefix+"weak_tolerance_relative")!=null)       this.weak_tolerance_relative=Double.parseDouble(properties.getProperty(prefix+"weak_tolerance_relative"));
		if (properties.getProperty(prefix+"weak_min_neibs")!=null)                this.weak_min_neibs=Integer.parseInt(properties.getProperty(prefix+"weak_min_neibs"));
		if (properties.getProperty(prefix+"strong_strength")!=null)               this.strong_strength=Double.parseDouble(properties.getProperty(prefix+"strong_strength"));
		if (properties.getProperty(prefix+"weak_strength")!=null)                 this.weak_strength=Double.parseDouble(properties.getProperty(prefix+"weak_strength"));
		
		if (properties.getProperty(prefix+"sky_extract")!=null)          this.sky_extract=Boolean.parseBoolean(properties.getProperty(prefix+"sky_extract"));		
		if (properties.getProperty(prefix+"sky_recalc")!=null)           this.sky_recalc=Boolean.parseBoolean(properties.getProperty(prefix+"sky_recalc"));		
		if (properties.getProperty(prefix+"sky_highest_min")!=null)      this.sky_highest_min=Double.parseDouble(properties.getProperty(prefix+"sky_highest_min"));
		if (properties.getProperty(prefix+"cold_frac")!=null)            this.cold_frac=Double.parseDouble(properties.getProperty(prefix+"cold_frac"));
		if (properties.getProperty(prefix+"hot_frac")!=null)             this.hot_frac=Double.parseDouble(properties.getProperty(prefix+"hot_frac"));
		if (properties.getProperty(prefix+"cold_scale")!=null)           this.cold_scale=Double.parseDouble(properties.getProperty(prefix+"cold_scale"));
		if (properties.getProperty(prefix+"sky_seed")!=null)             this.sky_seed=Double.parseDouble(properties.getProperty(prefix+"sky_seed"));
		if (properties.getProperty(prefix+"lma_seed")!=null)             this.lma_seed=Double.parseDouble(properties.getProperty(prefix+"lma_seed"));
		if (properties.getProperty(prefix+"seed_temp")!=null)            this.seed_temp=Double.parseDouble(properties.getProperty(prefix+"seed_temp"));

		
		if (properties.getProperty(prefix+"sky_shrink")!=null)           this.sky_shrink=Integer.parseInt(properties.getProperty(prefix+"sky_shrink"));
		if (properties.getProperty(prefix+"seed_rows")!=null)            this.seed_rows=Integer.parseInt(properties.getProperty(prefix+"seed_rows"));
		if (properties.getProperty(prefix+"max_disparity")!=null)        this.max_disparity=Double.parseDouble(properties.getProperty(prefix+"max_disparity"));
		if (properties.getProperty(prefix+"max_disparity_strength")!=null)this.max_disparity_strength=Double.parseDouble(properties.getProperty(prefix+"max_disparity_strength"));
		if (properties.getProperty(prefix+"sky_lim")!=null)              this.sky_lim=Double.parseDouble(properties.getProperty(prefix+"sky_lim"));
		if (properties.getProperty(prefix+"lim_temp")!=null)             this.lim_temp=Double.parseDouble(properties.getProperty(prefix+"lim_temp"));
		if (properties.getProperty(prefix+"sky_expand_extra")!=null)     this.sky_expand_extra=Integer.parseInt(properties.getProperty(prefix+"sky_expand_extra"));
		if (properties.getProperty(prefix+"sky_bottleneck")!=null)       this.sky_bottleneck=Integer.parseInt(properties.getProperty(prefix+"sky_bottleneck"));
		if (properties.getProperty(prefix+"sky_reexpand_extra")!=null)   this.sky_reexpand_extra=Integer.parseInt(properties.getProperty(prefix+"sky_reexpand_extra"));
		if (properties.getProperty(prefix+"min_strength")!=null)         this.min_strength=Double.parseDouble(properties.getProperty(prefix+"min_strength"));
		if (properties.getProperty(prefix+"lowest_sky_row")!=null)       this.lowest_sky_row=Integer.parseInt(properties.getProperty(prefix+"lowest_sky_row"));
		if (properties.getProperty(prefix+"sky_bottom_override")!=null)  this.sky_bottom_override=Double.parseDouble(properties.getProperty(prefix+"sky_bottom_override"));
		if (properties.getProperty(prefix+"sky_override_shrink")!=null)  this.sky_override_shrink=Integer.parseInt(properties.getProperty(prefix+"sky_override_shrink"));

		if (properties.getProperty(prefix+"clouds_en")!=null)            this.clouds_en=Boolean.parseBoolean(properties.getProperty(prefix+"clouds_en"));		
		if (properties.getProperty(prefix+"clouds_fom")!=null)           this.clouds_fom=Double.parseDouble(properties.getProperty(prefix+"clouds_fom"));
		if (properties.getProperty(prefix+"clouds_spread")!=null)        this.clouds_spread=Double.parseDouble(properties.getProperty(prefix+"clouds_spread"));
		if (properties.getProperty(prefix+"clouds_disparity")!=null)     this.clouds_disparity=Double.parseDouble(properties.getProperty(prefix+"clouds_disparity"));
		if (properties.getProperty(prefix+"clouds_weak")!=null)          this.clouds_weak=Double.parseDouble(properties.getProperty(prefix+"clouds_weak"));
		if (properties.getProperty(prefix+"clouds_strength")!=null)      this.clouds_strength=Double.parseDouble(properties.getProperty(prefix+"clouds_strength"));
		if (properties.getProperty(prefix+"clouds_not_strength")!=null)  this.clouds_not_strength=Double.parseDouble(properties.getProperty(prefix+"clouds_not_strength"));
		if (properties.getProperty(prefix+"clouds_strong")!=null)        this.clouds_strong=Boolean.parseBoolean(properties.getProperty(prefix+"clouds_strong"));		

		if (properties.getProperty(prefix+"wall_en")!=null)              this.wall_en=Boolean.parseBoolean(properties.getProperty(prefix+"wall_en"));		
		if (properties.getProperty(prefix+"wall_dflt")!=null)            this.indoors_en=Boolean.parseBoolean(properties.getProperty(prefix+"wall_dflt"));		
		if (properties.getProperty(prefix+"wall_str")!=null)             this.wall_str=Double.parseDouble(properties.getProperty(prefix+"wall_str"));
		if (properties.getProperty(prefix+"wall_far")!=null)             this.wall_far=Double.parseDouble(properties.getProperty(prefix+"wall_far"));
		if (properties.getProperty(prefix+"wall_near")!=null)            this.wall_near=Double.parseDouble(properties.getProperty(prefix+"wall_near"));
		
		if (properties.getProperty(prefix+"treeline_en")!=null)          this.treeline_en=Boolean.parseBoolean(properties.getProperty(prefix+"treeline_en"));		
		if (properties.getProperty(prefix+"treeline_wide")!=null)        this.treeline_wide=Boolean.parseBoolean(properties.getProperty(prefix+"treeline_wide"));		
		if (properties.getProperty(prefix+"treeline_height")!=null)      this.treeline_height=Integer.parseInt(properties.getProperty(prefix+"treeline_height"));
		if (properties.getProperty(prefix+"treeline_width")!=null)       this.treeline_width=Integer.parseInt(properties.getProperty(prefix+"treeline_width"));
		if (properties.getProperty(prefix+"treeline_lim_high")!=null)    this.treeline_lim_high=Boolean.parseBoolean(properties.getProperty(prefix+"treeline_lim_high"));		
		if (properties.getProperty(prefix+"treeline_str")!=null)         this.treeline_str=Double.parseDouble(properties.getProperty(prefix+"treeline_str"));
		if (properties.getProperty(prefix+"treeline_far")!=null)         this.treeline_far=Double.parseDouble(properties.getProperty(prefix+"treeline_far"));
		if (properties.getProperty(prefix+"treeline_near")!=null)        this.treeline_near=Double.parseDouble(properties.getProperty(prefix+"treeline_near"));
		if (properties.getProperty(prefix+"treeline_fg_str")!=null)      this.treeline_fg_str=Double.parseDouble(properties.getProperty(prefix+"treeline_fg_str"));
		if (properties.getProperty(prefix+"treeline_fg_far")!=null)      this.treeline_fg_far=Double.parseDouble(properties.getProperty(prefix+"treeline_fg_far"));
		if (properties.getProperty(prefix+"treeline_fg_near")!=null)     this.treeline_fg_near=Double.parseDouble(properties.getProperty(prefix+"treeline_fg_near"));
		
		if (properties.getProperty(prefix+"indoors_en")!=null)           this.indoors_en=Boolean.parseBoolean(properties.getProperty(prefix+"indoors_en"));		
		if (properties.getProperty(prefix+"indoors_str")!=null)          this.indoors_str=Double.parseDouble(properties.getProperty(prefix+"indoors_str"));
		if (properties.getProperty(prefix+"indoors_disp")!=null)         this.indoors_disp=Double.parseDouble(properties.getProperty(prefix+"indoors_disp"));
		if (properties.getProperty(prefix+"indoors_min_out")!=null)      this.indoors_min_out=Integer.parseInt(properties.getProperty(prefix+"indoors_min_out"));
		
		if (properties.getProperty(prefix+"disp_boost_min")!=null)       this.disp_boost_min=Double.parseDouble(properties.getProperty(prefix+"disp_boost_min"));
		if (properties.getProperty(prefix+"disp_boost_diff")!=null)      this.disp_boost_diff=Double.parseDouble(properties.getProperty(prefix+"disp_boost_diff"));
		if (properties.getProperty(prefix+"disp_boost_neibs")!=null)     this.disp_boost_neibs=Integer.parseInt(properties.getProperty(prefix+"disp_boost_neibs"));
		if (properties.getProperty(prefix+"disp_boost_amount")!=null)    this.disp_boost_amount=Double.parseDouble(properties.getProperty(prefix+"disp_boost_amount"));
		if (properties.getProperty(prefix+"scale_combo_strength")!=null) this.scale_combo_strength=Double.parseDouble(properties.getProperty(prefix+"scale_combo_strength"));
		if (properties.getProperty(prefix+"save_debug_images")!=null)    this.save_debug_images=Boolean.parseBoolean(properties.getProperty(prefix+"save_debug_images"));		
		
		if (properties.getProperty(prefix+"margin")!=null)               this.margin=Integer.parseInt(properties.getProperty(prefix+"margin"));
		if (properties.getProperty(prefix+"sensor_mask_inter")!=null)    this.sensor_mask_inter=Integer.parseInt(properties.getProperty(prefix+"sensor_mask_inter"));
		if (properties.getProperty(prefix+"use_partial")!=null)          this.use_partial=Boolean.parseBoolean(properties.getProperty(prefix+"use_partial"));		
		if (properties.getProperty(prefix+"run_poly")!=null)             this.run_poly=Boolean.parseBoolean(properties.getProperty(prefix+"run_poly"));
		if (properties.getProperty(prefix+"centroid_radius")!=null)      this.centroid_radius=Double.parseDouble(properties.getProperty(prefix+"centroid_radius"));
		if (properties.getProperty(prefix+"n_recenter")!=null)           this.n_recenter=Integer.parseInt(properties.getProperty(prefix+"n_recenter"));
		if (properties.getProperty(prefix+"min_ref_str")!=null)          this.min_ref_str=Double.parseDouble(properties.getProperty(prefix+"min_ref_str"));
		if (properties.getProperty(prefix+"min_ref_frac")!=null)         this.min_ref_frac=Double.parseDouble(properties.getProperty(prefix+"min_ref_frac"));
		if (properties.getProperty(prefix+"td_weight")!=null)            this.td_weight=Double.parseDouble(properties.getProperty(prefix+"td_weight"));
		if (properties.getProperty(prefix+"td_neib_weight")!=null)       this.td_neib_weight=Double.parseDouble(properties.getProperty(prefix+"td_neib_weight"));
		if (properties.getProperty(prefix+"pd_weight")!=null)            this.pd_weight=Double.parseDouble(properties.getProperty(prefix+"pd_weight"));
		if (properties.getProperty(prefix+"td_nopd_only")!=null)         this.td_nopd_only=Boolean.parseBoolean(properties.getProperty(prefix+"td_nopd_only"));
		if (properties.getProperty(prefix+"neib_notd_only")!=null)       this.neib_notd_only=Boolean.parseBoolean(properties.getProperty(prefix+"neib_notd_only"));

		if (properties.getProperty(prefix+"use_neibs")!=null)            this.use_neibs=Boolean.parseBoolean(properties.getProperty(prefix+       "use_neibs"));
		if (properties.getProperty(prefix+"neibs_nofpn_only")!=null)     this.neibs_nofpn_only=Boolean.parseBoolean(properties.getProperty(prefix+"neibs_nofpn_only"));
		if (properties.getProperty(prefix+"neibs_nofpn_init")!=null)     this.neibs_nofpn_init=Boolean.parseBoolean(properties.getProperty(prefix+"neibs_nofpn_init"));
		if (properties.getProperty(prefix+"redo_both")!=null)            this.redo_both=Boolean.parseBoolean(properties.getProperty(prefix+       "redo_both"));
		if (properties.getProperty(prefix+"min_num_neibs")!=null)        this.min_num_neibs=Integer.parseInt(properties.getProperty(prefix+       "min_num_neibs"));
		if (properties.getProperty(prefix+"scale_neibs_pd")!=null)       this.scale_neibs_pd=Double.parseDouble(properties.getProperty(prefix+    "scale_neibs_pd"));
		if (properties.getProperty(prefix+"scale_neibs_td")!=null)       this.scale_neibs_td=Double.parseDouble(properties.getProperty(prefix+    "scale_neibs_td"));
		if (properties.getProperty(prefix+"scale_avg_weight")!=null)     this.scale_avg_weight =Double.parseDouble(properties.getProperty(prefix+ "scale_avg_weight"));
		if (properties.getProperty(prefix+"min_str")!=null)              this.min_str=Double.parseDouble(properties.getProperty(prefix+"min_str"));
		if (properties.getProperty(prefix+"min_str_fpn")!=null)          this.min_str_fpn=Double.parseDouble(properties.getProperty(prefix+"min_str_fpn"));
		if (properties.getProperty(prefix+"min_str_sum")!=null)          this.min_str_sum=Double.parseDouble(properties.getProperty(prefix+"min_str_sum"));
		if (properties.getProperty(prefix+"min_str_sum_fpn")!=null)      this.min_str_sum_fpn=Double.parseDouble(properties.getProperty(prefix+"min_str_sum_fpn"));
		if (properties.getProperty(prefix+"min_str_neib")!=null)         this.min_str_neib=Double.parseDouble(properties.getProperty(prefix+"min_str_neib"));
		if (properties.getProperty(prefix+"min_str_neib_fpn")!=null)     this.min_str_neib_fpn=Double.parseDouble(properties.getProperty(prefix+"min_str_neib_fpn"));
		
		if (properties.getProperty(prefix+"min_neibs")!=null)            this.min_neibs=Integer.parseInt(properties.getProperty(prefix+"min_neibs"));
		if (properties.getProperty(prefix+"weight_zero_neibs")!=null)    this.weight_zero_neibs=Double.parseDouble(properties.getProperty(prefix+"weight_zero_neibs"));
		if (properties.getProperty(prefix+"half_disparity")!=null)       this.half_disparity=Double.parseDouble(properties.getProperty(prefix+"half_disparity"));
		if (properties.getProperty(prefix+"half_avg_diff")!=null)        this.half_avg_diff=Double.parseDouble(properties.getProperty(prefix+"half_avg_diff"));
		if (properties.getProperty(prefix+"use_combo_relaible")!=null)   this.use_combo_relaible=Boolean.parseBoolean(properties.getProperty(prefix+"use_combo_relaible"));		
		if (properties.getProperty(prefix+"ref_need_lma")!=null)         this.ref_need_lma=Boolean.parseBoolean(properties.getProperty(prefix+"ref_need_lma"));		
		if (properties.getProperty(prefix+"ref_need_lma_combo")!=null)   this.ref_need_lma_combo=Boolean.parseBoolean(properties.getProperty(prefix+"ref_need_lma_combo"));		
		if (properties.getProperty(prefix+"pix_step")!=null)             this.pix_step=Integer.parseInt(properties.getProperty(prefix+"pix_step"));
		if (properties.getProperty(prefix+"search_rad")!=null)           this.search_rad=Integer.parseInt(properties.getProperty(prefix+"search_rad"));
		if (properties.getProperty(prefix+"maybe_sum")!=null)            this.maybe_sum=Double.parseDouble(properties.getProperty(prefix+"maybe_sum"));
		if (properties.getProperty(prefix+"sure_sum")!=null)             this.sure_sum=Double.parseDouble(properties.getProperty(prefix+"sure_sum"));
		if (properties.getProperty(prefix+"maybe_avg")!=null)            this.maybe_avg=Double.parseDouble(properties.getProperty(prefix+"maybe_avg"));
		if (properties.getProperty(prefix+"sure_avg")!=null)             this.sure_avg=Double.parseDouble(properties.getProperty(prefix+"sure_avg"));
		if (properties.getProperty(prefix+"max_search_rms")!=null)       this.max_search_rms=Double.parseDouble(properties.getProperty(prefix+"max_search_rms"));
		if (properties.getProperty(prefix+"maybe_fom")!=null)            this.maybe_fom=Double.parseDouble(properties.getProperty(prefix+"maybe_fom"));
		if (properties.getProperty(prefix+"sure_fom")!=null)             this.sure_fom=Double.parseDouble(properties.getProperty(prefix+"sure_fom"));
		if (properties.getProperty(prefix+"treat_serch_fpn")!=null)      this.treat_serch_fpn=Boolean.parseBoolean(properties.getProperty(prefix+"treat_serch_fpn"));		
		if (properties.getProperty(prefix+"use_combo_dsi")!=null)        this.use_combo_dsi=Boolean.parseBoolean(properties.getProperty(prefix+"use_combo_dsi"));		
		if (properties.getProperty(prefix+"use_lma_dsi")!=null)          this.use_lma_dsi=Boolean.parseBoolean(properties.getProperty(prefix+"use_lma_dsi"));
		if (properties.getProperty(prefix+"fpn_remove")!=null)           this.fpn_remove=Boolean.parseBoolean(properties.getProperty(prefix+"fpn_remove"));
		if (properties.getProperty(prefix+"fpn_max_offset")!=null)       this.fpn_max_offset=Double.parseDouble(properties.getProperty(prefix+"fpn_max_offset"));
		if (properties.getProperty(prefix+"fpn_radius")!=null)           this.fpn_radius=Double.parseDouble(properties.getProperty(prefix+"fpn_radius"));
		if (properties.getProperty(prefix+"fpn_ignore_border")!=null)    this.fpn_ignore_border=Boolean.parseBoolean(properties.getProperty(prefix+"fpn_ignore_border"));
		
		if (properties.getProperty(prefix+"min_offset")!=null)           this.min_offset=Double.parseDouble(properties.getProperty(prefix+"min_offset"));
		if (properties.getProperty(prefix+"max_rel_offset")!=null)       this.max_rel_offset=Double.parseDouble(properties.getProperty(prefix+"max_rel_offset"));
		if (properties.getProperty(prefix+"max_roll_deg")!=null)         this.max_roll_deg=Double.parseDouble(properties.getProperty(prefix+"max_roll_deg"));
		if (properties.getProperty(prefix+"max_zoom_diff")!=null)        this.max_zoom_diff=Double.parseDouble(properties.getProperty(prefix+"max_zoom_diff"));
		if (properties.getProperty(prefix+"fpn_skip")!=null)             this.fpn_skip=Boolean.parseBoolean(properties.getProperty(prefix+"fpn_skip"));
		if (properties.getProperty(prefix+"fpn_rematch")!=null)          this.fpn_rematch=Boolean.parseBoolean(properties.getProperty(prefix+"fpn_rematch"));
		
		if (properties.getProperty(prefix+"mov_en")!=null)               this.mov_en=Boolean.parseBoolean(properties.getProperty(prefix+"mov_en"));
		if (properties.getProperty(prefix+"mov_sigma")!=null)            this.mov_sigma=Double.parseDouble(properties.getProperty(prefix+"mov_sigma"));
		if (properties.getProperty(prefix+"mov_max_std")!=null)          this.mov_max_std=Double.parseDouble(properties.getProperty(prefix+"mov_max_std"));
		if (properties.getProperty(prefix+"mov_thresh_rel")!=null)       this.mov_thresh_rel=Double.parseDouble(properties.getProperty(prefix+"mov_thresh_rel"));
		if (properties.getProperty(prefix+"mov_thresh_abs")!=null)       this.mov_thresh_abs=Double.parseDouble(properties.getProperty(prefix+"mov_thresh_abs"));
		if (properties.getProperty(prefix+"mov_clust_max")!=null)        this.mov_clust_max=Double.parseDouble(properties.getProperty(prefix+"mov_clust_max"));
		if (properties.getProperty(prefix+"mov_grow")!=null)             this.mov_grow=Integer.parseInt(properties.getProperty(prefix+"mov_grow"));
		if (properties.getProperty(prefix+"mov_max_len")!=null)          this.mov_max_len=Integer.parseInt(properties.getProperty(prefix+"mov_max_len"));

		if (properties.getProperty(prefix+"mov_show")!=null)             this.mov_show=Boolean.parseBoolean(properties.getProperty(prefix+"mov_show"));
		if (properties.getProperty(prefix+"mov_debug_level")!=null)      this.mov_debug_level=Integer.parseInt(properties.getProperty(prefix+"mov_debug_level"));
		if (properties.getProperty(prefix+"adjust_atr_0")!=null)         this.adjust_atr[0]=Boolean.parseBoolean(properties.getProperty(prefix+"adjust_atr_0"));		
		if (properties.getProperty(prefix+"adjust_atr_1")!=null)         this.adjust_atr[1]=Boolean.parseBoolean(properties.getProperty(prefix+"adjust_atr_1"));		
		if (properties.getProperty(prefix+"adjust_atr_2")!=null)         this.adjust_atr[2]=Boolean.parseBoolean(properties.getProperty(prefix+"adjust_atr_2"));		
		if (properties.getProperty(prefix+"adjust_xyz_0")!=null)         this.adjust_xyz[0]=Boolean.parseBoolean(properties.getProperty(prefix+"adjust_xyz_0"));		
		if (properties.getProperty(prefix+"adjust_xyz_1")!=null)         this.adjust_xyz[1]=Boolean.parseBoolean(properties.getProperty(prefix+"adjust_xyz_1"));		
		if (properties.getProperty(prefix+"adjust_xyz_2")!=null)         this.adjust_xyz[2]=Boolean.parseBoolean(properties.getProperty(prefix+"adjust_xyz_2"));		
		if (properties.getProperty(prefix+"exit_change_atr")!=null)      this.exit_change_atr=Double.parseDouble(properties.getProperty(prefix+"exit_change_atr"));
		if (properties.getProperty(prefix+"exit_change_xyz")!=null)      this.exit_change_xyz=Double.parseDouble(properties.getProperty(prefix+"exit_change_xyz"));
		if (properties.getProperty(prefix+"max_cycles")!=null)           this.max_cycles=Integer.parseInt(properties.getProperty(prefix+"max_cycles"));
		if (properties.getProperty(prefix+"max_LMA")!=null)              this.max_LMA=Integer.parseInt(properties.getProperty(prefix+"max_LMA"));
		if (properties.getProperty(prefix+"max_rms")!=null)              this.max_rms=Double.parseDouble(properties.getProperty(prefix+"max_rms"));
		
		if (properties.getProperty(prefix+"eq_en")!=null)                this.eq_en=Boolean.parseBoolean(properties.getProperty(prefix+"eq_en"));		
		if (properties.getProperty(prefix+"eq_stride_hor")!=null)        this.eq_stride_hor=Integer.parseInt(properties.getProperty(prefix+"eq_stride_hor"));
		if (properties.getProperty(prefix+"eq_stride_vert")!=null)       this.eq_stride_vert=Integer.parseInt(properties.getProperty(prefix+"eq_stride_vert"));
		if (properties.getProperty(prefix+"eq_min_stile_weight")!=null)  this.eq_min_stile_weight=Double.parseDouble(properties.getProperty(prefix+"eq_min_stile_weight"));
		if (properties.getProperty(prefix+"eq_min_stile_number")!=null)  this.eq_min_stile_number=Integer.parseInt(properties.getProperty(prefix+"eq_min_stile_number"));
		if (properties.getProperty(prefix+"eq_min_stile_fraction")!=null)this.eq_min_stile_fraction=Double.parseDouble(properties.getProperty(prefix+"eq_min_stile_fraction"));
		if (properties.getProperty(prefix+"eq_min_disparity")!=null)     this.eq_min_disparity=Double.parseDouble(properties.getProperty(prefix+"eq_min_disparity"));
		if (properties.getProperty(prefix+"eq_max_disparity")!=null)     this.eq_max_disparity=Double.parseDouble(properties.getProperty(prefix+"eq_max_disparity"));
		if (properties.getProperty(prefix+"eq_weight_add")!=null)        this.eq_weight_add=Double.parseDouble(properties.getProperty(prefix+"eq_weight_add"));
		if (properties.getProperty(prefix+"eq_weight_scale")!=null)      this.eq_weight_scale=Double.parseDouble(properties.getProperty(prefix+"eq_weight_scale"));
		if (properties.getProperty(prefix+"eq_level")!=null)             this.eq_level=Double.parseDouble(properties.getProperty(prefix+"eq_level"));
		
		if (properties.getProperty(prefix+"mb_en")!=null)                this.mb_en=Boolean.parseBoolean(properties.getProperty(prefix+"mb_en"));
		if (properties.getProperty(prefix+"mb_tau")!=null)               this.mb_tau=Double.parseDouble(properties.getProperty(prefix+"mb_tau"));
		if (properties.getProperty(prefix+"mb_max_gain")!=null)          this.mb_max_gain=Double.parseDouble(properties.getProperty(prefix+"mb_max_gain"));
		if (properties.getProperty(prefix+"mb_max_gain_inter")!=null)    this.mb_max_gain_inter=Double.parseDouble(properties.getProperty(prefix+"mb_max_gain_inter"));
		
		if (properties.getProperty(prefix+"stereo_merge")!=null)         this.stereo_merge=Boolean.parseBoolean(properties.getProperty(prefix+"stereo_merge"));
		if (properties.getProperty(prefix+"stereo_gap")!=null)           this.stereo_gap=Integer.parseInt(properties.getProperty(prefix+"stereo_gap"));
		if (properties.getProperty(prefix+"stereo_intereye")!=null)      this.stereo_intereye=Double.parseDouble(properties.getProperty(prefix+"stereo_intereye"));
		if (properties.getProperty(prefix+"stereo_phone_width")!=null)   this.stereo_phone_width=Double.parseDouble(properties.getProperty(prefix+"stereo_phone_width"));
		
		if (properties.getProperty(prefix+"extra_hor_tile")!=null)       this.extra_hor_tile=Integer.parseInt(properties.getProperty(prefix+"extra_hor_tile"));
		if (properties.getProperty(prefix+"extra_vert_tile")!=null)      this.extra_vert_tile=Integer.parseInt(properties.getProperty(prefix+"extra_vert_tile"));
		if (properties.getProperty(prefix+"crop_3d")!=null)              this.crop_3d=Boolean.parseBoolean(properties.getProperty(prefix+"crop_3d"));		
		if (properties.getProperty(prefix+"sensor_mask")!=null)          this.sensor_mask=Integer.parseInt(properties.getProperty(prefix+"sensor_mask"));
		if (properties.getProperty(prefix+"merge_all")!=null)            this.merge_all=Boolean.parseBoolean(properties.getProperty(prefix+"merge_all"));		
		
		if (properties.getProperty(prefix+"video_fps")!=null)            this.video_fps=Double.parseDouble(properties.getProperty(prefix+"video_fps"));
		if (properties.getProperty(prefix+"sensor_fps")!=null)           this.sensor_fps=Double.parseDouble(properties.getProperty(prefix+"sensor_fps"));
		if (properties.getProperty(prefix+"mode_avi")!=null)             this.mode_avi=Integer.parseInt(properties.getProperty(prefix+"mode_avi"));
		if (properties.getProperty(prefix+"avi_JPEG_quality")!=null)     this.avi_JPEG_quality=Integer.parseInt(properties.getProperty(prefix+"avi_JPEG_quality"));
		if (properties.getProperty(prefix+"run_ffmpeg")!=null)           this.run_ffmpeg=Boolean.parseBoolean(properties.getProperty(prefix+"run_ffmpeg"));
		if (properties.getProperty(prefix+"video_ext")!=null)            this.video_ext=(String) properties.getProperty(prefix+"video_ext");
		if (properties.getProperty(prefix+"video_codec")!=null)          this.video_codec=(String) properties.getProperty(prefix+"video_codec");
		if (properties.getProperty(prefix+"video_crf")!=null)            this.video_crf=Integer.parseInt(properties.getProperty(prefix+"video_crf"));
		if (properties.getProperty(prefix+"remove_avi")!=null)           this.remove_avi=Boolean.parseBoolean(properties.getProperty(prefix+"remove_avi"));
		if (properties.getProperty(prefix+"video_codec_combo")!=null)    this.video_codec_combo=(String) properties.getProperty(prefix+"video_codec_combo");
		if (properties.getProperty(prefix+"video_crf_combo")!=null)      this.video_crf_combo=Integer.parseInt(properties.getProperty(prefix+"video_crf_combo"));
		if (properties.getProperty(prefix+"um_mono")!=null)              this.um_mono=Boolean.parseBoolean(properties.getProperty(prefix+"um_mono"));
		if (properties.getProperty(prefix+"um_sigma")!=null)             this.um_sigma=Double.parseDouble(properties.getProperty(prefix+"um_sigma"));
		if (properties.getProperty(prefix+"um_weight")!=null)            this.um_weight=Double.parseDouble(properties.getProperty(prefix+"um_weight"));
		if (properties.getProperty(prefix+"mono_fixed")!=null)           this.mono_fixed=Boolean.parseBoolean(properties.getProperty(prefix+"mono_fixed"));
		if (properties.getProperty(prefix+"mono_range")!=null)           this.mono_range=Double.parseDouble(properties.getProperty(prefix+"mono_range"));
		
		if (properties.getProperty(prefix+"anaglyth_en")!=null)           this.anaglyth_en=Boolean.parseBoolean(properties.getProperty(prefix+"anaglyth_en"));
		if  (properties.getProperty(prefix+"anaglyph_left") != null) {
			try {
				this.anaglyph_left = setLongColor(Long.parseLong(properties.getProperty(prefix+"anaglyph_left")));
			} catch(NumberFormatException e){
				this.anaglyph_left = anaglyph_left_default;
			}
		}
		if  (properties.getProperty(prefix+"anaglyph_right") != null) {
			try {
				this.anaglyph_right = setLongColor(Long.parseLong(properties.getProperty(prefix+"anaglyph_right")));
			} catch(NumberFormatException e){
				this.anaglyph_right = anaglyph_right_default;
			}
		}
		
		if (properties.getProperty(prefix+"annotate_color")!=null)       this.annotate_color=Boolean.parseBoolean(properties.getProperty(prefix+"annotate_color"));
		if (properties.getProperty(prefix+"annotate_mono")!=null)        this.annotate_mono=Boolean.parseBoolean(properties.getProperty(prefix+"annotate_mono"));

		if  (properties.getProperty(prefix+"annotate_color_color")!=null) {
			long lcolor_annotate = Long.parseLong(properties.getProperty(prefix+"annotate_color_color"));
			if (lcolor_annotate < 0) this.annotate_color_color = null;
			else this.annotate_color_color = setLongColor(lcolor_annotate);
		}
		if  (properties.getProperty(prefix+"annotate_color_mono")!=null) {
			long lcolor_annotate = Long.parseLong(properties.getProperty(prefix+"annotate_color_mono"));
			if (lcolor_annotate < 0) this.annotate_color_mono = null;
			else this.annotate_color_mono = setLongColor(lcolor_annotate);
		}
		if (properties.getProperty(prefix+"annotate_transparent_mono")!=null) this.annotate_transparent_mono=Boolean.parseBoolean(properties.getProperty(prefix+"annotate_transparent_mono"));
	}
	
	@Override
	public IntersceneMatchParameters clone() throws CloneNotSupportedException {
		IntersceneMatchParameters imp =     new IntersceneMatchParameters();
 		imp.ims_use                       = this.ims_use;
 		imp.ims_rebuild                   = this.ims_rebuild;
 		imp.ims_offset                    = this.ims_offset;
 		imp.gmt_plus                      = this.gmt_plus;
 		imp.ims_ortho                     = this.ims_ortho.clone();
 		imp.ims_mount_atr                 = this.ims_mount_atr.clone();
 		imp.ims_mount_xyz                 = this.ims_mount_xyz.clone();
 		imp.ims_scale_xyz                 = this.ims_scale_xyz.clone();
 		imp.ims_scale_atr                 = this.ims_scale_atr.clone();
 		imp.sfm_use                       = this.sfm_use;
 		imp.sfm_min_base                  = this.sfm_min_base;
 		imp.sfm_min_gain                  = this.sfm_min_gain;
 		imp.sfm_min_frac                  = this.sfm_min_frac;
 		imp.sfm_num_pairs 	              = this.sfm_num_pairs;
 		imp.sfp_tolerance 	              = this.sfp_tolerance;
 		imp.sfm_readjust                  = this.sfm_readjust;
 		imp.sfm_prev_frac 	              = this.sfm_prev_frac;
 		imp.sfm_same_weight               = this.sfm_same_weight;
 		imp.sfm_shrink                    = this.sfm_shrink;
 		imp.sfm_fade_sigma	              = this.sfm_fade_sigma;
 		imp.sfm_min_str1   	              = this.sfm_min_str1;
 		imp.sfm_min_str16   	          = this.sfm_min_str16;
 		imp.sfm_use_neibs 	              = this.sfm_use_neibs;
 		imp.sfm_neib_too_str1              = this.sfm_neib_too_str1;
 		imp.sfm_neib_too_str16            = this.sfm_neib_too_str16;
 		imp.sfm_neib_str1   	          = this.sfm_neib_str1;
 		imp.sfm_neib_str16                = this.sfm_neib_str16;
		imp.sfm_extrap_steps              = this.sfm_extrap_steps;
		imp.sfm_extrap_radius             = this.sfm_extrap_radius;
		imp.sfm_average_neibs             = this.sfm_average_neibs;
		imp.sfm_fill_weak                 = this.sfm_fill_weak;
		imp.sfm_extrapolate               = this.sfm_extrapolate;

		imp.center_reference              = this.center_reference;
		imp.reset_photometric             = this.reset_photometric;
		imp.force_ref_dsi                 = this.force_ref_dsi;
		imp.force_orientations            = this.force_orientations;
		imp.run_ly                        = this.run_ly;
		imp.min_num_orient                = this.min_num_orient;
		imp.min_num_interscene            = this.min_num_interscene;
		imp.calc_quat_corr                = this.calc_quat_corr;
		imp.apply_quat_corr               = this.apply_quat_corr;
		imp.generate_egomotion            = this.generate_egomotion;
		imp.generate_mapped               = this.generate_mapped;
		imp.reuse_video                   = this.reuse_video;
		imp.save_mapped_color             = this.save_mapped_color;
		imp.save_mapped_mono              = this.save_mapped_mono;
		imp.gen_avi_color                 = this.gen_avi_color;
		imp.gen_avi_mono                  = this.gen_avi_mono;
		imp.show_mapped_color             = this.show_mapped_color;
		imp.show_mapped_mono              = this.show_mapped_mono;
		
		imp.generate_raw                  = this.generate_raw;
		imp.generate_inf                  = this.generate_inf;
		imp.generate_fg                   = this.generate_fg;
		imp.generate_bg                   = this.generate_bg;
		
		imp.generate_stereo               = this.generate_stereo;
		
//		imp.stereo_bases                  = this.stereo_bases.clone();
		imp.stereo_views                  = this.stereo_views.clone();
		for (int i = 0; i < this.stereo_views.length; i++) {
			imp.stereo_views[i]           = this.stereo_views[i].clone();
		}
		imp.generate_stereo_var           = this.generate_stereo_var.clone();
		
		imp.export_images                 = this.export_images;
		imp.show_images                   = this.show_images;
		imp.show_images_bgfg              = this.show_images_bgfg;
		imp.show_images_mono              = this.show_images_mono;
		imp.export_ranges                 = this.export_ranges;
		imp.debug_ranges                  = this.debug_ranges;		
		imp.show_ranges                   = this.show_ranges;
		imp.export_ml_files               = this.export_ml_files;
		imp.show_color_nan                = this.show_color_nan;
		imp.show_mono_nan                 = this.show_mono_nan;
		imp.scale_extrap_atr              = this.scale_extrap_atr;
		imp.scale_extrap_xyz              = this.scale_extrap_xyz;
		imp.avg_len                       = this.avg_len;
		imp.min_num_scenes                = this.min_num_scenes;
		imp.max_num_scenes                = this.max_num_scenes;
		imp.blur_egomotion                = this.blur_egomotion;
		imp.range_disparity_offset        = this.range_disparity_offset;
		imp.range_min_strength            = this.range_min_strength;
		imp.range_max                     = this.range_max;
		imp.export3d                      = this.export3d;
		
		
		
		imp.scene_is_ref_test     = this.scene_is_ref_test;
		imp.render_ref            = this.render_ref;
		imp.render_scene          = this.render_scene;
		imp.toRGB                 = this.toRGB;
		imp.show_2d_correlations  = this.show_2d_correlations;
		imp.show_motion_vectors   = this.show_motion_vectors ;
		imp.debug_level           = this.debug_level;

		imp.test_ers =              this.test_ers;         
		imp.test_ers0 =             this.test_ers0;
		imp.test_ers1 =             this.test_ers1;

		imp.num_bottom                    = this.num_bottom;
		imp.num_passes                    = this.num_passes;
		imp.max_change                    = this.max_change;
		imp.min_disparity                 = this.min_disparity;
		imp.max_sym_disparity             = this.max_sym_disparity;
		imp.min_strength_lma              = this.min_strength_lma;
		imp.min_strength_replace          = this.min_strength_replace;
		imp.min_strength_blur             = this.min_strength_blur;
		imp.sigma                         = this.sigma;
		imp.num_blur                      = this.num_blur;
		imp.disparity_corr                = this.disparity_corr;
		imp.outliers_nth_fromextrem       = this.outliers_nth_fromextrem;
		imp.outliers_tolerance_absolute   = this.outliers_tolerance_absolute;
		imp.outliers_tolerance_relative   = this.outliers_tolerance_relative;
		imp.outliers_max_iter             = this.outliers_max_iter;
		imp.outliers_max_strength2        = this.outliers_max_strength2;
		imp.outliers_nth_fromextrem2      = this.outliers_nth_fromextrem2;
		imp.outliers_tolerance_absolute2  = this.outliers_tolerance_absolute2;
		imp.outliers_tolerance_relative2  = this.outliers_tolerance_relative2;
		imp.outliers_lma_max_strength     = this.outliers_lma_max_strength;
		imp.outliers_max_strength         = this.outliers_max_strength;
		imp.outliers_from_lma_max_strength= this.outliers_from_lma_max_strength;
		imp.search_radius                 = this.search_radius;
		imp.remove_no_lma_neib            = this.remove_no_lma_neib;
		imp.diff_from_lma_pos             = this.diff_from_lma_pos;
		imp.diff_from_lma_neg             = this.diff_from_lma_neg;
		imp.outliers_lma_nth_fromextrem   = this.outliers_lma_nth_fromextrem;
		imp.filter_margin                 = this.filter_margin;
		imp.weak_tolerance_absolute       = this.weak_tolerance_absolute;
		imp.weak_tolerance_relative       = this.weak_tolerance_relative;
		imp.weak_min_neibs                = this.weak_min_neibs;
		imp.strong_strength               = this.strong_strength;
		imp.weak_strength                 = this.weak_strength;
		
		imp.sky_extract =           this.sky_extract;
		imp.sky_recalc =            this.sky_recalc;
		imp.sky_highest_min =       this.sky_highest_min;
		imp.cold_frac =             this.cold_frac;
		imp.hot_frac =              this.hot_frac;
		imp.cold_scale =            this.cold_scale;
		imp.sky_seed =              this.sky_seed;
		imp.lma_seed =              this.lma_seed;
		imp.seed_temp =             this.seed_temp;
		imp.sky_shrink =            this.sky_shrink;
		imp.seed_rows =             this.seed_rows;
		imp.max_disparity =         this.max_disparity;
		imp.max_disparity_strength= this.max_disparity_strength;
		imp.sky_lim =               this.sky_lim;
		imp.lim_temp =              this.lim_temp;
		imp.sky_expand_extra =      this.sky_expand_extra;
		imp.sky_bottleneck =        this.sky_bottleneck;
		imp.sky_reexpand_extra =    this.sky_reexpand_extra;
		imp.min_strength =          this.min_strength;
		imp.lowest_sky_row =        this.lowest_sky_row;
		imp.sky_bottom_override =   this.sky_bottom_override;
		imp.sky_override_shrink =   this.sky_override_shrink;

		imp.clouds_en =             this.clouds_en;
		imp.clouds_fom =            this.clouds_fom;
		imp.clouds_spread =         this.clouds_spread;
		imp.clouds_disparity =      this.clouds_disparity;
		imp.clouds_weak =           this.clouds_weak;
		imp.clouds_strength =       this.clouds_strength;
		imp.clouds_not_strength =   this.clouds_not_strength;
		imp.clouds_strong =         this.clouds_strong;
		
		imp.wall_en =               this.wall_en;
		imp.wall_dflt =             this.wall_dflt;
		imp.wall_str =              this.wall_str;
		imp.wall_far =              this.wall_far;
		imp.wall_near =             this.wall_near;
		
		imp.treeline_en =     	    this.treeline_en;
		imp.treeline_wide =   	    this.treeline_wide;
		imp.treeline_height =       this.treeline_height;
		imp.treeline_width =        this.treeline_width;  
		imp.treeline_lim_high =     this.treeline_lim_high;
		imp.treeline_str =          this.treeline_str;    
		imp.treeline_far =          this.treeline_far;    
		imp.treeline_near =         this.treeline_near;  
		imp.treeline_fg_str =       this.treeline_fg_str; 
		imp.treeline_fg_far = 	    this.treeline_fg_far; 
		imp.treeline_fg_near =      this.treeline_fg_near;		
		
		imp.indoors_en =            this.indoors_en;
		imp.indoors_str =           this.indoors_str;
		imp.indoors_disp =          this.indoors_disp;
		imp.indoors_min_out =       this.indoors_min_out;
		
		imp.disp_boost_min =        this.disp_boost_min;
		imp.disp_boost_diff =       this.disp_boost_diff;
		imp.disp_boost_neibs =      this.disp_boost_neibs;
		imp.disp_boost_amount =     this.disp_boost_amount;
		imp.scale_combo_strength =  this.scale_combo_strength;
		imp.save_debug_images =     this.save_debug_images;
		
		imp.margin                = this.margin;
		imp.sensor_mask_inter     = this.sensor_mask_inter;
		imp.use_partial           = this.use_partial;
		imp.run_poly              = this.run_poly;
		imp.centroid_radius       = this.centroid_radius;
		imp.n_recenter            = this.n_recenter;
		imp.min_ref_str           = this.min_ref_str;
		imp.min_ref_frac          = this.min_ref_frac;
				
		imp.td_weight             = this.td_weight;
		imp.td_neib_weight        = this.td_neib_weight;
		imp.pd_weight             = this.pd_weight;
		imp.td_nopd_only          = this.td_nopd_only;
		imp.neib_notd_only        = this.neib_notd_only;
		
		imp.use_neibs             = this.use_neibs;                                                                
		imp.neibs_nofpn_only      = this.neibs_nofpn_only;
		imp.neibs_nofpn_init      = this.neibs_nofpn_init;
		imp.redo_both             = this.redo_both;
		imp.min_num_neibs         = this.min_num_neibs;
		imp.scale_neibs_pd        = this.scale_neibs_pd;
		imp.scale_neibs_td        = this.scale_neibs_td;
		imp.scale_avg_weight      = this.scale_avg_weight;
		
		imp.min_str               = this.min_str;
		imp.min_str_fpn           = this.min_str_fpn;
		imp.min_str_sum           = this.min_str_sum;
		imp.min_str_sum_fpn       = this.min_str_sum_fpn;
		imp.min_str_neib          = this.min_str_neib;
		imp.min_str_neib_fpn      = this.min_str_neib_fpn;
		
		
		imp.min_neibs             = this.min_neibs;
		imp.weight_zero_neibs     = this.weight_zero_neibs;
		imp.half_disparity        = this.half_disparity;
		imp.half_avg_diff         = this.half_avg_diff;
		imp.use_combo_relaible    = this.use_combo_relaible;
		imp.ref_need_lma          = this.ref_need_lma;
		imp.ref_need_lma_combo    = this.ref_need_lma_combo;
		imp.pix_step =              this.pix_step;
		imp.search_rad =            this.search_rad;
		imp.maybe_sum =             this.maybe_sum;
		imp.sure_sum =              this.sure_sum;
		imp.maybe_avg =             this.maybe_avg;
		imp.sure_avg =              this.sure_avg;
		imp.max_search_rms =        this.max_search_rms;
		imp.maybe_fom =             this.maybe_fom;
		imp.sure_fom =              this.sure_fom;
		imp.treat_serch_fpn       = this.treat_serch_fpn;
		imp.use_combo_dsi         = this.use_combo_dsi;
		imp.use_lma_dsi           = this.use_lma_dsi;
		imp.fpn_remove            = this.fpn_remove;
		imp.fpn_max_offset        = this.fpn_max_offset;
		imp.fpn_radius            = this.fpn_radius;
		imp.fpn_ignore_border     = this.fpn_ignore_border;
		
		imp.min_offset            = this.min_offset;
		imp.max_rel_offset        = this.max_rel_offset;
		imp.max_roll_deg          = this.max_roll_deg;
		imp.max_zoom_diff         = this.max_zoom_diff;
		imp.fpn_skip              = this.fpn_skip;
		imp.fpn_rematch           = this.fpn_rematch;
		
		imp.mov_en                = this.mov_en;
		imp.mov_sigma             = this.mov_sigma;
		imp.mov_max_std           = this.mov_max_std;
		imp.mov_thresh_rel        = this.mov_thresh_rel;
		imp.mov_thresh_abs        = this.mov_thresh_abs;
		imp.mov_clust_max         = this.mov_clust_max;
		imp.mov_grow              = this.mov_grow;
		imp.mov_max_len           = this.mov_max_len;
		imp.mov_show              = this.mov_show;
		imp.mov_debug_level       = this.mov_debug_level;
		imp.adjust_atr[0]         = this.adjust_atr[0];
		imp.adjust_atr[1]         = this.adjust_atr[1];
		imp.adjust_atr[2]         = this.adjust_atr[2];
		imp.adjust_xyz[0]         = this.adjust_xyz[0];
		imp.adjust_xyz[1]         = this.adjust_xyz[1];
		imp.adjust_xyz[2]         = this.adjust_xyz[2];
		imp.exit_change_atr       = this.exit_change_atr;
		imp.exit_change_xyz       = this.exit_change_xyz;
		imp.max_cycles            = this.max_cycles;
		imp.max_LMA               = this.max_LMA;
		imp.max_rms               = this.max_rms;
		
		imp.eq_en =                 this.eq_en;
		imp.eq_stride_hor =         this.eq_stride_hor;
		imp.eq_stride_vert =        this.eq_stride_vert;
		imp.eq_min_stile_weight =   this.eq_min_stile_weight;
		imp.eq_min_stile_number =   this.eq_min_stile_number;
		imp.eq_min_stile_fraction = this.eq_min_stile_fraction;
		imp.eq_min_disparity =      this.eq_min_disparity;
		imp.eq_max_disparity =      this.eq_max_disparity;
		imp.eq_weight_add =         this.eq_weight_add;
		imp.eq_weight_scale =       this.eq_weight_scale;
		imp.eq_level =              this.eq_level;
		
		imp.mb_en =                 this.mb_en;
		imp.mb_tau =                this.mb_tau;
		imp.mb_max_gain =           this.mb_max_gain;
		imp.mb_max_gain_inter =     this.mb_max_gain_inter;

		imp.stereo_merge          = this.stereo_merge;
		imp.stereo_gap            = this.stereo_gap;
		imp.stereo_intereye       = this. stereo_intereye;
		imp.stereo_phone_width    = this. stereo_phone_width;
		
		imp.extra_hor_tile        = this.extra_hor_tile;
		imp.extra_vert_tile       = this.extra_vert_tile;
		imp.crop_3d               = this.crop_3d;
		imp.sensor_mask           = this.sensor_mask;
		imp.merge_all             = this.merge_all;

		imp.video_fps =             this. video_fps;
		imp.sensor_fps =            this. sensor_fps;
		imp.mode_avi =              this. mode_avi;
		imp.avi_JPEG_quality =      this. avi_JPEG_quality;
		imp.run_ffmpeg =            this. run_ffmpeg;
		imp.video_ext =             this. video_ext;
		imp.video_codec =           this. video_codec;
		imp.video_crf =             this. video_crf;
		imp.remove_avi =            this. remove_avi;
		imp.video_codec_combo =     this. video_codec_combo;
		imp.video_crf_combo =       this. video_crf_combo;
		
		imp.um_mono =               this. um_mono;
		imp.um_sigma =              this. um_sigma;
		imp.um_weight =             this. um_weight;
		imp.mono_fixed =            this. mono_fixed;
		imp.mono_range =            this. mono_range;
		
		imp.anaglyth_en =           this. anaglyth_en;
		imp.anaglyph_left =         this. anaglyph_left;
		imp.anaglyph_right =        this. anaglyph_right;
		
		imp.annotate_color =        this. annotate_color;
		imp.annotate_mono =         this. annotate_mono;
		imp.annotate_color_color =  this. annotate_color_color;
		imp.annotate_color_mono =   this. annotate_color_mono;
		imp.annotate_transparent_mono = this. annotate_transparent_mono;
		
		
		return imp;
	}
	public static long getLongColor(Color color) {
		return ((long) color.getRGB()) & 0xffffffffL;
	}
	public static Color setLongColor(long lcolor) {
		if (lcolor < (1 << 24)) { // no alpha
			return new Color((int) lcolor);
		} else { // has alpha, may or may not fit into int
			if (lcolor > Integer.MAX_VALUE) {
				lcolor -= (1L << 32);
			}
			return new Color((int) lcolor, true);
		}
	}
	
	public void orderStereoViews(){
		boolean ordered;
		do {
			ordered=true;
			for (int i = 0; i < (stereo_views.length - 1); i++) {
				if (stereo_views[i+1][0] > stereo_views[i][0]) {
					continue;
				}
				if (    (stereo_views[i+1][0] == stereo_views[i][0]) &&
						(stereo_views[i+1][1] >  stereo_views[i][1])) {
					continue;
				}
				if (    (stereo_views[i+1][0] == stereo_views[i][0]) &&
						(stereo_views[i+1][1] == stereo_views[i][1]) &&
						(stereo_views[i+1][2] >  stereo_views[i][2])) {
					continue;
				}
				if (    (stereo_views[i+1][0] == stereo_views[i][0]) &&
						(stereo_views[i+1][1] == stereo_views[i][1]) &&
						(stereo_views[i+1][2] == stereo_views[i][2])) {
					// all same values - remove extra
					generate_stereo_var[i] |= generate_stereo_var[i+1];
					for (int j = i+1; j < (stereo_views.length - 1); j++) {
						generate_stereo_var[j] = generate_stereo_var[j+1];
						stereo_views[j] = stereo_views[j + 1]; 
					}
					ordered = false;
					break; // next while
				}
				boolean en = generate_stereo_var[i+1];
				generate_stereo_var[i+1] = generate_stereo_var[i];
				generate_stereo_var[i] = en;
				double [] view =  stereo_views[i+1];
				stereo_views[i+1] = stereo_views[i];
				stereo_views[i] = view;
				ordered = false;
			}

		} while (!ordered);
		return;
	}

	public void addStereoView(String stereo_view_string, boolean en) {
		double[] stereo_view = StringToDoubles(stereo_view_string,3);
		if (stereo_view != null) {
			addStereoView(stereo_view, en);
		}
	}
	
	public void addStereoView(double[] stereo_view, boolean en) {
		double [][]  views = new double [stereo_views.length + 1][];
		boolean [] ens = new boolean [stereo_views.length + 1];
		views[0] = stereo_view;
		ens[0] = en;
		System.arraycopy(stereo_views, 0, views, 1,      stereo_views.length);
		System.arraycopy(generate_stereo_var, 0, ens, 1, stereo_views.length);
		stereo_views = views;
		generate_stereo_var = ens;
		orderStereoViews();
	}
	
	public void removeStereoView(int indx) {
		if ((indx >=0) && (indx <stereo_views.length)) {
			double [][] views = new double [stereo_views.length - 1][];
			boolean [] ens = new boolean [stereo_views.length - 1];
			if (indx > 0) {
				System.arraycopy(stereo_views,        0, views, 0, indx);
				System.arraycopy(generate_stereo_var, 0, ens,   0, indx);
			}
			if (indx < (stereo_views.length - 1)) {
				System.arraycopy(stereo_views,        indx+1, views, indx, stereo_views.length - indx - 1);
				System.arraycopy(generate_stereo_var, indx+1, ens,   indx, stereo_views.length - indx - 1);
			}
			stereo_views = views;
			generate_stereo_var = ens;
		}
	}

	public static String doublesToString(double [] data) {
		return doublesToString(data, null);
	}
	public static String doublesToString(double [] data, String fmt) {
//		if ((fmt == null) || (fmt.trim().length()==0)) {
//			fmt = "%.0f";
//		}
		String s = "";
		for (int i = 0; i < data.length; i++) {
			if (fmt==null) {
				s += data[i]; // unformatted
			} else { 
				s+=String.format(fmt,data[i]);
			}
			if (i < (data.length - 1)) {
				s+= ", ";
			}
		}
		return s;
	}
	
	public static double [] StringToDoubles(String s, int len) {
		StringTokenizer st = new StringTokenizer(s, " \t\n\r\f,");
		if (st.countTokens() == 0) {
			return null;
		}
		if (len <= 0) {
			len = st.countTokens();
		}
		double [] data = new double [len]; 
		int i = 0;
		while (st.hasMoreTokens() && (i < len)) {
			double d = 0;
			try {
				d = Double.parseDouble(st.nextToken());
			} catch(NumberFormatException e){
				d = 0;
			}

			data[i++] = d;
		}
		return data;
	}
	
	public static float [] StringToFloats(String s, int len) {
		StringTokenizer st = new StringTokenizer(s, " \t\n\r\f,");
		if (st.countTokens() == 0) {
			return null;
		}
		if (len <= 0) {
			len = st.countTokens();
		}
		float [] data = new float [len]; 
		int i = 0;
		while (st.hasMoreTokens() && (i < len)) {
			double d = 0;
			try {
				d = Double.parseDouble(st.nextToken());
			} catch(NumberFormatException e){
				d = 0;
			}

			data[i++] = (float) d;
		}
		return data;
	}
		
	public static String floatsToString(float [] data) {
		return floatsToString(data, null);
	}
	
	public static String floatsToString(float [] data, String fmt) {
//		if ((fmt == null) || (fmt.trim().length()==0)) {
//			fmt = "%.0f";
//		}
		String s = "";
		for (int i = 0; i < data.length; i++) {
			if (fmt==null) {
				s += data[i]; // unformatted
			} else { 
				s+=String.format(fmt,data[i]);
			}
			if (i < (data.length - 1)) {
				s+= ", ";
			}
		}
		return s;
	}
	
	
}
