#!/usr/bin/env bash
set -euo pipefail

BASE_URL="${EYESIS_MCP_BASE_URL:-http://127.0.0.1:48888}"
TIMEOUT="${EYESIS_MCP_TIMEOUT_SEC:-10}"

usage() {
  cat <<'USAGE'
Usage:
  scripts/mcp_http.sh status
  scripts/mcp_http.sh dialog
  scripts/mcp_http.sh button --label TEXT
  scripts/mcp_http.sh set --label TEXT --value TEXT [--id UUID]
  scripts/mcp_http.sh submit --ok 0|1 [--id UUID]
  scripts/mcp_http.sh interrupt --confirm 0|1 [--asap 0|1]
  scripts/mcp_http.sh confirm-stop --stop 0|1

Options:
  --base URL         MCP base URL (default: $EYESIS_MCP_BASE_URL or http://127.0.0.1:48888)
  --timeout SEC      curl max-time seconds (default: $EYESIS_MCP_TIMEOUT_SEC or 10)
  -h, --help         Show help

Examples:
  scripts/mcp_http.sh status
  scripts/mcp_http.sh button --label "Restore"
  scripts/mcp_http.sh set --label "Selected path" --value "/path/file.corr-xml"
  scripts/mcp_http.sh submit --ok 1
USAGE
}

cmd_get() {
  local path="$1"
  curl -sS --fail --max-time "${TIMEOUT}" "${BASE_URL}${path}"
}

cmd_post() {
  local path="$1"
  shift
  curl -sS --fail --max-time "${TIMEOUT}" -X POST "$@" "${BASE_URL}${path}"
}

require_nonempty() {
  local name="$1"
  local value="$2"
  if [[ -z "${value}" ]]; then
    echo "Missing required ${name}" >&2
    exit 2
  fi
}

if [[ $# -lt 1 ]]; then
  usage
  exit 2
fi

subcmd="$1"
shift

# parse optional common flags before subcommand-specific flags
while [[ $# -gt 0 ]]; do
  case "$1" in
    --base)
      BASE_URL="$2"
      shift 2
      ;;
    --timeout)
      TIMEOUT="$2"
      shift 2
      ;;
    --)
      shift
      break
      ;;
    *)
      break
      ;;
  esac
done

case "${subcmd}" in
  status)
    cmd_get "/mcp/status"
    ;;
  dialog)
    cmd_get "/mcp/dialog"
    ;;
  button)
    label=""
    while [[ $# -gt 0 ]]; do
      case "$1" in
        --label)
          label="$2"
          shift 2
          ;;
        *)
          echo "Unknown option for button: $1" >&2
          exit 2
          ;;
      esac
    done
    require_nonempty "--label" "${label}"
    cmd_post "/mcp/button" --data-urlencode "label=${label}"
    ;;
  set)
    label=""
    value=""
    id=""
    while [[ $# -gt 0 ]]; do
      case "$1" in
        --label)
          label="$2"
          shift 2
          ;;
        --value)
          value="$2"
          shift 2
          ;;
        --id)
          id="$2"
          shift 2
          ;;
        *)
          echo "Unknown option for set: $1" >&2
          exit 2
          ;;
      esac
    done
    require_nonempty "--label" "${label}"
    require_nonempty "--value" "${value}"
    if [[ -n "${id}" ]]; then
      cmd_post "/mcp/dialog/values" --data-urlencode "id=${id}" --data-urlencode "label=${label}" --data-urlencode "value=${value}"
    else
      cmd_post "/mcp/dialog/values" --data-urlencode "label=${label}" --data-urlencode "value=${value}"
    fi
    ;;
  submit)
    ok=""
    id=""
    while [[ $# -gt 0 ]]; do
      case "$1" in
        --ok)
          ok="$2"
          shift 2
          ;;
        --id)
          id="$2"
          shift 2
          ;;
        *)
          echo "Unknown option for submit: $1" >&2
          exit 2
          ;;
      esac
    done
    require_nonempty "--ok" "${ok}"
    if [[ -n "${id}" ]]; then
      cmd_post "/mcp/dialog/submit" --data-urlencode "id=${id}" --data-urlencode "ok=${ok}"
    else
      cmd_post "/mcp/dialog/submit" --data-urlencode "ok=${ok}"
    fi
    ;;
  interrupt)
    confirm=""
    asap="0"
    while [[ $# -gt 0 ]]; do
      case "$1" in
        --confirm)
          confirm="$2"
          shift 2
          ;;
        --asap)
          asap="$2"
          shift 2
          ;;
        *)
          echo "Unknown option for interrupt: $1" >&2
          exit 2
          ;;
      esac
    done
    require_nonempty "--confirm" "${confirm}"
    cmd_post "/mcp/interrupt" --data-urlencode "confirm=${confirm}" --data-urlencode "asap=${asap}"
    ;;
  confirm-stop)
    stop=""
    while [[ $# -gt 0 ]]; do
      case "$1" in
        --stop)
          stop="$2"
          shift 2
          ;;
        *)
          echo "Unknown option for confirm-stop: $1" >&2
          exit 2
          ;;
      esac
    done
    require_nonempty "--stop" "${stop}"
    cmd_post "/mcp/interrupt/confirm" --data-urlencode "stop=${stop}"
    ;;
  -h|--help)
    usage
    ;;
  *)
    echo "Unknown subcommand: ${subcmd}" >&2
    usage
    exit 2
    ;;
esac
