package com.elphel.imagej.mcp;

import java.io.IOException;
import java.io.OutputStream;
import java.net.InetSocketAddress;
import java.net.URI;
import java.net.URLDecoder;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import com.elphel.imagej.correction.Eyesis_Correction;
import com.elphel.imagej.mcp.McpFsAccess;
import com.sun.net.httpserver.Headers;
import com.sun.net.httpserver.HttpExchange;
import com.sun.net.httpserver.HttpHandler;
import com.sun.net.httpserver.HttpServer;

public class McpServer {
    private static McpServer INSTANCE;

    private final Eyesis_Correction owner;
    private final int port;
    private HttpServer server;

    public static synchronized McpServer startIfNeeded(Eyesis_Correction owner, int port) {
        if (INSTANCE != null) {
            return INSTANCE;
        }
        McpServer instance = new McpServer(owner, port);
        instance.start();
        INSTANCE = instance;
        return instance;
    }

    private McpServer(Eyesis_Correction owner, int port) {
        this.owner = owner;
        this.port = port;
    }

    private void start() {
        try {
            server = HttpServer.create(new InetSocketAddress("127.0.0.1", port), 0);
        } catch (IOException e) {
            System.out.println("MCP: failed to start HTTP server on port " + port + ": " + e.getMessage());
            return;
        }
        server.createContext("/mcp/status", new StatusHandler());
        server.createContext("/mcp/dialog", new DialogHandler());
        server.createContext("/mcp/dialog/values", new DialogValuesHandler());
        server.createContext("/mcp/dialog/submit", new DialogSubmitHandler());
        server.createContext("/mcp/button", new ButtonHandler());
        server.createContext("/mcp/interrupt", new InterruptHandler());
        server.createContext("/mcp/interrupt/confirm", new InterruptConfirmHandler());
        server.createContext("/mcp/fs/roots", new FsRootsHandler());
        server.createContext("/mcp/fs/list", new FsListHandler());
        server.createContext("/mcp/fs/read", new FsReadHandler());
        server.createContext("/mcp/fs/head", new FsHeadHandler());
        server.createContext("/mcp/fs/tail", new FsTailHandler());
        server.createContext("/mcp/fs/glob", new FsGlobHandler());
        server.setExecutor(null);
        server.start();
        if (Eyesis_Correction.MCP_DEBUG_LEVEL >= Eyesis_Correction.MINIMAL_DEBUG_MCP) {
            System.out.println("MCP: server started on http://127.0.0.1:" + port);
        }
    }

    private class StatusHandler implements HttpHandler {
        @Override
        public void handle(HttpExchange exchange) throws IOException {
            String response = buildStatusJson();
            sendJson(exchange, 200, response);
        }
    }

    private class DialogHandler implements HttpHandler {
        @Override
        public void handle(HttpExchange exchange) throws IOException {
            String response = buildDialogJson();
            sendJson(exchange, 200, response);
        }
    }

    private class DialogValuesHandler implements HttpHandler {
        @Override
        public void handle(HttpExchange exchange) throws IOException {
            if (!"POST".equalsIgnoreCase(exchange.getRequestMethod())) {
                sendJson(exchange, 405, "{\"ok\":false,\"error\":\"POST required\"}");
                return;
            }
            Map<String, String> params = parseParams(exchange);
            String id = params.get("id");
            String label = params.get("label");
            String value = params.get("value");
            if (label == null) {
                sendJson(exchange, 400, "{\"ok\":false,\"error\":\"Missing label\"}");
                return;
            }
            boolean applied = (id == null) ? McpDialogRegistry.setValue(label, value) : McpDialogRegistry.setValueForId(id, label, value);
            if (applied) {
                if (Eyesis_Correction.MCP_DEBUG_LEVEL >= Eyesis_Correction.MINIMAL_DEBUG_MCP) {
                    System.out.println("MCP: dialog value label=\"" + label + "\"");
                }
                sendJson(exchange, 200, "{\"ok\":true}");
            } else {
                sendJson(exchange, 409, "{\"ok\":false,\"error\":\"No active dialog or id mismatch\"}");
            }
        }
    }

    private class DialogSubmitHandler implements HttpHandler {
        @Override
        public void handle(HttpExchange exchange) throws IOException {
            if (!"POST".equalsIgnoreCase(exchange.getRequestMethod())) {
                sendJson(exchange, 405, "{\"ok\":false,\"error\":\"POST required\"}");
                return;
            }
            Map<String, String> params = parseParams(exchange);
            String id = params.get("id");
            boolean ok = parseBool(params.get("ok"), true);
            boolean applied = (id == null) ? McpDialogRegistry.submitCurrent(ok) : McpDialogRegistry.submitForId(id, ok);
            if (applied && Eyesis_Correction.MCP_DEBUG_LEVEL >= Eyesis_Correction.MINIMAL_DEBUG_MCP) {
                System.out.println("MCP: dialog submit ok=" + ok);
            }
            sendJson(exchange, 200, applied ? "{\"ok\":true}" : "{\"ok\":false,\"error\":\"No active dialog\"}");
        }
    }

    private class ButtonHandler implements HttpHandler {
        @Override
        public void handle(HttpExchange exchange) throws IOException {
            if (!"POST".equalsIgnoreCase(exchange.getRequestMethod())) {
                sendJson(exchange, 405, "{\"ok\":false,\"error\":\"POST required\"}");
                return;
            }
            Map<String, String> params = parseParams(exchange);
            String label = params.get("label");
            if (label == null) {
                sendJson(exchange, 400, "{\"ok\":false,\"error\":\"Missing label\"}");
                return;
            }
            owner.triggerCommand(label, true);
            if (Eyesis_Correction.MCP_DEBUG_LEVEL >= Eyesis_Correction.MINIMAL_DEBUG_MCP) {
                System.out.println("MCP: button \"" + label + "\"");
            }
            sendJson(exchange, 200, "{\"ok\":true}");
        }
    }

    // codex 2026-01-27: request interrupt via MCP
    private class InterruptHandler implements HttpHandler {
        @Override
        public void handle(HttpExchange exchange) throws IOException {
            if (!"POST".equalsIgnoreCase(exchange.getRequestMethod())) {
                sendJson(exchange, 405, "{\"ok\":false,\"error\":\"POST required\"}");
                return;
            }
            Map<String, String> params = parseParams(exchange);
            boolean confirm = parseBool(params.get("confirm"), true);
            boolean asap = parseBool(params.get("asap"), false);
            owner.requestSyncStop(confirm, asap);
            sendJson(exchange, 200, "{\"ok\":true}");
        }
    }

    // codex 2026-01-27: confirm or cancel interrupt via MCP
    private class InterruptConfirmHandler implements HttpHandler {
        @Override
        public void handle(HttpExchange exchange) throws IOException {
            if (!"POST".equalsIgnoreCase(exchange.getRequestMethod())) {
                sendJson(exchange, 405, "{\"ok\":false,\"error\":\"POST required\"}");
                return;
            }
            Map<String, String> params = parseParams(exchange);
            boolean stop = parseBool(params.get("stop"), true);
            boolean applied = owner.confirmSyncStop(stop);
            sendJson(exchange, 200, applied ? "{\"ok\":true}" : "{\"ok\":false,\"error\":\"No pending confirm\"}");
        }
    }

    // codex 2026-01-28: list allowed roots for MCP file access
    private class FsRootsHandler implements HttpHandler {
        @Override
        public void handle(HttpExchange exchange) throws IOException {
            List<java.nio.file.Path> roots = McpFsAccess.getAllowedRoots();
            StringBuilder sb = new StringBuilder();
            sb.append("{\"ok\":true,\"roots\":[");
            for (int i = 0; i < roots.size(); i++) {
                if (i > 0) {
                    sb.append(",");
                }
                sb.append("\"").append(jsonEscape(roots.get(i).toString())).append("\"");
            }
            sb.append("]}");
            sendJson(exchange, 200, sb.toString());
        }
    }

    private class FsListHandler implements HttpHandler {
        @Override
        public void handle(HttpExchange exchange) throws IOException {
            Map<String, String> params = parseParams(exchange);
            String pathStr = params.get("path");
            if (pathStr == null) {
                sendJson(exchange, 400, "{\"ok\":false,\"error\":\"Missing path\"}");
                return;
            }
            java.nio.file.Path path = McpFsAccess.normalizePath(pathStr);
            if (path == null || !McpFsAccess.isAllowed(path)) {
                sendJson(exchange, 403, "{\"ok\":false,\"error\":\"Path not allowed\"}");
                return;
            }
            if (!java.nio.file.Files.isDirectory(path)) {
                sendJson(exchange, 400, "{\"ok\":false,\"error\":\"Not a directory\"}");
                return;
            }
            int max = parseInt(params.get("max"), 1000);
            StringBuilder sb = new StringBuilder();
            sb.append("{\"ok\":true,\"entries\":[");
            int count = 0;
            try (java.util.stream.Stream<java.nio.file.Path> stream = java.nio.file.Files.list(path)) {
                java.util.Iterator<java.nio.file.Path> it = stream.iterator();
                while (it.hasNext()) {
                    if (count >= max) {
                        break;
                    }
                    java.nio.file.Path entry = it.next();
                    if (count > 0) {
                        sb.append(",");
                    }
                    sb.append("{");
                    sb.append("\"name\":\"").append(jsonEscape(entry.getFileName().toString())).append("\"");
                    sb.append(",\"path\":\"").append(jsonEscape(entry.toString())).append("\"");
                    boolean isDir = java.nio.file.Files.isDirectory(entry);
                    sb.append(",\"type\":\"").append(isDir ? "dir" : "file").append("\"");
                    if (!isDir) {
                        try {
                            sb.append(",\"size\":").append(java.nio.file.Files.size(entry));
                        } catch (IOException e) {
                            sb.append(",\"size\":-1");
                        }
                    }
                    try {
                        sb.append(",\"mtime\":").append(java.nio.file.Files.getLastModifiedTime(entry).toMillis());
                    } catch (IOException e) {
                        sb.append(",\"mtime\":-1");
                    }
                    sb.append("}");
                    count++;
                }
            }
            sb.append("]}");
            sendJson(exchange, 200, sb.toString());
        }
    }

    private class FsReadHandler implements HttpHandler {
        @Override
        public void handle(HttpExchange exchange) throws IOException {
            Map<String, String> params = parseParams(exchange);
            String pathStr = params.get("path");
            if (pathStr == null) {
                sendJson(exchange, 400, "{\"ok\":false,\"error\":\"Missing path\"}");
                return;
            }
            java.nio.file.Path path = McpFsAccess.normalizePath(pathStr);
            if (path == null || !McpFsAccess.isAllowed(path)) {
                sendJson(exchange, 403, "{\"ok\":false,\"error\":\"Path not allowed\"}");
                return;
            }
            if (!java.nio.file.Files.isRegularFile(path)) {
                sendJson(exchange, 400, "{\"ok\":false,\"error\":\"Not a file\"}");
                return;
            }
            long offset = parseLong(params.get("offset"), 0L);
            int maxBytes = parseInt(params.get("maxBytes"), 262144);
            if (maxBytes < 1) {
                maxBytes = 1;
            }
            byte[] data;
            try (java.io.RandomAccessFile raf = new java.io.RandomAccessFile(path.toFile(), "r")) {
                if (offset > 0) {
                    raf.seek(offset);
                }
                int len = (int) Math.min(maxBytes, raf.length() - raf.getFilePointer());
                if (len < 0) {
                    len = 0;
                }
                data = new byte[len];
                int read = raf.read(data);
                if (read < len) {
                    data = java.util.Arrays.copyOf(data, Math.max(read, 0));
                }
            }
            String text = new String(data, StandardCharsets.UTF_8);
            StringBuilder sb = new StringBuilder();
            sb.append("{\"ok\":true,\"bytes\":").append(data.length).append(",\"data\":\"");
            sb.append(jsonEscape(text)).append("\"}");
            sendJson(exchange, 200, sb.toString());
        }
    }

    private class FsHeadHandler implements HttpHandler {
        @Override
        public void handle(HttpExchange exchange) throws IOException {
            handleHeadTail(exchange, true);
        }
    }

    private class FsTailHandler implements HttpHandler {
        @Override
        public void handle(HttpExchange exchange) throws IOException {
            handleHeadTail(exchange, false);
        }
    }

    private void handleHeadTail(HttpExchange exchange, boolean head) throws IOException {
        Map<String, String> params = parseParams(exchange);
        String pathStr = params.get("path");
        if (pathStr == null) {
            sendJson(exchange, 400, "{\"ok\":false,\"error\":\"Missing path\"}");
            return;
        }
        java.nio.file.Path path = McpFsAccess.normalizePath(pathStr);
        if (path == null || !McpFsAccess.isAllowed(path)) {
            sendJson(exchange, 403, "{\"ok\":false,\"error\":\"Path not allowed\"}");
            return;
        }
        if (!java.nio.file.Files.isRegularFile(path)) {
            sendJson(exchange, 400, "{\"ok\":false,\"error\":\"Not a file\"}");
            return;
        }
        int lines = parseInt(params.get("lines"), 200);
        if (lines < 1) {
            lines = 1;
        }
        List<String> result = new ArrayList<String>();
        if (head) {
            try (java.io.BufferedReader reader = java.nio.file.Files.newBufferedReader(path, StandardCharsets.UTF_8)) {
                String line;
                while ((line = reader.readLine()) != null && result.size() < lines) {
                    result.add(line);
                }
            }
        } else {
            java.util.ArrayDeque<String> deque = new java.util.ArrayDeque<String>(lines);
            try (java.io.BufferedReader reader = java.nio.file.Files.newBufferedReader(path, StandardCharsets.UTF_8)) {
                String line;
                while ((line = reader.readLine()) != null) {
                    if (deque.size() >= lines) {
                        deque.pollFirst();
                    }
                    deque.addLast(line);
                }
            }
            result.addAll(deque);
        }
        StringBuilder sb = new StringBuilder();
        sb.append("{\"ok\":true,\"lines\":[");
        for (int i = 0; i < result.size(); i++) {
            if (i > 0) {
                sb.append(",");
            }
            sb.append("\"").append(jsonEscape(result.get(i))).append("\"");
        }
        sb.append("]}");
        sendJson(exchange, 200, sb.toString());
    }

    private class FsGlobHandler implements HttpHandler {
        @Override
        public void handle(HttpExchange exchange) throws IOException {
            Map<String, String> params = parseParams(exchange);
            String pattern = params.get("pattern");
            if (pattern == null) {
                sendJson(exchange, 400, "{\"ok\":false,\"error\":\"Missing pattern\"}");
                return;
            }
            int max = parseInt(params.get("max"), 1000);
            int maxDepth = parseInt(params.get("maxDepth"), 6);
            java.nio.file.FileSystem fs = java.nio.file.FileSystems.getDefault();
            java.nio.file.PathMatcher matcher = fs.getPathMatcher("glob:" + pattern);
            List<java.nio.file.Path> roots = McpFsAccess.getAllowedRoots();
            StringBuilder sb = new StringBuilder();
            sb.append("{\"ok\":true,\"matches\":[");
            int count = 0;
            for (java.nio.file.Path root : roots) {
                if (count >= max) {
                    break;
                }
                if (!java.nio.file.Files.exists(root)) {
                    continue;
                }
                try (java.util.stream.Stream<java.nio.file.Path> stream = java.nio.file.Files.walk(root, maxDepth)) {
                    java.util.Iterator<java.nio.file.Path> it = stream.iterator();
                    while (it.hasNext()) {
                        if (count >= max) {
                            break;
                        }
                        java.nio.file.Path path = it.next();
                        boolean matched = matcher.matches(path);
                        if (!matched) {
                            try {
                                java.nio.file.Path rel = root.relativize(path);
                                matched = matcher.matches(rel);
                            } catch (IllegalArgumentException e) {
                                matched = false;
                            }
                        }
                        if (matched) {
                            if (count > 0) {
                                sb.append(",");
                            }
                            sb.append("\"").append(jsonEscape(path.toString())).append("\"");
                            count++;
                        }
                    }
                }
            }
            sb.append("]}");
            sendJson(exchange, 200, sb.toString());
        }
    }

    private String buildStatusJson() {
        int stopRequested = owner.getSyncStopRequested();
        StringBuilder sb = new StringBuilder();
        sb.append("{");
        sb.append("\"running\":").append(owner.isSyncRunning());
        sb.append(",\"stopRequested\":").append(stopRequested);
        sb.append(",\"confirmPending\":").append(owner.isSyncConfirmPending());
        sb.append(",\"confirmConvenient\":").append(owner.isSyncConfirmConvenient());
        sb.append(",\"buttonLabel\":\"").append(jsonEscape(owner.getSyncButtonLabel())).append("\"");
        sb.append("}");
        return sb.toString();
    }

    private String buildDialogJson() {
        McpDialogSession session = McpDialogRegistry.getCurrent();
        if (session == null) {
            return "{\"ok\":true,\"dialog\":null}";
        }
        StringBuilder sb = new StringBuilder();
        sb.append("{\"ok\":true,\"dialog\":{");
        sb.append("\"id\":\"").append(jsonEscape(session.getId())).append("\",");
        sb.append("\"title\":\"").append(jsonEscape(session.getTitle())).append("\",");
        sb.append("\"fields\":[");
        List<McpDialogField> fields = session.getFields();
        for (int i = 0; i < fields.size(); i++) {
            McpDialogField f = fields.get(i);
            if (i > 0) {
                sb.append(",");
            }
            sb.append("{");
            sb.append("\"type\":\"").append(f.type.name().toLowerCase()).append("\"");
            if (f.label != null) {
                sb.append(",\"label\":\"").append(jsonEscape(f.label)).append("\"");
            }
            if (f.tab != null) {
                sb.append(",\"tab\":\"").append(jsonEscape(f.tab)).append("\"");
            }
            if (f.tooltip != null) {
                sb.append(",\"tooltip\":\"").append(jsonEscape(f.tooltip)).append("\"");
            }
            if (f.units != null) {
                sb.append(",\"units\":\"").append(jsonEscape(f.units)).append("\"");
            }
            if (f.defaultValue != null) {
                sb.append(",\"default\":\"").append(jsonEscape(f.defaultValue)).append("\"");
            }
            if (f.choices != null) {
                sb.append(",\"choices\":[");
                for (int c = 0; c < f.choices.length; c++) {
                    if (c > 0) {
                        sb.append(",");
                    }
                    sb.append("\"").append(jsonEscape(f.choices[c])).append("\"");
                }
                sb.append("]");
            }
            sb.append("}");
        }
        sb.append("]}}" );
        return sb.toString();
    }

    private static void sendJson(HttpExchange exchange, int code, String body) throws IOException {
        // codex 2026-01-27: add newline to avoid shell prompt merging with JSON
        if (!body.endsWith("\n")) {
            body = body + "\n";
        }
        byte[] data = body.getBytes(StandardCharsets.UTF_8);
        Headers headers = exchange.getResponseHeaders();
        headers.set("Content-Type", "application/json; charset=utf-8");
        exchange.sendResponseHeaders(code, data.length);
        OutputStream os = exchange.getResponseBody();
        os.write(data);
        os.close();
    }

    private static Map<String, String> parseParams(HttpExchange exchange) throws IOException {
        Map<String, String> params = new HashMap<String, String>();
        URI uri = exchange.getRequestURI();
        if (uri != null && uri.getQuery() != null) {
            parseQueryString(uri.getQuery(), params);
        }
        if ("POST".equalsIgnoreCase(exchange.getRequestMethod())) {
            byte[] body = exchange.getRequestBody().readAllBytes();
            if (body.length > 0) {
                String raw = new String(body, StandardCharsets.UTF_8);
                parseQueryString(raw, params);
            }
        }
        return params;
    }

    private static int parseInt(String value, int fallback) {
        if (value == null) {
            return fallback;
        }
        try {
            return Integer.parseInt(value.trim());
        } catch (NumberFormatException e) {
            return fallback;
        }
    }

    private static long parseLong(String value, long fallback) {
        if (value == null) {
            return fallback;
        }
        try {
            return Long.parseLong(value.trim());
        } catch (NumberFormatException e) {
            return fallback;
        }
    }

    private static void parseQueryString(String raw, Map<String, String> out) {
        if (raw == null || raw.isEmpty()) {
            return;
        }
        String[] pairs = raw.split("&");
        for (String pair : pairs) {
            if (pair.isEmpty()) {
                continue;
            }
            int idx = pair.indexOf('=');
            String key = idx >= 0 ? pair.substring(0, idx) : pair;
            String value = idx >= 0 ? pair.substring(idx + 1) : "";
            out.put(urlDecode(key), urlDecode(value));
        }
    }

    private static boolean parseBool(String value, boolean defaultValue) {
        if (value == null || value.isEmpty()) {
            return defaultValue;
        }
        String v = value.trim().toLowerCase();
        return v.equals("1") || v.equals("true") || v.equals("yes") || v.equals("y");
    }

    private static String urlDecode(String value) {
        try {
            return URLDecoder.decode(value, StandardCharsets.UTF_8.name());
        } catch (Exception e) {
            return value;
        }
    }

    private static String jsonEscape(String value) {
        if (value == null) {
            return "";
        }
        StringBuilder sb = new StringBuilder(value.length() + 10);
        for (int i = 0; i < value.length(); i++) {
            char c = value.charAt(i);
            switch (c) {
                case '\\': sb.append("\\\\"); break;
                case '"': sb.append("\\\""); break;
                case '\n': sb.append("\\n"); break;
                case '\r': sb.append("\\r"); break;
                case '\t': sb.append("\\t"); break;
                default: sb.append(c); break;
            }
        }
        return sb.toString();
    }
}
