/**
 **
 ** IntersceneMatchParameters - Class for handling multiple configuration parameters
 ** related to the interscene match 
 **
 ** Copyright (C) 2022 Elphel, Inc.
 **
 ** -----------------------------------------------------------------------------**
 **
 **  IntersceneMatchParameters.java is free software: you can redistribute it and/or modify
 **  it under the terms of the GNU General Public License as published by
 **  the Free Software Foundation, either version 3 of the License, or
 **  (at your option) any later version.
 **
 **  This program is distributed in the hope that it will be useful,
 **  but WITHOUT ANY WARRANTY; without even the implied warranty of
 **  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 **  GNU General Public License for more details.
 **
 **  You should have received a copy of the GNU General Public License
 **  along with this program.  If not, see <http://www.gnu.org/licenses/>.
 ** -----------------------------------------------------------------------------**
 **
 */
package com.elphel.imagej.tileprocessor;
import java.awt.Color;
import java.awt.Rectangle;
import java.io.IOException;
import java.util.Properties;
import java.util.StringTokenizer;

import com.elphel.imagej.common.GenericJTabbedDialog;
import com.elphel.imagej.cuas.CuasMotion;
import com.elphel.imagej.orthomosaic.ComboMatch;
import com.elphel.imagej.vegetation.VegetationLMA;

public class IntersceneMatchParameters {
	public static String [] MODES3D = {"RAW", "INF", "FG", "BG"}; // RAW:-1
	public static String [] MODES_AVI = {"RAW", "JPEG", "PNG"};
	public static String [] FLT_ALT_MODES = {"--- (no ALT filter)", "ALT only", "no ALT only"};
	public static String [] FLT_ORIENT_MODES = {"--- (no ORIENT filter)", "ORIENT only", "no ORIENT only"};
	public static Color DEFAULT_TARGET_TEXT_COLOR = new Color (0, 255, 255); // 220);
	public static Color   anaglyph_left_default = new Color (255, 0, 0); // red
	public static Color   anaglyph_right_default =new Color (0, 255, 255); // cyan

	public static Color DEFAULT_cuas_text_color =    new Color (  0, 255,   0);//  220);
	public static Color DEFAULT_cuas_text_bg =       new Color (127, 127, 127);//  220);
	public static Color DEFAULT_cuas_selected_color= new Color (127,   0, 255);//  220);
	public static Color DEFAULT_cuas_radar_uas_color =new Color(  0,100,140);
	public static Color DEFAULT_cuas_radar_color =   new Color(  0,255,100);
	public static Color DEFAULT_cuas_grid_color =    new Color(100,100,100);
	
	
	
	// Maybe add parameters to make sure there is enough data? Enough in each zone? Enough spread?
	public  boolean ims_use =            true;   // use IMS data
	public  boolean ims_rebuild =        false;  // recalculate INS data (after lag change)
	public  double  ims_offset =         0.0;    // IMS data lead relative to images (seconds) 
	public  double  gmt_plus =           3.0;    // (hrs) filename timestamp is this lower than in IMU log
	public  double  [] ims_ortho =       {0.5, -0.5, 0.5, -0.5}; // approximate (90-deg) IMS to camera
	public  double  [] ims_mount_atr =   {0, 0, 0}; // IMS mount fine correction (A,T,R around camera axes)
	public  double  [] ims_mount_xyz =   {0, 0, 0}; // IMS center in camera coordinates
	// gyros offsets in camera ATR order
	// Azimuth -> -pimu_omega[2] rad/s, average did_pimu.theta[2]/did_pimu.dt
	// Tilt ->    -pimu_omega[1] rad/s, average did_pimu.theta[1]/did_pimu.dt
	// Roll ->    +pimu_omega[0] rad/s, average did_pimu.theta[0]/did_pimu.dt
	public double [] pimu_gyr_offs = {-0.000360, -0.001173, -0.001418}; 
	public double [] pimu_acc_offs = {0, 0, 0}; // camera x,y,z, (raw_ins_vxyz/corrected_vxyz -1)  
	
	
	@Deprecated
	public  double  [] ims_scale_xyz =   {1.0,1.0,1.0}; // {1.1, 1.1, 1.1};
	@Deprecated
	public  double  [] ims_scale_atr =   {1.0,1.0,1.0}; // {1.1, 1.1, 1.1};
		
	public  boolean fmg_initial_en =     true; // enable IMS-based FPN mitigation for initial orientation
	public  boolean fmg_reorient_en =    true; // enable IMS-based FPN mitigation for readjustmnet orientation
	public  double  fmg_distance =       10.0; // try to find other reference scene not closer than this pixels
	public  double  fmg_max_quad =       10.0; // estimate offset by 4 points (rooll-aware, 25% from center) if center
	                                           // offset is too small
	public  boolean fmg_rectilinear =    false;// use rectilinear model for scene offset estimation 
	
	
	public  boolean sfm_use =            true;   // use SfM to improve depth map
	public  double  sfm_min_base =       0.6;    // 2.0;    // use SfM if baseline exceeds this
	public  double  sfm_min_gain =       2.0;    // 5.0;    // Minimal SfM gain to apply SfM to the depth map
	public  double  sfm_min_frac =       0.5;    // Minimal fraction of defined tiles to have SfM correction
	
    public  int     sfm_num_pairs =     32;      // desired number of SfM pairs to average
    public  double  sfp_tolerance =      0.05;   // average SfM pairs if their baselines differ less
	public  int     sfm_readjust =       5;      // number of SfM readjustment cycles 
	public  double  sfm_prev_frac =      0.6;    // update if new sfm gain > this fraction of the old one
	public  double  sfm_same_weight =    0.8;    // correction weight when new SfM gain is the same as the old one
	public  int     sfm_shrink =         2;      // shrink sfm gain area before applying sfm_fade_sigma
    public  double  sfm_fade_sigma =     3.0;    // fade SfM gains at the edges 
	public  double  sfm_min_str1 =       0.15;   // 0.22;    // update if correction strength exceeds
	public  double  sfm_min_str16 =      0.22;   // 0.4;    // update if correction strength exceeds (for >=16 pairs)
	public  boolean sfm_use_neibs =      true;   // use neighbors if individual corr is too weak
	public  double  sfm_neib_too_str1=   0.3;    // do not count neighbors stronger than that
	public  double  sfm_neib_too_str16=  0.4;    // do not count neighbors stronger than that  (for >=16 pairs)
	public  double  sfm_neib_str1=       0.2;    // update if no-individual and neibs correction strength exceeds 
	public  double  sfm_neib_str16=      0.5;    // update if no-individual and neibs correction strength exceeds  (for >=16 pairs) 
	public  int     sfm_extrap_steps =   5;      // extrapolate disparity this number over undefined by SfM area
	public  int     sfm_extrap_radius =  5;      // find tilt for extrapolation using this radius circle around the new tile
	
	public boolean  sfm_average_neibs = false;   // Average neighbors disparity among them  
	public boolean  sfm_fill_weak =     false;   // Fill too weak tiles from nearest neighbors
	public boolean  sfm_extrapolate =   false;   // Extrapolate tiles with no SfM data (near edges)
	
	// Filter SfM reference disparity
	public double   sfm_max_dip =       0.1;     // maximal dip relative to lowest neighbor (0 - any)
	public double   sfm_max_bump =      0.5;     // maximal bump relative to highest neighbor (0 - any)
	public boolean  sfm_filter_nan =    true;    // replace NaN by defined neighbors average
	public boolean  sfm_only =          false;   // process only series with SfM (temporary feature)
	
	// debug
	public boolean  sfm_save_seq =      true;    // Save SfM sequence image in model directory 
	public boolean  sfm_show_seq =      true;    // Show SfM sequence 
	public boolean  sfm_show_corr_ind = false;   // Show SfM for each pairs group, each refinement run 
	public boolean  sfm_show_corr =     false;   // Show combined SfM for each refinement run
	
	// rectilinear matching parameters
	public String   omtch_img_set =   ComboMatch.FILES_LISTS_PATHS[0];       // image s
	
	
	public int      rln_gpu_width =     4096;    // horizontal GPU image size (applied during init)
	public int      rln_gpu_height =    4096;    // vertical GPU image size (applied during init)
	
	public double   rln_fat_zero =      10000.0; // phase correlation fat zero
	public boolean  rln_use_neibs =     true;    // calculate TD neighbors in phase correlation 
	public boolean  rln_neibs_fill =    false;   // fill empty neighbors centers
	public double   rln_neib_radius =   5.9;     // neighbors radius (tiles)
	public double   rln_radius_frac =   0.1;     // rln_neib_radius not less than this fraction of woi width/height   
	public double   rln_cent_radius =   4.0;     // centroids center radius
	public int      rln_n_recenter =    2;       // when cosine window, re-center window these many times
    public double   rln_sngl_rstr =     0.25;    // minimal single-tile phase correlation maximums relative to max str
    public double   rln_neib_rstr =     0.35;    // minimal neighbors phase correlation maximums relative to max str
	
    // Pairwise scenes matching
    public boolean  pmtch_use_affine =  false;   // when matching pairs, start with known scene affine matrices, false - use unity
    public double   pmtch_frac_remove = 0.1;
	public double   pmtch_metric_err =  0.05;    // 0.02;//  2 cm
	public double   pmtch_max_std =     1.5;     // maximal standard deviation to limit center area  
	public double   pmtch_min_std_rad = 2.0;     // minimal radius of the central area (if less - fail)
	public boolean  pmtch_ignore_rms =  true; 
	public int      pmtch_num_iter =   10;
    
	public double   pmtch_pull_skew =   0.0;     // ~rotation, = 0 fraction of the total weight == 1
	public double   pmtch_pull_tilt =   0.1;     // > 0
	public double   pmtch_pull_scale =  0.0;     // = 0
	
	public double   pmtch_cent_rad =    0.6;     // center circle radius fraction of 0.5* min(width, height) in tiles
	public double   pmtch_max_cent_rad =40;      //  30; // maximal center radius in tiles (limit pmtch_cent_rad)
	public double   pmtch_cent_fill =   0.2;     // 25; // should be populated not less than this
	public double   pmtch_cent_final =  0.23;    // 4; // should be populated not less than this
	public double   pmtch_ease_nosfm =  2.0;     // ease metric_error when no SfM gain == 0;
	
    // spiral search parameters
    public boolean  ospir_existing =    false;   // use existing pair
    public boolean  ospir_invert =      false;   // invert existing pair
    public double   ospir_step =        8.0;     // spiral step (pix)
    public double   ospir_range =      50.0;     // spiral radius (pix)
	public double   ospir_double =      0.5;     // if overlap < this, increase zoom by 1, range - twice	
    public double   ospir_good_rms =    0.3; // 27;    // maximal immediately acceptable LMA RMS for the initial search
    public double   ospir_max_rms =     0.35;    // maximal acceptable LMA RMS - best during initial search 
    public int      ospir_overlap =  3000;       // do not try to match if there is too small overlap (scaled pixels)
    public int      ospir_num_iter =    5;       // maximal number of LMA iterations during initial search
    public double []ospir_rms_iter = {1.0, 0.6};// maximal rms per iteration
    public boolean  ospir_ignore_rms =  false;   // ignore RMS worsening during spiral search
    public int      ospir_debug =       0;       // Debug level during sppiral search 
    public boolean  ospir_augment =     true;    // Use low-res matching during augmenting (false - skip, go to high-res) 
    
    // pairwise match parameters
	public int      pwise_zoom =         -5;
	public double   pwise_overlap =      0.25;
    public double   pwise_max_rms =      0.6;    // maximal acceptable LMA RMS - best during refine 
	public boolean  pwise_skip_exist =   true;
	public boolean  pwise_refine_exist = true;  // if false, start from scratch, true - start from previous
	public boolean  pwise_delete_fail =  true;  // delete existing match if now failed 
	public boolean  pwise_gen_inverse =  false; // generate inverse matches
	public boolean  pwise_save_each =    true;       // save state file after each match
	public boolean  pwise_log_append =   true;      //
	public String   pwise_log_path =     "/media/elphel/NVME/lwir16-proc/ortho_videos/debug/sept12-13/overlaps/pairwise_01.log";
	public int      pwise_debug =        0;
    
    // alt_data-related parameters
	public boolean  alt_overwrite =      false;   // re-calculate alt_data even if it exists
	public boolean  alt_pairwise =       true;    // use pairwise affines if available ** not used?
	public double   alt_sigma =          25.0;    // reduce weight of the border tiles, Gaussian sigma in pixels (after scaling) to apply to weights
	public double   alt_abs_outliers =   3.0;     // remove absolute outliers when fitting planes 
	public double   alt_outliers =       0.1;     // remove outliers when fitting planes 
	public int      alt_refine=          1;       // refine plane after removing outliers (0 - no outlier removal, 1 - remove outliers and refine once, ...) 
	
	public double   alt_weight_rot =     1.0;     // >0 weight of pairs errors in qn3
	public double   alt_weight_tilt =    0.3;     // >0 weight of pairs errors in qn1, qn2
	public double   alt_weight_scale =   0.1;     // >0 weight in pairs scale-1.0 errors
	public double   alt_pull =           0.2;     // 0 <= pull <1 - fraction of all RMS contributors
	public double   alt_pull_rots =      1.0;     // >=0 weight of sum of rotations, may be 0, normalized by pull value 
	public double   alt_pull_tilts =     0.3;     // >=0 weights of sum of qn1 and qn2 of scenes, normalized by pull value
	public double   alt_pull_scales =    0.1;     // >=0 weights of scales of scenes, normalized by pull value
	
	public boolean  pmap_move_only =      false;
	public boolean  pmap_ignore_affines = false;
	public boolean  pmap_use_inv =        false;
	public double   pmap_skew_pull =      1.0;
	public double   pmap_tilt_pull =      1.0;
	public double   pmap_scale_pull =     0.01; // was 0.1;
	public double   pmap_position_pull =  0.0001;
	public double   pmap_overlap_pow =    2.0; // match weight as overlap fraction to this power
	public double   pmap_rms_diff =       0.000001;
	public int      pmap_num_iter =     100; //  50;
	public boolean  pmap_show_result =    false;
	public int      pmap_debugLevel =     2;
    
    public boolean  pequ_ignore_equalize= false; // ignore previous equalization
    public boolean  pequ_use_inv =        false;
    public double   pequ_scale_weight = 500;     // relative weight of scale differences compared to offset differences
    public double   pequ_pull_weight =    0.001; // relative weight of offsets and scales differences from 1.0 to pairs mismatch
    public double   pequ_half_weight_sec =300.0;   //  - time difference to reduce weight twice
    public double   pequ_min_weight_sec = 0.0; // 01;  //  weight of pairs at very different time
    public double   pequ_overlap_pow =    2.0; // match weight as overlap fraction to this power
    public double   pequ_rms_diff =       0.000001;
    public int      pequ_num_iter =     100;    //  50;
    public int      pequ_debugLevel =     2;
	
 	
    // pairs selection/display parameters
    public boolean  flt_list =            true;
    public boolean  flt_undef_only =      false;
    public double   flt_min_overlap =     0.0;
    public double   flt_max_overlap =     1.0;
    public double   flt_min_rms =         0.0;
    public double   flt_max_rms =         2.0;
    public boolean  flt_nan_rms =         false;
    
	public	boolean flt_filt_zoom =       true;
	public	int     flt_min_zoom =       -2;
	public	int     flt_max_zoom =       10;
    public double   flt_min_sfm =         0.0;  // minimal minimal SfM gain of a pair
    public double   flt_max_sfm =      1000.0; // maximal minimal SfM gain of a pair
	public	int     flt_alt =             0; // 0 - do not filter, 1 - keep only with alt data, 2 - keep only without alt data
	public	int     flt_orient =          0; // 0 - do not filter, 1 - keep only with orientation data, 2 - keep only without orientation data
    
    public boolean  flt_show_names =      true;
    public boolean  flt_show_overlaps =   true;
    public boolean  flt_show_rms =        true;
    public boolean  flt_show_zoom =       true;
    public boolean  flt_show_alt =        true;
    

    public String   patt_save_top = "/media/elphel/NVME/lwir16-proc/ortho_videos/debug/sept12-13/pattern_match/";
    public String   patt_save_subdir = "combo_maps";
    
    
	public double [] getImsMountATR() {
		return new double [] {
				ims_mount_atr[0] * Math.PI/180,
				ims_mount_atr[1] * Math.PI/180,
				ims_mount_atr[2] * Math.PI/180};
	}
	public void setImsMountATR( double [] atr) {
		ims_mount_atr[0] = atr[0] * 180/Math.PI;
		ims_mount_atr[1] = atr[1] * 180/Math.PI;
		ims_mount_atr[2] = atr[2] * 180/Math.PI;
	}

	
	public  boolean center_reference =   false;
	public  boolean lock_position =      false; // camera position is not changing, only orientation (cuas mode)
	public  boolean cuas_rotation =      false; // assumes lock_position, camera is rotating around the center
	public  boolean manual_correction =  false; // once used for foliage to merge two sequences
	public  boolean overlap_sequences =  false; // overlap sequences: scan down from the previous center  
	public  boolean reset_photometric =  true;  // reset photometric calibration - once for each new series
	public  boolean force_ref_dsi =      false; // true;
	public  boolean force_orientations = false;
	public  boolean run_ly =             false; // will return just after LY adjustments, skipping all output generation
	public  int     run_ly_mode =        3;     // +1 - lazy eye, +2 - infinity
	public  boolean run_ly_lma =         true;
	public  int min_num_orient =         2; // make from parameters, should be >= 1
	public  int min_num_interscene =     2; // make from parameters, should be >= 1
	public  boolean adjust_imu_orient =  false; // adjust IMU misalignment to the camera
	public  boolean apply_imu_orient =   true;  // apply  IMU misalignment to the camera if adjusted
	public  boolean orient_by_move =     false; // use translation data to adjust IMU orientation  
	public  boolean orient_by_rot =      true;  // use rotation data to adjust IMU orientation
	public  boolean orient_combo =       true;  // use combined rotation+orientation for IMU/camera matching
	public  boolean adjust_gyro =        false; // adjust qyro omegas offsets
	public  boolean apply_gyro =         true;  // apply adjusted qyro omegas offsets
	public  boolean adjust_accl =        false; // adjust IMU velocities scales
	public  boolean apply_accl =         true;  // apply IMU velocities scales
	
	public  boolean calc_quat_corr =     true; // calculate camera orientation correction from predicted by IMS 
    public  boolean apply_quat_corr =    true; // apply camera orientation correction from predicted by IMS
    public  boolean use_quat_corr =      true; // Use orientation correction everywhere if available
    public  boolean inertial_only =      true; // Use only inertial data (from DID_PIMU), not the IMS for initial poses
	
	public  boolean generate_egomotion = true; // generate egomotion table (image-based and ims)
	public  boolean generate_mapped =    true; // generate any of the sequences - Tiff, video, mono or stereo
	public  boolean reuse_video =        false; // dry-run video generation, just reassemble video fragments (active if !generate_mapped)
	public  boolean save_mapped_color =  false; // may be not needed when using AVI
	public  boolean save_mapped_mono =   true;  // may be not needed when using AVI
	public  boolean generate_raw =       false;
	public  boolean generate_inf =       false;
	public  boolean generate_fg =        true;
	public  boolean generate_bg =        false;

	public  boolean generate_stereo =    false; // applies to all stereobases, including 0
	
	public  boolean export_images =      true;  // pseudo-color 16-slice images (same disparity, COMBO_DSN_INDX_DISP_FG and COMBO_DSN_INDX_DISP_BG_ALL,
	public  boolean show_images =        false; // color, infinity
	public  boolean show_images_bgfg =   false; // bg and fg
	public  boolean show_images_mono =   false; // float, monochrome 16-slice images (same disparity, COMBO_DSN_INDX_DISP_FG and COMBO_DSN_INDX_DISP_BG_ALL,
	public  boolean show_color_nan =     true;  // use NAN background for color images (sharp, but distinct black)
	public  boolean show_mono_nan =      false; // use NAN background for monochrome images (sharp, but distinct black)
	
	public  double [][] stereo_views = {  // base, up, back
			{   0.0,   0.0,    0.0},
			{ 200.0,   0.0,    0.0},
			{ 500.0,   0.0, 2000.0},
			{1000.0, 500.0, 3000.0}};
	
	public  boolean [] generate_stereo_var = new boolean[stereo_views.length];

	// Other parameters
	public  double  scale_extrap_atr =    1.0; // scale angular extrapolation
	public  double  scale_extrap_xyz =    0.0; // scale linear extrapolation (1.0 for drones)
	public  int     avg_len =             1;   // average pose over number of previous scenes
	public  int     min_num_scenes =     10;   // abandon series if there are less than this number of scenes in it 
	public  int     max_num_scenes =    100;   // abandon series if longer than 
	public  double  blur_egomotion =      2.0;
	
	
	public  boolean export_ranges =      true;
	public  boolean debug_ranges =       false;
	public  boolean show_ranges =        true;
	
	public  boolean export_ml_files =    false; // export files for DNN training
	
	public  double  range_disparity_offset =   -0.08;
	public  double  range_min_strength = 0.5;
	public  double  range_max =       5000.0;
	// Export 3D model
	public  boolean export3d =           false; // true;
	public  boolean export3dterrain =    false; // true;	
	// Generate CT scan
	public  boolean export_CT =          false; // true;
	public  double  ct_min =             3.7; //
	public  double  ct_max =             4.5; //
	public  double  ct_step =            0.02;  // pix
	public  int     ct_expand =          -4;    // pix // expand section area by 1 pixel in 8 directions
	
	// Generate terrain surface (cut trees)
	// flat surface extraction
//	public boolean   terr_enable =     true;
	public double    terr_rmse_above = 1.0; // from average
	public double    terr_rmse_below = 1.0; // from average, // positive value
	public int       terr_num_refine = 5;
	public double    terr_frac_above = 0.4;
	public double    terr_frac_below = 0.3;
	
	public double    terr_rthrsh_abv = 0.01; // 08; // later may be relative to SfM ?
	public boolean   terr_flat_brd =   true; // when filling NaN, replace NaN border tiles with flat ground
	public int       terr_expand =     4; // 2;
	public boolean   terr_save =       true; // save debug data in model directory

	
	
	// Debug and visualization
	public  boolean scene_is_ref_test=              false; // correlate ref-2-ref for testing
	private boolean render_ref =                    true;  // render reference scene
	private boolean render_scene =                  true;  // render scene to be correlated with ref
	public  boolean toRGB =                         true;  // render scenes in pseudo-colors DOES NOT WORK - use ColorProcParameters
	private boolean show_2d_correlations =          true;  // show raw 2D correlations (individual and combined)
	private boolean show_motion_vectors =           true;  // show calculated motion vectors
	public  int     debug_level =                     -1;  // all renders are disable for debug_level < 0, scene "renders" for for debug_level < 1

	// Pairwise ERS testing
	public boolean  test_ers =                     false;
	public  int     test_ers0 =                       -1; // try adjusting a pair of scenes with ERS. Reference scene index
	public  int     test_ers1 =                       -1; // try adjusting a pair of scenes with ERS. Other scene index

	// Other parameters for filtering depth maps	
	public  int         num_bottom =                      3; // 6; // average this number of lowest disparity neighbors (of 8)
	public  int         num_passes =                    100;
	public  double      max_change =                      1.0e-3;
	public  double      min_disparity =                  -0.15; // 0.2;
	public  double      max_sym_disparity =               0.1; // 0.2;
	// 2 next are wrong currently - minimal strength is ~0.25
	public  double      min_strength_lma =                0.0; // 0.3;  // no real filtering
	public  double      min_strength_replace =            0.05; ///  0.14; /// Before /// - LWIR, after - RGB
	public  double      min_strength_blur =               0.06; ///  0.2;
	public  double      sigma =                           2; /// 5;
	public  int         num_blur =                        1; // 3;
	public  double      disparity_corr =                  0.0;
	public  int         outliers_nth_fromextrem =         1; // second from min/max - removes dual-tile max/mins
	public  double      outliers_tolerance_absolute =     0.2;
	public  double      outliers_tolerance_relative =     0.02;
	public  int         outliers_max_iter =             100;
	public  double      outliers_max_strength2 =          1.0; // 0.5; any
	public  int         outliers_nth_fromextrem2 =        0; // second from min/max - removes dual-tile max/mins
	public  double      outliers_tolerance_absolute2 =    0.5;
	public  double      outliers_tolerance_relative2 =    0.1;
	public  double      outliers_lma_max_strength =       0.4; // 0.5;
	public  double      outliers_max_strength =           0.1; ///  0.25;
	public  double      outliers_from_lma_max_strength =  0.8;
	public  int         search_radius =                   3;  // Search farther if no LMA neighbor is found closer. Original value - 1 (8 neighbors)
	public  boolean     remove_no_lma_neib =              true;
	public  double      diff_from_lma_pos =             100.0;
	public  double      diff_from_lma_neg =               2.0;
	public  int         outliers_lma_nth_fromextrem =     0; // 1; 
	public  int         filter_margin =                  -8; // 8; // pixels

	public  double      weak_tolerance_absolute=          0.25;
	public  double      weak_tolerance_relative=          0.025;
	public  int         weak_min_neibs =                  5;
	public  double      strong_strength=                  0.5;
	public  double      weak_strength=                    0.2; // none is below 
	//------------	
	
	// Sky detection/filtering
	public  boolean sky_extract =                    true;  // do not try to extract blue sky
	public  boolean sky_recalc =                     false; // force blue sky recalculation even if it exists
	public  double  sky_highest_min =                -50; // lowest absolute value should not be higher (requires photometric) 
	public  double  cold_frac =                        0.005; // this and lower will scale fom by  cold_scale
	public  double  hot_frac =                         0.9;    // this and above will scale fom by 1.0
	public  double  cold_scale =                       0.2;  // <=1.0. 1.0 - disables temperature dependence
	
	public  double  sky_seed =                         5.0;  // 5.0 start with product of strength by diff_second below this
	public  double  lma_seed =                         2.0;  // seed - disparity_lma limit		double sky_lim  =      15.0; // then expand to product of strength by diff_second below this
	public  double  seed_temp =                        0.5;  // seed colder that this point between min and max temp
	
	public  int     sky_shrink =                       2;
	public  int     seed_rows =                        5; // sky should appear in this top rows 
	public  double  max_disparity =                    0.3;  // maximal strong sky disparity 
	public  double  max_disparity_strength =           0.2;  // maximal strength of high disparity sky
	public  double  sky_lim =                         15.0; // then expand to product of strength by diff_second below this
	public  double  lim_temp =                         0.5;  // sky colder that this point between min and max temp
	public  int     sky_expand_extra =                 0; // 1?
	public  int     sky_bottleneck =                   5;
	public  int     sky_reexpand_extra =               9;//  re-expand after bottleneck in addition to how it was shrank	
	public  double  min_strength =                     0.08;
	public  int     lowest_sky_row =                  50; // appears that low - invalid, remove completely
	public  double  sky_bottom_override =           -300; // maximal average sky value to override lowest_sky_row test
	public  int     sky_override_shrink =             6; // shrink detected sky before finding hottest tile there
	
	
	// processing clouds in the sky
	public boolean  clouds_en =                       true;  // enable clouds in the sky detection / processing
	public  double  clouds_fom =                      30.0;  // maximal FOM for clouds (must be <=)         
	public  double  clouds_spread =                   60.0;  // maximal spread for clouds (must be <=)         
	public  double  clouds_disparity =                 0.1;  // maximal disparity for strong clouds         
	public  double  clouds_weak =                      0.18; // maximal strength for near definite clouds         
	public  double  clouds_strength =                  0.25; // minimal strength for far strong clouds (definitely cloud)
	public  double  clouds_not_strength =              0.4;  // maximal strength for near maybe clouds (if it has strong cloud neighbor)
	public boolean  clouds_strong =                   true;  // allow weak cloud if it has strong (1.5x) cloud neib
	
	// process smooth walls mistaken for sky (disable for natural environments)
	public boolean  wall_en =                         true;  // enable smooth walls detection/processing
	public boolean  wall_dflt =                       false; // default (undetected) is wall (false - sky)
	public  double  wall_str =                        0.1;   // minimal strength of the far object (small - just non-NaN disparity)         
	public  double  wall_far =                        0.2;   // maximal disparity to consider cluster to be sky         
	public  double  wall_near =                       1.0;   // minimal disparity to consider cluster to be wall         

	// processing far treeline that may be confused with clouds. Only behind far objects such as far horizontal surface.
	// Maybe add temperature (treeline warmer than sky)?
	public boolean  treeline_en =                     true;  // enable treeline processing
	public boolean  treeline_wide =                   true;  // look not only under, but diagonal too.
	public int      treeline_height =                  8;    // maximal height of the treeline (tiles)
	public int      treeline_width =                   3;    // minimal horizontal width of the treeline (tiles)
	public boolean  treeline_lim_high =               false; // limit too high treeline (false - delete completely)
	public double   treeline_str =                     0.3;  // treeline minimal strength 
	public double   treeline_far =                     0.03; // treeline min disparity (pix)
	public double   treeline_near =                    0.4;  // treeline max disparity (pix)
	public double   treeline_fg_str =                  0.4;  // pre-treeline FG objects (such as flat ground) minimal strength 
	public double   treeline_fg_far =                  0.2;  // pre-treeline FG objects  min disparity (pix)
	public double   treeline_fg_near =                 0.6;  // pre-treeline FG objects  max disparity (pix)
	
	// suspecting indoors (disabling sky)
	public boolean  indoors_en =                      true; // allow weak cloud if it has strong (1.5x) cloud neib
	public  double  indoors_str =                      0.5; // minimal strength of the far object         
	public  double  indoors_disp =                     0.8; // maximal minimal outdoor strong disparity         
	public  int     indoors_min_out  =                10;   // minimal strong far tiles to deny indoors
	
	// Increase fom if there are enough LMA-defined similar tiles around
	public  double  disp_boost_min  =                  0.5;
	public  double  disp_boost_diff  =                 0.35;
	public  int     disp_boost_neibs  =                2;
	public  double  disp_boost_amount  =               2.0;
	public  double  scale_combo_strength  =            0.4; // reduce strength when it comes from combo, not DSI-MAIN
	public  boolean save_debug_images =               true;  // Save blue sky debug images in model directory
	/// IMS intergration
	
	
	
	// Some "AGC" to adjust how much to discard
	public  int     margin =                       1;      // do not use tiles if their centers are closer to the image edge
	public  int     sensor_mask_inter =           -1;      // bitmask of the sensors to use (-1 - all)
	public  boolean use_partial =                   true;  // find motion vectors for individual pairs, false - for sum only
	public  boolean run_poly =                      false; // not yet implemented
	public  double  centroid_radius =               4.0;   // 
	public  int     n_recenter =                    2;     // when cosine window, re-center window this many times
	// filtering motion vectors
	// TD accumulation of the inter-scene correlations  demonstrated artifacts (horizontally offset by 8 pixels
	// false maximum that is sharper than the real one. Still not understood - maybe float precision related. 
	public  double  td_weight =                     0.25; // 0.75;   // mix correlations accumulated in TD with 
	public  double  td_neib_weight =                0.6; // 0.75;   // mix correlations accumulated in TD (neibs) 
	public  double  pd_weight =                     0.15; // 0.25;   // correlations (post) accumulated in PD
	public  boolean td_nopd_only =                  false; // true;  // only use TD accumulated data if no safe PD is available for the tile.
	public  boolean neib_notd_only =                false; // true;  // use neighbors only if individual TD is too weak

	// averaging with neighbors for interscene matching
	public  boolean use_neibs =                     true;  // false; // true;
	public  boolean neibs_nofpn_only =              false; // consolidate neighbors for non-fpn tiles only!
	public  boolean neibs_nofpn_init =              true;  // Non-fpn only for initial adjustment
	public  boolean redo_both =                     true;  // use average of neighbors for both pd,td if any of the center tile tests (td, pd) fails
	public  int min_num_neibs =                     4;     // plus center, total number >= (min_num_neibs+1)
	public  double scale_neibs_pd =                 0.5;   // scale threshold for the pixel-domain average maximums
	public  double scale_neibs_td =                 0.5;   // scale threshold for the transform-domain average maximums
	public  double scale_avg_weight =               0.5;   // reduce influence of the averaged correlations compared to the single-tile ones
	
	public  double  min_str_fpn =                   0.2;   // 0.25; // minimal correlation strength for all but TD-accumulated layer
	public  double  min_str_sum_fpn =               0.42;  // 0.5; // 0.8;  // minimal correlation strength for TD-accumulated layer
	public  double  min_str_neib_fpn =              0.5;   // 0.8;  // minimal correlation strength for TD-accumulated neighbors layer

	public  double  min_str =                       0.15;  // 12; //18;  // tiles w/o FPN: minimal correlation strength for all but TD-accumulated layer
	public  double  min_str_sum =                   0.3;   //  0.2;  // 0.33; // tiles w/o FPN: minimal correlation strength for TD-accumulated layer
	public  double  min_str_neib =                  0.3;   // 0.33; // tiles w/o FPN: minimal correlation strength for TD-accumulated layer

	public  int     min_neibs =                     2;	   // minimal number of strong neighbors (> min_str)
	public  double  weight_zero_neibs =             0.2;   // Reduce weight for no-neib (1.0 for all 8)
	public  double  half_disparity =                5.0;   // Reduce weight twice for this disparity
	public  double  half_avg_diff =                 0.2;   // when L2 of x,y difference from average of neibs - reduce twice
	
// new method with alignment to eigenvectors. Keep above  for comparison, and possibly for front-looking
	
	public boolean  eig_use =                      false;  // use eigenvectors alignment, false for old settings
	public boolean  eig_xy_lma =                   false;  // do not use eigenvectors for LMA (prepare the same, debug feature)
	public double   eig_str_sum =                   0.22;  //  0.2;  // 0.33; // tiles w/o FPN: minimal correlation strength for TD-accumulated layer
	public double   eig_str_sum_fpn =               0.33;  // 0.5; // 0.8;  // minimal correlation strength for TD-accumulated layer
	public double   eig_str_neib =                  0.1;   // 0.33; // tiles w/o FPN: minimal correlation strength for TD-accumulated layer
	public double   eig_str_neib_fpn =              0.15;  // 0.8;  // minimal correlation strength for TD-accumulated neighbors layer
	public double   eig_min_abs =                   0.05;  // minimal absolute corr value to keep for centroid and eigen filtering of connected
	public double   eig_min_rel =                   0.25;  // minimal relative to max corr value to keep for centroid and eigen filtering of connected
	public double   eig_sub_frac =                  1.0;   // subtract fraction of threshold {eig_min_abs,eig_min_rel} after selecting by them (0 - select only, will have pedestal)
	public double   eig_max_sqrt =                  6.0;   // for sqrt(lambda) - consider infinity (infinite linear feature, a line)
	public double   eig_min_sqrt =                  1.0;   // for sqrt(lambda) - limit minimal sqrt(lambda) - can be sharp for very small max
	// refining offset with elliptical masks
	public  int     eig_recenter =                  2;     // Re-center window around new maximum. 0 -no refines (single-pass)
	public double   eig_sub_frac1 =                 0.0;   // subtract during refine (may be 0)
	// Using eigenvectors/values to create ellipse (slightly larger) and use it for a cosine mask (as round before) to refine centers
	public double   eig_scale_axes =                1.2;   // scale half-axes of the ellipse: 1.0 <-> sqrt(eigenvalue)
	public double   eig_inc_axes =                  1.0;   // add do half-axes
	
	public boolean  eig_use_neibs =                true;   // use correlation from 9 tiles with neibs, if single-tile fails
	public  int     eig_min_weaks =                   4;   // Minimal number of weak neighbors to keep center weak tile
	public  int     eig_min_strongs =                 2;   // Minimal number of strong neighbors for stron tiles
	public double   eig_disp_diff =                   1.0; // maximal disparity difference from the closest (by disparity) neighbor
	public boolean  eig_remove_neibs =             true;   // remove weak (by-neibs) tiles if they have strong (by-single) neighbor
	public boolean  eig_filt_other =              false;   // apply other before-eigen filters
	public  double  eig_max_rms =                   2.0;   // eigen-normalized maximal RMS to consider adjustment to be a failure
	public boolean  eig_fast2x2 =                  true;   // use fast eigenvectors for 2x2 matrices
	

/*
min_str_sum_nofpn	0.22	
min_str_neib_nofpn	0.25	
min_str_fpn	0.2	
min_str_sum_fpn	0.22	
min_str_neib_fpn	0.35	

 */
	
	
	// Detect initial match
	public boolean  use_combo_reliable =           true;  // use combo dsi if available for relaible tiles
	public boolean  ref_need_lma =                 true;  // need LMA output for reliable tiles (no combo available)
	public boolean  ref_need_lma_combo =           true;  // need LMA output for reliable tiles (when combo is available)
	public  double  min_ref_str =                   0.33;  // 0.22;  // For orientations: use only tiles of the reference scene DSI_MAIN is stronger  
	public  double  min_ref_str_lma =               0.8;   // 0.22;  // For orientations: use only tiles of the reference scene DSI_MAIN is stronger  
	public  double  min_ref_frac =                  0.2;   // 0.22;  if fraction number of reliable tiles is less than this, use best possible
	
	// was overwritten in code to always be true ****
	public  boolean ref_smooth =                    true; // false; // smooth reference disparity for initial matching
	public  boolean ref_smooth_always =             true; // in older code it was hard-wired true inside adjustDiffPairsLMAInterscene
//	ref_smooth_always, when true mimics old behavior
	public  double  ref_sigma =                     2.0;   // Gaussian sigma to smooth reference disparity for initial matching
	public  double  ref_smooth_diff =               0.15;  // discard smooth disparity if it differs by more from measured
	
// SfM-related filtering (remove tiles without SfM)
	public boolean  sfm_filter =                   true;   // use SfM filtering if available 
	public  double  sfm_minmax =                    3.0;   // 10.0 // minimal value of the SfM gain maximum to consider available
	public  double  sfm_fracmax =                   0.75;  // minimal fraction of the SfM maximal gain
	public  double  sfm_fracall =                   0.3;    // minimal relative area of the SfM-enabled tiles (do not apply filter if less)
	
	
	public  int     pix_step =                      4;     // Azimuth/tilt search step in pixels
	public  int     search_rad =                   10;     // Search radius in steps
	public  double  maybe_sum =                     1.0;   // minimal sum of strengths (will search for the best)
	public  double  sure_sum =                     50.0;   // definitely good sum of strengths (no more search) too high, will be FOM-defined
	public  double  maybe_avg =                     0.01;  // maybe average strength
	public  double  sure_avg =                      0.15;  // 0.015; // sure average strength disabling (was 0.03)
	public  double  max_search_rms =                1.5;   // good - 0.34, so-so - 0.999
	public  double  maybe_fom =                     1.0;   // good - 38, second good - 4.5
	public  double  sure_fom =                     12.0;   // good - 38, second good - 4.5
	public  boolean treat_serch_fpn =              false;  // use FPN (higher) thresholds during search (even if offset is not small)
	
	
	// Reference scene disparity 
	public  boolean use_combo_dsi =                 true;  // use interscene DSI if available (instead of the single-scene)
	public  boolean use_lma_dsi =                   true;  // only use reference DSI tiles that have LMA (strong) disparity
	
	// Remove correlation caused by FPN
	public  boolean fpn_remove =                    true;  // only use reference DSI tiles that have LMA (strong) disparity
	public  double  fpn_max_offset =                8.0;   // pix - ignore larger FPN offsets
	public  double  fpn_min_offset =                1.8;   // pix - ignore tile if maximum differs less from fpn offset 
	public  double  fpn_radius =                    0.75;  // pix - zero around center 
	public  boolean fpn_ignore_border =             false; // only if fpn_mask != null - ignore tile if maximum touches fpn_mask
	
	public  double  min_offset =                    3.0;   // pixels - minimal average pixel offset between images to consider interscene matching
	public  double  max_pull_jump =                 3.0;   // pixels - maximal offset from the current pose to "pull" pose.  
	public  double  max_rel_offset =                0.25;  // maximal interscene offset as a fraction of image width
	public  double  max_roll_deg =                  5.0;   // maximal interscene roll to consider matching
	public  double  max_zoom_diff =                 0;     // for down-views when changing altitude (0 - ignore)
	public  boolean fpn_skip =                      true;  // skip too close scenes (false - abort, previous behavior)
	public  boolean fpn_rematch =                   true;  // match fpn-failed scenes to later scenes with larger difference
	// still not clear why it sometimes fails without refine_invert (too large initial mismatch)
	public  boolean refine_invert =                 true;  // Refine with LMA while inverting relative poses from other reference
	public  boolean use_precomp =                   false; // try to predict initial error from previous scenes
	
	// Remove moving objects (goal is not to detect slightest movement, but to improve pose matching
	public  boolean mov_en =                        true;  // enable detection/removal of the moving objects during pose matching
	public  double  mov_sigma =                     1.5;   // pix - weighted-blur offsets before detection
	// next two to prevent motion detection while errors are too big
	public  double  mov_max_std =                   1.2;   // pix
	public  double  mov_thresh_rel =                3.5;   // .0;   // exceed average error
	public  double  mov_thresh_abs=                 0.5;   // sqrt(dx^2+dy^2) in moving areas 
	public  double  mov_clust_max =                 1.5;   // cluster maximum should exceed threshold this times
	public  int     mov_grow =                      4;     // grow detected moving area
	public  int     mov_max_len =                   0;     // (0 - no limit) do not remove moving objects if they do not fit into the square   
	public  boolean mov_show =                      true;  // show debug images for movement detection
	public  int     mov_debug_level =               1;     // >0 verbose
	//LMA parameters
	public  boolean [] adjust_atr = new boolean [] {true,true,true};
	public  boolean [] adjust_xyz = new boolean [] {true,true,true};
	public  double  exit_change_atr =               1.0E-5;//rad,  L2 norm for difference to ext LMA 
	public  double  exit_change_xyz =               1.0E-3;//meters, L2 norm for difference to ext LMA 
	public  int     max_cycles =                   10;     // hard limit on full correlation/LMA cycles
	public  int     max_LMA =                      25;     // hard limit on LMA iterations
	public  double  max_rms =                       2.0;   // maximal RMS to consider adjustment to be a failure
	
// equalization of the inter-scene correlations	
	public  boolean eq_en =                 true;// equalize "important" FG tiles for better camera XYZ fitting
	public  int     eq_stride_hor =         8;   // half of a equalization supertile width
	public  int     eq_stride_vert =        8;   // half of a equalization supertile height
	// Supertile qualification
	public  double  eq_min_stile_weight =   0.2; // minimal total weight of the tiles in a supertile (lower will not be modified)
	public  int     eq_min_stile_number =   10;  // minimal number of defined tiles in a supertile
	public  double  eq_min_stile_fraction = 0.02;// minimal total tile strength compared to the average one
	// Individual tiles qualification
	public  double  eq_min_disparity =      5; // minimal disparity of tiles to consider (applies after filtering / boosting)
	public  double  eq_max_disparity =   1000; // maximal disparity of tiles to consider (applies after filtering / boosting)
	// Boost amount 
	public  double  eq_weight_add =         0.03; // calculate from min-strengths
	public  double  eq_weight_scale =      20.0;   // maximal boost ratio 
	public  double  eq_level =              0.9; // equalization level (0.0 - leave as is, 1.0 - boost to have the same supertile strength as average)
	
	public  boolean mb_en =                 true;
	public  double  mb_tau =                0.008; // time constant, sec
	public  double  mb_max_gain =           5.0;   // motion blur maximal gain (if more - move second point more than a pixel
	public  double  mb_max_gain_inter =     2.0;   // same for interscene correlation for pose adjustment
	//CUAS mode
	public  boolean cuas_update_existing=   false;  // re-create center_CLT if it exists (FIXME: accumulates errors - need fixing)
	public  int     cuas_discard_border =   8;     // Discard this number of pixels from each side when merging
	public  double  cuas_max_fold =         50;
	public  int     cuas_min_in_row_col =   4;     // Minimal number of defined tiles in a row/column
	public  double  cuas_clt_variant =      50; // 10;    // threshold from the nearest to create a variant for the tile
	public  double  cuas_clt_threshold =    70; //  20;    // threshold for CLT tile difference not merged when selecting the best
	public  double  cuas_clt_decrease =     0.01;  // Scale CLT outliers weight 
	public  double  cuas_decay_average =   30.0;   // Decay in seconds for cimulative CLT 
	public  double  cuas_keep_fraction =    0.75;   // Filter CLT variants for tiles keeping at least this fraction of the total weight
	public  boolean cuas_step =             true;  // recalculate template image after each fitting step

	public  boolean cuas_subtract_fpn=      true;  // Subtract FPN
	public  boolean cuas_calc_fpn =         false; // Recalculate+save FPN unconditionally, if false - try to read saved one first
	public  double  cuas_rot_period =       175.0; // rotation period of te gimbal mount (in scenes)
	public  double  cuas_min_series =       180.0; // discard shorter series // FIXME: allow shorter, but modify FPN/row_col to copy parent's files if less than cuas_rot_period
	public  boolean cuas_subtract_rowcol=   true;  // Subtract row/column noise
	public  boolean cuas_calc_rowcol=       false;  // Recalculate+save  row/column noise, if false - try to read saved one first
	public  double  cuas_um_sigma =         10.0;  // Apply Unsharp Mask filter sigma when calculating row/column noise
	public  double  cuas_max_abs_rowcol =   50.0;  // consider pixels with abs(UM difference) does not exceed this value 
	public  double  cuas_outliers_rowcol =  0.001; // scale weight of the outliers with high difference (to prevent undefined values 
	public  boolean cuas_reset_first=       false; // true;  // false; // Reset average in first scene (for large diffirence in unfinished row/col)
	public  int     cuas_invert_margins =   0;     // Expand image each side when inverting tasks 
	public  int     cuas_invert_iters =     4;     // Enhance inversion iterations 
	public  double  cuas_invert_tolerance = 0.001; // Finish enhancing when last change was lower than 
	public  int     cuas_invert_gap2 =      10;    // Maximal dual gap size for inversion (depends on scanning radius in tiles) <0 = use maximal possible
	
	// Calculation of FPN weights for averaging different poses, detecting high gradients of the real image on back-propagated average where image difference
	// may have large errors causes my discrete pixels 
	public double   cuas_w_um_sigma =       1.5;   // run UM to approximate LoG
	public int      cuas_w_good_margins =  12;     // consider defined values near image margins always good (do not have anything better anyway). Does not apply to NaN
	                                               // areas that will be filled from different poses
	public double   cuas_w_threshold=      20;	   // threshold to cut off high variations (it will be squared to compare to squared difference values)
	public double   cuas_w_scale_thresh =  1.1;    // Allow over-threshold values before GB, to block them completely after GB thresholding
	public boolean  cuas_w_expand_max =    true;   // replace diff values by max of neighbors (fill zero-square gaps) 
	public double   cuas_w_outliers =      0.0001; // Assign weights to outliers to avoid NaNs in bad areas
	public double   cuas_w_blur =          1.0;    // blur thresholded squared values to smooth transitions. 

	// filters for intersceneExport
	public boolean  cuas_dsi_parent =     false;   // Ignore this scene sequence DSI, use parent's
	public double   cuas_fg_strength =    0.65;    // minimal FG strenth to limit disparity with lower Y
	public double   cuas_disp_add =       0.015;   // allow BG (sky) disparity to be slightly higher
	public int      cuas_skip_bottom =    2;       // skip bottom lines while looking for FG
	public double   cuas_multi_strength = 0.45;    // maximal strength to use multi-tile DSI
	public double   cuas_reliable_str   = 0.8;     // use for relaible tiles if INTER-INTRA-LMA is available, not just DSI_MAIN
	
	public double   cuas_fat_zero =      50.0; // phase correlation fat zero 
	public double   cuas_cent_radius =   3.0;     // centroids center radius
	public int      cuas_n_recenter =    2;       // when cosine window, re-center window these many times
    public double   cuas_rstr =          0.01;   // minimal phase correlation maximums relative to max str
    public boolean  cuas_smooth =        true;    // used cosine window when averaging correlations
    public int      cuas_corr_pairs =     50;     // number of correlation pairs to accumulate
    public int      cuas_corr_offset =    20;     // offset between motion detection pairs
    public boolean  cuas_gaussian_ra =   true;    // use gaussian temporal Gaussian instead of running average
    public int      cuas_temporal_um =    100;    // temporal "unsharp mask" - subtract running
    public double   cuas_tum_threshold =  5.0;    // if >0, remove outliers frim the running average and recalculate RA
    
    public int      cuas_precorr_ra =     10;     // rolling average before correlation
    public int      cuas_corr_step =      2;      // correlation step when using rolling average
    
    public boolean  cuas_half_step =     true;   // half step (=cuas_corr_offset/2) when scanning for motion
    public int      cuas_max_range =     2;       // how far to extend local max: 1 3x3 neighbors, 2 - 5x5 neighbs
    public int      cuas_num_cycles =    25;      // number of cycles of testing and removing bad targets // will get out earlier 
    
    // boosting weight of moving targets
    public double   cuas_speed_min =     0.0;     // minimal pixels per range (per cuas_corr_offset)
    public double   cuas_speed_pref =    0.0;     // preferable speed (boost weights for faster targets)
    public double   cuas_speed_boost =   1.0;     // speed boost limit
	// target filtering after constant velocity accumulation
    
    // additional filtering before finding the local max
	public double   cuas_lmax_flt_neglim=-0.3;   // limit negative data to reduce ridge influence (>=0 to disable)
	public double   cuas_lmax_flt_hsigma= 1.0;   // HPF (~UM) subtract GB with this sigma from the data tile 
	public double   cuas_lmax_flt_lsigma= 1.0;   // LPF - GB result of the previous subtraction
	public double   cuas_lmax_flt_scale = 5.0;   // scale filtering result
	
	public String   cuas_sky_path =       "";    // edited not here
	public double   cuas_sky_threshold =  0.9;   // minimal value of the sky mask where target is possible
	public double [] cuas_sky_offset = {0.0, 0.0}; // add to coordinates when looking for the mask value 

	public double   cuas_lmax_fraction =  0.55;    // Check if local maximum is separated from the surrounding by this fraction of the maximum value
    public double   cuas_lmax_hack_ridge= 0.45;    // Hack for ridges: remove horizontal streaks, where average(abs()) for the line exceeds fraction of abs max 0.32-good, 0.55 - bad
    public double   cuas_lmax_radius =    4.5; // 4.5;     // look inside ((int)cuas_lmax_radius) * 2 + 1 square for the local maximum isolation 
	public boolean  cuas_lmax_zero =     false;   // true;    // zero all data outside this radius from the maximum
	
    public double   cuas_centered_radius= 3.5; // in centered mode (target placed at [8,8] - zero out data outside peripheral areas, keep 0.5 at the radius 
    public double   cuas_centered_blur =  0.5; // relative transition radius range (for 0.5 the mask inside (1-0.5) radius will be 1.0, outside (1+0.5) radius will be 0   
    public double   cuas_duplicate_tolerance = 2.0; // (pix) Remove weaker maximums closer than this to a stronger one   
    
	public double   cuas_target_radius = 2.0;  // target centroids center radius
	public double   cuas_target_strength =0.2; // target centroids strength
	public double [][] cuas_target_frac = {{0,0.04},{2.5,0.1},{5,0.3}};
	public boolean  cuas_no_border =     true;   // exclude targets with centers on the 16x16 tile edges
	// CUAS Motion LMA parameters
	public double   cuas_lma_sigma =     2.5;
	public double   cuas_wnd_pedestal =  0.1; // Add constant to Gaussian weights
	public double   cuas_lma_r0 =        3.0;  //maximum with with overshoot
	public double   cuas_lma_ovrsht =    1.75;

	public boolean  cuas_lma_fit_xy =    true;
	public boolean  cuas_lma_fit_a =     true;
	public boolean  cuas_lma_fit_c =     true;
	public boolean  cuas_lma_fit_r =     true;
	public boolean  cuas_lma_fit_k =     true;
	public double   cuas_lambda =        0.1;
	public double   cuas_lambda_good =   0.5;
	public double   cuas_lambda_bad =    8;
	public double   cuas_lambda_max =  100;
	public double   cuas_rms_diff =      0.001; // relative RMS improvement
	public int      cuas_num_iter =     20;

	// CUAS Motion LMA filter parameters
	public double   cuas_lma_rms =       1.5; // 1.5; // Maximal RMS (should always match, regardless if A)
	public double   cuas_lma_arms =      0.37; // 0.06; // Maximal absolute RMS (should match one of cuas_lma_arms OR cuas_lma_rrms (0.484)
	public double   cuas_lma_rrms =      0.28; // 0.15; // Maximal relative to A rms. OK is when (RMS < cuas_lma_arms) || (RMS < cuas_lma_rrms * A)
	public double   cuas_lma_mina =      0.4; // 1.0;  // Minimal A (amplitude) = 1.0 (< 2.0)
	public double   cuas_lma_pre_mina =  0.5; // scale  cuas_lma_mina to filter initial candidates (if there are > one of them)
	public int      cuas_min_keep =      2; // keep at least this number of candidates before using cuas_lma_pre_mina filter
	
	public double   cuas_lma_maxr =      7.0;  // Maximal R (radius)    =5.0  (> 3.8)
	public double   cuas_lma_minr1 =     0.75;  // Minimal R1 (radius of positive peak) 
	public double   cuas_lma_mink =      0.0;  // Minimal K (overshoot) = 0.0 (<0.007)
	public double   cuas_lma_maxk =      5.0;  // Maximal K (overshoot) = 5.0 (>3.8)
	public double   cuas_lma_a2a =       0.5;  // Minimal ratio of the maximal pixel to the amplitude
	
	
	
	public double   cuas_slow_ra =      50.0;  // running averaging for slow targets
	public double   cuas_slow_score =    1.0;  // scale scores for slow target mode
	public double   cuas_score_mv =      0.0;  // minimal score for the motion vectors
	public double   cuas_score_lma =     1.6;  // minimal score for the target LMA
	public double   cuas_factor_lim =    5.0;  // limit each individual score factor
	public double   cuas_factor_pow =    1.0;  // raise score factor to this power before combining
	public double [] cuas_score_coeff = {1.0, 0.05, 0.7, 1.5, 2.0, 1.0, 1.0}; //weights of amplitude, RMSE and RMSE/amplitude, center,  mismatch, match_length, travel 

	public boolean  cuas_center_targ =   true;  // re-run target extraction with targets centered to their tiles 
	public boolean  cuas_multi_targ =    true;  // Use multi-target for each tile mode 
	
	
	
	public boolean  cuas_isolated =      true;  // remove targets that do not have neighbors before/after 
	public int      cuas_min_seq =       5;     // minimal number of consecutive key frames for the same target
	public int      cuas_enough_seq =   10;     // number of consecutive key frames considered good regardless of individual scores (maybe add minimal travel in pixels)?
	public double   cuas_seq_travel =    2.5;   // minimal diagonal of the bounding box that includes sequence to be considered "cuas_enough_seq". Filtering out atmospheric fluctuations
	public int      cuas_min_seq_fl =    4;     // minimal number of consecutive key frames for the same target that includes first or last key frame
//	public boolean  cuas_remove_2seq =   true;  // Remove 2-long target sequences consisting of 2 consecutive key frames 
//	public boolean  cuas_remove_2seq_all=true;  // Remove 2-long first/last target sequences
	public double   cuas_max_mismatch =  2.0;   // maximal position error between consecutive scene sequences 
	public double   cuas_good_mismatch = 1.0;   // maximal position error between consecutive scene sequences that adds to the score
	public double   cuas_slow_fast_mismatch = 1.5; // allow larger mismatch between slow and fast
	public double   cuas_match_len_pwr = 0.5;   // raise matching length to this power for calculating score
	
	public boolean  cuas_fail_mismatch = false; // fail high mismatch early
	public boolean  cuas_ignore_mismatch = false; // calculate mismatch but do not remove

	// May be obsolete by now as there is an image mask of the sky.
	public boolean  cuas_by_horizon =      true;  // remove targets that are below horizon
	public double   cuas_horizon =         380.0; // maximal Y - position of the sky 
	
	public double   cuas_mask_width =      9;
	public double   cuas_mask_blur =       3;
	public boolean  cuas_mask_round =   true;
	
	public int      cuas_target_type =     0; // 0 - unknown, 1 - known, 2 - friend, 3 - foe
	public int      cuas_known_type =      2; // 0 - unknown, 1 - known, 2 - friend, 3 - foe
	
	public double   cuas_input_range =     5;
	public boolean  cuas_scale2x =       true;
	public boolean  cuas_ra_background = true; // apply running average to the background of the moving targets (false - use high-noise no-averaged images
	// which files to save in the model directory:

	public boolean  cuas_targets =       true; // enable targets generation together with the source UM images

	public int      cuas_iter_show =        0; // 1; // Maximal enhancement iteration to show intermediate result (0 - none) 
	public boolean  cuas_2d_save_show =  true; // save/show 2D correlation  
	public boolean  cuas_intermed_low =  true; // save intermediate vector fields and target coordinates from the LMA (80x64 layers) 
	public boolean  cuas_intermed_high = true; // save pixel-resolution accumulated images (640x512) 
	public boolean  cuas_intermed_giga = false;// save huge (gigabyte) intermediate files 
	public boolean  cuas_debug_more =    true;// save/show more images 
	public boolean  cuas_generate =      true; // generate and save targets Tiff and/or video files
	public boolean  cuas_generate_csv =  true; // generate and save targets as csv file
	public boolean  cuas_save_mono =     false; // save 32-bit monochrome target+backgrounds Tiffs (before optional scaling)
	public boolean  cuas_save_color =    true; // save color  rendered images (same as videos)
	public boolean  cuas_save_video =    true; // save color  rendered images (same as videos)
	public boolean  cuas_clean_video =   true; // save video without any debug information for targets, output in TIFF files. False - same output for video and TIFFs
	
	
	public boolean  cuas_annotate =      true; // save color  rendered images (same as videos)
	public boolean  cuas_annotate_uas =  true; // indicate uas from log if available
	public Color    cuas_text_color =    DEFAULT_cuas_text_color; // new Color (  0, 255,   0);//  220);
	public Color    cuas_text_bg =       DEFAULT_cuas_text_bg; // new Color (127, 127, 127);//  220);
	public Color    cuas_selected_color= DEFAULT_cuas_selected_color; // new Color (127,   0, 255);//  220);
	public boolean  cuas_transparent =   true;   // text bg for other targets
	public boolean  cuas_transparent_uas=false; // text bg for UAS
	
	public String   cuas_font_name =     "Monospaced";  
	public int      cuas_font_size =     8; // before scaling
	public int      cuas_font_type =     1; // 0 - PLAIN, 1 - BOLD, 2 - ITALIC

	public double   cuas_font_spacing =  1.2; //height to size ratio. If 0 - will use default spacing ( ~=1.5)
	public boolean  cuas_annot_missing = false; // Reserve a line for requested but missing parameters
	
	// AZ/EL calibration
	@Deprecated
	public double   cuas_ifov =          0.05;  // degree per pixel Use gc.getIFOVDegrees() and gc.getIFOV() instead
	public int      cuas_px0 =           283;   // pixel with known azimuth 
	public int      cuas_py0 =           386;   // pixel with known elevation
	public double   cuas_az0 =           201.5; // degrees for cuas_px0;
	public double   cuas_el0 =           0.0;   // degrees for cuas_px0;
	public boolean  cuas_show_disp =     true; // Show disparity (corrected) near target (not in clean)
	public boolean  cuas_show_rng =      true; // Show distance to target (range) in meters
	public boolean  cuas_show_inf =      true; // Show distance greater than max (or negativce) as infinity
	public boolean  cuas_show_inf_gt =   true; // Use ">max" instead of infinity symbol
	public boolean  cuas_show_true_rng = true;  // show true range (from the flight log)

	// ranging parameters
	public boolean  cuas_smooth_omegas =   true; // Recalculate omegas from continuing targets positions 
	public boolean  cuas_rng_um =          true; // apply UM to images for ranging (assumed true)
	public boolean  cuas_rng_um_all =      false;// Unsharp mask all images (after FPN and Row/Col), unchecked - when needed. Checked faster, but original data is lost. 
	public double   cuas_rng_um_sigma =    1.5;  // unsharp mask sigma for ranging images
	public boolean  cuas_rng_um2 =         true; // apply UM to images twice
	public double   cuas_rng_um_weight =   1.0;  // unsharp mask weight for ranging images
	public boolean  cuas_rng_coswnd =      false; // Use cosine scenes window (false - rectangular) 
	public int      cuas_rng_combine =     1;    // combine multiple scenes before intrascene correlation
	public double   cuas_rng_radius0 =     3.0;  // mask out data outside peripheral areas, keep 0.5 at the radius - first iteration 
	public double   cuas_rng_radius =      2.5;  // mask out data outside peripheral areas, keep 0.5 at the radius  - next iteration
	public double   cuas_rng_blur =        0.5;  // relative transition radius range (for 0.5 the mask inside (1-0.5) radius will be 1.0, outside (1+0.5) radius will be 0
	public double   cuas_rng_fz0 =        300.0; // Fat zero in target ranging mode, first iteration (first 2?)
	public double   cuas_rng_fz =         30.0;  // Fat zero in target ranging mode
	public double   cuas_rng_scale =       8.0;  // Scale alt_data to use same strength as for normal ranging because targets

	// will overwrite ImageDttParameters
	public boolean  cuas_dual_pass =       true;  // First pass - do not adjust disparity 
	public boolean  cuas_disp_only2 =      false; // Disparity only on second pass (keep scales and shape - some CUAS where failing on second pass) 
	public boolean  cuas_lma_bypass =      false; // Bypass solution tests, OK weak 
	public int      cuas_mcorr_sel =       3;     // all pairs and diagonals (as was for non-cuas ranging)
	public int      cuas_mcorr_sel_lma =   3;     // all pairs (will use all selected by cuas_mcorr_sel)
	
	
	public double   cuas_max_rel_rms =      1.2;   // maximal relative (to average max/min amplitude LMA RMS) // May be up to 0.3)
	public double   cuas_min_strength =     0.2;   // minimal composite strength (sqrt(average amp squared over absolute RMS)
	public double   cuas_ac_offset =        0.01;  // add to a,c coefficients for near-lines where A,C could become negative because of window 
	public double   cuas_min_max_ac =       0.02;  // 0.175; // 0.14;  // LWIR16: 0.01  maximal of a and C coefficients minimum (measures sharpest point/line)
	public double   cuas_min_min_ac =       0.003; // LWIR16: 0.007 minimal of a and C coefficients minimum (measures sharpest point)
	
	public boolean  cuas_reset_disparity =  true;  // reset target disparities from infinity
	public double   cuas_initial_disparity = 1.0;  // Start correlation with this disparity (in addition to infinity) after reset
	public double   cuas_infinity =         0.6466;// disparity at infinity for targets
	
	public boolean  cuas_rng_img =         false;  // Generate/save per-sensor target images
	public boolean  cuas_rng_glob=         true;   // Generate/save integrated target disparities (one per target) TODO: add first/second half
	public int      cuas_glob_ends =       0;      // 0 - same as internal, 1 - cosine extended ends, 2 - rectangular extended ends // make a parameter
	public boolean  cuas_rng_disp=         true;   // Generate/save target disparities
	public boolean  cuas_rng_vfy=          false;  // Generate/save ranging verification images (per-sensor and combined rendering from the same data)
	
	public int      cuas_rng_niterate =    6;      // number of disparity iterations
	public double   cuas_rng_diff =        0.002;  // exit when disparity difference is smaller
	
	// following targets parameter
	public int      cuas_flw_levels =      2;       // 1 - all, 2 - all and two halves, 3 - all, two halves and 4 quaters
	public int      cuas_flw_len =        10;       // Minimal number of key frames in each segment after division;
	public double   cuas_flw_diff =        0.03;    // (pix) Minimal difference between halves to use "tilted" non-constant disparity 
	
	// targets matching parameters
	public double   cuas_known_err =      20;      // pix, mark as "identified" when closer to the flight log position 
	public double   cuas_tmtch_frac =      0.8; // target should match flight log in theis fraction of keyframes
	public int      cuas_tmtch_ends =      6; // 2;   // maximal number of undetected first/last keyframes to compare to previous/next sequence
	public int      cuas_tmtch_gaps =     10; //  4;   // maximal number of consecutive undetected keyframes of the same target
	public double   cuas_tmtch_pix  =     25.0; // maximal pixel mismatch for UAS target 
	public double   cuas_local_apix  =     2.0;  // maximal absolute pixel mismatch for local targets matching (tightened)
	public double   cuas_tmtch_apix  =    5.0;  // 10.0; // maximal absolute pixel mismatch for the same target extrapolated with the average of the before/after lateral speeds
	public double   cuas_tmtch_rpix  =    1.0;  // maximal pixel/s mismatch for the same target extrapolated with the average of the before/after lateral speeds
	public double   cuas_tmtch_axv =      50.0; //20.0; // m/s maximal axial (range) velocity in m/s
	public double   cuas_tmtch_axv_k =     3.0; // maximal axial (range) velocity to lateral velocity ratio (take max)
	public double   cuas_local_diff =      1.5;  // maximal disparity difference to match 3d for local target match (loosened for local)
	public double   cuas_tmtch_diff =      0.06;// maximal disparity difference to match 3d (regardless of absolute disparity)
	public double   cuas_tmtch_disp =      0.2; // 0.1; // minimal disparity (any should be above) to estimate lateral velocity (smaller - match2d only)
	public boolean  cuas_tmtch_short =    true; // when multiple conflicting matches, prefer shortest gap (false - longest combo pair length)
	
	// parameters for "radar" image generation 
	public double   cuas_radar_range =    1500.0; // maximal radar range in meters
	public int      cuas_radar_height =  950; // height of the radar finite range view (pix)
	public int      cuas_radar_bottom =   10;  // radar window bottom margin
	public int      cuas_radar_top =      24;  // infinity targets from window top
    public Color    cuas_radar_uas_color = DEFAULT_cuas_radar_uas_color; // new Color(  0,100,140); True UAS position sircle
    public Color    cuas_radar_color =   DEFAULT_cuas_radar_color; //new Color(  0,255,100); Detected color
    public double   cuas_radar_radius =    2.5;
    public double   cuas_radar_uas_rad =   4.0;
	// parameters for "radar" grid generation 
	public double   cuas_grid_ring = 100.0; // (m) range rings step in meters
	public double   cuas_grid_line =   5.0; // (deg) grid axial directions step
	public boolean  cuas_grid_annot =true; // annotate grid
	public Color    cuas_grid_color =   DEFAULT_cuas_grid_color; // new Color(100,100,100); // Radar grid and grid annotations color
	public int      cuas_grid_font =   7; // before scaling
	public int      cuas_grid_az_top = 5; // where to put azimuth
	public int      cuas_grid_sides =  5; // grid annotation from left/right
	
	// Annotation selection
	public int []   cuas_annot_sel = CuasMotion.ANNOT_DEFAULTS.clone();
	
	// debug parameters
	public boolean  cuas_debug =           false;  // save debug images (and show them if not in batch mode)
	public int      cuas_dbg_rng_seq =    -1;     // Debug single sequence ranging (-1 - process all)
	public int      cuas_dbg_rng_tgt =    -1;     // Debug single target ranging (-1 - process all), >0 - target id
	public int      cuas_dbg_show_mode =   0;     // Debug show ranging debug images: +1 - disparity, +2 - correlations +4 - debug single run
	public boolean  cuas_step_debug =      false;  // save debug images during per-step cuas recalculation (and show them if not in batch mode)
	public boolean  cuas_save_stats =      false;  // Save target statistics data to find out which test failed and when (or when tile was found good)
	public boolean  cuas_target_debug =    false; // save debug images during per-step cuas recalculation (and show them if not in batch mode)
	public boolean  cuas_target_score =    true;  // show target score and sequence length in the final tiff/video (not in clean video)
	public boolean  cuas_target_color =    true;  // annotate known target with different color
	public boolean  cuas_log_ranging =     true;  // Log ranging iterations
	
	public boolean  cuas_overwrite =       false; // overwrite num_orient and num_accum
	public int      cuas_num_orient =      2;     // initial value for num_orient
	public int      cuas_num_accum =       1;     // initial value for num_accum
	
	public boolean  cuas_reuse_targets =   true;  // read previously calculated non-conflict (one per tile) targets
	public String   cuas_reuse_path =      "-TARGETS_SINGLE-FINAL";    // either suffix (all parameters the same) or the full path (contains "/") 
	public boolean  cuas_reuse_disparity = true;  // read previously calculated non-conflict targets data with disparities (recalculate ranges)
	public boolean  cuas_reuse_globals =   true;  // if globally combined data is available, read it. If successful show 16:9 video
	public double   cuas_max_disp_diff =   0.05;  // Maximal disparity difference during last change to consider disparity valid
	public double   cuas_min_disp_str =    0.4;   // Minimal disparity strength to consider disparity valid
	public double   cuas_rng_limit =       5000;  // maximal displayed distance to target
	
	
	// TODO: move next parameters elsewhere - they are not the motion blur ones.
	public  int     mb_gain_index_pose =    5;     // pose readjust pass to switch to full mb_max_gain from mb_max_gain_inter
	public  int     mb_gain_index_depth =   5;     // depth map refine pass (SfM) to switch to full mb_max_gain from mb_max_gain_inter
	public  int     mb_ers_index =          3;     // pose readjust pass to enable ERS 
	public  int     mb_ers_y_index =        3;     // pose readjust pass to enable ERS in vertical (Y) direction - it competes with SfM
	public  int     mb_ers_r_index =        3;     // pose readjust pass to enable ERS in vertical (Y) direction - it competes with SfM
	public  int     mb_all_index =          4;     // simultaneously LMA-adjust XYZATR plus optional angular ERS
    public  boolean mb_pref_orient =        true;  // rely on IMU orientation before position
    public  boolean lma_use_Z =             true;  // LMA adjust Z
    public  boolean lma_use_R =             true;  // LMA adjust R
    
    // FIXME: some of the following parameters will be modified or removed 
    public  boolean configured_lma =         false;
    public  boolean freeze_xy_pull =         true;  // false; // true; // debugging freezing xy to xy_pull
    public  boolean copy_pull_current =      false; // true;
    public  boolean restore_imu =            false; // restore imu omega-az and omega-tl, freeze ERS, adjust X,Y,Z,Az,Tl,Rl
    public  boolean lpf_xy =                 false; // lpf x and y, re-adjust X,Y,Z,A,T,R with pull for X,Y. Disables  
    public  boolean readjust_xy_ims =        true;  // false;
    public  double  reg_weight_xy =          0;     // 10.0; // 1.0; // 10.0; // 0.05; // TODO: find out reasonable values
    
    // Quaternion LMA parameters
    public  double  quat_lambda =            0.1;
    public  double  quat_lambda_scale_good = 0.5;
    public  double  quat_lambda_scale_bad =  8.0;
    public  double  quat_lambda_max =      100;
    public  double  quat_rms_diff =          0.001;
    public  int     quat_num_iter =         20;
    public  double  quat_reg_w =             0.25;

    public  double  quat_max_ratio =         5.0; // max derivatives difference 
    public  double  quat_max_change =        0.1; // radians
    public  double  quat_min_transl =        5.0; // meters to adjust by translation
    public  double  quat_min_rot =           0.1; // radians to adjust by rotation
    public  double  quat_min_lin =          10.0; // meters, minimal distance per axis to adjust IMS velocity scale 
	
	public  boolean stereo_merge =       true;
	public  int     stereo_gap =         32;    // pixels between right and left frames
	public  double  stereo_intereye =    63.5;  // mm 
	public  double  stereo_phone_width = 143.0; // mm if 0 - no padding. Will be padded only when merging

	public  int     extra_hor_tile =     15;
	public  int     extra_vert_tile =    10;
	public  boolean crop_3d =            true; // do not show extra of FG/BG views (currently they only ref scene has disparity)
	public  int     sensor_mask =        1; // -1 - all
	public  boolean merge_all =          true; // merge all 16 channels for 3D modes 

	public  boolean show_mapped_color =  true;
	public  boolean show_mapped_mono =   true;

	public  boolean gen_avi_color =      true; // will use save_mapped_color
	public  boolean gen_avi_mono =       true; // will use save_mapped_mono
	public  double  video_fps =          15; // 4x slowmo
	public  double  sensor_fps =         60; // sensor fps
	public  int     mode_avi =           0;  // 0 - raw, 1 - JPEG, 2 - PNG
	public  int     avi_JPEG_quality =  90;
	public  boolean run_ffmpeg =         true; // only after AVI
	public  String  video_ext =          ".webm";
	public  String  video_codec =        "vp8";
	public  int     video_crf =         40;    // lower - better, larger file size
	public  boolean remove_avi =         true; // remove avi after conversion to webm
	public  double  video_bitrate_m =    50; // video bitrate in megabits/sec
	public  String  video_codec_combo =  "vp8"; // applies when combining videos
	public  int     video_crf_combo =   40;    // lower - better, larger file size applies when combining videos
	public  boolean add_average =        true;
	public  boolean calculate_average =  false; // Calculate average from the slices. False - use CLT average if available.
	public  boolean subtract_average =   false;
	public  int     running_average =    0;
	public  boolean extract_center_orientation = true; // in lock_position mode only - debug feature
	public  boolean um_mono =            true; // applies to both TIFF and AVI 
	public  double  um_sigma =          10;
	public  double  um_weight =          0.97; //
	public  boolean um_mono_linear =     true; // save linear (no-UM) in addition to UM
	public  boolean mono_fixed =         true; // normalize to fixed range when converting to 8 bits 
	public  double  mono_range =       500.0;  // monochrome full-scale range (+/- half)
	public  boolean anaglyth_en =        true; // applies to both TIFF and AVI 
	public Color   anaglyph_left =    anaglyph_left_default;
	public Color   anaglyph_right =   anaglyph_right_default; // cyan
	
	public  boolean annotate_color =     true; // annotate pseudo-color video frames with timestamps 
	public  boolean annotate_mono =      true; // annotate monochrome video frames with timestamps 
	public Color   annotate_color_color =new Color( 255, 255, 255); // greenish over "fire"
	public Color   annotate_color_mono =   new Color( 255, 180,  50); // reddish over grey
	public boolean annotate_transparent_mono =  false; // // black if not transparent
	
	public String    terr_model_path =  "/media/elphel/SSD3-4GB/lwir16-proc/berdich3/models/models_1697875868-1697879449-b/1697877487_245877/v35";
	public String    terr_model_state = "1697877487_245877-TERR-VEG-STATE"; 
	public String    terr_segments_dir = "sections"; 
	public String    terr_segments_suffix = "-new"; 
	public String    terr_par_dir =           "";      // directory path to intermediate parameters
	public String    terr_par_file =          "";      // file name of the  intermediate parameters
	public boolean   terr_par_restore =       false;   // restore parameters from file
	public int       terr_step_restore =        2;     // If restore, start with step after restored 
	// terrain/vegetation parameters
	@Deprecated
	public boolean   terr_um_en =             true;
//	@Deprecated
	public double    terr_um_sigma =           1.0; // use for render 
//	@Deprecated
	public double    terr_um_weight =          0.8; // use for render
	@Deprecated
	public double    terr_nan_tolerance =      0.001;  // set undefined terrain (out of capture area) to nan, it is almost zero after UM. if !terr_um_en will be forced ==0.0
	public int       terr_nan_grow =          20;      // grow undefined scenes terrain after detection 
    public int       terr_shrink_veget =      20;      // shrink accumulated vegetation for filling terrain
    public int       terr_shrink_terrain=     20;      // shrink accumulated terrain after cutting shrunk vegetation (added to terr_shrink_veget)
	public double    terr_vegetation_over =   35;      // initial vegetation over (hotter) filled terrain
    public int       terr_filter_veget =      10;      // shrink+grow shrunk vegetation to remove small clusters
    public int       terr_veget_grow =         2;      // grow (fill NaN holes) vegetation when extracting from multi-scene sequences
	
	public boolean   terr_tile_woi =          true; // if false - use woi50;
	public Rectangle terr_woi_enclos =        new Rectangle(80,  210, 210, 230); // will be tiled, using width/height from woi_step;
	public Rectangle terr_woi_step =          new Rectangle(10,10,20,20);
	public Rectangle terr_woi_last =          new Rectangle(0,0,0,0); // also single 
	public boolean   terr_skip_exist=         true; // skip existing woi/parameters, already saved.
	public boolean   terr_continue=           false; // start from last/single
	
	public boolean   terr_fit_terr =          true;  // adjust terrain pixels
	public boolean   terr_fit_veget =         true;  // adjust vegetation pixels
	public boolean   terr_fit_alpha =         true;  // adjust vegetation alpha pixels
	public boolean   terr_fit_scenes =        true;  // adjust scene offsets (start from 0 always?)
	public boolean   terr_fit_elevations =    false; // adjust elevation pixels (not yet implemented)
	public boolean   terr_fit_terr_elev =     false; // adjust terrain elevation (common, maybe add per-pixel later)
	public boolean   terr_fit_terr_elev_pix = false; // adjust terrain elevation (common, maybe add per-pixel later)
	public boolean [][] terr_fits_disable =      {{false},{false},{false},{true,false,true},{false},{false},{false}};
	public boolean [][] terr_only_fits_disable = {{false},{false},{false},{true,false,true},{false},{false},{false}};
	
	public double    terr_max_warp =          1.8;
	public int       terr_max_elevation =     22;    // maximal offset to consider when looking for vegetation influence
	public int       terr_max_elev_terr =      2;    // maximal offset to consider when looking for terrain of variable elevation 
	public double    terr_max_elev_chng =    0.5;  // maximal terrain elevation change from last successfully used 

	
	@Deprecated
	public int       terr_min_scenes =         1; // minimal number of scenes (inside woi) vegetation pixel must influence
	
	@Deprecated
	public int       terr_min_samples_scene = 10;
	@Deprecated
	public int       terr_min_total_scenes =   2;
	@Deprecated
	public int       terr_min_pixels =        10;
	
	@Deprecated
	public boolean   terr_warm_veget =        true; // start with vegetation warmer than terrain
	@Deprecated
	public double    terr_warmest =           90.0;  // above - vegetation. below - terrain
	@Deprecated
	public double    terr_initial_split =      0.1;  // initial alpha: terrain 0.0+, vegetation 1.0-
	@Deprecated
	public double    terr_min_split_frac =     0.15; // minimal modality fraction to use split by temperature
	
	public double    terr_difference =        100.0;     // vegetation is 100 warmer (target)
	public double    terr_pull_cold =          0.001;    // pull vegetations to warm, terrain to cold
	public double    terr_elevation_radius =   1.5;      // Radius of elevation/vegetation influence
	public double    terr_terr_elev_radius =   1.5;      // Terrain elevation radius
	public double    terr_elev_radius_extra =  1.2;      // scale both radii when setupElevationLMA(), and setupTerrainElevationPixLMA() (not setupTerrainElevationLMA()) 
	
	public double    terr_alpha_contrast =     1.0; // initial alpha contrast (>=1.0)
	public double    terr_alpha_dflt =         0.5; // now unused
	
    public double    terr_alpha_sigma =        8.0;  // Initial alpha: Gaussian blur sigma to find local average for vegetation temperature. 
	public double    terr_alpha_init_min =     0.7;  // Initial alpha: fraction for transparent 
	public double    terr_alpha_init_max =     0.95; // Initial alpha: fraction for opaque 
	public double    terr_alpha_init_offs =    0.01; // Initial alpha: opaque/transparent offset from 1.0/0.0 
    
	public double    terr_alpha_loss =       100.0;
	public double    terr_alpha_loss_lin =     0.0;
	public double    terr_alpha_offset =       0.1; // alpha offset near alpha ~=1.0
	public double    terr_alpha_0offset =      0.0; // New - a separate alpha offset nea alpha ~= 0.0
	public double    terr_alpha_min_veg =      0.5; // old version 0.5;  // Minimal vegetation alpha. If (alpha-alpha_offset)/(1-2*alpha_offset) < alpha_min_veg, pull down to lpha_offset
	public double    terr_alpha_max_terrain =  0.75;// increase pull vegetation if below
	public double    terr_alpha_pull_pwr =     1.0; // raise extra pull to that power
	
	public double    terr_alpha_lpf =          2.5;  // pull to average of 4 neighbors
	public double    terr_alpha_lpf_border =   1.0;  // pull to average of 4 neighbors for border tiles (to keep stable)
	public boolean   terr_alpha_piece_linear = true;
	public double    terr_alpha_scale_avg =    1.0;  // scale average alpha (around 0.5) when pulling to it
	public double    terr_alpha_push =        12;    // push from alpha==0.5
	public double    terr_alpha_push_neutral = 0.5;  // alpha point from which push (closer to opaque)
	public double    terr_alpha_weight_center =1.5;  // weight of center alpha pixel relative to each of the 4 ortho ones
	public boolean   terr_en_holes =           true; // enable small holes // maybe second pass after good fit with vegetation and search for correct offset?
	public double    terr_alpha_mm_hole    =   0.1;  // NaN to disable. Local "almost minimum" (lower than this fraction between min and max neighbor) is not subject to alpha_lpf
	public double    terr_alpha_diff_hole  =   0.01; // Minimal alpha difference between min and max neighbor to be considered a hole
	public double    terr_terr_lpf =           0.1;  // pull terrain to average of 4 neighbors
	public double    terr_veget_lpf =         10.0;  // pull vegetation to average of 4 neighbors
	public double    terr_elev_lpf =          50.0;  // pull elevation to average of 4 neighbors
	public double    terr_terr_elev_lpf =     10.0;  // pull terrain elevation to average of 4 neighbors
	public double    terr_terr_pull0 =         0.1;  // pull terrain to initial (pre-adjustment) values
	public double    terr_terr_pull_up=        0.25; // pull terrain to initial when it is colder
	public double    terr_terr_pull_avg=       0.1;  // Pull terrain to the initial offset by the average offset of all terrain pixels  
	public double    terr_veget_pull0 =        5.0;  // pull vegetation initial (pre-adjustment) values
	public double    terr_veget_pull_low_alpha = 10; // scale (boost) pull0 for low alpha (mostly terrain) 
	public double    terr_elev_pull0 =         1.0;  // pull elevation to initial (pre-adjustment) values
	public double    terr_terr_elev_pull0 =    1.0;  // pull terrain elevation to average of all segment terrain elevation
	
	public boolean   terr_elev_alpha_en =      false;// Enable loss for low vegetation with high opacity
	public double    terr_elev_alpha =         1.0;  // multiply alpha by under-low elevation for loss
	public double    terr_elev_alpha_pwr =     2.0;  // raise alpha to this power (when alpha > 0)

	
	public double    terr_low_veget =          2.0;  // (pix) Elevation considered low (lower loss for high alpha)
	public double    terr_scenes_pull0 =       1.0;  // pull average scene offset to zero
    // scaling elevation losses for high elevations (decrease pull and/or lpf for high elevations)
	public double    terr_elev_scale_thresh =  1.0;  // reduce losses for higher (initial) elevations TODO: consider actual elevations
	public boolean   terr_elev_scale_pull =   false;  // scale elevation pull losses for high elevations 
	public boolean   terr_elev_scale_lpf =    false;  // scale elevation diffusion losses for high elevations 
	// tree-top removal
	public boolean   terr_ttop_en =           false; // remove tree tops from transparency weights
	public double    terr_ttop_gb =           1.0;   // Elevation Gaussian blur sigma to detect tree tops
	public double    terr_ttop_min =          3.0;   // Minimal tree top elevation
	public double    terr_ttop_rel_lev =      0.9;   // Relative (to the top height) sample level
	public double    terr_ttop_rel_rad =      0.25;  // Relative (to the top height) sample ring radius
	public double    terr_ttop_frac =         0.5;   // Minimal fraction of the ring pixels below sample level
	public double    terr_ttop_rem_rad =      0.25;  // Relative (to the top height) remove transparency radius
	
	
	
	

	// LMA parameters        
	public double    terr_boost_parallax =     3.0;  //
	public double    terr_max_parallax =      10.0;  // parallax limit when evaluating boost parallax
	public double    terr_hifreq_weight =     10.0; // 22.5; //  0 - do not use high-freq. Relative weight of laplacian components differfences to the DC ones
	public double    terr_terr_corr =          1.0; // (~>30- unstable) relative weight of average mismatch between images and model (terrain corrections)
	public double    terr_reg_weights =        0.2; // fraction of the total weight used for regularization
	
	public double    terr_lambda =             5.0;  // 
	public double    terr_lambda_scale_good =  0.5;
	public double    terr_lambda_scale_bad =   8.0;
	public double    terr_lambda_max =      1000;
	public double    terr_rms_diff =           1e-8; // 0.0001; virtually forever
	public int       terr_last_series =       -1;  // Last LMA adjustment series (<0 - to the terr_num_iters.length -1)
	public int []    terr_num_iters =        {30}; //sequence on numbers of iterations
///	public boolean   terr_only_special =       true; // special sequences for terrain-only tiles
///	public boolean   terr_only_pix =           true; // force per-pixel terrain elevation in terrain-only mode, overwrite fits_disable[TVAO_TERR_ELEV_PIX]
	public int       terr_only_series =       -1;   // similar to terr_last_series but for terrain-only mode (<0 - length of terr_only_num_iters)
	public int []    terr_only_num_iters =   {25};  // number of iterations
	public int       terr_cholesky =           0;
	
// second run
	public boolean [] terr_recalc_weights =    {false,false,true}; // recalculate weight depending on terrain visibility
	
	
	public double    terr_recalc_opaque =      0.9;  // above is opaque  
	public double    terr_recalc_pedestal =    0.05; // weight of opaque tiles - apply after normalization
	public double    terr_recalc_frac =        0.5;  // discard transparency below this fraction of the maximal one
	public double    terr_recalc_dist =        0.0;  // increase weight for far pixels (double if scale differece == this)
	public double    terr_recalc_pwr =         1.0;  // raise transparency to this power
	public double    terr_recalc_gb =          1.5;  // blur transparency-based confidence
	public double    terr_recalc_boost =       3.0;  // maximal weight boost for texture with low visibility 
	public boolean   terr_recalc_average =    false; // apply transparency to average mismatch
	
	
	// combine  parameters
	public boolean   terr_crop =               true; // Crop output image to the bounds of all segments
	public boolean   terr_keep_partial =       true; // Keep partial images (false - only accumulated)
	public int       terr_border_width =       6;
	public boolean   terr_render_open =        true; // render open areas (no vegetation offset)
	public boolean   terr_render_no_alpha =    true; // render where no opacity is available
	public double    terr_alpha_min =          0.1;  // below - completely transparent vegetation
	public double    terr_alpha_max =          0.8;  // above - completely opaque
	public double    terr_weight_opaque =      0.02; // render through completely opaque vegetation
	public double    terr_boost_render =       3.0;  // boost parallax during render
	public double    terr_max_render =        10.0;  // maximal parallax during render
	
	public int       terr_num_exaggerate =     3;    // generate exaggerated rendering   
	
	// Experimental reconstruction
	public boolean   terr_rebuild_elev =       false; // rebuild elevations and scales
	public int       terr_elev_grow =          1024; // was 200
	public double    terr_threshold_terrain =  0.05;
	public double    terr_min_max_terrain=     0.1;
	public double    terr_min_terrain =        0.001;
	public double    terr_min_vegetation =     0.5;
	public int       terr_debug =              0;
	public String    terr_debug_path =        "/media/elphel/SSD3-4GB/lwir16-proc/berdich3/debug/vegetation/lma_um/";
	public int       terr_debug_length =       200;
	public boolean   terr_debug_improved =     true; // Save debug data if LMA step improved
	public boolean   terr_debug_worsened =     true; // Save debug data if LMA step improved
	
	// synthesis
	public String    synth_directory =       "/media/elphel/SSD3-4GB/lwir16-proc/berdich3/restore_foliage/01/";
	public String    synth_scene =           "1697877490_630338";
	public String    synth_model =           "-preview_segments-0-0-640-512-ACCUM.tiff";
	public String    synth_segments =        ".vegetation-segments"; // serialized: number of models, array of models
	public String    synth_render =          "_synth-render.tiff";
	public boolean   synth_crop_real =       true; // crop synthetic by real 
	public String    synth_scene_offs =      "-0-0-640-512-SCENE_OFFSETS.tiff";
	public boolean   synth_add_offs =        true; // crop synthetic by real 
	
	
//	public String ter_rendered_path = "/media/elphel/SSD3-4GB/lwir16-proc/berdich3/debug/vegetation/essential/1697877491_613999-terrain_vegetation_render.tiff";
//	public String ter_warp_path =     "/media/elphel/SSD3-4GB/lwir16-proc/berdich3/debug/vegetation/essential/1697877491_613999-combo_offset.tiff";
	
	
	
	
	public boolean renderRef()            {return renderRef             (debug_level);}
	public boolean renderScene()          {return renderScene           (debug_level);}
	public boolean show2dCorrelations()   {return show2dCorrelations    (debug_level);}
	public boolean showMotionVectors()    {return showMotionVectors     (debug_level);}
	public boolean showCorrMotion()       {return showCorrMotion        (debug_level);}
	public boolean showMovementDetection(){return showMovementDetection (debug_level);}
	public int movDebugLevel()            {return movDebugLevel         (debug_level);}

	public boolean renderRef(int dlev)            {return (dlev>1) && render_ref;}
	public boolean renderScene(int dlev)          {return (dlev>1) && render_scene;}
	public boolean show2dCorrelations(int dlev)   {return (dlev>1) && show_2d_correlations;}
	public boolean showMotionVectors(int dlev)    {return (dlev>1) && show_motion_vectors;}
	public boolean showCorrMotion(int dlev)       {return (dlev>0) && show_motion_vectors;}
	public boolean showMovementDetection(int dlev){return (dlev>0) && mov_show;}
	public int movDebugLevel(int dlev)            {return (dlev > -1) ? mov_debug_level : 0;}
	
	
	public IntersceneMatchParameters() {
		
	}

	public double [][] get_pimu_offs(){
		// TODO: use ims_ortho, ims_mount_atr
		return new double [][] {
			{ pimu_acc_offs[0] + 1,  pimu_acc_offs[1] + 1, pimu_acc_offs[2] + 1},
			{-pimu_gyr_offs[2],     -pimu_gyr_offs[1],     pimu_gyr_offs[0]}}; // {XYZ,ATR}
	}
	
	public void set_pimu_omegas(double [] atr) {
		pimu_gyr_offs[2] = - atr[0];
		pimu_gyr_offs[1] = - atr[1];
		pimu_gyr_offs[0] =   atr[2];
	}
	
	public void set_pimu_velocities_scales(double [] vxyz) {
		pimu_acc_offs[0] = vxyz[0] - 1;
		pimu_acc_offs[1] = vxyz[1] - 1;
		pimu_acc_offs[2] = vxyz[2] - 1;
	}
	
	
	public void dialogQuestions(GenericJTabbedDialog gd) {
		//		gd.addMessage  ("Scene parameters selection");
//		gd.addTab         ("Inter-Match", "Parameters for full-resolution (no decimation/macrotiles) scene matching");
		gd.addTab         ("IMS", "IMS Integration for interscene matching");
		gd.addCheckbox ("Use IMS data",                              this.ims_use,
				"Use IMS data for egomotion.");
		gd.addCheckbox ("Recalculate IMS",                           this.ims_rebuild,
				"Recalculate scene INS data.");
		gd.addNumericField("IMS offset",                             this.ims_offset, 5,8,"s",
				"Positive if the IMS data has lower lag than images.");
		gd.addNumericField("GMT plus",                               this.gmt_plus, 5,8,"hrs",
				"Timestamp filename is lower by this hours than in the IMU log.");
		
		gd.addStringField ("IMS ortho mount quaterion", IntersceneMatchParameters.doublesToString(ims_ortho), 80,
				"Approximate (90-deg) IMS to camera {0.5, -0.5, 0.5, -0.5}.");
		gd.addStringField ("IMS mount ATR correction (deg)",  IntersceneMatchParameters.doublesToString(ims_mount_atr), 80,
				"IMS mount fine correction (A,T,R around camera axes).");
		gd.addStringField ("IMS mount XYZ correction (m)",  IntersceneMatchParameters.doublesToString(ims_mount_xyz), 80,
				"MS center (X,Y,Z m) in camera coordinates.");

		gd.addStringField ("IMU angular corrections (rad/s)",  IntersceneMatchParameters.doublesToString(pimu_gyr_offs), 80,
				"Average  from did_pimu.theta[]/did_pimu.dt when not moving.");

		gd.addStringField ("IMU accelerators gain",  IntersceneMatchParameters.doublesToString(pimu_acc_offs), 80,
				"IMU accelerators gain difference from 1. Negative values when IMU-reported velocities are smaller than actual");

		gd.addStringField ("scale XYZ from IMS", IntersceneMatchParameters.doublesToString(ims_scale_xyz), 80,
				"Scales for X, Y, and Z velocities when converting from IMS data, such as '1.1 1.1 1.1'.");
		gd.addStringField ("scale ATR from IMS", IntersceneMatchParameters.doublesToString(ims_scale_atr), 80,
				"Scales for Azimuth, Tilt, and Roll velocities when converting from IMS data, such as '1.1 1.1 1.1' (not needed).");
		
		gd.addMessage  ("IMS-based FPN mitigation for interscene matching");
		gd.addCheckbox ("FPN mitigation for initial orientation",   this.fmg_initial_en,
				"Enable IMS-based FPN mitigation for initial orientation.");
		gd.addCheckbox ("FPN mitigation for re-orientation",        this.fmg_reorient_en,
				"Enable IMS-based FPN mitigation for readjustmnet orientation.");
		gd.addNumericField("Maximal inter-scene offset",            this.fmg_distance, 5,8,"pix",
				"Try to find other reference scene not closer than this pixels.");
		gd.addNumericField("Maximal offset for quad-point use",     this.fmg_max_quad, 5,8,"pix",
				"Estimate offset by 4 points (rooll-aware, 25% from center) if center offset is too small.");
		gd.addCheckbox ("Rectilinear model for offset estimation",  this.fmg_rectilinear,
				"Use rectilinear model for scene offset estimation.");
		
		gd.addTab         ("SfM", "Structure from Motion to improve depth map for the lateral views");
		gd.addCheckbox ("Use SfM",                                   this.sfm_use,
				"Use SfM for the depth map enhancement for laterally moving camera.");
		gd.addNumericField("Minimal SfM baseline",                   this.sfm_min_base, 5,8,"m",
				"Use SfM only if the baseline (lateral offset between scenes in a series) exceeds this value.");
		gd.addNumericField("Minimal SfM gain",                       this.sfm_min_gain, 5,8,"",
				"Minimal SfM gain to apply SfM to the depth map.");
		gd.addNumericField("Minimal SfM fraction",                   this.sfm_min_frac, 5,8,"",
				"Minimal fraction of defined tiles to have SfM correction.");
		gd.addNumericField("Number of SfM pairs to average",         this.sfm_num_pairs, 0,3,"",
				"Desired number of SfM pairs to average, will be reduced if not enough or tolerance (below) is violated.");
		gd.addNumericField("SfP inter-pair baseline tolerance",      this.sfp_tolerance, 5,8,"",
				"Average SfM pairs if their baselines differ less.");
		gd.addNumericField("Readjust SfM cycles",                    this.sfm_readjust, 0,3,"",
				"Number of SfM readjustment cycles.");
		gd.addNumericField("Minimal SfM gain from previous",         this.sfm_prev_frac, 5,8,"",
				"Update if new sfm gain > this fraction of the old one.");
		gd.addNumericField("Correction scale for equal gains",         this.sfm_same_weight, 5,8,"",
				"Correction scale when new SfM gain is the same as the old one.");
		gd.addNumericField("Shrink SfM area",                        this.sfm_shrink, 0,3,"",
				"Shrink sfm gain area before applying sfm_fade_sigmas.");
		gd.addNumericField("Fade SfM area sigma",                    this.sfm_fade_sigma, 5,8,"",
				"Fade SfM gains at the edges.");
		gd.addNumericField("Minimal correlation strength",           this.sfm_min_str1, 5,8,"",
				"Update if correction strength of individual tiles exceeds.");
		gd.addNumericField("Minimal correlation strength >= 16 pairs", this.sfm_min_str16, 5,8,"",
				"Update if correction strength of individual tiles exceeds (for >=16 pairs).");
		gd.addCheckbox ("Use neighbors",                             this.sfm_use_neibs,
				"Use neighbors if individual corr is too weak.");
		gd.addNumericField("Too strong neighbors >= 16 pairs",       this.sfm_neib_too_str1, 5,8,"",
				"Do not accumulate neighbors stronger than that.");
		gd.addNumericField("Too strong neighbors",                   this.sfm_neib_too_str16, 5,8,"",
				"Do not accumulate neighbors stronger than that (for >=16 pairs).");
		gd.addNumericField("Minimal neighbors strength",             this.sfm_neib_str1, 5,8,"",
				"Update if no-individual and neighbors correction strength exceeds this.");
		gd.addNumericField("Minimal neighbors strength >=16 pairs",  this.sfm_neib_str16, 5,8,"",
				"Update if no-individual and neighbors correction strength exceeds this (for >=16 pairs).");
		gd.addNumericField("Extrapolation steps",                    this.sfm_extrap_steps, 0,3,"",
				"Extrapolate disparity this number of times over the undefined by SfM area.");
		gd.addNumericField("Extrapolation radius",                   this.sfm_extrap_radius, 0,3,"",
				"Find tilt for extrapolation using this radius circle around the new tile.");
		gd.addCheckbox ("Average neighbors",                         this.sfm_average_neibs,
				"Average neighbors disparity among them.");
		gd.addCheckbox ("Fill too weak tiles",                       this.sfm_fill_weak,
				"Fill too weak tiles from nearest neighbors.");
		gd.addCheckbox ("Extrapolate no SfM data",                   this.sfm_extrapolate,
				"Extrapolate tiles with no SfM data (near edges).");

		gd.addMessage  ("SfM filter reference disparity");
		gd.addNumericField("Maximal dip",                            this.sfm_max_dip, 5,8,"pix",
				"Maximal dip relative to lowest neighbor (0 - any).");
		gd.addNumericField("Maximal bump",                           this.sfm_max_bump, 5,8,"pix",
				"Maximal bump relative to highest neighbor (0 - any).");
		gd.addCheckbox ("Replace NaNds",                             this.sfm_filter_nan,
				"Replace NaNs by defined neighbors average.");
		gd.addCheckbox ("Process SfM only",                          this.sfm_only,
				"Process only series with SfM (temporary feature to fix previously wrong SfM).");
		
		gd.addMessage  ("SfM debug images");
		gd.addCheckbox ("Save SfM sequence image",                   this.sfm_save_seq,
				"Save SfM sequence image in model directory.");
		gd.addCheckbox ("Show SfM sequence",                         this.sfm_show_seq,
				"Show SfM sequence (one slice per refinement run).");
		gd.addCheckbox ("Show SfM details per run",                  this.sfm_show_corr_ind,
				"Show SfM for each pairs group, each refinement run.");
		gd.addCheckbox ("Show SfM combo per run",                    this.sfm_show_corr,
				"Show combined SfM for each refinement run.");

		gd.addTab      ("Mosaic", "Combining scene series into a composite map map");
		
		gd.addChoice      ("Files list/data path (w/o extension):", ComboMatch.FILES_LISTS_PATHS, omtch_img_set);
		
		gd.addNumericField("GPU window width",                       this.rln_gpu_width, 0,5,"pix",
				"Horizontal GPU image size (applied during init.)");
		gd.addNumericField("GPU window height",                      this.rln_gpu_height, 0,5,"pix",
				"Vertical GPU image size (applied during init).");
		
		
		
		gd.addMessage  ("Rectilinear image matching");
		gd.addNumericField("Phase correlation fat zero",             this.rln_fat_zero, 5,8,"",
				"Phase correlation fat zero - was set to 10000.");
		gd.addCheckbox ("Calculate neighbors",                       this.rln_use_neibs,
				"Calculate TD neighbors in phase correlation .");
		gd.addCheckbox ("Fill empty neighbors centers",              this.rln_neibs_fill,
				"Fill empty neighbors centers (false - only non-empty).");
		gd.addNumericField("Neighbors radius",                       this.rln_neib_radius, 5,8,"tiles",
				"Use these tiles around the center one.");
		gd.addNumericField("Neighbors radius fraction woi",          this.rln_radius_frac, 5,8,"",
				"Naighbors radius not less than this fraction of tiles WOI.");
		gd.addNumericField("Centroids radius",                       this.rln_cent_radius, 5,8,"",
				"Centroids radius for maximums isolation.");
		gd.addNumericField("Recenter centroid",                      this.rln_n_recenter, 0,3,"",
				"when cosine window, re-center window these many times");
		gd.addNumericField("Minimal relative strength (single tiles)",this.rln_sngl_rstr, 5,8,"",
				"Minimal single-tile phase correlation maximums relative to maximal strength.");
		gd.addNumericField("Minimal relative strength (neighbors)",  this.rln_neib_rstr, 5,8,"",
				"Minimal neighbors phase correlation maximums relative to maximal strength.");

		gd.addMessage  ("Pairwise scenes matching");
		gd.addCheckbox    ("Use scenes' affine",               this.pmtch_use_affine, "Use known scenes' affine matrices, false - start from scratch (unity) ones.");
		gd.addNumericField("Remove fraction of worst matches", this.pmtch_frac_remove,  3,7,"",	"When fitting scenes remove this fraction of worst match tiles.");
		gd.addNumericField("Maximal metric error",             this.pmtch_metric_err,  3,7,"m",	"Maximal tolerable fitting error caused by elevation variations.");
		gd.addNumericField("Central area standard deviation",  this.pmtch_max_std,  3,7,"",	"Central area limit by the standard deviation.");
		gd.addNumericField("Central area minimal radius",      this.pmtch_min_std_rad,  3,7,"tile",	"Minimal radius of the central area after all LMA passes.");
		gd.addCheckbox    ("Ignore previous RMSE",             this.pmtch_ignore_rms, "Do not exit full fitting cycles if the RMSE worsened/not improved.");
		gd.addNumericField("Number of fitting iterations",     this.pmtch_num_iter, 0,3,"","number of full fittng iterations.");
		
		gd.addNumericField("Pull skew (rotation)",             this.pmtch_pull_skew, 3,7,"",	"Prevent pairwise match from rotation.");
		gd.addNumericField("Pull tilt",                        this.pmtch_pull_tilt, 3,7,"",	"Prevent pairwise match from tilt.");
		gd.addNumericField("Pull scale",                       this.pmtch_pull_scale, 3,7,"",	"Prevent pairwise match from scaling.");
		
		gd.addNumericField("Central area radius as fraction",  this.pmtch_cent_rad,   3,7,"",	"Central area radius as fraction of half minimal WOI dimension.");
		gd.addNumericField("Maximal central area radius",      this.pmtch_max_cent_rad,   3,7,"tiles",	"Absolute limit to the center area radius (eases bad peripheral matching).");
		gd.addNumericField("Central area minimal fill",        this.pmtch_cent_fill,  3,7,"",	"Central area minimal fill for all but the last iteration.");
		gd.addNumericField("Central area minimal fill final",  this.pmtch_cent_final, 3,7,"",	"Central area minimal fill for the last iteration.");
		gd.addNumericField("Relax metric error for no-SfM",    this.pmtch_ease_nosfm, 3,7,"",	"Relax metric error for no-SfM scenes (sfm_gain==0).");
		
		
		gd.addMessage  ("Initial spiral search for image matching");
		gd.addCheckbox    ("Use existing image pair",     this.ospir_existing, "Use existing affine settings for this pair, do not use spiral search.");
		gd.addCheckbox    ("Invert existing image pair",  this.ospir_invert, "Invert existing image pair affine transform, do not use spiral search.");
		gd.addNumericField("Spiral search step",          this.ospir_step,  3,7,"scaled pix",	"Distance between spiral search probes, in scaled pixels.");
		gd.addNumericField("Spiral search radius",        this.ospir_range, 3,7,"scaled pix",	"Maximal radius of the spiral search, in scaled pixels.");
		gd.addNumericField("Mitigate small overlap",      this.ospir_double, 3,7,"","For small overlaps increase zoom by 1 and range - twice.");
		gd.addNumericField("RMSE to end search",          this.ospir_good_rms,  3,7,"scaled pix",	"Maximal RMSE to consider match, in scaled pixels.");
		gd.addNumericField("Satisfactory RMSE, spiral",           this.ospir_max_rms,  3,7,"scaled pix",	"Maximal RMSE to consider match, in scaled pixels, during spiral.");
		gd.addNumericField("Minimal overlap",             this.ospir_overlap,  0,4,"scaled pix ^ 2","Minimal overlap area in square scaled pixels.");
		gd.addNumericField("LMA iterations",              this.ospir_num_iter, 0,2,"",              "Number of LMA iterations during spiral search.");
		gd.addNumericField("RMSE at first iteration",     this.ospir_rms_iter[0],  3,7,"scaled pix","Maximal RMSE at first iteration.");
		gd.addNumericField("RMSE at second iteration",    this.ospir_rms_iter[1],  3,7,"scaled pix","Maximal RMSE at second iteration.");
		gd.addCheckbox    ("Ignore worsening RMSE",       this.ospir_ignore_rms, "Ignore worsening/not improving RMSE during spiral search.");
		gd.addNumericField("Spiral search debug level",   this.ospir_debug, 0,3,"","Debug level during Spiral search.");
		gd.addCheckbox    ("Use low-res in augmentation", this.ospir_augment, "Use low-res matching during augmenting (false - skip, go to high-res).");

		gd.addMessage  ("Pairwise image matching");
		gd.addNumericField("Zoom level",                  this.pwise_zoom, 0,3,"","Low-res images to determine overlaps.");
		gd.addNumericField("Minimal overlap fraction",    this.pwise_overlap,  3,7,"",	"Minimal overlap area as a fraction of the smaller of the two images.");
		gd.addNumericField("Satisfactory RMSE, refine",   this.pwise_max_rms,  3,7,"scaled pix",	"Maximal RMSE to consider match, in scaled pixels, during refine (high res).");
		gd.addCheckbox    ("Skip existing",               this.pwise_skip_exist, "Do not regenerate if match with same or higher resolution exists.");
		gd.addCheckbox    ("Refine existing",             this.pwise_refine_exist, "Refine existing matches (false - start from scratch with spiral search).");
		gd.addCheckbox    ("Delete failed",               this.pwise_delete_fail, "Delete previous matches if it failed now.");
		gd.addCheckbox    ("Generate inverse matches",    this.pwise_gen_inverse, "Generate (refine if exist and enabled) inverse matches.");
		gd.addCheckbox    ("Save state after each match", this.pwise_save_each, "Update state file after each match generation to mitigate possible crashes.");
		gd.addCheckbox    ("Write log file",              this.pwise_log_append, "Enable writing log file with matching results.");
		gd.addStringField ("Log file full path",          this.pwise_log_path, 150, "Path of the log file to be appended.");	
		gd.addNumericField("Pairwise match debug level",  this.pwise_debug, 0,3,"","Debug level during Spiral search.");
		
		gd.addMessage  ("Parameters for elevation maps matching");
		gd.addCheckbox    ("Overwrite existing alt_data", this.alt_overwrite, "Overwrite already defined altitude match pairs.");
		gd.addCheckbox    ("Use pairwise affines",        this.alt_pairwise, "Use pairwise affines if available (false - always recalculate from individual).");
		gd.addNumericField("Border sigma",                this.alt_sigma,  3,7,"scaled pix",	"Reduce weight of the border tiles, Gaussian sigma in tiles to apply to weights.");
		gd.addNumericField("Absolute outliers offset",    this.alt_abs_outliers,  3,7,"m","Remove absolute outliers when fitting planes.");
		gd.addNumericField("Fraction of ouliers",         this.alt_outliers,  3,7,"",	"Remove outliers when fitting planes, removed fraction.");
		gd.addNumericField("Number of alt plane refines", this.alt_refine, 0,3,"",      "Refine altitude difference plane after removing outliers (0 - no outlier removal, 1 - remove outliers and refine once, ...)");
		
		gd.addMessage  ("Parameters for scenes orientation LMA fitting");
		gd.addNumericField("Rotation mismatches weights", this.alt_weight_rot, 3,7,"",  "Weights of rotation mismatches.");
		gd.addNumericField("Tilt mismatches weights",     this.alt_weight_tilt, 3,7,"", "Weights of tilt mismatches.");
		gd.addNumericField("Scale mismatches weights",    this.alt_weight_scale, 3,7,"","Weights of scale mismatches.");
		gd.addNumericField("Pull strength",               this.alt_pull, 3,7,"",        "Relative weight of combined regularization pull (0<=pull<1.0)");
		gd.addNumericField("Rotations pull",              this.alt_pull_rots, 3,7,"",   "Weight of sum of all rotations pull to zero - will be normalized to make sum of pulls a specified fraction of totla weights ");
		gd.addNumericField("Tilts pull",                  this.alt_pull_tilts, 3,7,"",  "Weight of sum of all tilts pull to zero - will be normalized to make sum of pulls a specified fraction of totla weights ");
		gd.addNumericField("Scales pull",                 this.alt_pull_scales, 3,7,"", "Weight of sum of all scales pull to zero - will be normalized to make sum of pulls a specified fraction of totla weights ");
		
		gd.addMessage  ("Build map with LMA from pairwise matches");
		gd.addCheckbox    ("Moves only",                  this.pmap_move_only,  "Moves only, no affine transform.");
		gd.addCheckbox    ("Ignore existing affines",     this.pmap_ignore_affines, "Start from unity matrices, ignore saved affines.");
		gd.addCheckbox    ("Use reversed pairs",          this.pmap_use_inv,     "Use reversed (late-early timestamps) pairs.");
		gd.addNumericField("Skew pull",                   this.pmap_skew_pull,    7,11,"",	"Intention to have axes perpendicular.");
		gd.addNumericField("Tilt pull",                   this.pmap_tilt_pull,    7,11,"",	"Intention to have axes same length.");
		gd.addNumericField("Scale pull",                  this.pmap_scale_pull,   7,11,"",	"Intention to have average scale of images = 1.0.");
		gd.addNumericField("Position pull",               this.pmap_position_pull,7,11,"",	"Intension to match IMS (GNSS) coordinates.");
		gd.addNumericField("Overlap inportance",          this.pmap_overlap_pow,  3,7,"",	"Raise overlap fraction (of the smaller image) to this power before using as weight.");
		gd.addNumericField("RMSE relative improvement",   this.pmap_rms_diff,     8,11,"",	"Relative RMSE improvement to exit LMA.");
		gd.addNumericField("LMA iterations",              this.pmap_num_iter,      0,3,"",".Maximal number of the LMA iterations.");
		gd.addCheckbox    ("Show result image",           this.pmap_show_result, ".");
		gd.addNumericField("Debug level for maps",        this.pmap_debugLevel, 0,3,"","Debug building for LMA building maps.");
		
		gd.addMessage  ("Equalize map intensitieswith LMA from pairwise intensity matches");
		gd.addCheckbox    ("Ignore existing equalization",this.pequ_ignore_equalize, "Ignore existing equalization, start from scratch (a=1, b=0).");
		gd.addCheckbox    ("Use reversed pairs",          this.pequ_use_inv,     "Use reversed (late-early timestamps) pairs.");
		gd.addNumericField("Relative scale weight",       this.pequ_scale_weight,  3,7,"",	"Importance of a-coefficient (scale) relative to b-coefficient (offset).");
		gd.addNumericField("Pull weight",                 this.pequ_pull_weight,  3,7,"",	"Relative weight of offsets and scales differences from 1.0 to pairs mismatch.");
		gd.addNumericField("Half-weight time difference", this.pequ_half_weight_sec,  3,7,"s",	"Time difference (in a pair) to reduce weight twice.");
		gd.addNumericField("Weight for large time offset",this.pequ_min_weight_sec,  3,7,"",	"Weight of pairs at very different time.");
		gd.addNumericField("Overlap inportance",          this.pequ_overlap_pow,  3,7,"",	"Raise overlap fraction (of the smaller image) to this power before using as weight.");
		gd.addNumericField("RMSE relative improvement",   this.pequ_rms_diff,     3,7,"",	"Relative RMSE improvement to exit LMA.");
		gd.addNumericField("LMA iterations",              this.pequ_num_iter, 0,3,"",".Maximal number of the LMA iterations.");
		gd.addNumericField("Debug level for equalization",this.pequ_debugLevel, 0,3,"","Debug level for global (LMA-based) intensity equalization.");

		gd.addMessage  ("Pairs selection/display parameters");
		gd.addCheckbox    ("Filter pairs",                this.flt_list, "Filter available pairs.");
		gd.addCheckbox    ("Keep undefined pairs only",   this.flt_undef_only, "Keep only undefined pairs (affines== null).");
		gd.addNumericField("Minimal scene overlap (0..1)",this.flt_min_overlap,  3,7,"", "Minimal overlap of the scenes to keep (0-no overlap, 1.0 - smaller scene is inside the parger one.");
		gd.addNumericField("Minimal scene overlap (0..1)",this.flt_max_overlap,  3,7,"", "Maximal overlap of the scenes to keep (0-no overlap, 1.0 - smaller scene is inside the parger one.");
		gd.addNumericField("Minimal RMSE",                this.flt_min_rms,  3,7,"",	 "Minimal LMA RMSE of the scene pair.");
		gd.addNumericField("Maximal RMSE",                this.flt_max_rms,  3,7,"",	 "Maximal LMA RMSE of the scene pair.");
		gd.addCheckbox    ("NaN RMS (failed match)",      this.flt_nan_rms, "Keep only failed matches with RMSE=NaN.");
		
		gd.addCheckbox    ("Filter by zoom level"  ,      this.flt_filt_zoom, "Filter by the zoom level used for matching.");
		gd.addNumericField("Minimal zoom",                this.flt_min_zoom, 0,3,"","Minimal zoom level used for matching.");
		gd.addNumericField("Maximal zoom",                this.flt_max_zoom, 0,3,"","Maximal zoom level used for matching.");
		gd.addNumericField("Minimal SfM gain",            this.flt_min_sfm,  3,7,"","Minimal SfM gain of the minimum in the scene pair.");
		gd.addNumericField("Maximal SfM gain",            this.flt_max_sfm,  3,7,"","Maximal SfM gain of the minimum in the scene pair.");
		gd. addChoice("Filter by pairwise ALT availability",FLT_ALT_MODES, FLT_ALT_MODES[this.flt_alt],"Filter by pairwise ALT availability.");
		gd. addChoice("Filter by orientation availability",FLT_ORIENT_MODES, FLT_ORIENT_MODES[this.flt_orient],"Filter by the scene orientation availability.");
		
		gd.addCheckbox    ("Show scene names",            this.flt_show_names, "Show scene full names (timestamps) in selection drop-down list.");
		gd.addCheckbox    ("Show scene overlaps",         this.flt_show_overlaps, "Show scene overlaps (in percents) in selection drop-down list.");
		gd.addCheckbox    ("Show pairs RMSE",             this.flt_show_rms, "Show scene match RMSE in selection drop-down list.");
		gd.addCheckbox    ("Show zoom level",             this.flt_show_zoom, "Show zoom level.");
		gd.addCheckbox    ("Show ALT",                    this.flt_show_alt,  "Show altitude data availability.");
		
		gd.addMessage  ("Pattern match directories");
		gd.addStringField ("Pattern match save directory",this.patt_save_top, 120, "Top directory to save combo maps");
		gd.addStringField ("Save subdirectory",           this.patt_save_subdir, 80, "Subdirectory for versions of the same scene/pair of scenes");
		
		gd.addTab         ("Scene Series", "Processing series of scenes and multi-series sets");
		gd.addMessage  ("Build series options");
		
		gd.addCheckbox ("Use center scene as a reference",           this.center_reference,
				"True for mapping with post-processing. If false, use the last scene as a reference (driving).");
		gd.addCheckbox ("Lock camera position",                      this.lock_position,
				"Only camera orientation changes, position is locked (cUAS mode).");
		gd.addCheckbox ("CUAS mode",                                 this.cuas_rotation,
				"Assumes lock_position, camera is rotating around the center.");
		gd.addCheckbox ("Manual correction",                         this.manual_correction,
				"Once used for foliage to merge two sequences.");
		gd.addCheckbox ("Overlap sequences",                         this.overlap_sequences,
				"Overlap sequences by half - start (down) from the previous center.");
		gd.addCheckbox ("Reset photometric calibration",             this.reset_photometric,
				"Reset photometric calibration, will use basic one before re-calibrating.");
		gd.addCheckbox ("Force reference scene DSI calculation",     this.force_ref_dsi,
				"Calculate reference scene DSI even if the file exists.");
		gd.addCheckbox ("Force egomotion calculation",               this.force_orientations,
				"Calculate relative poses of each scene camera relative to the reference scene even if the data exists.");
		gd.addCheckbox ("Run LY adjustments",                        this.run_ly,
				"Adjust cameras orientations (after building DSI) and exit without generatin result files.");
		gd.addNumericField("LY mode",this.run_ly_mode, 0,3,"", 	"+1 - Lazy Eye, +2 - infinity");
		gd.addCheckbox ("Use LMA for LY adjustments",                this.run_ly_lma,
				"Use LMA for LY adjustments.");
		gd.addNumericField("Minimal number of egomotion calculations",this.min_num_orient, 0,3,"",
				"Minimal number of fitting scenes cycles, should be >=1. First cycle includes spiral search for the first scene");
		gd.addNumericField("Minimal number of interscene accumulations", this.min_num_interscene, 0,3,"",
				"Minimal required number of re-calculations of the interscene-accumulated DSI.");
		gd.addCheckbox ("Adjust IMU orientation",                    this.adjust_imu_orient,
				"Adjust IMU misalignment to the camera.");
		gd.addCheckbox ("Apply IMU orientation",                     this.apply_imu_orient,
				"Apply  IMU misalignment to the camera if adjusted.");
		gd.addCheckbox ("Use translation for IMU orientation",       this.orient_by_move,
				"Use translation data to adjust IMU orientation .");
		gd.addCheckbox ("Use rotation for IMU orientation",          this.orient_by_rot,
				"Use rotation data to adjust IMU orientation.");
		gd.addCheckbox ("Use combo mode IMU orientation",            this.orient_combo,
				"Use combined Z/h, R, A-X/h, T+Y/h for IMU mount-to-camera orientation correction. False - use X,Y,Z,A,T,R");
		gd.addCheckbox ("Adjust gyro offsets",                       this.adjust_gyro,
				"Adjust qyro omegas offsets.");
		gd.addCheckbox ("Apply gyro offsets",                        this.apply_gyro,
				"Apply adjusted qyro omegas offsets.");
		
		gd.addCheckbox ("Adjust accelerometers scales",              this.adjust_accl,
				"Adjust IMU linear velocities scales.");
		gd.addCheckbox ("Apply accelerometers scales",               this.apply_accl,
				"Apply IMU linear velocities scales.");
		
		gd.addCheckbox ("Calculate IMS orientation correction",      this.calc_quat_corr,
				"Calculate camera orientation correction from predicted by IMS .");
		gd.addCheckbox ("Apply IMS orientation correction ",         this.apply_quat_corr,
				"Apply camera orientation correction from predicted by IMS when generating output files.");
		gd.addCheckbox ("Use IMS orientation correction internally", this.use_quat_corr,
				"Depreceted? Use camera orientation correction from predicted by IMS internally from the previous calculations.");
		gd.addCheckbox ("Inertial only initial poses",               this.inertial_only,
				"Use only IMU (DID_PIMU) data for initial egomotion, not the GNSS-aided IMS output.");
		
		gd.addMessage  ("Generate/show scene sequences");
		gd.addCheckbox ("Generate egomotion table",                  this.generate_egomotion,
				"Generate table with image-based and IMS egomotion data.");
		gd.addCheckbox ("Generate mapped scene sequence",            this.generate_mapped,
				"Generate scene sequence mapped to the reference scene.");
		gd.addCheckbox ("Reuse existing video fragments, disable other options",this.reuse_video,
				"Dry-run video generation, just re-assemble earlier generated video fragments. Disables all other file generation.");
		gd.addCheckbox ("Save scene sequences in (pseudo)colors as TIFF",this.save_mapped_color,
				"Save generated scene sequences in (pseudo)color mode as a multi-slice TIFF.");
		gd.addCheckbox ("Save scene sequences in monochrome as TIFF",    this.save_mapped_mono,
				"Show generated scene sequences in monochrome mode as a multi-slice TIFF. May use Unsharp Mask.");
		gd.addCheckbox ("Generate color video",                      this.gen_avi_color,
				"Generate video for color scene sequences.");
		gd.addCheckbox ("Generate monochrome video",                 this.gen_avi_mono,
				"Generate video for monochrome scene sequences.");
		gd.addCheckbox ("Show scene sequences in (pseudo) colors",    this.show_mapped_color,
				"Show generated scene sequences in (pseudo) color mode. Disabled in batch mode");
		gd.addCheckbox ("Show scene sequences in monochrome",        this.show_mapped_mono,
				"Show generated scene sequences in monochrome mode. May use Unsharp Mask. Disabled in batch mode.");
		
		
		gd.addCheckbox ("Generate RAW images",                       this.generate_raw,
				"Raw images from single (top) camera using original view - just after aberration correction and aligning sensors .");
		gd.addCheckbox ("Generate INF images",                       this.generate_inf,
				"Single-camera images aligned at infinity - 2D correction functionally similar to camera stabilization.");
		gd.addCheckbox ("Generate 3D-corrected FG images",           this.generate_fg,
				"Correct in 3D scene images (from all 16 sensors) matching reference (last in sequence) scene with foreground (FG) priority.");
		gd.addCheckbox ("Generate 3D-corrected BG images",           this.generate_bg,
				"Correct in 3D scene images (from all 16 sensors) matching reference (last in sequence) scene with background (BG) priority.");
		gd.addCheckbox ("Generate binocular stereo pairs",           this.generate_stereo,
				"Generate stereo-pairs for 3D-corrected videos (FG,BG). Ebables specific modes (including 0-baseline / mono).");

		for (int i = 0; i < stereo_views.length; i++) {
			double base = stereo_views[i][0]; // stereo_bases[i];
			String ub = String.format("(%.0fmm up, %.0fmm back) ",stereo_views[i][1],stereo_views[i][2]);
			if ((stereo_views[i][1]==0) && (stereo_views[i][2]==0)){
				ub="";
			}
			String title = (base == 0.0)?
					"Generate mono (single camera) scene sequences"+ub:
						"Generate "+base+"mm-baseline stereo scene sequences"+ub;
			String tooltip = (base == 0.0)?
					"Generate mono (single camera) scene sequences "+ub+"as Tiff and/or video.":
						"Generate "+base+"mm-baseline stereo scene sequences "+ub+"as Tiff and/or video.";
			gd.addCheckbox (title,                                   this.generate_stereo_var[i], tooltip);
		}
		
		
		
		
		gd.addMessage  ("Generate/save reference (last) scene images");
		gd.addCheckbox ("Export all-sensor images",                  this.export_images,
				"Export multi-slice images: with constant disparity, with foreground disparity, and with background disparity");
		gd.addCheckbox ("Show exported images (same disparity)",     this.show_images,
				"Display generated/saved image set, pseudocolors");
		gd.addCheckbox ("Show exported FG/BG 3d-corrected",          this.show_images_bgfg,
				"Show foreground and background exported images");
		gd.addCheckbox ("Show floating-point monochrome images",     this.show_images_mono,
				"Display generated/saved monochrome images (in addition to color) ");
		gd.addCheckbox ("Generate metric depth map",                 this.export_ranges,
				"Calculate strength, distance, X, and Y in meters");
		gd.addCheckbox ("Debug metric depth map",                    this.debug_ranges,
				"Debug metric depth maps");
		gd.addCheckbox ("Show metric depth map",                     this.show_ranges,
				"Display calculated depth map in meters");
		gd.addCheckbox ("Export files for DNN training/testing",     this.export_ml_files,
				"Display calculated depth map in meters");
		
		gd.addMessage  ("Additional parameters");
		gd.addCheckbox ("Color NaN background",                      this.show_color_nan,
				"Use NaN for undefined tiles (false - 0.0f). NaN produces sharp distinct result, 0.0f - blended");
		gd.addCheckbox ("Mono NaN background",                       this.show_mono_nan,
				"Use NaN for undefined tiles (false - 0.0f). NaN produces sharp distinct result, 0.0f - blended");

		gd.addNumericField("Scale angular pose extrapolation",       this.scale_extrap_atr, 3,5,"",
				"Scale angular extrapolation relative to the previous pair");
		gd.addNumericField("Scale linear pose extrapolation",        this.scale_extrap_xyz, 3,5,"",
				"Scale linear extrapolation relative to the previous pair (use 1.0 for a drone)");
		gd.addNumericField("Number of previous scenes to extrapolate",this.avg_len, 0,3,"",
				"Average this number of previous scenes movement for next scene prediction");

		gd.addNumericField("Minimal number of scenes to keep series",this.min_num_scenes, 0,3,"",
				"Scrap all seriest if less numer of scenes can be matched to the reference scene (including reference itself)");
		gd.addNumericField("Cut longer series",                      this.max_num_scenes, 0,3,"",
				"Cut longer series");
		
		gd.addNumericField("LPF egomotion sigma",                    this.blur_egomotion, 3,5,"scenes",
				"LPF egomotion components with this sigma before using as ERS (not implemented).");

		gd.addMessage  ("Metric distance map generation");
		gd.addNumericField("Disparity at infinity",                  this.range_disparity_offset, 5,7,"pix",
				"Disparity at infinity - subtract from measured disparity when converting to ranges.");
		gd.addNumericField("Minimal strength for range calculation", this.range_min_strength, 5,7,"",
				"Disregard weaker results when measuring range.");
		gd.addNumericField("Maximal displayed range",                this.range_max, 5,7,"m",
				"Do not display extremely far objects.");

		gd.addMessage  ("3D model generation");
		gd.addCheckbox ("Generate 3D model",                         this.export3d,
				"Generate textures and model.");
		gd.addCheckbox ("Generate 3D model of terrain",              this.export3dterrain,
				"Generate textures and model, remove vegetation.");
		
		gd.addMessage  ("CT scan generation");
		gd.addCheckbox ("Generate CT scan",                          this.export_CT,	"Generate \"CT scan\" of the vegetation.");
		gd.addNumericField("Minimal CT scan disparity",              this.ct_min, 5,7,"pix","Start disparity (lowest altitude).");
		gd.addNumericField("Maximal CT scan disparity",              this.ct_max, 5,7,"pix","End disparity (highest altitude).");
		gd.addNumericField("CT scan step",                           this.ct_step, 5,7,"pix","Scan step.");
		gd.addNumericField("CT scan expand section",                 this.ct_expand, 0,3,"tiles","expand section area, 2 - a tile in 8 directions.");

		gd.addMessage  ("Terrain generation (w/o vegetation)");
		
//        gd.addCheckbox    ("Generate terrain",                       this.terr_enable,	"Generate \"CT scan\" of the vegetation.");
		gd.addNumericField("Plain fit, limit above",                 this.terr_rmse_above,5,7,"std","Discard higher disparity, relative to stadard deviation.");
		gd.addNumericField("Plain fit, limit below",                 this.terr_rmse_below,5,7,"std","Discard lower disparity, relative to stadard deviation.");
		gd.addNumericField("Plain fitting passes",                   this.terr_num_refine, 0,3,"","Number of passes to refine ground plane fitting.");
		gd.addNumericField("Plain fit, outliers above",              this.terr_frac_above*100,5,7,"%","Discard this fraction of outliers above plane.");
		gd.addNumericField("Plain fit, outliers below",              this.terr_frac_below*100,5,7,"%","Discard this fraction of outliers above plane.");
		gd.addNumericField("Relative vegetation disparity",          this.terr_rthrsh_abv,5,7,"x","Relative to disparity threshold of vegetation height.");
		gd.addCheckbox    ("Assign border to fitted plane",          this.terr_flat_brd,"Assign NaN border tiles to fitted plane disparity.");
		gd.addNumericField("Expand detected vegetation area",        this.terr_expand, 0,3,"tiles","Expand section area, 2 - a tile in 8 directions.");
		gd.addCheckbox    ("Save debug terrain images",              this.terr_save,	"Save terrain-related debug image in the model directory.");
		
		gd.addMessage  ("Debug and visialization parameters");
		gd.addCheckbox ("Replace scene with reference scene",        this.scene_is_ref_test,
				"Correlate reference scene with itself for testing (may want to manually change scene_atr and scene_xyz in debug mode)");
		gd.addCheckbox    ("Render reference scene",                 this.render_ref,
				"Render reference scene as a multi-layer (per-port) image");
		gd.addCheckbox    ("Render compared scene",                  this.render_scene,
				"Render reference compared as a multi-layer (per-port) image");
		gd.addCheckbox    ("Pseudo-color render (does not work)",    this.toRGB,
				"Use pseudo-color palette for scene render  - DOES NOT WORK - use ColorProcParameters");
		gd.addCheckbox    ("Show 2D correlations",                   this.show_2d_correlations,
				"Show raw 2D correlations (per channel and combined)");
		gd.addCheckbox    ("Show motion vectors",                    this.show_motion_vectors,
				"Show motion vectors and strengths per channel and combines, NaN-s are when strengths corresponding are below thresholds");
		gd.addNumericField("Debug Level for interscene match",       this.debug_level, 0,3,"",
				"Debug Level for the above parameters: renders and raw correlations need >1,  motion vectors > 0");

		gd.addMessage  ("Pairwise ERS testing");
		gd.addCheckbox ("Test pairwise matching with ERS",           this.test_ers,
				"Test pairwise dispaly and pose correction to test ERS compensation");
		gd.addNumericField("Test scene reference index",             this.test_ers0, 0,3,"",
				"Reference scene index in a scene sequence");
		gd.addNumericField("Test scene other scene index",           this.test_ers1, 0,3,"",
				"Other scene index in a scene sequence (should have a very different angular/linear velocity component)");
		
		
		gd.addTab("Fiter DSI","Filter DSI before use (interscene fitting, interscene accumulation, ML generation");
//		gd.addMessage  ("Depth map filtering parameters");
		gd.addNumericField("Average lowest disparity neighbors",     this.num_bottom, 0,3,"",
				"Average this number of lowest disparity neighbors (of 8)");
		gd.addNumericField("Number of filter iterations",            this.num_passes, 0,3,"",
				"Number of filter iterations");
		gd.addNumericField("Maximal change to exit",                 this.max_change, 5,7,"pix",
				"Maximal change to exit iterations");
		gd.addNumericField("Minimal disparity",                      this.min_disparity, 5,7,"pix",
				"Minimal acceptable disparity");
		gd.addNumericField("Disparity range for symmetrical pull",   this.max_sym_disparity, 5,7,"pix",
				"For larger disparities undefined tiles are pull predominantly down (to average of lowest), "+
		        "but for small disparities (sky) pull is to average of all neighbors.");
		gd.addNumericField("Minimal strength LMA",                   this.min_strength_lma, 5,7,"",
				"Lower strength for LMA tiles - treat as non-LMA");
		gd.addNumericField("Minimal strength to replace",            this.min_strength_replace, 5,7,"",
				"Minimal strength to replace (now not used)");
		gd.addNumericField("Minimal strength to blur",               this.min_strength_blur, 5,7,"",
				"Minimal strength to blur  (now not used)");
		gd.addNumericField("Blur weak sigma",                        this.sigma, 5,7,"",
				"Blur weak sigma");
		gd.addNumericField("Number of blur passes",                  this.num_blur, 0,3,"",
				"Number of blur passes");
		gd.addNumericField("*** Disparity offset (used 04/07/2023) ***",     this.disparity_corr, 5,7,"",
				"for setInterTasks() - used 04/07/2023.");
		gd.addNumericField("Outlier N-th from extreme",              this.outliers_nth_fromextrem, 0,3,"",
				"0 - use min/max, 1 - use second min/max, ... second from min/max - removes dual-tile clusters max/mins");
		gd.addNumericField("Outliers tolerance absolute",            this.outliers_tolerance_absolute, 5,7,"pix",
				"Absolute difference from extreme neighbor");
		gd.addNumericField("Outliers tolerance realtive",            this.outliers_tolerance_relative, 5,7,"pix/pix",
				"Add to tolerance as a fraction of absolute tile disparity");
		gd.addNumericField("Maximal number of outlier iterations",   this.outliers_max_iter, 0,3,"",
				"Maximal number of iterations for removing outliers");
		gd.addNumericField("Maximal outlier strength 2",             this.outliers_max_strength2, 5,7,"",
				"Maximal outlier strength for second pass filtering");
		gd.addNumericField("Outlier N-th from extreme 2",            this.outliers_nth_fromextrem2, 0,3,"",
				"Second filter: 0 - use min/max, 1 - use second min/max, ... second from min/max - removes dual-tile clusters max/mins");
		gd.addNumericField("Outliers tolerance absolute 2",          this.outliers_tolerance_absolute2, 5,7,"",
				"Absolute difference from extreme neighbor, second filter");
		gd.addNumericField("Outliers tolerance realtive 2",          this.outliers_tolerance_relative2, 5,7,"",
				"Add to tolerance as a fraction of absolute tile disparity, second filter");
		gd.addNumericField("Outliers LMA maximal strength",          this.outliers_lma_max_strength, 5,7,"",
				"Maximal LMA strength for an outlier");
		gd.addNumericField("Outliers non-LMA maximal strength",      this.outliers_max_strength, 5,7,"",
				"Outliers non-LMA (centroid) maximal strength");
		gd.addNumericField("Outliers from LMA max strength",         this.outliers_from_lma_max_strength, 5,7,"",
				"Non-LMA outliers from LMA neighbors maximal strength");
		
		gd.addNumericField("Search for LMA radius",                  this.search_radius, 0,3,"tile",
				"Search farther if no LMA tiles are found around this non-LMA tile");
		gd.addCheckbox ("Remove non-LMA tile if no LMA near",        this.remove_no_lma_neib,
				"Remove non-LMA tile if no LMA one is found within specified radius");
		
		gd.addNumericField("LMA difference positive",                this.diff_from_lma_pos, 5,7,"",
				"Maximal non-LMA tile positive difference from the LMA neighbor");
		gd.addNumericField("LMA difference negative",                this.diff_from_lma_neg, 5,7,"",
				"Maximal non-LMA tile negative difference from the LMA neighbor");
		gd.addNumericField("LMA outliers N-th from extreme",         this.outliers_lma_nth_fromextrem, 0,3,"",
				"0 - use min/max, 1 - use second min/max, ... ");
		gd.addNumericField("Margin from the sensor FOV edge ",       this.filter_margin, 0,3,"pix",
				"Disregard tiles with centers closer to the sensor FoV in pixels");
		
		gd.addMessage  ("Removing weak tiles with insufficient matching neighbors");
		gd.addNumericField("Neighbor disparity tolerance (absolute)",this.weak_tolerance_absolute, 5,7,"pix",
				"Absolute disparity difference to be considered a neighbor");
		gd.addNumericField("Neighbor disparity tolerance (relative)", this.weak_tolerance_relative, 5,7,"",
				"Relative disparity difference to be considered a neighbor");
		gd.addNumericField("Minimal matching neighbors",              this.weak_min_neibs, 0,3,"",
				"Minimal number of weak matching neighbors (at least one is needed if there are strong non-neighbors).");
		gd.addNumericField("Minimal strength to be strong",           this.strong_strength, 5,7,"",
				"Minimal strength to be considered strong(not weak). LMA are also considered strong.");
		gd.addNumericField("Minimal weak strength",                   this.weak_strength, 5,7,"",
				"Weaker are removed unconditionally (not used now).");
		
		gd.addTab("Sky","Featureless sky areas detection and filtering (LWIR only)");
		gd.addMessage  ("Temperature-related filtering (requires photometric calibration)");
		gd.addCheckbox ("Try blue sky calculation/use",              this.sky_extract,
				"Disable if no sky will appear in the image (such as for down-pointed camera).");
		gd.addCheckbox ("Force blue sky recalculation",              this.sky_recalc,
				"Recalculate blue sky even if it already exists.");
		gd.addNumericField("High limit of the scene coldest tile",   this.sky_highest_min, 5,7,"",
				"Sky is normally cold. If the (globally offset) minimal scene absolutre pixel value is above - no sky in the scene (debug images still show it)");
		gd.addNumericField("Cold level fraction",                    this.cold_frac, 5,7,"",
				"All tiles below this percentile are considered \"cold\".");
		gd.addNumericField("Hot level fraction",                     this.hot_frac, 5,7,"",
				"All tiles above this percentile are considered \"hot\".");
		gd.addNumericField("Cold FOM scale",                         this.cold_scale, 5,7,"",
				"Multiply (strength*spread) by this value (reduce), and linearly scale up to 1.0 for hot ones.");
		gd.addMessage  ("Generate seed tiles for the potential sky area");
		gd.addNumericField("Seed FOM level",                         this.sky_seed, 5,7,"",
				"Maximal FOM (strength*spread*temp_scale) level to seed sky area, temp_scale is generated from average pixel value "+
		        "at infinity and 3 above parameters");
		gd.addNumericField("Seed maximal disparity",                 this.lma_seed, 5,7,"pix",
				"Seed tiles should not have LMA-found disparity above this");
		gd.addNumericField("Seed maximal relative temperature",      this.seed_temp, 5,7,"",
				"Seed tiles should be  colder than this point between min and max scene temperature.");
		gd.addNumericField("Shrink sky seed",                        this.sky_shrink, 0,3,"",
				"Shrink sky seed tiles to eliminate too small (false) clusters. One shrinks in hor/vert, 2 - with diagonals, ...");
		gd.addNumericField("Seed rows",                              this.seed_rows, 0,3,"",
				"Sky is above and normally should appear in the top image rows. Applies after shrinking.");
		gd.addMessage  ("Expand seed tiles");
		gd.addNumericField("Maximal strong sky disparity",           this.max_disparity, 5,7,"",
				"Maximal disparity of the sky tiles if they are strong enough.");
		gd.addNumericField("Strength of the high disparity sky",     this.max_disparity_strength, 5,7,"",
				"Required tile strenghth to be considered strong to test against maximal disparity.");
		gd.addNumericField("Expand FOM limit",                       this.sky_lim, 5,7,"",
				"Expand while FOM is below this value (usually to a high-contrast skyline).");
		gd.addNumericField("Sky maximal relative temperature",      this.lim_temp, 5,7,"",
				"Sky tiles should be  colder than this point between min and max scene temperature.");
		gd.addNumericField("Expand extra",                           this.sky_expand_extra, 0,3,"",
				"Additionally expand sky area after reaching threshold in the previous step.");
		gd.addNumericField("Bottleneck width",                       this.sky_bottleneck, 0,3,"",
				"Shrink/reexpand from the seed detected sky to prevent \"leaks\" through narrow gaps in the high-contrast sky limit.");
		gd.addNumericField("Re-expand extra after bottleneck",       this.sky_reexpand_extra, 0,3,"",
				"Additionally expand sky area after bottleneck shrink in excess of re-expanding by the same amount as shrank.");
		gd.addNumericField("Modify strength to be at least this",    this.min_strength, 5,7,"",
				"Input strength has some with zero values resulting in zero FOM. Make them at least this.");
		gd.addNumericField("Lowest sky row",                         this.lowest_sky_row, 0,3,"",
				"Last defense - if the detected sky area reaches near-bottom of the page - it is invalid, remove it (but keep in debug images).");
		gd.addNumericField("Hottest sky tile to override lowest row",this.sky_bottom_override, 5,7,"",
				"If the detected sky is all cold enough, bypass lowest row test, allow to raise camera.");
		gd.addNumericField("Shrink before finding hottest sky",      this.sky_override_shrink, 0,3,"",
				"Shrink detected sky before looking for the hottest sky tile (blurred skyline in wet atmosphere).");
		
		gd.addMessage  ("Detect/process clouds in the sky");
		gd.addCheckbox ("Enable clouds",                             this.clouds_en,
				"Enable clouds in the sky detection / processing.");
		gd.addNumericField("Clouds maximal FOM",                     this.clouds_fom, 5,7,"",
				"Maximal FOM for clouds (must be <=).");
		gd.addNumericField("Clouds maximal spread",                  this.clouds_spread, 5,7,"",
				"Maximal spread for clouds (must be <=).");
		gd.addNumericField("Clouds disparity",                       this.clouds_disparity, 5,7,"pix",
				"Maximal disparity for strong clouds.");
		gd.addNumericField("Clouds weak",                            this.clouds_weak, 5,7,"",
				"Maximal strength for near definite clouds.");
		gd.addNumericField("Clouds sure strength",                   this.clouds_strength, 5,7,"",
				"Minimal strength for far strong clouds (definitely cloud).");
		gd.addNumericField("Non-clouds strength",                    this.clouds_not_strength, 5,7,"",
				"Maximal strength for near maybe clouds (if it has strong cloud neighbor). Stronger near - definitely not clouds.");
		gd.addCheckbox ("Clouds by strong neighbor",                 this.clouds_strong,
				"Treat weak non-cloud as cloud if it has strong (1.5x strength) cloud neighbor.");

		gd.addMessage  ("Detect smooth walls mistaken for sky. Disable for natural environments");
		gd.addCheckbox ("Detect smooth walls",                       this.wall_en,
				"Suspect isolated sky clusters to be smooth walls.");
		gd.addCheckbox ("Walls default",                             this.wall_dflt,
				"If unsure (both or none) - treat as a wall (false - as sky).");
		gd.addNumericField("Wall strength",                          this.wall_str, 5,7,"",
				"Minimal strength to consider cluster to be a wall or sky. Use very small to use any non-NAN disparity.");
		gd.addNumericField("Wall far",                               this.wall_far, 5,7,"pix",
				"Maximal disparity to designate cluster as sky.");
		gd.addNumericField("Wall near",                              this.wall_near, 5,7,"pix",
				"Minimal disparity to designate cluster as a wall.");
		
		gd.addMessage  ("Processing far treeline/mountains that may be confused with clouds. Only behind far objects such as far horizontal surface");
		gd.addCheckbox ("Enable treeline processing",                this.treeline_en,
				"Enable treeline processing (reduces sky/clouds).");
		gd.addCheckbox ("Look not only under, but diagonal too",     this.treeline_wide,
				"Look not only under, but diagonal too.");
		gd.addNumericField("Maximal treeline height",                this.treeline_height, 0,3,"tiles",
				"Maximal height of the treeline. Increase for mountain ridges.");
		gd.addNumericField("Minimal treeline width",                 this.treeline_width, 0,3,"tiles",
				"Minimal horizontal width of the treeline.");
		gd.addCheckbox ("Limit treeline height (not delete it)",     this.treeline_lim_high,
				"Limit too high treeline (false - delete completely).");
		gd.addNumericField("Treeline min strength",                  this.treeline_str, 5,7,"",
				"Treeline min strength.");
		gd.addNumericField("Treeline min disparity",                 this.treeline_far, 5,7,"pix",
				"Treeline min (far) disparity.");
		gd.addNumericField("Treeline max disparity",                 this.treeline_near, 5,7,"pix",
				"Treeline max (near) disparity.");
		gd.addNumericField("Pre-treeline FG min strength",           this.treeline_fg_str, 5,7,"",
				"Pre-treeline FG objects (such as flat ground) minimal strength.");
		gd.addNumericField("Pre-treeline FG min disparity",          this.treeline_fg_far, 5,7,"pix",
				"Pre-treeline FG min (far) disparity.");
		gd.addNumericField("Pre-treeline FG max disparity",          this.treeline_fg_near, 5,7,"pix",
				"Pre-treeline FG max (near) disparity.");
		
		gd.addMessage  ("Detect indoors (and disable sky)");
		gd.addCheckbox ("Enable indoors",                             this.indoors_en,
				"Enable indoors detection and disable sky (if detected).");
		gd.addNumericField("Outdoors objects minimal strength",       this.indoors_str, 5,7,"",
				"Minimal strength of outdoors (far) objects.");
		gd.addNumericField("Outdoors disparity",                      this.indoors_disp, 5,7,"pix",
				"Maximal disparity for strong outdoors objects.");
		gd.addNumericField("Minimal outdoor tiles",                  this.indoors_min_out, 0,3,"tiles",
				"Minimal number of far strong tiles to deny indoor mode.");
		
		gd.addMessage  ("Boost FOM if there are enough LMA-defined good neighbors around (Thin wires over the sky)");
		gd.addNumericField("Minimal disparity",                      this.disp_boost_min, 5,7,"pix",
				"Minimal LMA-defined disparity to boost FOM.");
		gd.addNumericField("Disparity difference",                   this.disp_boost_diff, 5,7,"pix",
				"Maximal disparity difference to neighbor to count.");
		gd.addNumericField("Number of good neighbors",               this.disp_boost_neibs, 0,3,"",
				"Number of neighbors (of 8) to have small disparity difference to boost FOM.");
		gd.addNumericField("Boost amount",                           this.disp_boost_amount, 5,7,"x",
				"Multiply FOM by this value if number of neighbors is exactly minimal. Scale proportional to the total number of neighbors.");
		gd.addNumericField("Scale if strength is combo, not main",   this.scale_combo_strength, 5,7,"x",
				"Reduce strength when it comes from combo, not DSI-MAIN.");
		gd.addCheckbox ("Save debug images",                          this.save_debug_images,
				"Save debug images for Blue Sky generation in model directory.");

		gd.addTab         ("Inter-Match", "Parameters for full-resolution scene matching");
		gd.addMessage  ("Interscene match parameters");
		gd.addNumericField("Image margin",                           this.margin, 0,5,"pix",
				"Do not use tiles if their centers are closer to the virtual image edge");
		gd.addNumericField("Used sensors mask",                      this.sensor_mask_inter, 0,5,"",
				"Bitmask of the sensors to use, -1 - use all");
		gd.addCheckbox ("Preserve partial",                          this.use_partial,
				"Preserve motion vectors for each of the selected sensors, fallse - use only combined (in transform domain and in pixel domain)");
		gd.addCheckbox ("Poly maximums",                             this.run_poly,
				"Use polinomial interpolationn to determin correlation maximums instead of centroids (not yet implemented, ignored)");
		gd.addNumericField("Centroid radius",                        this.centroid_radius, 5,7,"pix",
				"Calculate centroids after multiplication by a half-cosine window. All correlation data farther than this value from the center is ignored");
		gd.addNumericField("Refine centroids",                       this.n_recenter, 0,5,"",
				"Repeat centroids after moving the window center to the new centroid location this many times (0 - calculate once)");
		gd.addMessage  ("Mixing TD and PD accumulation of 2d correlations");
		gd.addNumericField("TD-accumulated weight",                  this.td_weight, 5,7,"",
				"Mix argmax from TD-accumulated correlation.");
		gd.addNumericField("TD-accumulated neighbors weight",       this.td_neib_weight, 5,7,"",
				"Mix argmax from TD-accumulated neighbors (center plus 8-neighbors) correlation.");
		gd.addNumericField("PD-accumulated weight",                  this.pd_weight, 5,7,"",
				"Mix argmax from PD-accumulated correlation.");
		gd.addCheckbox ("TD when no PD only",                        this.td_nopd_only,
				"Use argmax from TD only if PD data is not available for this tile.");
		gd.addCheckbox ("Use neighbors only if no individual",       this.neib_notd_only,
				"Use argmax from averaged neighbors TD only if this tile own maximum is not strong enough.");
		
		gd.addMessage  ("Averaging inter-scene 2D correlations with 8 immediate neighbors");
		gd.addCheckbox ("Use averaging neighbors",              this.use_neibs,
				"Average tile 2D correlation with 8 immediate neighbors if single-tile correlation is not strong enough.");
		// TODO: Consider/Implement pure TD averaging
		gd.addCheckbox ("No-FPN averaging only (all passes)",   this.neibs_nofpn_only,
				"Use averaging with neighbors only if tile offset is large enough not to have FPN interference (for all passes).");
		gd.addCheckbox ("No-FPN averaging only during init",    this.neibs_nofpn_init,
				"Additionally disable FPN-prone tiles during initial scene pose adjustment.");
		gd.addCheckbox ("Use averaging for both PD and TD",     this.redo_both,
				"Recalculate tile average for both TD and PD if at least one of them is not strong enough.");
		gd.addNumericField("Minimal neighbors to average",      this.min_num_neibs, 0,3,"",
				"Minimal number of defined (non-null) neighbors 2D correlations (of 8) to use averaging.");
		gd.addNumericField("Scale PD thershold",                this.scale_neibs_pd, 5,7,"",
				"Use averaging for tiles that are stronger than scaled threshold for a single-tile motion vector for pixel-domain values.");
		gd.addNumericField("Scale TD threshold",                this.scale_neibs_td, 5,7,"",
				"Use averaging for tiles that are stronger than scaled threshold for a single-tile motion vector for transform-domain values.");
		gd.addNumericField("Scale averaged strengths",          this.scale_avg_weight, 5,7,"",
				"Scale averaged motion vector strengths (for each of the TD and PD).");
		
		gd.addMessage  ("Filtering motion vectors");
		gd.addNumericField("Minimal correlation strength (non-sum)",       this.min_str, 5,7,"",
				"Minimal correlation strength for individual correlation and for pixel-domain averaged one. Weeker tiles results are removed.");
		gd.addNumericField("Minimal correlation strength (non-sum) w/FPN", this.min_str_fpn, 5,7,"",
				"Similar to above, but for small offsets where FPN correlation may be present");
		
		gd.addNumericField("Minimal correlation strength (sum only)",      this.min_str_sum, 5,7,"",
				"Minimal correlation strength for transform-domain averaging among sensors. Weeker tiles results are removed.");
		gd.addNumericField("Minimal correlation strength (sum only) w/FPN",this.min_str_sum_fpn, 5,7,"",
				"Similar to above, but for small offsets where FPN correlation may be present");
		
		gd.addNumericField("Minimal correlation neighbors strength",       this.min_str_neib, 5,7,"",
				"Minimal correlation strength for transform-domain averaging among sensors and neighbors. Weeker tiles results are removed.");
		gd.addNumericField("Minimal correlation neighbors strength w/FPN",this.min_str_neib_fpn, 5,7,"",
				"Similar to above, but for small offsets where FPN correlation may be present");
		
		gd.addNumericField("Minimal number of neighbors (of 8)",     this.min_neibs, 0,3,"",
				"Remove motion vectors with less than this number of defined (passing min_str) neighbors.");
		gd.addNumericField("No-neighbors weight (<1.0)",             this.weight_zero_neibs, 5,7,"",
				"Punish for reduced neighbors - weigh for no-neighbors), weight of 8 neighbors = 1.0.");
		gd.addNumericField("Disparity to reduce weight twice from infinity", this.half_disparity, 5,7,"",
				"Weight at this disparity is 0.5, at infinity - 1.0.");
		gd.addNumericField("Difference from neighbors average ",     this.half_avg_diff, 5,7,"",
				"Reduce twice for high difference from neighbors average.");
		

		gd.addMessage  ("Align to eigenvectors, replaces above group");
		gd.addCheckbox ("Use eigenvectors", this.eig_use,
				"Use adjustment based on aligned/normalized offset vectors, errors sqrt(eigenvalue) normalized");
		gd.addCheckbox ("No eigenvectors LMA", this.eig_xy_lma,
				"Do not use eigenvectors for LMA (prepare the same, debug feature).");
		gd.addNumericField("Minimal correlation strength (intrascene)",this.eig_str_sum, 5,7,"",
				"Minimal correlation strength for transform-domain averaging among sensors. Weeker tiles results are removed.");
		gd.addNumericField("Minimal correlation strength (intrascene) w/FPN",this.eig_str_sum_fpn, 5,7,"",
				"Similar to above, but for small offsets where FPN correlation may be present");
		gd.addNumericField("Minimal correlation neighbors strength", this.eig_str_neib, 5,7,"",
				"Minimal correlation strength for transform-domain averaging among sensors and neighbors. Weeker tiles results are removed.");
		gd.addNumericField("Minimal correlation neighbors strength w/FPN", this.eig_str_neib_fpn, 5,7,"",
				"Similar to above, but for small offsets where FPN correlation may be present");
		gd.addNumericField("Minimal correlation absolute",           this.eig_min_abs, 5,7,"",
				"Minimal absolute corr value to keep for centroid and eigen filtering of connected.");
		gd.addNumericField("Minimal correlation relative",           this.eig_min_rel, 5,7,"",
				"Minimal relative to max corr value to keep for centroid and eigen filtering of connected.");
		gd.addNumericField("Pedestal supress",                       this.eig_sub_frac, 5,7,"",
				"1.0 - subract threshold value before centroid and covariance, 0.0 - keep original correlation values.");
		gd.addNumericField("Infinity radius",                        this.eig_max_sqrt, 5,7,"",
				"For sqrt(egenvalue)==radius - consider infinity (infinite linear feature, a line).");
		gd.addNumericField("Minimal radius",                         this.eig_min_sqrt, 5,7,"",
				"For sqrt(egenvalue)==radius - do not trust too sharp (caused by small spot over threshold).");
		gd.addNumericField("Number of recentering passes",           this.eig_recenter, 0,3,"",
				"Mask data with cosine window determined from eigenvectors/values and refine dx, dy this number of times");
		gd.addNumericField("Pedestal subtract for recentering",      this.eig_sub_frac1, 5,7,"",
				"Subtract this fraction of the threshold from the 2D correlation data during recentering (dx, dy refinement).");
		gd.addNumericField("Scale ellipse half-axes",                this.eig_scale_axes, 5,7,"x",
				"Scale ellipse half-axes (1/sqrt(lambda1), 1/sqrt(lambda1).");
		gd.addNumericField("Increment ellipse half-axes",            this.eig_inc_axes, 5,7,"pix",
				"Add to half-axes.");
		gd.addCheckbox ("Use neighbors if single fails",             this.eig_use_neibs,
				"Use correlation from 9 tiles with neibs, if single-tile fails");
		gd.addNumericField("Min weak neighbors",                     this.eig_min_weaks, 0,3,"",
				"Minimal number of weak neighbors to keep center weak tile. Too few will not reduce noise by averaging");
		gd.addNumericField("Min strong neighbors of strong tiles",   this.eig_min_strongs, 0,3,"",
				"Minimal number of strong (w/o averaging with neighbors) tiles.");
		gd.addNumericField("Maximal disparity difference to neighbors",this.eig_disp_diff, 5,7,"pix",
				"Maximal disparity difference from the closest (by disparity) neighbor.");
		
		gd.addCheckbox ("Remove weak by strong and too few weak neighbors", this.eig_remove_neibs,
				"Remove weak (by-neibs) tiles if they have strong (by-single) neighbor. Or if they have too few weak ones");
		gd.addCheckbox ("Apply other filters",                       this.eig_filt_other,
				"Apply other (before-eigen) filters");
		gd.addNumericField("Maximal eigen-normalized RMS",           this.eig_max_rms, 5,7,"",
				"Maximal eigen-normalized RMSE for LMA adjustment. Replaces \"Maximal RMS to fail\" setting below.");
		gd.addCheckbox ("Use fast 2x2 eigenvectors",             this.eig_fast2x2,
				"Use specific 2x2 calculation, false - use Jama");
		
		
		
		gd.addMessage  ("Filtering tiles for interscene matching");
		
		gd.addCheckbox ("Use combo DSI (if available) for reliable", this.use_combo_reliable,
				"Use interscene DSI if available (instead of the single-scene) for selecting reliable tiles");
		gd.addCheckbox ("Need LMA",                                  this.ref_need_lma,
				"Require LMA result for reliable reference");
		gd.addCheckbox ("Need LMA if combo",                         this.ref_need_lma_combo,
				"Require LMA result for reliable reference");
		
		gd.addNumericField("DSI_MAIN minimal strength",              this.min_ref_str, 5,7,"",
					"Match only tiles where DSI_MAIN is stronger than that (and has LMA).");
		gd.addNumericField("INTER-INTRA-LMA minimal strength",       this.min_ref_str_lma, 5,7,"",
				"Match only tiles where INTER-INTRA-LMA is stronger than that (and has LMA).");
		gd.addNumericField("DSI_MAIN minimal fraction",              this.min_ref_frac, 5,7,"",
				"If relative number of the reliable tiles is less than this - use this best fraction.");
		
		gd.addMessage  ("Smooth reference disparity for initial matching");
		gd.addCheckbox ("Smooth reference disparity",                this.ref_smooth,
				"Smooth reference disparity for initial matching.");
		gd.addCheckbox ("Smooth always",                             this.ref_smooth_always,
				"In older code it was hard-wired true inside adjustDiffPairsLMAInterscene. Checked mimics that legacy behavior.");
		gd.addNumericField("Reference disparity sigma",              this.ref_sigma, 5,7,"pix",
				"Gaussian sigma to smooth reference disparity for initial matching.");
		gd.addNumericField("Max smooth offset",                      this.ref_smooth_diff, 5,7,"pix",
				"Discard smooth disparity if it differs by more from measured.");
		

		gd.addCheckbox ("Use SfM filtering if available",            this.sfm_filter,
				"Mask out tiles without sufficient SfM gain (if SfM data is available).");
		gd.addNumericField("Min SfM gain maximum to consider",       this.sfm_minmax, 5,7,"",
				"Minimal value of the SfM gain maximum to consider SfM available.");
		gd.addNumericField("Threshold fraction of the SfM maximum",  this.sfm_fracmax, 5,7,"",
				"Threshold (minimal) fraction of the SfM maximal gain to enable tile.");
		gd.addNumericField("Minimal relative SfM area",              this.sfm_fracall, 5,7,"",
				"Minimal relative area of the SfM-enabled tiles (do not apply filter if less).");

		
		
		gd.addMessage  ("Initial search for the interscene match");
		
		
		gd.addNumericField("Azimuth/tilt step",                      this.pix_step, 0,3,"pix",
				"Search in a spiral starting with no-shift with this step between probes, in approximate pixels");
		gd.addNumericField("Search spiral radius",                   this.search_rad, 0,3,"steps",
				"Maximal search radius in steps");
		gd.addNumericField("\"Maybe\" sum of strengths",             this.maybe_sum, 5,7,"",
				"Minimal acceptable sum of defined tiles strengths (will look for the best among matching)");
		gd.addNumericField("\"Sure\" sum of strengths",              this.sure_sum, 5,7,"",
				"Definitely sufficient sum of defined tiles strengths (will not continue looking for better).");
		gd.addNumericField("\"Maybe\" average of strengths",         this.maybe_avg, 5,7,"",
				"Minimal acceptable average of defined tiles strengths (will look for the best among matching)");
		gd.addNumericField("\"Sure\" average of strengths",          this.sure_avg, 5,7,"",
				"Definitely sufficient average of defined tiles strengths (will not continue looking for better).");
		
		gd.addNumericField("Maximal offset standard deviation",      this.max_search_rms, 5,7,"pix",
				"Standard deviation for X,Y offsets.");
		gd.addNumericField("\"Maybe\" FOM",                          this.maybe_fom, 5,7,"",
				"Minimal acceptable Figure of Merit (semi-total strength divided by standard deviation of offsets), will look for the best among matching.");
		gd.addNumericField("\"Sure\" FOM",                           this.sure_fom, 5,7,"",
				"Definitely sufficient FOM (semi-total strength divided by standard deviation of offsets), will non continue looking for better.");
		gd.addCheckbox ("Treat search all FPN",                      this.treat_serch_fpn,
				"Use FPN (higher) thresholds during search (even if offset is not small).");
		
		
		gd.addMessage  ("Reference scene disparity");
		gd.addCheckbox ("Use combo DSI (if available)",              this.use_combo_dsi,
				"Use interscene DSI if available (instead of the single-scene)");
		gd.addCheckbox ("LMA tiles only",                            this.use_lma_dsi,
				"Use only strong correlation tiles (with LMA available) for interscene correlation (pose matching)");

		gd.addMessage  ("Supress FPN in interscene correlations");
		gd.addCheckbox ("Enable FPN suppression",                    this.fpn_remove,
				"Zero-out integrated inter-scene correlation around zero-shift offset");
		gd.addNumericField("Maximal FPN offset to consider",         this.fpn_max_offset, 6,7,"pix",
				"Maximal offset from the zero shift to consider (normally just 8 pix)");
		gd.addNumericField("Closest peak from the FPN offset",       this.fpn_min_offset, 6,7,"pix",
				"Discard tile if maximum is closer to the fpn offset");
		gd.addNumericField("FPN suppression radius",                 this.fpn_radius, 6,7,"pix",
				"Blank correlation pixels closer than this distance from the FPN offset");
		gd.addCheckbox ("Ignore maximums \"touching\" FPN",          this.fpn_ignore_border,
				"Discard TD integrated tiles where local maximum is a neighbor (including diagonal) to blanked FPN correlation pixels");

		gd.addMessage  ("Limit series, handle FPN-related problems");
		gd.addNumericField("Minimal inter-scene offset",             this.min_offset, 6,7,"pix",
				"Minimal average pixel offset between images to consider interscene matching");
		gd.addNumericField("Maximal \"pull\" jump",                  this.max_pull_jump, 6,7,"pix",
				"Maximal offset from the current pose to the \"pull\" pose.");
		gd.addNumericField("Maximal interscene offset fraction of width", this.max_rel_offset, 6,7,"x width",
				"Maximal interscene offset as a fraction of image width to handle low overlap");
		gd.addNumericField("Maximal interscene roll",                this.max_roll_deg, 6,7,"degrees",
				"Maximal interscene roll to consider matching");
		gd.addNumericField("Maximal interscene relative zoom difference", this.max_zoom_diff, 6,7,"",
				"Applicable for the down views from a drone. Saet to 0 to ignore.");
		gd.addCheckbox ("Skip too close to reference scenes",        this.fpn_skip,
				"Skip too close to reference scenes (false - abort, previous behavior)");
		gd.addCheckbox ("Match FPN-failed with other scenes",        this.fpn_rematch,
				"Match fpn-failed scenes to later scenes with larger difference.");
		gd.addCheckbox ("Refine inversion",                          this.refine_invert,
				"Refine with LMA while inverting relative poses from other reference.");
		gd.addCheckbox ("Precompensate orientation readjustment",    this.use_precomp,
				"Guess needed initial precompensation from the previously processed scenes.");
		
		gd.addMessage  ("Detect and remove moving objects from pose matching");
		gd.addCheckbox ("Enable movement detection/elimination",     this.mov_en,
				"Detect and mask areas with detected movement to improve pose matching");
		gd.addNumericField("Detection blur sigma",                   this.mov_sigma, 6,7,"pix",
				"Blur squared difference (dx^2+dy^2) befopre thresholding for movement detection");
		gd.addNumericField("Max weighted mismatch std",              this.mov_max_std, 6,7,"pix",
				"Do not try to detect moving objects until approximate pose match is achieved (standard deviation)");
		gd.addNumericField("Relative threshold",                     this.mov_thresh_rel, 6,7,"x",
				"Moving areas over standard deviation. Both relative and absolute should be exceeded.");
		gd.addNumericField("Absolute threshold",                     this.mov_thresh_abs, 6,7,"pix",
				"Moving areas sqrt(dx*dx+dy*dy). Both relative and absolute should be exceeded.");
		gd.addNumericField("Cluster max over threshold",             this.mov_clust_max, 6,7,"",
				"Moving cluster should contain tile with this exceed over thresholds");
		gd.addNumericField("Moving cluster grow",                    this.mov_grow, 0,3,"",
				"Standard grow values - 1 - ortho, 2 - diagonal, 3 - twice orto, 4 - twice diagonal.");
		gd.addNumericField("Maximal movement size",                    this.mov_max_len, 0,3,"tiles",
				"Do not remove moving objects if they do not fit into the square with this side.");
		
		gd.addCheckbox ("Show movement debug images",                this.mov_show,
				"Disabled if 'Debug Level for interscene match' < 1");
		gd.addNumericField("Debug level for movement detection (0/1)", this.mov_debug_level, 0,3,"",
				"Disabled if 'Debug Level for interscene match' < 0");
		
		gd.addMessage  ("LMA parameters");
		gd.addCheckbox ("Azimuth",                                   this.adjust_atr[0],
				"Adjust scene camera azimuth with LMA");
		gd.addCheckbox ("Tilt",                                      this.adjust_atr[1],
				"Adjust scene camera tilt with LMA");
		gd.addCheckbox ("Roll",                                      this.adjust_atr[2],
				"Adjust scene camera roll with LMA");
		gd.addCheckbox ("X",                                         this.adjust_xyz[0],
				"Adjust scene camera X with LMA");
		gd.addCheckbox ("Y",                                         this.adjust_xyz[1],
				"Adjust scene camera Y with LMA");
		gd.addCheckbox ("Z",                                         this.adjust_xyz[2],
				"Adjust scene camera Z with LMA");

		gd.addNumericField("Exit ATR change",                        this.exit_change_atr, 6,7,"pix",
				"L2 of azimuth, tilt, roll change (in pixels at infinity) to exit fitting");
		gd.addNumericField("Exit XYZ change",                        this.exit_change_xyz, 5,7,"m",
				"L2 of linear scene camera position change (in meters) to exit fitting");


		gd.addNumericField("Max full iterations",                    this.max_cycles, 0,3,"",
				"Hard limit on interscene correlations plus LMA cycles");
		gd.addNumericField("Max LMA iterations",                     this.max_LMA, 0,3,"",
				"Hard limit on LMA iterations");
		gd.addNumericField("Maximal RMS to fail",                    this.max_rms, 5,7,"pix",
				"Maximal fitting RMS to consider fitting failure");
		
		
		gd.addMessage  ("Interscene Equalization (Equalization of the interscene correlation confidence to improve camera X,Y,Z matching)");
		gd.addCheckbox ("Enable equalization",                       this.eq_en,
				"Enable boosting of the weak but important interscene correlation tiles strength by equalizing average strength of the \"supertiles\"");
		gd.addMessage  ("Equalization supertiles dimensions");
		gd.addNumericField("Supertile horizontal stride",            this.eq_stride_hor, 0,3,"tiles",
				"Half of a equalization supertile width - supertiles have a 50% overlap in each direction.");
		gd.addNumericField("Supertile vertical stride",              this.eq_stride_hor, 0,3,"tiles",
				"Half of a equalization supertile height - supertiles have a 50% overlap in each direction.");
		gd.addMessage  ("Supertile qualification (not qualified will keep the original strength value)");
		gd.addNumericField("Minimal supertile total weight",         this.eq_min_stile_weight, 5,7,"",
				"Minimal total weight of the tiles in a supertile (lower will not be modified).");
		gd.addNumericField("Minimal number of defined tiles",        this.eq_min_stile_number, 0,3,"tiles",
				"Minimal number of defined tiles in a supertile.");
		gd.addNumericField("Minimal weight fraction of average",     this.eq_min_stile_fraction, 5,7,"",
				"Minimal total supertile strength RELATIVE to the average one.");
		gd.addMessage  ("Individual tiles qualification (applied after calculating the amount of boost).");
		gd.addNumericField("Minimal equalized tile disparity",       this.eq_min_disparity, 5,7,"pix",
				"Allows to disqualify infinity and far tiles (do not contribute to camera X,Y,Z) freom boosting.");
		gd.addNumericField("Maximal equalized tile disparity",       this.eq_max_disparity, 5,7,"pix",
				"Unlikely to have meaning, introduced for symmetry. Can be set to a large number.");
		gd.addMessage  ("Boost amount. Each defined tile will be increased by approximately previous subtracted offset and then scaled");
		gd.addNumericField("Add to strength before scaling",         this.eq_weight_add, 5,7,"",
				"Add to strength, then scale.");
		gd.addNumericField("Maximal boost ratio",                    this.eq_weight_scale, 5,7,"",
				"Strength scale. If the new supertile strength exceed the target value, each tile will be scaled down to match.");
		gd.addNumericField("Equalization level",                     this.eq_level, 5,7,"",
				"Target supertile strength will be set to: 0 - original strength (no modification), 1.0 - average supertile strength.");
		
		gd.addTab("MB","Motion Blur");
		gd.addCheckbox ("Compensate motion blur",                    this.mb_en,
				"Enable motion blur correction.");
		gd.addNumericField("Sensor time constant",                   this.mb_tau, 5,7,"s",
				"Sensor exponential decay in seconds.");
		gd.addNumericField("Maximal gain",                           this.mb_max_gain, 5,7,"x",
				"Maximal gain for motion blur correction (if needed more for 1 pixel, increase offset). Will be forced fro the last adjustment");
		gd.addNumericField("Maximal gain pose",                      this.mb_max_gain_inter, 5,7,"x",
				"Maximal gain for motion blur correction during interscene correlation. Will be used for all but the last adjustment.");
		gd.addTab("CUAS","CUAS Parameters");
		gd.addCheckbox ("Re-calculate center CLT",                    this.cuas_update_existing,
				"Re-create center_CLT if it exists (FIXME: accumulates errors - need fixing).");
		gd.addNumericField("Discard margins",                        this.cuas_discard_border, 0,3,"pix",
				"Discards this number of pixels from each side when merging images.");
		gd.addNumericField("Maximal X,Y fold",                       this.cuas_max_fold, 5,7,"pix",
				"Maximal non-monotonic Px, Py in PxPyD (usually near image edges).");
		gd.addNumericField("Minimal tiles in a row/column",          this.cuas_min_in_row_col, 0,3,"tiles",
				"Discards rows then columns that have less defined tiles (noticed in a diagonal after folds removal).");
		gd.addNumericField("CLT tile variant thershold",             this.cuas_clt_variant, 5,7,"",
				"Threshold from the nearest in teplate to create a variant for the tile (RMS).");
		gd.addNumericField("CLT tile diff merge",                    this.cuas_clt_threshold, 5,7,"",
				"Threshold to the CLT tile difference to a template (RMS) when selectin a single best.");
		gd.addNumericField("CLT outliers scale",                     this.cuas_clt_decrease, 5,7,"x",
				"Scale CLT outliers tiles weight when averaging.");
		gd.addNumericField("Decay of average",                       this.cuas_decay_average, 5,7,"s",
				"Reduce weight of old scenes while averaging (in e times) after this time.");
		gd.addNumericField("Keep weight fraction",                   this.cuas_keep_fraction, 5,7,"x",
				"Filter tile CLT variants that keep fraction of the total weight.");
		gd.addCheckbox ("Recalculate after each tuning step",        this.cuas_step,
				"Recalciulate correlation pattern after each position/orientation and disparity adjustment step.");
		gd.addMessage("=== Subtract FPN ===");
		gd.addCheckbox ("Subtract FPN",                              this.cuas_subtract_fpn,
				"Subtract FPN from the channel images.");
		gd.addCheckbox ("(Re)calculate FPN",                         this.cuas_calc_fpn,
				"Recalculate+save FPN unconditionally, if false - try to read saved one.");
		gd.addNumericField("Rotation period",                        this.cuas_rot_period, 5,7,"scene periods",
				"Used for averaging FPN for integer number of rotation periods. TODO: calculate from the actual rotation.");
		gd.addNumericField("Minimal series length",                  this.cuas_min_series, 5,7,"scene periods",
				"Skip shorter series (should be not much less than rotation period.");
		gd.addMessage("=== Row/column noise (on top of the FPN) ===");
		gd.addCheckbox ("Subtract row/column noise",                 this.cuas_subtract_rowcol,
				"Subtract per-scene row/column noise (on top of the per scene sequence FPN).");
		gd.addCheckbox ("(Re)calculate row/column noise",            this.cuas_calc_rowcol,
				"Recalculate per-scene row column noise even if it exists. If false, try to read first.");
		gd.addNumericField("Row/column unsharp mask filter",         this.cuas_um_sigma, 5,7,"pix",
				"Apply Unsharp mask filter to the difference between individual scenes and per-sequence FPN before calculating row/column noise.");
		gd.addNumericField("Row/column maximal difference",          this.cuas_max_abs_rowcol, 5,7,"counts",
				"Consider pixels were abs(UM difference) does not exceed this value during row/column noise calcualtion.");
		gd.addNumericField("Row/column scale outliers",              this.cuas_outliers_rowcol, 5,7,"x",
				"Scale weight of the outliers with high difference than above (to prevent undefined values).");
		gd.addCheckbox ("Reset average on first sequence",           this.cuas_reset_first,
				"Reset average in first scene (for large diffirence in unfinished row/col.");
		gd.addMessage("=== Tasks inversion ===");
		gd.addNumericField("Inversion: add margins",                 this.cuas_invert_margins, 0,3,"tiles",
				"Increase processing area each side when calculating tasks inversion.");
		gd.addNumericField("Number of inverion iterations",          this.cuas_invert_iters, 0,3,"",
				"Number of iterations when enhancing tasks inversion.");
		gd.addNumericField("Inversion tolerance",                    this.cuas_invert_tolerance, 5,7,"pix",
				"Finish enhancing tasks inversion when last change was lower than this value.");
		gd.addNumericField("Inversion gap (dual size)",              this.cuas_invert_gap2, 0,3,"tiles",
				"Maximal dual gap size for inversion (depends on scanning radius in tiles)  .");
		
		gd.addMessage("=== Masking out high LoG areas for averaging for different poses ===");
		gd.addNumericField("UM sigma",                               this.cuas_w_um_sigma, 5,7,"pix",
				"Run UM to approximate LoG for detection of areas with sharp changes.");
		gd.addNumericField("Good margins",                           this.cuas_w_good_margins, 0,3,"pix",
				"Consider defined values near image margins always good (do not have anything better anyway).  Does not apply to NaN areas that will be filled from different poses.");
		gd.addNumericField("Blanking threshold",                     this.cuas_w_threshold, 5,7,"",
				"Threshold to cut off high variations (it will be squared to compare to squared difference values).");
		gd.addNumericField("Scale threshold",                        this.cuas_w_scale_thresh, 5,7,"x",
				"Finish enhancing tasks inversion when last change was lower than this value.");
		gd.addCheckbox ("Expand squared diffs",                      this.cuas_w_expand_max,
				"Replace squared diff values by max of 8 neighbors.");
		gd.addNumericField("Outliers weight",                        this.cuas_w_outliers, 5,7,"x",
				"Assign weights to outliers to avoid NaNs in bad areas (remaining undefined with all poses combined).");
		gd.addNumericField("Blur weights",                           this.cuas_w_blur, 5,7,"pix",
				"Blur thresholded squared values to smooth transitions.");

		gd.addMessage("=== Sky filters for intersceneExport() ===");
		gd.addCheckbox ("Use parent's DSI",                          this.cuas_dsi_parent,
				"Ignore this scene sequence DSI, use parent'se.");
		gd.addNumericField("Minimal FG strength",                    this.cuas_fg_strength, 5,7,"",
				"All higher (lower Y) than strong FG should have lower disparity.");
		gd.addNumericField("Disparity allowance",                    this.cuas_disp_add, 5,7,"pix",
				"Allow disparity for lower Y to be slightly higher than limit from the strong FG objects.");
		gd.addNumericField("Skip bottom rows",                       this.cuas_skip_bottom, 0,3,"pix",
				"Skip bottom rows when determining FG.");
		gd.addNumericField("Multi-tile DSI strength",                this.cuas_multi_strength, 5,7,"",
				"Maximal strength to use multi-tile DSI.");
		gd.addNumericField("Reliable strength",                      this.cuas_reliable_str, 5,7,"",
				"Reliable tile strength when combo-dsi is available.");
		
		gd.addMessage("=== CUAS motion detection ===");
		gd.addNumericField("Phase correlation fat zero",             this.cuas_fat_zero, 5,8,"",
				"Phase correlation fat zero for motion detection.");
		gd.addNumericField("Centroids radius",                       this.cuas_cent_radius, 5,8,"",
				"Centroids radius for maximums isolation.");
		gd.addNumericField("Recenter centroid",                      this.cuas_n_recenter, 0,3,"",
				"when cosine window, re-center window these many times");
		gd.addNumericField("Minimal correlation strength",            this.cuas_rstr, 5,8,"",
				"Minimal phase correlation maximums (negative value - relative to the maximal strength of the whole images correlations).");
		gd.addCheckbox    ("Smooth weights",                         this.cuas_smooth,
				"Apply cosine weights when averaging a sequence of correlation pairs.");
		gd.addNumericField("Number of pairs",                        this.cuas_corr_pairs, 0,3,"",
				"The number of correlation pairs to accumulate.");
		gd.addNumericField("Pairs offset",                           this.cuas_corr_offset, 0,3,"scenes",
				"Offset between the correlation pairs");
		gd.addCheckbox    ("Gaussian instead of Running Average",    this.cuas_gaussian_ra,
				"Use gaussian temporal Gaussian instead of running average.");
		gd.addNumericField("Temporal unsharp mask length",           this.cuas_temporal_um, 0,3,"scenes",
				"Subtract running average this long.");
		gd.addNumericField("Temporal UM threshold",                  this.cuas_tum_threshold, 5,8,"",
				"Remove outlier pixels that differ from the running average by more than this (to later re-calculate running average excluding those pixels.");

		gd.addNumericField("Pre-correlation running average",        this.cuas_precorr_ra, 0,3,"scenes",
				"Smoothing input data by running average before correlation for motion vectors calculations. Target extraction wil still use individual scenes.");
		gd.addNumericField("Correlation step after RA",              this.cuas_corr_step, 0,3,"scenes",
				"Decimate correlations after running average");
		gd.addCheckbox    ("Half scan step",                         this.cuas_half_step,
				"Reduce step for motion detection = offset/2, if false = offset.");
		gd.addNumericField("Local max range",                        this.cuas_max_range, 0,3,"",
				"While filtering local correlation maximums: 1 - 3x3 neighbors, 2 - 5x5 ones.");
		gd.addNumericField("Number of enhancement cycles",           this.cuas_num_cycles, 0,3,"",
				"Number of cycles of testing and removing bad targets from compoetition with weaker neighbors.");

		gd.addMessage("=== Moving target filtering ===");
		gd.addNumericField("Minimal speed",                          this.cuas_speed_min, 5,8,"ppr",
				"Minimal target speed in pixels per range (per cuas_corr_offset).");
		gd.addNumericField("Preferable speed",                       this.cuas_speed_pref, 5,8,"ppr",
				"Boost effective strength when speed is above this.");
		gd.addNumericField("Maximal speed boost",                    this.cuas_speed_boost, 5,8,"",
				"Maximal speed-caused effective strength boost.");

		gd.addNumericField("Negative tile data limit",               this.cuas_lmax_flt_neglim, 5,8,"",
				"Limit negative data to reduce ridge influence (>=0 to disable). Only used if filtering is ON (see below).");
		gd.addNumericField("HPF sigma for the data tile",            this.cuas_lmax_flt_hsigma, 5,8,"pix",
				"HPF (~UM) subtract GB with this sigma from the data tile.");
		gd.addNumericField("LPF sigma for the data tile",            this.cuas_lmax_flt_lsigma, 5,8,"pix",
				"LPF - GB result of the previous subtraction.");
		gd.addNumericField("Scale filtering result",                 this.cuas_lmax_flt_scale, 5,8,"x",
				"Scale filtering result.");
		
		gd.addMessage("Cuas sky mask path: \""+cuas_sky_path+"\"");
		gd.addNumericField("Cuas sky threshold",                 this.cuas_sky_threshold, 5,8,"",
				"Minimal value of the sky mask where target is possible.");
		gd.addStringField ("Sky mask x,y offsets",                  IntersceneMatchParameters.doublesToString(cuas_sky_offset), 80,
				"Add to coordinates when looking for the mask value .");
		
		gd.addNumericField("Maximums separation fraction",           this.cuas_lmax_fraction, 5,8,"",
				"Check if local maximum is separated from tye surrounding by this fraction of the maximum value.");
		gd.addNumericField("Hack for ridges",                        this.cuas_lmax_hack_ridge, 5,8,"",
				"Remove horizontal streaks, where average(abs()) for the line exceeds fraction of abs max(0.32-good max, 0.55 - bad max).");
		gd.addNumericField("Maximums separation radius",             this.cuas_lmax_radius, 5,8,"pix",
				"Look inside ((int)cuas_lmax_radius) * 2 + 1 square for the local maximum isolation .");

		gd.addCheckbox    ("Zero far from maximums pixels",          this.cuas_lmax_zero,
				"Zero all data outside this radius from the maximum.");
		
		
		gd.addNumericField("Mask radius in the centered mode ",      this.cuas_centered_radius, 5,8,"pix",
				"In centered mode (target placed at [8,8]) - zero out data outside peripheral areas, keep 0.5 at the radius.");
		gd.addNumericField("Centered mask blur fraction",            this.cuas_centered_blur, 5,8,"",
				"Relative transition radius range (for 0.5 the mask inside (1-0.5) radius will be 1.0, outside (1+0.5) radius will be 0.");
		gd.addNumericField("Duplicates offset tolerance",            this.cuas_duplicate_tolerance, 5,8,"pix",
				"Remove weaker maximums closer than this to a stronger one.");
		
		
		gd.addNumericField("Target radius",                          this.cuas_target_radius, 5,8,"pix",
				"Target radius, also used to calculate fraction of totals inside (windowed) to all positive values.");
		gd.addNumericField("Minimal target strength",                this.cuas_target_strength, 5,8,"",
				"Minimal value of the target image.");
		gd.addStringField ("Fraction for strengths", double2dToString(this.cuas_target_frac), 100,
				"Variable number of (strength, fraction) pairs separated by \":\". Each pair of strength, minimal fraction is separated by \",\".");
		gd.addCheckbox    ("Target not on the tile edge",            this.cuas_no_border,
				"Exclude targets with centers on the edge of 16x16 tiles.");
		
		gd.addMessage("=== Moving target LMA ===");
		gd.addNumericField("Weight Gaussian sigma",                  this.cuas_lma_sigma, 5,8,"pix",
				"Weights to emphasize maximum center area when fitting.");
		gd.addNumericField("Weight pedestal",                        this.cuas_wnd_pedestal, 5,8,"",
				"Add constant to Gaussian weights.");
		gd.addNumericField("Target typical radius",                  this.cuas_lma_r0, 5,8,"pix",
				"Typical target radius including negative overshoot (caused by UM filter).");
		gd.addNumericField("Target maximum overshoot",               this.cuas_lma_ovrsht, 5,8,"",
				"Hos much smaller is the first zero than total maximum with overshoot (2.0 - first zero radius 1/2 of the full.");
		
		gd.addCheckbox    ("LMA fit X,Y",                            this.cuas_lma_fit_xy,
				"Fit target center position.");
		gd.addCheckbox    ("LMA fit amplitude (strength)",           this.cuas_lma_fit_a,
				"Fit maximum amplitude.");
		gd.addCheckbox    ("LMA fit offset",                         this.cuas_lma_fit_c,
				"Fit out-of-maximum level (offset).");
		gd.addCheckbox    ("LMA fit radius",                         this.cuas_lma_fit_r,
				"Fit target total radius - includes negative overshoot caused by UM filter.");
		gd.addCheckbox    ("LMA fit overshoot",                      this.cuas_lma_fit_k,
				"Fit target overshoot (2.0 - first zero crossing at half radius.");
		
		
		
		
		
		
		gd.addMessage("--- Moving target LMA fitting parameters ---");
		gd.addNumericField("LMA lambda",                             this.cuas_lambda, 5,8,"",
				"LMA initial lambda.");
		gd.addNumericField("Scale lambda after success",             this.cuas_lambda_good, 5,8,"",
				"Multiply lambda if RMS decreaed.");
		gd.addNumericField("Scale lambda after failure",             this.cuas_lambda_bad, 5,8,"",
				"Multiply lambda if RMS increaed.");
		gd.addNumericField("Maximal lambda",                         this.cuas_lambda_max, 5,8,"",
				"Give up if lambda gets higher value.");
		gd.addNumericField("Relative RMS improvement",               this.cuas_rms_diff, 5,8,"",
				"Finish fitting when the relative RMS improvement drops below this value.");
		gd.addNumericField("LMA iterations",                         this.cuas_num_iter, 0,3,"",
				"Maximal number of the LMA iterations.");
		gd.addMessage("--- Moving target discrimination parameters thresholds ---");
		gd.addNumericField("Maximal RMS",                            this.cuas_lma_rms, 5,8,"",
				"Maximal RMS for target that should match always, regardless of the amplitude.");
		gd.addNumericField("Maximal sufficient RMS",                 this.cuas_lma_arms, 5,8,"",
				"Maximal sufficient RMS for target. Satisfying any of the sufficient or relative is enough");
		gd.addNumericField("Maximal relative RMS",                   this.cuas_lma_rrms, 5,8,"",
				"Maximal relative (to amplitude) RMS for target. Satisfying any of the absolute and relative is sufficient");
		gd.addNumericField("Minimal target amplitude",               this.cuas_lma_mina, 5,8,"",
				"Filter out weak targets.");
		gd.addNumericField("Scale minimal amplitude",                this.cuas_lma_pre_mina, 5,8,"",
				"Scale cuas_lma_mina to filter initial candidates (if there are > one of them).");
		gd.addNumericField("Keep this number of weak",               this.cuas_min_keep, 0,3,"",
				"Keep at least this number of candidates before using cuas_lma_pre_mina filter.");
		gd.addNumericField("Maximal radius",                         this.cuas_lma_maxr, 5,8,"pix",
				"Maximal target radius including negative overshoot.");
		gd.addNumericField("Mimimal inner radius",                   this.cuas_lma_minr1, 5,8,"pix",
				"Minimal radius of positive peak.");
		gd.addNumericField("Minimal overshoot ratio",                this.cuas_lma_mink, 5,8,"",
				"Minimal ratio of the overshoot radius to the first 0 radius (typical 1.0).");
		gd.addNumericField("Maximal overshoot ratio",                this.cuas_lma_maxk, 5,8,"",
				"Maximal ratio of the overshoot radius to the first 0 radius (typical 3.0).");
		gd.addNumericField("Minimal max pixel to amplitude ratio",   this.cuas_lma_a2a, 5,8,"",
				"Minimal ratio of the maximal pixel near fitted maximum to the LMA amplitude.");
		gd.addMessage("=== Slow targets detection  ===");
		gd.addNumericField("Slow targets averaging",                 this.cuas_slow_ra, 5,8,"frames",
				"Running averaging for slow targets.");
		gd.addNumericField("Slow targets score scale",               this.cuas_slow_score, 5,8,"",
				"Scale scores for slow target detection mode relative to fast target.");
		
		gd.addMessage("=== Handling overall target scores  ===");
		gd.addNumericField("Minimal MV score",                       this.cuas_score_mv, 5,8,"",
				"Remove potential motion tiles that have lower scores.");
		gd.addNumericField("Minimal target LMA score",               this.cuas_score_lma, 5,8,"",
				"Remove potential target LMA tiles that have lower scores.");
		gd.addNumericField("Limit each individual score factor",     this.cuas_factor_lim, 5,8,"",
				"Limit each contributing score factor by this value.");
		gd.addNumericField("Raise score factor to this power",       this.cuas_factor_pow, 5,8,"",
				"Raise each score factor to this power before combining them.");
		gd.addStringField ("Score factors weights",                  IntersceneMatchParameters.doublesToString(cuas_score_coeff), 80,
				"Relative importance of LMA amplitude, RMSE and RMSE/ampitude, center,  before/after match, sqrt(match length), travel.");
		gd.addCheckbox ("Center targets in their tiles",             this.cuas_center_targ,
				"Re-run target extraction placing the targets at the center of their tiles.");
		gd.addCheckbox ("Multi-target mode",                         this.cuas_multi_targ,
				"Use multi-target (for each tile) mode.");
		
		
		gd.addMessage("=== Post-processing targets filtering  ===");
		gd.addCheckbox ("Remove single-frame targets",              this.cuas_isolated,
				"Remove targets that do not have neighbors before/after.");

		gd.addNumericField("Minimal sequence length",               this.cuas_min_seq, 0,3,"key frames",
				"Minimal number of consecutive key frames for the target to be detected.");
		gd.addNumericField("Sufficient consecutive key frames",     this.cuas_enough_seq, 0,3,"key frames",
				"Number of consecutive key frames considered good regardless of their scores.");
		gd.addNumericField("Minimal sequence travel",               this.cuas_seq_travel, 5,8,"pix",
				"Minimal diagonal of the bounding box that includes sequence to be considered \"cuas_enough_seq\". Filtering out atmospheric fluctuations.");
		
		gd.addNumericField("Minimal end sequence length",           this.cuas_min_seq_fl, 0,3,"key frames",
				"Minimal number of consecutive key frames for the target to be detected if the sequence includes first or last key frame.");
		
		gd.addNumericField("Maximal allowed mismatch",              this.cuas_max_mismatch, 5,8,"pix",
				"Maximal position error between consecutive scene sequences.");
		gd.addNumericField("Good mismatch",                         this.cuas_good_mismatch, 5,8,"pix",
				"Maximal position error between consecutive scene sequences to add to the score.");
		gd.addNumericField("Good mismatch between slow and fast",   this.cuas_slow_fast_mismatch, 5,8,"pix",
				"Allow larger mismatch between slow and fast.");
		gd.addNumericField("Matching length power",                 this.cuas_match_len_pwr, 5,8,"",
				"Raise matching length to this power for calculating score.");

		gd.addCheckbox ("Fail mismatch early",                      this.cuas_fail_mismatch,
				"Fail after score calculation if there is not match both before and after.");
		gd.addCheckbox ("Ignore mismatch",                          this.cuas_ignore_mismatch,
				"Calculate mismatch, but keep targets for debugging (see cuas_target_debug).");
		gd.addCheckbox   ("Filter by horizon",                      this.cuas_by_horizon,
				"Remove targets that are below the horizon.");
		gd.addNumericField("Pixel Y of the horizon",                this.cuas_horizon, 5,8,"pix",
				"Pixel Y of the horizon.");
		
		gd.addMessage("=== Targets video generation (see also Vidoe/Stereo tab)  ===");
		

		gd.addNumericField("Mask diameter",                         this.cuas_mask_width, 5,8,"pix",
				"Taget mask to replace static background with moving target.");
		gd.addNumericField("Mask transition width",                 this.cuas_mask_blur, 5,8,"pix",
				"Transition between opaque and transparent mask.");
		gd.addCheckbox ("Mask round",                               this.cuas_mask_round,
				"Use round mask. Unchesked - use sharp square mask without any transition.");
		
		gd.addNumericField("Unidentified targets icon type",        this.cuas_target_type, 0,3,"",
				"0 - unknown (dashed square), 1 - known (square), 2 - friend (circle), 3 - foe (diamond).");
		gd.addNumericField("Identified Targets icon type",          this.cuas_known_type, 0,3,"",
				"0 - unknown (dashed square), 1 - known (square), 2 - friend (circle), 3 - foe (diamond).");
		
		gd.addNumericField("Image range",                           this.cuas_input_range, 5,8,"10mK",
				"Dispaly pixel values range (1.0 ~= 10 mK).");
		gd.addCheckbox ("Scale images twice",                       this.cuas_scale2x,
				"Scale imges to 1280x1024 for higher annotation quality.");
		gd.addCheckbox ("Smooth background with runnong average",   this.cuas_ra_background,
				"Apply running average to the background of the moving targets (false - use high-noise no-averaged images.");

		gd.addMessage("=== Saving intermediate and final images and video ===");
		gd.addCheckbox ("Generate and save targets images/video",   this.cuas_targets,
				"Generate after differential images are output");
		
		gd.addNumericField("Maximal iteration to save/show",        this.cuas_iter_show, 0,3,"",
				"Maximal enhancement iteration to save/ show intermediate result (0 - none).");
		gd.addCheckbox ("Save/show 2D correlations",                this.cuas_2d_save_show,
				"Save and optionally show 2D correlations.");
		gd.addCheckbox ("Save tile-resolution intermediate images", this.cuas_intermed_low,
				"Save intermediate vector fields and target coordinates from the LMA (80x64 layers).");
		gd.addCheckbox ("Save pixel-resolution intermediate images",this.cuas_intermed_high,
				"Save pixel-resolution accumulated images (640x512).");
		gd.addCheckbox ("Save gigabyte files",                      this.cuas_intermed_giga,
				"Save pixel-resolution huge files, such as hyperstack comparison.");
		gd.addCheckbox ("Save/show additional images",              this.cuas_debug_more,
				"Save/show more images.");

		
		gd.addCheckbox ("Generate and save detected targets",       this.cuas_generate,
				"Generate and save targets Tiff and/or video files (all images and videos).");
		gd.addCheckbox ("Save target data as a CSV file",           this.cuas_generate_csv,
				"Generate and save targets as a spreadsheet data (.csv).");
		
		gd.addCheckbox ("Save monochrome targets+background",       this.cuas_save_mono,
				"Save 32-bit monochrome targets+background Tiffs (before optional scaling).");
		gd.addCheckbox ("Save color targets+background",            this.cuas_save_color,
				"Save color rendered images (same as videos), optionally scaled.");
		gd.addCheckbox ("Save videos",                              this.cuas_save_video,
				"Save video with moving targets.");
		gd.addCheckbox ("Clean videos",                             this.cuas_clean_video,
				"Save video without any debug information for targets, output in TIFF files. False - same output for video and TIFFs.");
		
		gd.addMessage("=== Target Annotation ===");
		gd.addCheckbox ("Annotate targets",                         this.cuas_annotate,
				"Add textual annotations to the target icons.");
		gd.addCheckbox ("Show UAS from flighgt log",               this.cuas_annotate,
				"Indicate UAS position from the UAS flight log, if available.");
		{
			String scolor = String.format("%08x", getLongColor(this.cuas_text_color));
			gd.addStringField ("Target annotations color",scolor, 8, "Any invalid hex number disables annotation");
		}
		{
			String scolor = String.format("%08x", getLongColor(this.cuas_text_bg));
			gd.addStringField ("Target annotations background color",scolor, 8, "Any invalid hex number disables annotation");
		}
		{
			String scolor = String.format("%08x", getLongColor(this.cuas_selected_color));
			gd.addStringField ("Selected target annotations color",scolor, 8, "Any invalid hex number disables annotation");
		}
		
		gd.addCheckbox ("Transparent text background (other)",     this.cuas_transparent,
				"Annotate other targets over image (unchecked - over solid color).");
		gd.addCheckbox ("Transparent text background (UAS)",       this.cuas_transparent_uas,
				"Annotate UAS over image (unchecked - over solid color).");
		
		gd.addStringField ("Font name",                             this.cuas_font_name, 20,
				"Font name, currently \"Monospaced\".");
		gd.addNumericField("Font size",                             this.cuas_font_size, 0,3,"pnt",
				"Font size befoerr image scaling. Scaled images use proportionally scaled font .");
		gd.addNumericField("Font type",                             this.cuas_font_type, 0,3,"",
				"Font type: 0 - PLAIN, 1 - BOLD, 2 - ITALIC.");
		gd.addNumericField("Font spacing",                          this.cuas_font_spacing, 5,8,"x",
				"Height to size ratio. If 0 - will use default spacing ( ~=1.5).");
		gd.addCheckbox ("Reserve lines for undefined parameters",   this.cuas_annot_missing,
				"Reserve line for requested but missing parameters.");
		gd.addNumericField("Known image pixel X coordinate",        this.cuas_px0, 0,3,"",
				"Image pixel X corresponding to the known azimuth.");
		gd.addNumericField("Known image pixel X coordinate",        this.cuas_py0, 0,3,"",
				"Image pixel Y corresponding to the known elevation.");
		gd.addNumericField("Known pixel azimuth",                   this.cuas_az0, 5,8,"degree",
				"Azimuth corresponding to the known pixel X.");
		gd.addNumericField("Known pixel elevation",                 this.cuas_el0, 5,8,"degree",
				"Elevation corresponding to the known pixel Y.");
		gd.addCheckbox ("Show target disparity",                    this.cuas_show_disp,
				"Show disparity before infinity correction (not in clean mode).");
		gd.addCheckbox ("Show target distance (meters)",            this.cuas_show_rng,
				"Show range true from the UAS log.");
		gd.addCheckbox ("Show infinity distance to target",         this.cuas_show_inf,
				"Show when distance is above limit (unchecked - skip the line).");
		gd.addCheckbox ("Show infinity as >range_limit",            this.cuas_show_inf_gt,
				"Show infinity as >range_limit, unchecked - as infinity symbol.");
		gd.addCheckbox ("Show true range", this.cuas_show_true_rng,
				"Show range true from the UAS log.");
		
		gd.addMessage("=== Ranging parameters ===");
		gd.addCheckbox ("Smooth omegas",                             this.cuas_smooth_omegas,
				"Recalculate omegas from continuing targets positions.");
		gd.addCheckbox ("Apply unsharp mask to images for ranging",  this.cuas_rng_um,
				"Apply unsharp mask to images before ranging.  Assumed true.");
		gd.addCheckbox ("Apply unsharp mask to all images before ranging", this.cuas_rng_um_all,
				"Unsharp mask all images (after FPN and Row/Col), unchecked - when needed. Checked faster, but original data is lost.");
		gd.addNumericField("Unsharp mask sigma (radius)",            this.cuas_rng_um_sigma, 5,7,"pix",
				"Unsharp mask Gaussian sigma for ranging.");
		gd.addCheckbox ("Apply unsharp mask twice",                  this.cuas_rng_um2,
				"Apply unsharp mask twice to fight gradients appearing as level shifts.");
		gd.addNumericField("Unsharp mask weight",                    this.cuas_rng_um_weight, 5,7,"",
				"Unsharp mask weightt (multiply blurred version before subtraction from the original).");
		gd.addCheckbox ("Cosine scenes window",                      this.cuas_rng_coswnd,
				" Use cosine scenes window (false - rectangular) .");
		gd.addNumericField("Combine scenes before correlating",      this.cuas_rng_combine, 0,3,"",
				"Combine multiple scenes before intrascene correlation.");
		
		gd.addNumericField("Mask radius for ranging, first iteration",this.cuas_rng_radius0, 5,8,"pix",
				"Mask out data outside peripheral areas, keep 0.5 at the radius .");
		gd.addNumericField("Mask radius for ranging, next iterations",this.cuas_rng_radius, 5,8,"pix",
				"Mask out data outside peripheral areas, keep 0.5 at the radius .");
		gd.addNumericField("Mask blur fraction for ranging",         this.cuas_rng_blur, 5,8,"",
				"Relative transition radius range (for 0.5 the mask inside (1-0.5) radius will be 1.0, outside (1+0.5) radius will be 0.");
		gd.addNumericField("Fat zero (first iteration)",             this.cuas_rng_fz0, 5,8,"",
				"Fat zero for target ranging (larger, use for the initial fitting).");
		gd.addNumericField("Fat zero (next iterations)",             this.cuas_rng_fz, 5,8,"",
				"Fat zero for target ranging (smaller, use for the next LMA iterations).");
		gd.addNumericField("Scale alt image data",                   this.cuas_rng_scale, 5,8,"x",
				"Scale alt_data to use same strength as for normal ranging because targets are smaller than tile.");
		
		
		gd.addMessage("=== Overwrite image_dtt parameters ===");
		gd.addCheckbox    ("Dual-pass LMA (no disparity on first)",  this.cuas_dual_pass,
				"First adjust other parameters (keeping disparities), then add disparities");
		gd.addCheckbox    ("Disparity only on second pass",          this.cuas_disp_only2,
				"Sometimes failed for CUAS on second pass, mitigate by disparity only (keep others from pass1).");
		gd.addCheckbox ("Bypass LMA solution tests, use anyway",     this.cuas_lma_bypass,
				"Use to find out which test fails in the logs. Disable when done.");
		
		gd.addNumericField("Select correlation pairs to calculate",  this.cuas_mcorr_sel,  0, 3, "",
				"Will select pairs to process and use for CM in CUAS mode:  +1 - all, +2 - dia, +4 - sq, +8 - neibs, +16 - hor + 32 - vert");
		gd.addNumericField("Select correlation pairs for LMA",       this.cuas_mcorr_sel_lma,  0, 3, "",
				"Will select from the calculate pairs, so 1 will behave as before 09/2025.  +1 - all, +2 - dia, +4 - sq, +8 - neibs, +16 - hor + 32 - vert");
		
		gd.addNumericField("Maximal relative RMS",                   this.cuas_max_rel_rms,  6, 8, "",
				"Discard tile if ratio of RMS to average of min and max amplitude exceeds this value");
		gd.addNumericField("Minimal composite strength",             this.cuas_min_strength,  6, 8, "",
				"Discard tile if composite strength (average amplitude over SQRT of RMS) is below");
		gd.addNumericField("Offset A,C coefficients",                this.cuas_ac_offset,  6, 8, "",
				"Add to A, C coefficients for near-lines where A,C could become negative because of window ");
		gd.addNumericField("Minimal max (A,C)",                      this.cuas_min_max_ac,  6, 8, "",
				"Minimal value of max (A,C) coefficients to keep the tile (measures sharpest point/line correlation maximum)");
		gd.addNumericField("Minimal min (A,C)",                      this.cuas_min_min_ac,  6, 8, "",
				"Minimal value of min (A,C) coefficients to keep the tile (measures sharpest point correlation maximum)");
		gd.addCheckbox ("Reset target disparity",                    this.cuas_reset_disparity,
				"Reset target disparity, start from infinity.");
		gd.addNumericField("Initial disparity relative to infinity", this.cuas_initial_disparity,  6, 8, "pix",
				" Start correlation with this disparity (in addition to infinity) after reset.");
		gd.addNumericField("Disparity at infinity for ranging",      this.cuas_infinity,  6, 8, "pix",
				"Disparity correction in target ranging mode.");
		
		
		gd.addCheckbox ("Generate/save per-sensor target images",    this.cuas_rng_img,
				"Generate/save per-sensor images using same data as for ranging.");
		gd.addCheckbox("Generate/save target integrated disparities",this.cuas_rng_glob,
				"Generate/save integrated target disparities (one per target), average for all scenes.");
		gd.addNumericField("Ends integration mode",                  this.cuas_glob_ends, 0,3,"",
				"0 - same as internal, 1 - cosine extended ends, 2 - rectangular extended ends.");
		gd.addCheckbox ("Generate/save target disparities",          this.cuas_rng_disp,
				"Measure per-target, per key-frame disparity and save it in target structures.");
		gd.addCheckbox ("Ranging verification images",               this.cuas_rng_vfy,
				"Generate/save ranging verification images (per-sensor and combined rendering from the same data).");
		
		gd.addNumericField("Number of disparity iterations",         this.cuas_rng_niterate, 0,3,"",
				"Number of re-running disparity enhancements.");
		gd.addNumericField("Disparity difference",                   this.cuas_rng_diff, 5,8,"pix",
				"Exit disparity iterations when difference for each target is below.");

		gd.addMessage("=== Parameters for ranging individual targets ===");
		gd.addNumericField("Sequence subdivision levels (>=1)",      this.cuas_flw_levels, 0,3,"",
				"1 - all, 2 - all and two halves, 3 - all, two halves and 4 quaters.");
		gd.addNumericField("Minimal sudivided length",               this.cuas_flw_len, 0,3,"",
				"Minimal number of key frames in each segment after division.");
		gd.addNumericField("Disparity halves disparity difference",  this.cuas_flw_diff, 5,8,"pix",
				"Minimal difference between halves to use \"tilted\" (non-constant) disparity.");
		
		gd.addMessage("=== Targets matching between scene sequences ===");
		gd.addNumericField("Maximal distance from log position",     this.cuas_known_err, 5,8,"pix",
				"Maximal distance between the detected target and UAS log position (in raw image pixels). Also used for annotations.");
		gd.addNumericField("Match flight log fraction",              this.cuas_tmtch_frac, 5,8,"x",
				"Target should match flight log in this fraction of keyframes to be identified as UAS.");
		gd.addNumericField("Maximal undetected first/last",          this.cuas_tmtch_ends, 0,3,"keyframes",
				"Maximal number of undetected first/last keyframes to compare to previous/next sequence.");
		gd.addNumericField("Maxiaml undetected gaps in the same sequence",this.cuas_tmtch_gaps, 0,3,"keyframes",
				"Maxiaml undetected target gaps in the same sequence.");
		gd.addNumericField("Maximal pixel mismatch for the UAS target", this.cuas_tmtch_pix, 5,8,"pix",
				"Maximal pixel mismatch for the UAS target extrapolated with the average of the before/after lateral speeds.");
		gd.addNumericField("Maximal lateral mismatch, same sequence", this.cuas_local_apix, 5,8,"pix",
				"Maximal pixel mismatch for the same non-UAS target in the same sequence (tighter than inter-sequence).");
		gd.addNumericField("Maximal lateral mismatch, other sequence", this.cuas_tmtch_apix, 5,8,"pix",
				"Maximal pixel mismatch for the same non-UAS target extrapolated with the average of the before/after lateral speeds.");
		gd.addNumericField("Maximal lateral mismatch, per second",   this.cuas_tmtch_rpix, 5,8,"pix/s",
				"Maximal pixel mismatch for the same target extrapolated with the average of the before/after lateral speeds.");
		gd.addNumericField("Maximal axial velocity",                 this.cuas_tmtch_axv, 5,8,"m/s",
				"Maximal axial (range) velocity in m/s.");
		gd.addNumericField("Maximal axial to lateral ratio",         this.cuas_tmtch_axv_k, 5,8,"pix",
				"Maximal axial (range) velocity to lateral velocity ratio (take max axial velocity).");
		gd.addNumericField("Maximal disparity difference (same sequence)", this.cuas_local_diff, 5,8,"pix",
				"Maximal disparity difference to match 3d for local target match (loosened for local).");
		gd.addNumericField("Maximal disparity difference (other sequence)", this.cuas_tmtch_diff, 5,8,"pix",
				"Maximal disparity difference to match 3d (regardless of absolute disparity.");
		gd.addNumericField("Minimal disparity for 3D matching",      this.cuas_tmtch_disp, 5,8,"pix",
				"Minimal disparity (any should be above) to estimate lateral velocity (smaller - match2d only).");
		gd.addCheckbox ("Prefer shortest gap",                       this.cuas_tmtch_short,
				"When multiple conflicting matches, prefer shortest gap (false - longest combo pair length).");
		
		gd.addMessage("=== Radar image generation (check Radar mode below) ===");
		gd.addNumericField("Maximal Radar distance",                 this.cuas_radar_range, 5,8,"m",
				"Maximal Radar range in meters.");
		gd.addNumericField("Radar finite range targets view height", this.cuas_radar_height, 0,3,"pix",
				"Height of the radar finite range view.");
		gd.addNumericField("Radar view bottom margin",               this.cuas_radar_bottom, 0,3,"pix",
				"Radar window bottom margin).");
		gd.addNumericField("Debug ranging scene (-1 - process all)", this.cuas_radar_top, 0,3,"pix",
				"Far (infinity) targets offset from window top.");
		gd.addStringField ("True UAS location color", getStringColor(cuas_radar_uas_color, DEFAULT_cuas_radar_uas_color), 8,
				"6-character hex value. Any wrong hex value will be replaced by the default - "+getStringColor(DEFAULT_cuas_radar_uas_color));
		gd.addStringField ("Targets location color",  getStringColor(cuas_radar_color,DEFAULT_cuas_radar_color), 8,
				"6-character hex value. Any wrong hex value will be replaced by the default - "+getStringColor(DEFAULT_cuas_radar_color));
		gd.addNumericField("Detected targets circle radius",         this.cuas_radar_radius, 5,8,"pix",
				"Radius of the detected targets location circle. Should be less than that of the true UAS circle");
		gd.addNumericField("True UAS location circle radius",        this.cuas_radar_uas_rad, 5,8,"pix",
				"Radius of the true UAS location circle. Should be greater than that of the detected targets rendered on top.");
		gd.addMessage("=== Radar grid generation ===");
		gd.addNumericField("Range circles interval",                 this.cuas_grid_ring, 5,8,"m",
				"Range rings step in meters."); 
		gd.addNumericField("Aziuth grid lines interval",             this.cuas_grid_line, 5,8,"degree",
				"Azimuth lines step in degrees.");
		gd.addCheckbox ("Annotate grid rings/lines",                 this.cuas_grid_annot,
				"Annotate grid circles/lines wityh numerical values.");
		gd.addStringField ("Grid lines/text color", getStringColor(cuas_grid_color,DEFAULT_cuas_grid_color), 8,
				"6-character hex value. Any wrong hex value will be replaced by the default - "+getStringColor(DEFAULT_cuas_grid_color));
		gd.addNumericField("Grid font size",                         this.cuas_grid_font, 0,3,"pix",
				"Grid font size (before scaling by 2 for compatibility).");
		gd.addNumericField("Azimuth annotations offset", this.cuas_grid_az_top, 0,3,"pix",
				"Absolute Y-position of the bottom of the azimuth annotations.");
		gd.addNumericField("Grid annotations margins from righht/left", this.cuas_grid_sides, 0,3,"pix",
				"Grid annotations margins from left/right of the Radar pane.");
		gd.addMessage("=== Annotation Selection (what to display in the video ===");
		gd.addMessage(CuasMotion.ANNOT_TIP);
		for (int i = 0; i < CuasMotion.ANNOT_LABELS.length; i++) {
			gd.addStringField(
					CuasMotion.ANNOT_LABELS[i][0],  // String label,
					CuasMotion.getCuasAnnot (       //String value,
							i, // int    indx,
							cuas_annot_sel),        // int [] annots),
					8,                              // int    width,
					CuasMotion.ANNOT_LABELS[i][1]); // String tooltip)
		}
		
		gd.addMessage("=== Debug ===");
		gd.addCheckbox ("Save/show debug images",                    this.cuas_debug,
				"Save CUAS-related debug images and show them in non-batch mode.");
		gd.addNumericField("Debug ranging scene (-1 - process all)", this.cuas_dbg_rng_seq, 0,3,"",
				"Debug single-sequence ranging (-1 - process all).");
		gd.addNumericField("Debug ranging target (-1 - process all)", this.cuas_dbg_rng_tgt, 0,3,"",
				"Debug single-target ranging (-1 - process all), > 0 - target ID.");
		gd.addNumericField("Debug show mode",                        this.cuas_dbg_show_mode, 0,3,"",
				"Debug show ranging debug images: +1 - disparity, +2 - correlations +4 - debug single run.");
		
		gd.addCheckbox ("Save/show debug images for each tuning step",this.cuas_step_debug,
				"Save CUAS-related debug images during per-step cuas recalculation and show them in non-batch mode.");
		gd.addCheckbox ("Save target statistics",                    this.cuas_save_stats,
				" Save target statistics data to find out which test failed and when (or when tile was found good).");
		gd.addCheckbox ("Add debug to the target text",              this.cuas_target_debug,
				"Add additional debug text to the .");
		gd.addCheckbox ("Show target score",                         this.cuas_target_score,
				"Show target score and sequence length in the final video.");
		gd.addCheckbox ("Use known target color",                    this.cuas_target_color,
				"Annotate known target with different color.");
		gd.addCheckbox ("Log ranging iterations",                    this.cuas_log_ranging,
				"Log ranging parameters and results.");
		
		gd.addCheckbox ("Overwrite number of performed adjustments", this.cuas_overwrite,
				"Debug feature: overwrite actual number of performed orientation and disparity refinements to re-run them.");
		gd.addNumericField("Value to overwrote num_orient",          this.cuas_num_orient, 0,3,"",
				"Overwrite value (when Overwrite... is checked).");
		gd.addNumericField("Value to overwrote num_accum",           this.cuas_num_accum, 0,3,"",
				"Overwrite value (when Overwrite... is checked).");
		
		gd.addCheckbox ("Reuse previosly calculated targets",        this.cuas_reuse_targets,
				"Read previously calculated non-conflict (one per tile) targets.");
		gd.addStringField("Calculated targets suffix/path",          this.cuas_reuse_path, 120,
				"Either suffix (if all the parameters the same) or the full path (contains \"/\").");
		gd.addCheckbox ("Reuse previosly calculated disparities",    this.cuas_reuse_disparity,
				"Read previously calculated non-conflict (one per tile) targets with disparities (recalculate ranges).");
		gd.addCheckbox ("Reuse combined series (Radar mode)",        this.cuas_reuse_globals,
				"Read target data updated after global processing of scene series \"CUAS Combine\" command. If successful, go directly to output generation with 16:9 and a radar image.");
		gd.addNumericField("Maximal disparity difference",           this.cuas_max_disp_diff, 6,8,"pix",
				"Maximal disparity difference during last change to consider disparity valid.");
		gd.addNumericField("Minimal target disparity strength",      this.cuas_min_disp_str, 6,8,"",
				"Minimal disparity strength to consider disparity valid.");
		gd.addNumericField("Maximal displayed distance",             this.cuas_rng_limit, 6,8,"m",
				"Maximal displayed distance to target.");

		gd.addTab("LMA sequence","Interscene LMA sequence control");
		gd.addMessage("Parameters for control of the LMA pose adjustment sequence");
		gd.addNumericField("Pose readjust number for full mb_gain",  this.mb_gain_index_pose, 0,3,"",
				"Pose readjust pass to switch to full mb_max_gain from mb_max_gain_inter.");
		gd.addNumericField("Depth map refine number for full mb_gain",this.mb_gain_index_depth, 0,3,"",
				"Depth map refine pass (SfM) to switch to full mb_max_gain from mb_max_gain_inter.");
		gd.addNumericField("Pose readjust number to enable ERS LMA", this.mb_ers_index, 0,3,"",
				"Pose readjust pass to enable ERS.");
		gd.addNumericField("Pose readjust number to enable tilt ERS LMA",this.mb_ers_y_index, 0,3,"",
				"Pose readjust pass to enable ERS in vertical (Y) direction - it competes with SfM.");
		gd.addNumericField("Pose readjust number to enable roll ERS LMA",this.mb_ers_r_index, 0,3,"",
				"Pose readjust pass to enable ERS roll adjustment.");
		gd.addNumericField("Pose readjust number to enable all XYZATR",this.mb_all_index, 0,3,"",
				"Enables 6 DOF adjustment for one pass, ERS parameters adjusted separately above.");
		gd.addCheckbox ("Rely on IMS orientation before position",      this.mb_pref_orient,
				"First use IMS orientation, later use IMS position. If false - reverse order.");

		gd.addCheckbox ("LMA adjust Z",                                this.lma_use_Z,
				"Always adjust Z (altitude) when running interscene LMA. False - same as X and Y.");
		gd.addCheckbox ("LMA adjust roll",                             this.lma_use_R,
				"Always adjust roll when running interscene LMA. False - same as Azimuth and Tilt.");
		
		
		gd.addMessage  ("Other Interscene LMA parameters to be modified/replaced/removed");
		gd.addCheckbox ("Use configuration LMA parameters",            this.configured_lma,
				"Use configuration LMA parameters, disregard onther parameters that influence LMA parameters.");
		gd.addCheckbox ("Freeze XY pull",                              this.freeze_xy_pull,
				"Freeze XY target values, approach with smaller steps if needed .");
		gd.addCheckbox ("Copy pull values to current",                 this.copy_pull_current,
				"Replace current (start) values with the \"pull\" ones.");
		gd.addCheckbox ("Restore ERS to IMU velocities",               this.restore_imu,
				"restore imu omega-az and omega-tl, freeze ERS, adjust X,Y,Z,Az,Tl,Rl.");
		gd.addCheckbox ("LPF X and Y",                                 this.lpf_xy,
				"LPF x and y, re-adjust X,Y,Z,A,T,R with pull for X,Y. Disables - what?.");
		gd.addCheckbox ("Readjust XY from IMS",                        this.readjust_xy_ims,
				"Readjust X,Y from IMS linear/angular and full X,Y movement.");
		gd.addNumericField("LMA regularization weight",                this.reg_weight_xy, 5,7,"",
				"Regularization weight for LMA");
		
		gd.addTab("Quaternion LMA","Parameters fitting rotation to two sets of vectors");
		gd.addNumericField("LMA lambda",                                  this.quat_lambda, 6,8,"",
				"Initial value of the LMA lambda");
		gd.addNumericField("Scale lambda after successful LMA iteration", this.quat_lambda_scale_good, 3,5,"",
				"Scale lambda (reduce) if the new RMSE is lower than the previous one.");
		gd.addNumericField("Scale lambda after failed LMA iteration",     this.quat_lambda_scale_bad, 3,5,"",
				"Scale lambda (increase) if the new RMSE is higher than the previous one.");
		gd.addNumericField("Maximal value of lambda to try",              this.quat_lambda_max, 2,7,"",
				"Fail LMA if the result is still worse than before parameters were updates.");
		gd.addNumericField("Minimal relative RMSE improvement",           this.quat_rms_diff, 5,7,"",
				"Exit LMA iterations if relative RMSE improvement drops below this value.");
		gd.addNumericField("Maximal number of LMA iterations",            this.quat_num_iter, 0,3,"",
				"A hard limit on LMA iterations.");
		gd.addNumericField("Regularization weight",                       this.quat_reg_w, 3,5,"",
				"Regularization weight [0..1) weight of q0^2+q1^2+q1^2+q3^2 -1.");
		
		gd.addNumericField("Maximal derivatives by axes ratio",           this.quat_max_ratio, 3,5,"x",
				"If difference is larger, add regularization to reduce it.");
		gd.addNumericField("Maximal correction angles change",            this.quat_max_change, 3,5,"rad",
				"Do not update corrections if any angle is above this limit. ");
		gd.addNumericField("Minimal translation for mount calibration",   this.quat_min_transl, 3,5,"m",
				"Do not use translation for IMS mount adjustment if it is too small.");
		gd.addNumericField("Minimal rotation for mount calibration",      this.quat_min_rot, 3,5,"rad",
				"Do not use rotations for IMS mount adjustment if it is too small.");
		gd.addNumericField("Minimal per-axis travel",                     this.quat_min_lin, 3,5,"m",
				"Minimal per-axis travel to adjust accelerometer sensitivity.");
	
		gd.addTab("Stereo/Video","Parameters for stereo video generation");
		gd.addMessage  ("Stereo");
		if (stereo_views.length > 0) {
			String [] stereo_choices = new String [stereo_views.length + 1];
			stereo_choices[0] = "--none--";
			for (int i = 0; i < stereo_views.length; i++) {
				stereo_choices[i+1] = doublesToString(stereo_views[i],"%.0f")+" mm";	
			}
			gd. addChoice("Remove stereo-view (base, up, back)",      stereo_choices, stereo_choices[0], 
					"Remove selected stereo-view, consisting of streo-base, viewpoint above camera, viewpoint behing camera - all in mm");
		}
		gd.addStringField("Add another stereo view (baseline, above, behind)",                 "", 40,
				"Add another stereo view by providing baseline, above camera, behind camera (mm).");
		gd.addCheckbox ("Stereo merge right, left",                  this.stereo_merge,
				"Combine stereo pair in a single (wide) frame. Unchecked - generate separate videos.");
		gd.addNumericField("Stereo gap",                             this.stereo_gap, 0,3,"pix",
				"Distance (pixels) between right and left sub-images in a stereo frame.");
		gd.addNumericField("VR headset stereo base",                 this.stereo_intereye, 5,7,"mm",
				"Inter-lens distance of the VR phone adapter (Google Cardboard - 63.5)");
		gd.addNumericField("Phone screen width",                     this.stereo_phone_width, 5,7,"mm",
				"Phone screen width in full screen video mode.");
		gd.addNumericField("Scene sequence horizontal extra",        this.extra_hor_tile, 0,3,"tiles",
				"Enlarge reference scene window horizontally in each direction to accommodate other scenes in a sequence");
		gd.addNumericField("Scene sequence vertical extra",          this.extra_vert_tile, 0,3,"tiles",
				"Enlarge reference scene window vertically in each direction to accommodate other scenes in a sequence");
		gd.addCheckbox ("Crop 3D",                                   this.crop_3d,
				"Do not enlarge reference scene windows fro 3D views (FG, BG)");
		
		gd.addNumericField("Sensor mask (bitmask, -1 - all sensors)",this.sensor_mask, 0,3,"",
				"Select which sensors to be included in each scene of the sequence");
		gd.addCheckbox ("Merge all channels in 3D modes",                                   this.merge_all,
				"Ignore sensor mask, use all channels and merge them into one in 3D modes (FG and BG)");
		
		gd.addMessage  ("Generate video for scene sequences");
		gd.addNumericField("Video output frame rate",                this.video_fps, 5,7,"fps",
				"Video frames per second (original is 60fps).");
		gd.addNumericField("Sensor frame rate (60 for Boson)",       this.sensor_fps, 5,7,"fps",
				"Sensor (real) frame rate to be used in realtime videos after slowmo.");
		gd. addChoice("AVI mode",                                     MODES_AVI, MODES_AVI[this.mode_avi], 
				"AVI generation mode");
		gd.addNumericField("AVI JPEG Quality",                       this.avi_JPEG_quality, 0,3,"",
				"Only used in JPEG compression mode");		
		
		gd.addCheckbox ("Convert AVI to WEBM",                       this.run_ffmpeg,
				"Run ffmpeg on AVI video to convert to smaller modern video.");
		gd.addStringField ("Result video extension (e.g. \".webm\")",this.video_ext, 60,
				"Converted video extension, starting with dot.");
		gd.addStringField ("Video encoder",                          this.video_codec, 60,
				"FFMPEG video encoder, such as \"VP8\" or \"VP9\".");
		gd.addNumericField("Video CRF",                              this.video_crf, 0,3,"",
				"Quality - the lower the better. 40 - OK");		

		gd.addNumericField("Video bitrate",                          this.video_bitrate_m, 5,7,"Mbps",
				"Specify video bitrate in megabits per second, latter M will be added (0 does not work anymore).");

		gd.addCheckbox ("Remove AVI",                                this.remove_avi,
				"Remove large AVI files after (and only) conversion with ffmpeg.");
		
		gd.addStringField ("Video encoder for combining",            this.video_codec_combo, 60,
				"FFMPEG video encoder, such as \"VP8\" or \"VP9\". Applies when merging segments.");
		gd.addNumericField("Video CRF for combining",                this.video_crf_combo, 0,3,"",
				"Quality - the lower the better. 40 - OK.  Applies when merging segments.");		
		gd.addCheckbox ("Insert average slice",                      this.add_average,
				"Insert average slice before scene slices");
		gd.addCheckbox ("Calculate average from slices",             this.calculate_average,
				"Calculate average slice from other slices. If false - try to use average CLT if available.");
		gd.addCheckbox ("Subtract average",                          this.subtract_average,
				"Subtract average slice from all scenes");
		gd.addNumericField("Running average length",                 this.running_average, 0,3,"",
				"Apply running average to the scene sequence, set length.");		
		gd.addCheckbox ("Calculate center orientation",              this.extract_center_orientation,
				"Calculate center of orientation rotation (in lock_position=true mode only). Debug feature.");
		gd.addCheckbox ("Apply unsharp mask to mono",                this.um_mono,
				"Apply unsharp mask to monochrome image sequences/video.  Applies to TIFF generatiojn too");
		gd.addNumericField("Unsharp mask sigma (radius)",            this.um_sigma, 5,7,"pix",
				"Unsharp mask Gaussian sigma.");
		gd.addNumericField("Unsharp mask weight",                    this.um_weight, 5,7,"",
				"Unsharp mask weightt (multiply blurred version before subtraction from the original).");
		gd.addCheckbox ("Save linear",                               this.um_mono_linear,
				"Save TIFF sequences without UM in addition to UM");
		
		gd.addCheckbox ("Fixed monochrome range",                    this.mono_fixed,
				"Normalize monochrome (after UM) to a fixed range when converting to 8 bit RGB.");
		gd.addNumericField("Monochrome full range",                  this.mono_range, 5,7,"",
				"Monochrome full range to convert to 0..255.");
		

		gd.addCheckbox ("Generate anaglyph stereo",                  this.anaglyth_en,
				"Apply unsharp mask to monochrome image sequences/video.  Applies to TIFF generatiojn too");

		{String scolor = String.format("%08x", getLongColor(this.anaglyph_left));
		gd.addStringField ("Anaglyph color left",scolor, 8, "Any invalid hex number sets default red");}
		
		{String scolor = String.format("%08x", getLongColor(this.anaglyph_right));
		gd.addStringField ("Anaglyph color right",scolor, 8, "Any invalid hex number sets default cyan");}
		gd.addCheckbox ("Timestamp color videos",                    this.annotate_color,
				"Annotate pseudo-color video frames with timestamps.");
		gd.addCheckbox ("Timestamp monochrome videos",               this.annotate_mono,
				"Annotate monochrome video frames with timestamps.");
		String scolor = (this.annotate_color_color==null)?"none":String.format("%08x", getLongColor(this.annotate_color_color));
		gd.addStringField ("Timestamp color for pseudocolor frames",scolor, 8, "Any invalid hex number disables annotation");
		scolor = (this.annotate_color_mono==null)?"none":String.format("%08x", getLongColor(this.annotate_color_mono));
		
		gd.addStringField ("Timestamp color for monochrome frames",scolor, 8, "Any invalid hex number disables annotation");
		gd.addCheckbox ("Transparent timestamp background (monochrome)",this.annotate_transparent_mono,
				"Put monochrome timestamp over image (unchecked - use black background). Color - always black.");
		
// vegetation	
		
		
		gd.addTab("Vegetation","See through foliage parameters");
		gd.addStringField ("Model path",         terr_model_path, 120, "Model directory path with version.");
		gd.addStringField ("Model state file",   terr_model_state, 50, "Model vegetation source data (w/o extension).");
		gd.addStringField ("Segments subdir",    terr_segments_dir, 50,"Model vegetation source data (w/o extension).");
		gd.addStringField ("Segments suffix",    terr_segments_suffix, 50,"suffix to select from multiple versions (w/o \".par-tiff\".");

		gd.addStringField ("Restore directory",  terr_par_dir, 120,     "Directory to resore initial parameters from.");
		gd.addStringField ("Restore file",       terr_par_file, 120,    "Filename to resore initial parameters from.");
		gd.addCheckbox    ("Restore parameters", terr_par_restore,     "Restore initial parameters from file before running LMA.");
		gd.addNumericField("Restore step",       terr_step_restore, 0,3,"","Go to LMA sequence step after restore.");
		
		gd.addMessage     ("Unsharp mask filter of the input data (currently not used as both DC and HF fitting are implemented simultaneously).");
		gd.addCheckbox    ("Enable UM",          terr_um_en,           "Enable unsharp mask filter.");
		gd.addNumericField("UM sigma",           terr_um_sigma, 5,7,   "pix", "Unsharp mask sigma.");
		gd.addNumericField("UM weight",          terr_um_weight, 5,7,  "",    "Unsharp mask weight.");

		gd.addMessage     ("Preparaion of the initial LMA approximation");
		gd.addNumericField("NaN tolerance",          terr_nan_tolerance, 5,7,"", "Replace continuous almost zeros with NaN-s (ignored and forced to 0 if no UM filter).");
		gd.addNumericField("NaN grow",               terr_nan_grow, 0,3,"pix",   "Grow zero/\"almost\" zero rendered scene areas (+2 - 8 directions, +1 - only ortho), replace with NaNs.");
		gd.addNumericField("Shrink vegetation",      terr_shrink_veget, 0,3,"pix",   "Shrink detected vegetation areas for terrain separation.");
		gd.addNumericField("Shrink terrain",         terr_shrink_terrain, 0,3,"pix",   "Shrink remaining detected terrain areas before filling with laplacian (terr_shrink_veget is added to start from the original size).");
		gd.addNumericField("Vegetation hotter",      terr_vegetation_over, 5,7,"", "Initially consider to be vegetation if it is this hotter than filled in terrain holes.");
		gd.addNumericField("Vegetation shrink+grow", terr_filter_veget, 0,3,"pix", "Shrink + grow hot enough vegetation to filter from small clusters - needed for initial alpha.");
		
		gd.addNumericField("Grow vegetation extract", terr_veget_grow, 0,3,"pix", "Grow vegetation when extracting from the multi-scene sequences.");
		
		gd.addMessage     ("Scan WOI parameters");
		gd.addCheckbox    ("Tiled WOI",          terr_tile_woi, "Process tiled WOI (False - a single one).");
		gd.addCheckbox    ("Continue WOI",       terr_continue, "Continue WOI scan after \"Single/last WOI\"");
		gd.addStringField ("Scan enclosing WOI", rectangleToString(terr_woi_enclos), 40, "Scan WOI position and size.");
		gd.addStringField ("Step WOI",           rectangleToString(terr_woi_step), 40, "Scan step (x,y) and size (width, hight).");
		gd.addStringField ("Single/last WOI",    rectangleToString(terr_woi_last), 40, "A single/last scanned WOI.");
        gd.addCheckbox    ("Skip existing WOI",  terr_skip_exist, "Skip already esisting WOIs during scanning.");
        
		gd.addMessage     ("LMA losses and goals for alpha, terrain and vegetation pixels");

		gd.addCheckbox    ("Adjust terrain",       terr_fit_terr,      "Adjust terrain pixels.");
		gd.addCheckbox    ("Adjust vegetation",    terr_fit_veget,     "Adjust vegetation pixels.");
		gd.addCheckbox    ("Adjust alpha",         terr_fit_alpha,     "Adjust vegetation alpha pixels.");
		gd.addCheckbox    ("Adjust elevation",     terr_fit_elevations,"Adjust elevation pixels.");
		gd.addCheckbox    ("Adjust ter elevation common",terr_fit_terr_elev,"Adjust terrain elevation common.");
		gd.addCheckbox    ("Adjust ter elevation pixel",terr_fit_terr_elev_pix,"Adjust terrain elevation per-pixel.");
		gd.addCheckbox    ("Adjust scene offsets", terr_fit_scenes,    "Adjust scene offsets (start from 0 always?).");
		
		
		for (int i = 0; i < terr_fits_disable.length; i++) {
			gd.addStringField ("Skip "+VegetationLMA.TVAO_NAMES[i], booleansToString(terr_fits_disable[i],2), 40,
					"Skip adjustment of "+VegetationLMA.TVAO_NAMES[i]+". Use comma/space separated list of true/false, 1/0 or +/-." );
		}

		for (int i = 0; i < terr_only_fits_disable.length; i++) {
			gd.addStringField ("Skip "+VegetationLMA.TVAO_NAMES[i]+" in terrain-only", booleansToString(terr_only_fits_disable[i],2), 40,
					"Skip adjustment of "+VegetationLMA.TVAO_NAMES[i]+" in terrain-only mode. Use comma/space separated list of true/false, 1/0 or +/-." );
		}
		
		gd.addNumericField("Maximal warp",         terr_max_warp, 5,7,"pix", 	"(1.8) Do not use scenes where distance between vegetation projection exceeds this.");
		gd.addNumericField("Max elevation/offset", terr_max_elevation, 0,3,"pix","Maximal offset to consider when looking for vegetation influence.");
		gd.addNumericField("Max terrain elevation",terr_max_elev_terr, 0,3,"pix","Maximal terrain offset to consider when looking for variable-elevation terrain.");
		gd.addNumericField("Max elevation cjange", terr_max_elev_chng, 5,7,"pix","Maximal terrain elevation change from the previous success.");
		
		gd.addNumericField("(Min influenced scenes)",terr_min_scenes, 0,3,  "",   "Deprecated: Minimal number of scenes (inside woi) vegetation pixel must influence.");
		gd.addNumericField("Minimal samples/scene",terr_min_samples_scene, 0,3,"","Minimal samples per scene used for fitting (skip scene if less).");
		gd.addNumericField("Minimum scenes used",  terr_min_total_scenes, 0,3,"", "Minimal total number of scenes used (skip segment if less).");
		
		gd.addNumericField("Minimal pixels fitted",terr_min_pixels, 0,3,"",   "Minimal number of terrain and vegetation pixels used for fitting in this segment(skip segment if less).");
		gd.addCheckbox    ("(Start by temperature)", terr_warm_veget,     "Start with vegetation warmer than terrain.");
		gd.addNumericField("(Warmest terrain)",      terr_warmest, 5,7,"", 	"Above - vegetation. below - terrain.");
		gd.addNumericField("(Initial split)",        terr_initial_split, 5,7,"", 	"Initial alpha: terrain 0.0+, vegetation 1.0-.");
		gd.addNumericField("(Min. split fraction)",  terr_min_split_frac, 5,7,"", "minimal modality fraction to use split by temperature (otherwise use default alpha).");
		gd.addNumericField("Vegetation warmer",    terr_difference, 5,7,"", 	"Pull vegetation to be this warmer.");
		
		gd.addNumericField("Pull terrain cold",    terr_pull_cold, 5,7,"", 	"Pull vegetations to warm, terrain to cold.");
		gd.addNumericField("Elevation radius",     terr_elevation_radius, 5,7,"pix","Radius of elevation/vegetation influence.");
		gd.addNumericField("Terrain elevation radius",terr_terr_elev_radius, 5,7,"pix","Radius of terrain elevation influence.");
		gd.addNumericField("Radius extra",         terr_elev_radius_extra, 5,7,"pix","Increase both radii when recalculating for variable elevation.");
		
		gd.addNumericField("Alpha initial contrast",terr_alpha_contrast, 5,7,"","Initial alpha contrast (>= 1.0).");
		gd.addNumericField("Defalt alpha",         terr_alpha_dflt, 5,7,"", 	"Default vegetation alpha.");
		
		gd.addNumericField("Initial alpha sigma",  terr_alpha_sigma, 5,7,"", 	"Initial alpha: Gaussian blur sigma to find local average for vegetation temperature.");
		gd.addNumericField("Initial alpha min frac",terr_alpha_init_min,5,7,"", "Initial alpha: fraction for transparent.");
		gd.addNumericField("Initial alpha max frac",terr_alpha_init_max,5,7,"", "Initial alpha: fraction for opaque.");
		gd.addNumericField("Initial alpha offset", terr_alpha_init_offs, 5,7,"","Initial alpha: opaque/transparent offset from 1.0/0.0.");
		
		gd.addNumericField("Alpha loss",           terr_alpha_loss, 5,7,"", 	"Alpha quadratic growing loss for when out of [0,1] range");
		gd.addNumericField("Alpha loss linear",    terr_alpha_loss_lin, 5,7,"", "Alpha linear growing loss for when out of [0,1] range and below minimal vegetation alpha.");
		
		gd.addNumericField("Alpha offset @1.0",    terr_alpha_offset, 5,7,"",   "Start alpha losses below 1.0 by this value.");
		gd.addNumericField("Alpha offset @0.0",    terr_alpha_0offset, 5,7,"",  "Start alpha losses above 0.0 by this value.");
		gd.addNumericField("Minimal vegetation alpha", terr_alpha_min_veg, 5,7,"","Minimal vegetation alpha. If (alpha-alpha_offset)/(1-2*alpha_offset) < alpha_min_veg, pull down to lpha_offset.");
		gd.addNumericField("Maximal terrain alpha",terr_alpha_max_terrain, 5,7,"","Increase pull vegetation if below.");
		gd.addNumericField("Extra pull power",     terr_alpha_pull_pwr, 5,7,"","Raise extra pull (below maximal terrain alpha) to that power.");
		
		gd.addNumericField("Alpha diffusion",      terr_alpha_lpf, 5,7,"", 	    "Alpha diffusion to 4 ortho neighbors.");
		gd.addNumericField("Alpha diffusion border",terr_alpha_lpf_border, 5,7,"","Alpha diffusion to ortho neighbors on the border (for stability with low alpha diffusion).");
		gd.addCheckbox    ("Alpha piece-linear",   terr_alpha_piece_linear,     "Piece-linear alpha (_/\u203E, false - 0.0-cosine-1.0.");
		gd.addNumericField("Scale alpha",          terr_alpha_scale_avg, 5,7,"","Scale target (average of neighbors) alpha before pulling to it (not used now).");
		gd.addNumericField("Push alpha",           terr_alpha_push, 5,7,"", 	"Quadratic loss for middle alpha (push to 0.0 or 1.0.");
		gd.addNumericField("Neutral alpha",        terr_alpha_push_neutral, 5,7,"", "Alpha point from which to push (default 0.5).");
		gd.addNumericField("Alpha center weight",  terr_alpha_weight_center, 5,7,"","Weight of center alpha pixel relative to each of the 4 ortho ones.");
		gd.addCheckbox    ("Hole search enable",   terr_en_holes,               "Search for small semi-transparent holes, disable diffusion of local alpha minimums.");
		gd.addNumericField("Alpha MM fraction",    terr_alpha_mm_hole, 5,7,"",	"Disable diffusion for local \"almost minimum\" (lower than this fraction between min and max neighbors).");
		gd.addNumericField("Alpha hole difference",terr_alpha_diff_hole, 5,7,"", "Minimal alpha difference between min and max neighbor to be considered a hole.");
		gd.addNumericField("Terrain diffusion",    terr_terr_lpf, 5,7,"",       "LPF for terrain pixels (diffusion to 4 neighbors).");
		gd.addNumericField("Vegetation diffusion", terr_veget_lpf, 5,7,"", 		"LPF for vegetation pixels (diffusion to 4 neighbors).");
		gd.addNumericField("Elevation diffusion",  terr_elev_lpf, 5,7,"", 		"LPF for elevation pixels (diffusion to 4 neighbors).");
		gd.addNumericField("Terrain elevation diffusion",  terr_terr_elev_lpf, 5,7,"","LPF for terrain elevation elevation pixels (diffusion to 4 neighbors).");
		gd.addNumericField("Terrain pull",         terr_terr_pull0, 5,7,"",		"Terrain pixels pull to initial (pre-adjustment) values.");
		gd.addNumericField("Terrain pull up",      terr_terr_pull_up, 5,7,"",	"Terrain pixels pull to initial (pre-adjustment) values when it is colder than initial (<0 - sqme as above).");
		gd.addNumericField("Terrain pull to average",terr_terr_pull_avg, 5,7,"","Pull terrain to the initial offset by the average offset of all terrain pixels.");
		gd.addNumericField("Vegetation pull",      terr_veget_pull0, 5,7,"",	"Vegetation pixels pull initial (pre-adjustment) values.");
		gd.addNumericField("Boost for low alpha",  terr_veget_pull_low_alpha, 5,7,"x",	"Scale vegetation pull (terr_veget_pull0) for low alpha (mostly terrain) .");

		gd.addNumericField("Elevation pull",       terr_elev_pull0, 5,7,"",  	"Elevation pixels pull to initial (pre-adjustment) values.");
		gd.addNumericField("Terrain elevation pull",terr_terr_elev_pull0, 5,7,"","Terrain elevation pixels pull to initial (pre-adjustment) values.");
		gd.addCheckbox    ("Enable low vegetation",terr_elev_alpha_en,          "Enable loss for low vegetation with high opacity.");
		gd.addNumericField("Low vegetation loss",  terr_elev_alpha, 5,7,"",  	"Multiply alpha by under-low elevation for loss.");
		gd.addNumericField("Low vegetation power", terr_elev_alpha_pwr, 5,7,"", "Raise alpha to this power for low vegetation loss.");
		gd.addNumericField("Low elevation",        terr_low_veget, 5,7,"pix",  	"Elevation considered low (lower loss for high alpha.");
		gd.addNumericField("Pull scene offset",    terr_scenes_pull0, 5,7,"",	"Pull average scene offset to zero.");
		
	    // scaling elevation losses for high elevations (decrease pull and/or lpf for high elevations)
		gd.addNumericField("High elevation",       terr_elev_scale_thresh, 5,7,"","Reduce losses for higher (initial) elevations threshold.");
		gd.addCheckbox    ("High elevation pull",  terr_elev_scale_pull,        "Scale elevation pull losses for high elevations.");
		gd.addCheckbox    ("High elevation diffusion",terr_elev_scale_lpf,      "Scale elevation diffusion losses for high elevations.");

		gd.addMessage     ("Tree tops detection/filtering - no (stray) transparency around tree tops");
		gd.addCheckbox    ("Filter tree tops",     terr_ttop_en,               "Filter tree tops transparency.");
		gd.addNumericField("Elevation blur sigma", terr_ttop_gb, 5,7,"",  	   "Elevation Gaussian blur sigma to detect tree tops.");
		gd.addNumericField("Minimal tree top elevation",terr_ttop_min, 5,7,"", "Minimal tree top elevation.");
		gd.addNumericField("Relative sample level",terr_ttop_rel_lev, 5,7,"",  "Relative (to the top height) sample level.");
		gd.addNumericField("Relative ring radius", terr_ttop_rel_rad, 5,7,"",  "Relative (to the top height) sample ring radius.");
		gd.addNumericField("Minimal fraction lower",terr_ttop_frac, 5,7,"",    "Minimal fraction of the ring pixels below sample level.");
		gd.addNumericField("Remove transparency radius", terr_ttop_rem_rad, 5,7,"","Relative (to the top height) remove transparency radius.");

	
		gd.addMessage     ("LMA parameters");
		gd.addNumericField("Boost parallax",       terr_boost_parallax, 5,7,"",	"Increase weight of scenes that have high parallax to the reference one.");
		gd.addNumericField("Limit parallax",       terr_max_parallax, 5,7,"",	"Parallax limit when evaluating boost parallax.");
		gd.addNumericField("High freq. weight",    terr_hifreq_weight, 5,7,"", 	"Relative weight of laplacian components differfences to the DC ones (0 - do not use).");
		gd.addNumericField("Terrain correction",   terr_terr_corr, 5,7,"",   	"Relative weight of terrain correction (mismatch between average image and average model).");
		
		gd.addNumericField("Losses weight",        terr_reg_weights, 5,7,"", 	"Fraction of other losses compared to the RMSE.");
		gd.addNumericField("Initial lambda",       terr_lambda, 5,7,"", 		"Initial LMA lambda.");
		gd.addNumericField("Lambda scale on good", terr_lambda_scale_good, 5,7,"","Scale lambda if RMSE improved.");
		gd.addNumericField("Lambda scale on bad",  terr_lambda_scale_bad, 5,7,"","Scale lambda if RMSE worsened.");
		gd.addNumericField("Lambda max to fail",   terr_lambda_max, 5,7,"",     "Fail if lambda gets larger than that.");
		gd.addNumericField("RMSE difference",      terr_rms_diff, 10,12,"",     "Exit if RMSE improvement is lower.");
		
		gd.addNumericField("Last LMA series",       terr_last_series, 0,3,"","Last LMA series, -1 - to last available in the sequence below.");
		gd.addStringField ("Maximal iterations", intsToString(terr_num_iters), 40, "Maximal number of LMA iterations per series, 1 or several values.");

		gd.addMessage     ("Terrain-only LMA");
///		gd.addCheckbox    ("Terrain-only mode",     terr_only_special,           "Treat terrain-only tiles differently (below).");
///		gd.addCheckbox    ("Adjust pixel elevation",terr_only_pix,               "Force per-pixel terrain elevation adjustment in terrain-only mode.");
		gd.addNumericField("Terrain-only iterations",terr_only_series, 0,3,"",   "Last LMA series in terrain-only mode, -1 - to last available in the sequence below.");
		gd.addStringField ("Maximal iterations", intsToString(terr_only_num_iters), 40, "Maximal number of LMA iterations per series, 1 or several values.");
		gd.addChoice      ("LMA matrix solution mode:", VegetationLMA.CHOLESKY_NAMES, VegetationLMA.CHOLESKY_NAMES[terr_cholesky]);
		gd.addMessage     ("Second LMA run");
		gd.addStringField ("Recalculate weighths", booleansToString(terr_recalc_weights,2), 40,
				"Recalculate weights depending on terrain visibility. Use comma/space separated list of true/false, 1/0 or +/-." );
		
		gd.addNumericField("Opaque alpha",         terr_recalc_opaque, 5,7,"", "Alpha above his means opaque.");
		gd.addNumericField("Opaque weight",        terr_recalc_pedestal, 5,7,"","Relative weight of opaque tiles.");
		gd.addNumericField("Transparency fraction",terr_recalc_frac, 5,7,"",    "Discard transparency below this fraction of the maximal one.");
		gd.addNumericField("Transparency distance",terr_recalc_dist, 5,7,"",    "Increase weight for far pixels (double if scale differece == this).");
		gd.addNumericField("Transparency power",   terr_recalc_pwr, 5,7,"",     "Raise transparency to this power when calculating weight.");
		gd.addNumericField("Confidence blur",      terr_recalc_gb, 5,7,"pix",   "Blur transparency-based confidence.");
		gd.addNumericField("Maximal boost",        terr_recalc_boost, 5,7,"",   "Maximal boost while increasing low-confidence pixel weights.");
		gd.addCheckbox    ("Transparency average", terr_recalc_average,         "Apply transparency to average mismatch.");
		
		
		gd.addMessage     ("Combining LMA results segments");
		gd.addCheckbox    ("Crop result image",    terr_crop,                   "Crop output image to the bounds of all segments.");
		gd.addCheckbox    ("Keep partial",         terr_keep_partial,           "Keep partial (per source window) images.");
		gd.addNumericField("Overlap width",        terr_border_width, 0,3,"","Width of the inter-tile oiverlap border.");
		gd.addCheckbox    ("Render open",          terr_render_open, "");
		gd.addCheckbox    ("Render no alpha",      terr_render_no_alpha, "");
		gd.addNumericField("Transparent alpha",    terr_alpha_min, 5,7,"", 			".");
		gd.addNumericField("Opaque alpha",         terr_alpha_max, 5,7,"", 			".");
		gd.addNumericField("Opaque weight",        terr_weight_opaque, 5,7,"", 			".");
		gd.addNumericField("Boost parallax render",terr_boost_render, 5,7,"", "Increase weight of scenes that have high parallax to the reference one (during render).");
		gd.addNumericField("Limit parallax render",terr_max_render, 5,7,"",	  "Parallax limit when evaluating boost parallax (during render).");
		gd.addNumericField("Exagerrate steps",     terr_num_exaggerate, 0,3,"", 			".");

		gd.addMessage     ("Experimental");
		gd.addCheckbox    ("Rebuild elevations",   terr_rebuild_elev, "Regenerate elevations/scales in *.terrveg-tiff file");
		gd.addNumericField("Grow elevations",      terr_elev_grow, 0,3,"","Grow elevations over NaNs.");
		
        gd.addNumericField("Terrain threshold alpha",terr_threshold_terrain, 5,7,"", 			".");
		gd.addNumericField("Min max terrain",      terr_min_max_terrain, 5,7,"", 			".");
		gd.addNumericField("Minimal terrain",      terr_min_terrain, 5,7,"", 			".");
		gd.addNumericField("Minimal vegetation",   terr_min_vegetation, 5,7,"", 			".");
		gd.addNumericField("LMA debug level",      terr_debug, 0,3,"", "Debug level.");
		gd.addStringField ("Debug directory",      terr_debug_path, 120,"Directory to save debug images.");
		gd.addNumericField("Debug length",         terr_debug_length, 0,3,"","Maximal number of debug image slices.");
		gd.addCheckbox    ("Debug improved",       terr_debug_improved, "Save debug image after successful LMA step.");
		gd.addCheckbox    ("Debug worsened",       terr_debug_worsened, "Save debug image after unsuccessful LMA step.");
		
		gd.addTab("Synthesis","Synthesising Terrain/Foloage vies from the model");
		gd.addStringField ("Synthesis directory",  synth_directory, 120, "Directory with the terrain/foliage model generation files.");
		gd.addStringField ("Scene name",           synth_scene,     40,  "Name (timestamp) of the model.");
		gd.addStringField ("Model suffix",         synth_model,     120, "File name suffix for the model.");
		gd.addStringField ("Vegetation segments",  synth_segments,    120, "File name suffix for partial terrain/vegetation segments.");
		gd.addStringField ("Rendered suffix",      synth_render,    120, "Result file name suffix for the rendered synthetic images.");
		gd.addCheckbox    ("Crop by real",         synth_crop_real, "Crop synthetic by real.");
		gd.addStringField ("Scene offsets",        synth_scene_offs,120, "Filenam suffix with scene offsets.");
		gd.addCheckbox    ("Add scene offsets",    synth_add_offs,  "Add scene offsets (vignetting corrcetion).");
	}

	public void dialogAnswers(GenericJTabbedDialog gd) {
		this.ims_use =                        gd.getNextBoolean();
		this.ims_rebuild =                    gd.getNextBoolean();
		this.ims_offset =                     gd.getNextNumber();
		this.gmt_plus =                       gd.getNextNumber();
		this.ims_ortho =     IntersceneMatchParameters. StringToDoubles(gd.getNextString(), 4);
		this.ims_mount_atr = IntersceneMatchParameters. StringToDoubles(gd.getNextString(), 3);
		this.ims_mount_xyz = IntersceneMatchParameters. StringToDoubles(gd.getNextString(), 3);
		this.pimu_gyr_offs = IntersceneMatchParameters. StringToDoubles(gd.getNextString(), 3);
		this.pimu_acc_offs = IntersceneMatchParameters. StringToDoubles(gd.getNextString(), 3);
		this.ims_scale_xyz = IntersceneMatchParameters. StringToDoubles(gd.getNextString(), 3);
		this.ims_scale_atr = IntersceneMatchParameters. StringToDoubles(gd.getNextString(), 3);
		this.fmg_initial_en =                 gd.getNextBoolean();
		this.fmg_reorient_en =                gd.getNextBoolean();
		this.fmg_distance =                   gd.getNextNumber();
		this.fmg_max_quad =                   gd.getNextNumber();
		this.fmg_rectilinear =                gd.getNextBoolean();
		this.sfm_use =                        gd.getNextBoolean();
		this.sfm_min_base =                   gd.getNextNumber();
		this.sfm_min_gain =                   gd.getNextNumber();
		this.sfm_min_frac =                   gd.getNextNumber();
        this.sfm_num_pairs =            (int) gd.getNextNumber(); 
        this.sfp_tolerance =                  gd.getNextNumber();  
        this.sfm_readjust =             (int) gd.getNextNumber(); 
        this.sfm_prev_frac =                  gd.getNextNumber();
        this.sfm_same_weight =                gd.getNextNumber();
        this.sfm_shrink =               (int) gd.getNextNumber();
        this.sfm_fade_sigma =                 gd.getNextNumber(); 
        this.sfm_min_str1 =                   gd.getNextNumber(); 
        this.sfm_min_str16 =                  gd.getNextNumber(); 
        this.sfm_use_neibs =                  gd.getNextBoolean(); 
        this.sfm_neib_too_str1 =              gd.getNextNumber(); 
        this.sfm_neib_too_str16 =             gd.getNextNumber(); 
        this.sfm_neib_str1 =	              gd.getNextNumber(); 
        this.sfm_neib_str16 =	              gd.getNextNumber(); 
        this.sfm_extrap_steps =         (int) gd.getNextNumber();
        this.sfm_extrap_radius =        (int) gd.getNextNumber();
		this.sfm_average_neibs =              gd.getNextBoolean();
		this.sfm_fill_weak =                  gd.getNextBoolean();
		this.sfm_extrapolate =                gd.getNextBoolean();
		
		this.sfm_max_dip =                    gd.getNextNumber();
		this.sfm_max_bump =                   gd.getNextNumber();
		this.sfm_filter_nan =                 gd.getNextBoolean();
		this.sfm_only =                       gd.getNextBoolean();

		this.sfm_save_seq =                   gd.getNextBoolean();
		this.sfm_show_seq =                   gd.getNextBoolean();
		this.sfm_show_corr_ind =              gd.getNextBoolean();
		this.sfm_show_corr =                  gd.getNextBoolean();
		
		this.omtch_img_set = ComboMatch.FILES_LISTS_PATHS[gd.getNextChoiceIndex()];
		
		this.rln_gpu_width =            (int) gd.getNextNumber();
		this.rln_gpu_height =           (int) gd.getNextNumber();
		this.rln_fat_zero =                   gd.getNextNumber();
		this.rln_use_neibs =                  gd.getNextBoolean();
		this.rln_neibs_fill =                 gd.getNextBoolean();
		this.rln_neib_radius =                gd.getNextNumber();
		this.rln_radius_frac =                gd.getNextNumber();		
		this.rln_cent_radius =                gd.getNextNumber();
		this.rln_n_recenter =           (int) gd.getNextNumber();
		this.rln_sngl_rstr =                  gd.getNextNumber();
		this.rln_neib_rstr =                  gd.getNextNumber();
		
		this.pmtch_use_affine=                gd.getNextBoolean();
		this.pmtch_frac_remove  =             gd.getNextNumber();
		this.pmtch_metric_err   =             gd.getNextNumber();
		this.pmtch_max_std      =             gd.getNextNumber();
		this.pmtch_min_std_rad  =             gd.getNextNumber();
		this.pmtch_ignore_rms   =             gd.getNextBoolean();
		this.pmtch_num_iter     =       (int) gd.getNextNumber();
		
		this.pmtch_pull_skew =                gd.getNextNumber();
		this.pmtch_pull_tilt =                gd.getNextNumber();
		this.pmtch_pull_scale =               gd.getNextNumber();
		
		this.pmtch_cent_rad   =               gd.getNextNumber();
		this.pmtch_max_cent_rad   =           gd.getNextNumber();
		this.pmtch_cent_fill   =              gd.getNextNumber();
		this.pmtch_cent_final   =             gd.getNextNumber();
		this.pmtch_ease_nosfm   =             gd.getNextNumber();
		
        this.ospir_existing =                 gd.getNextBoolean();
        this.ospir_invert =                   gd.getNextBoolean();
        this.ospir_step =                     gd.getNextNumber();
        this.ospir_range =                    gd.getNextNumber();
        this.ospir_double =                   gd.getNextNumber();
        this.ospir_good_rms =                 gd.getNextNumber();
        this.ospir_max_rms =                  gd.getNextNumber();
        this.ospir_overlap =            (int) gd.getNextNumber();
        this.ospir_num_iter =           (int) gd.getNextNumber();
        this.ospir_rms_iter[0] =              gd.getNextNumber();
        this.ospir_rms_iter[1] =              gd.getNextNumber();
        this.ospir_ignore_rms =               gd.getNextBoolean();
        this.ospir_debug =              (int) gd.getNextNumber();
        this.ospir_augment =                  gd.getNextBoolean();
        
        this.pwise_zoom =               (int) gd.getNextNumber();
        this.pwise_overlap =                  gd.getNextNumber();
        this.pwise_max_rms =                  gd.getNextNumber();
        this.pwise_skip_exist =               gd.getNextBoolean();
        this.pwise_refine_exist =             gd.getNextBoolean();
        this.pwise_delete_fail =              gd.getNextBoolean();
        this.pwise_gen_inverse =              gd.getNextBoolean();
        this.pwise_save_each =                gd.getNextBoolean();
        this.pwise_log_append =               gd.getNextBoolean();
        this.pwise_log_path =                 gd.getNextString();
        this.pwise_debug =              (int) gd.getNextNumber();
        
        this.alt_overwrite =                  gd.getNextBoolean();
        this.alt_pairwise =                   gd.getNextBoolean();
        this.alt_sigma =                      gd.getNextNumber();
        this.alt_abs_outliers =               gd.getNextNumber();
        this.alt_outliers =                   gd.getNextNumber();
        this.alt_refine =               (int) gd.getNextNumber();
        
        this.alt_weight_rot =                 gd.getNextNumber();  
        this.alt_weight_tilt =                gd.getNextNumber();  
        this.alt_weight_scale =               gd.getNextNumber();   
        this.alt_pull =                       gd.getNextNumber();
        this.alt_pull_rots =                  gd.getNextNumber();
        this.alt_pull_tilts =                 gd.getNextNumber();  
        this.alt_pull_scales =                gd.getNextNumber();        
        
        this.pmap_move_only =		          gd.getNextBoolean();
        this.pmap_ignore_affines =            gd.getNextBoolean();
        this.pmap_use_inv =                   gd.getNextBoolean();
        this.pmap_skew_pull =                 gd.getNextNumber();
        this.pmap_tilt_pull =                 gd.getNextNumber();
        this.pmap_scale_pull =                gd.getNextNumber();
        this.pmap_position_pull =             gd.getNextNumber();
        this.pmap_overlap_pow =               gd.getNextNumber();
        this.pmap_rms_diff =                  gd.getNextNumber();
        this.pmap_num_iter =            (int) gd.getNextNumber();
        this.pmap_show_result =               gd.getNextBoolean();
        this.pmap_debugLevel =          (int) gd.getNextNumber();
                                              
        this.pequ_ignore_equalize =           gd.getNextBoolean();
        this.pequ_use_inv =                   gd.getNextBoolean();
        this.pequ_scale_weight =              gd.getNextNumber();
        this.pequ_pull_weight =               gd.getNextNumber();
        this.pequ_half_weight_sec =           gd.getNextNumber();
        this.pequ_min_weight_sec =            gd.getNextNumber();
        this.pequ_overlap_pow =               gd.getNextNumber();
        this.pequ_rms_diff =                  gd.getNextNumber();
        this.pequ_num_iter =            (int) gd.getNextNumber();
        this.pequ_debugLevel =          (int) gd.getNextNumber();
        
        this.flt_list          =              gd.getNextBoolean();
        this.flt_undef_only    =              gd.getNextBoolean();
        this.flt_min_overlap   =              gd.getNextNumber();
        this.flt_max_overlap   =              gd.getNextNumber();
        this.flt_min_rms       =              gd.getNextNumber();
        this.flt_max_rms       =              gd.getNextNumber();
        this.flt_nan_rms       =              gd.getNextBoolean();
        this.flt_filt_zoom     =              gd.getNextBoolean();
        this.flt_min_zoom      =        (int) gd.getNextNumber(); 
        this.flt_max_zoom      =        (int) gd.getNextNumber();
        this.flt_min_sfm       =              gd.getNextNumber();
        this.flt_max_sfm       =              gd.getNextNumber();
        
        this.flt_alt           =              gd.getNextChoiceIndex();
        this.flt_orient        =              gd.getNextChoiceIndex();
        this.flt_show_names    =              gd.getNextBoolean();
        this.flt_show_overlaps =              gd.getNextBoolean();
        this.flt_show_rms      =              gd.getNextBoolean();        
        this.flt_show_zoom     =              gd.getNextBoolean();
        this.flt_show_alt      =              gd.getNextBoolean();
        
        this.patt_save_top=                   gd.getNextString();
        this.patt_save_subdir =               gd.getNextString();
        
		this.center_reference =               gd.getNextBoolean();
		this.lock_position =                  gd.getNextBoolean();
		this.cuas_rotation =                  gd.getNextBoolean();
		this.lock_position |= this.cuas_rotation; 
		this.manual_correction =              gd.getNextBoolean();
		this.overlap_sequences =              gd.getNextBoolean();
		this.reset_photometric =              gd.getNextBoolean();
		this.force_ref_dsi =                  gd.getNextBoolean();
		this.force_orientations =             gd.getNextBoolean();
		this.run_ly =                         gd.getNextBoolean();
		this.run_ly_mode =             ((int) gd.getNextNumber()) & 3;
		this.run_ly_lma =                     gd.getNextBoolean();
		this.min_num_orient =           (int) gd.getNextNumber(); if (min_num_orient < 1) min_num_orient = 1;
		this.min_num_interscene =       (int) gd.getNextNumber(); if (min_num_interscene < 1) min_num_interscene = 1;
		this.adjust_imu_orient =              gd.getNextBoolean();
		this.apply_imu_orient =               gd.getNextBoolean();
		this.orient_by_move =                 gd.getNextBoolean();
		this.orient_by_rot =                  gd.getNextBoolean();
		this.orient_combo =                   gd.getNextBoolean();
		this.adjust_gyro =                    gd.getNextBoolean();
		this.apply_gyro =                     gd.getNextBoolean();
		this.adjust_accl =                    gd.getNextBoolean();
		this.apply_accl =                     gd.getNextBoolean();
		
		
		this.calc_quat_corr =                 gd.getNextBoolean();
		this.apply_quat_corr =                gd.getNextBoolean();
		this.use_quat_corr =                  gd.getNextBoolean();
		this.inertial_only =                  gd.getNextBoolean();
		this.generate_egomotion =             gd.getNextBoolean();
		this.generate_mapped =                gd.getNextBoolean();
		this.reuse_video =                    gd.getNextBoolean();
		this.save_mapped_color =              gd.getNextBoolean(); // as tiff
		this.save_mapped_mono =               gd.getNextBoolean(); // as tiff
		this.gen_avi_color =                  gd.getNextBoolean(); // as video
		this.gen_avi_mono =                   gd.getNextBoolean(); // as video
		this.show_mapped_color =              gd.getNextBoolean();
		this.show_mapped_mono =               gd.getNextBoolean();
		this.generate_raw =                   gd.getNextBoolean();
		this.generate_inf =                   gd.getNextBoolean();
		this.generate_fg =                    gd.getNextBoolean();
		this.generate_bg =                    gd.getNextBoolean();
		this.generate_stereo =                gd.getNextBoolean();
		for (int i = 0; i < stereo_views.length; i++) {
			this.generate_stereo_var[i] =     gd.getNextBoolean();
		}
		
		this.export_images =                  gd.getNextBoolean();
		this.show_images =                    gd.getNextBoolean();
		this.show_images_bgfg =               gd.getNextBoolean();
		this.show_images_mono =               gd.getNextBoolean();
		this.export_ranges =                  gd.getNextBoolean();
		this.debug_ranges =                   gd.getNextBoolean();
		this.show_ranges =                    gd.getNextBoolean();
		this.export_ml_files =                gd.getNextBoolean();

		// additional parameters		
		this.show_color_nan =                 gd.getNextBoolean();
		this.show_mono_nan =                  gd.getNextBoolean();
		
		this.scale_extrap_atr =               gd.getNextNumber();
		this.scale_extrap_xyz =               gd.getNextNumber();
		this.avg_len =                  (int) gd.getNextNumber();
		this.min_num_scenes =           (int) gd.getNextNumber();
		this.max_num_scenes =           (int) gd.getNextNumber();
		this.blur_egomotion =                 gd.getNextNumber();
		this.range_disparity_offset =         gd.getNextNumber();
		this.range_min_strength =             gd.getNextNumber();
		this.range_max =                      gd.getNextNumber();
		
		this.export3d =                       gd.getNextBoolean();
		this.export3dterrain =                gd.getNextBoolean();
		
		this.export_CT =                      gd.getNextBoolean();
		this.ct_min =                         gd.getNextNumber();
		this.ct_max =                         gd.getNextNumber();
		this.ct_step =                        gd.getNextNumber();
		this.ct_expand =                (int) gd.getNextNumber();
		
//		this.terr_enable=                     gd.getNextBoolean();
		this.terr_rmse_above=	        	  gd.getNextNumber();
		this.terr_rmse_below=	        	  gd.getNextNumber();
		this.terr_num_refine=	        (int) gd.getNextNumber();	
		this.terr_frac_above=            0.01*gd.getNextNumber();
		this.terr_frac_below=	         0.01*gd.getNextNumber();
		this.terr_rthrsh_abv=	        	  gd.getNextNumber();
		this.terr_flat_brd=                   gd.getNextBoolean();
		this.terr_expand=               (int) gd.getNextNumber();
		this.terr_save=                       gd.getNextBoolean();

		this.scene_is_ref_test =              gd.getNextBoolean();
		this.render_ref =                     gd.getNextBoolean();
		this.render_scene =                   gd.getNextBoolean();
		this.toRGB =                          gd.getNextBoolean();
		this.show_2d_correlations =           gd.getNextBoolean();
		this.show_motion_vectors =            gd.getNextBoolean();
		this.debug_level =              (int) gd.getNextNumber();

		this.test_ers =                       gd.getNextBoolean();
		this.test_ers0 =                (int) gd.getNextNumber();
		this.test_ers1 =                (int) gd.getNextNumber();

		this.num_bottom =               (int) gd.getNextNumber();
		this.num_passes =               (int) gd.getNextNumber();
		this.max_change =                     gd.getNextNumber();
		this.min_disparity =                  gd.getNextNumber();
		this.max_sym_disparity =              gd.getNextNumber();
		this.min_strength_lma =               gd.getNextNumber();
		this.min_strength_replace =           gd.getNextNumber();
		this.min_strength_blur =              gd.getNextNumber();
		this.sigma =                          gd.getNextNumber();
		this.num_blur =                 (int) gd.getNextNumber();
		this.disparity_corr =                 gd.getNextNumber();
		this.outliers_nth_fromextrem =  (int) gd.getNextNumber();
		this.outliers_tolerance_absolute =    gd.getNextNumber();
		this.outliers_tolerance_relative =    gd.getNextNumber();
		this.outliers_max_iter =        (int) gd.getNextNumber();
		this.outliers_max_strength2 =         gd.getNextNumber();
		this.outliers_nth_fromextrem2 = (int) gd.getNextNumber();
		this.outliers_tolerance_absolute2 =   gd.getNextNumber();
		this.outliers_tolerance_relative2 =   gd.getNextNumber();
		this.outliers_lma_max_strength =      gd.getNextNumber();
		this.outliers_max_strength =          gd.getNextNumber();
		this.outliers_from_lma_max_strength = gd.getNextNumber();
		this.search_radius =            (int) gd.getNextNumber();
		this.remove_no_lma_neib =             gd.getNextBoolean();
		this.diff_from_lma_pos =              gd.getNextNumber();
		this.diff_from_lma_neg =              gd.getNextNumber();
		this.outliers_lma_nth_fromextrem=(int)gd.getNextNumber();
		this.filter_margin =            (int) gd.getNextNumber();
		
		this.weak_tolerance_absolute =        gd.getNextNumber();
		this.weak_tolerance_relative =        gd.getNextNumber();
		this.weak_min_neibs =           (int) gd.getNextNumber();
		this.strong_strength =                gd.getNextNumber();
		this.weak_strength =                  gd.getNextNumber();
		
		this.sky_extract =              gd.getNextBoolean();
		this.sky_recalc =               gd.getNextBoolean();
		this.sky_highest_min =          gd.getNextNumber(); 	
		this.cold_frac =                gd.getNextNumber();       	
		this.hot_frac =                 gd.getNextNumber();        	
		this.cold_scale =               gd.getNextNumber();      	
		this.sky_seed =                 gd.getNextNumber();        	
		this.lma_seed =                 gd.getNextNumber();
		this.seed_temp =                gd.getNextNumber();
		this.sky_shrink =         (int) gd.getNextNumber();      	
		this.seed_rows =          (int) gd.getNextNumber();
		this.max_disparity =            gd.getNextNumber();         	
		this.max_disparity_strength =   gd.getNextNumber();         	
		this.sky_lim =                  gd.getNextNumber(); 
		this.lim_temp =                 gd.getNextNumber(); 
		this.sky_expand_extra =   (int) gd.getNextNumber();	
		this.sky_bottleneck =     (int) gd.getNextNumber();	
		this.sky_reexpand_extra = (int) gd.getNextNumber();	
		this.min_strength =             gd.getNextNumber();    
		this.lowest_sky_row =     (int) gd.getNextNumber();
		this.sky_bottom_override =      gd.getNextNumber();    
		this.sky_override_shrink =(int) gd.getNextNumber();
		
		this.clouds_en =                gd.getNextBoolean();
		this.clouds_fom =               gd.getNextNumber();    
		this.clouds_spread =            gd.getNextNumber();    
		this.clouds_disparity =         gd.getNextNumber();    
		this.clouds_weak =              gd.getNextNumber();    
		this.clouds_strength =          gd.getNextNumber();    
		this.clouds_not_strength =      gd.getNextNumber();    
		this.clouds_strong =            gd.getNextBoolean();

		this.wall_en =                  gd.getNextBoolean();
		this.wall_dflt =                gd.getNextBoolean();
		this.wall_str =                 gd.getNextNumber();    
		this.wall_far =                 gd.getNextNumber();    
		this.wall_near =                gd.getNextNumber();    
		
		this.treeline_en =              gd.getNextBoolean();
		this.treeline_wide =            gd.getNextBoolean();
		this.treeline_height =    (int) gd.getNextNumber();
		this.treeline_width =     (int) gd.getNextNumber();
		this.treeline_lim_high =        gd.getNextBoolean();
		this.treeline_str =             gd.getNextNumber();
		this.treeline_far =             gd.getNextNumber();
		this.treeline_near =            gd.getNextNumber();
		this.treeline_fg_str =          gd.getNextNumber();
		this.treeline_fg_far =          gd.getNextNumber();
		this.treeline_fg_near =         gd.getNextNumber();
		
		this.indoors_en =               gd.getNextBoolean();
		this.indoors_str =              gd.getNextNumber();    
		this.indoors_disp =             gd.getNextNumber();    
		this.indoors_min_out =    (int) gd.getNextNumber();

		this.disp_boost_min =           gd.getNextNumber();    
		this.disp_boost_diff =          gd.getNextNumber();    
		this.disp_boost_neibs =   (int) gd.getNextNumber();
		this.disp_boost_amount =        gd.getNextNumber();    
		this.scale_combo_strength =     gd.getNextNumber();
		this.save_debug_images =        gd.getNextBoolean();
		
		this.margin =             (int) gd.getNextNumber();
		this.sensor_mask_inter=   (int) gd.getNextNumber();
		this.use_partial =              gd.getNextBoolean();
		this.run_poly =                 gd.getNextBoolean();
		this.centroid_radius =          gd.getNextNumber();
		this.n_recenter =         (int) gd.getNextNumber();
		this.td_weight =                gd.getNextNumber();
		this.td_neib_weight =           gd.getNextNumber();
		this.pd_weight =                gd.getNextNumber();
		this.td_nopd_only =             gd.getNextBoolean();
		this.neib_notd_only =           gd.getNextBoolean();
		
		this.use_neibs =                gd.getNextBoolean();
		this.neibs_nofpn_only =         gd.getNextBoolean();
		this.neibs_nofpn_init =         gd.getNextBoolean();
		this.redo_both =                gd.getNextBoolean();
		this.min_num_neibs =      (int) gd.getNextNumber();
		this.scale_neibs_pd =           gd.getNextNumber();
		this.scale_neibs_td =           gd.getNextNumber();
		this.scale_avg_weight =         gd.getNextNumber();
		
		this.min_str =                  gd.getNextNumber();
		this.min_str_fpn =              gd.getNextNumber();
		this.min_str_sum =              gd.getNextNumber();
		this.min_str_sum_fpn =          gd.getNextNumber();
		this.min_str_neib =             gd.getNextNumber();
		this.min_str_neib_fpn =         gd.getNextNumber();
		this.min_neibs =          (int) gd.getNextNumber();
		this.weight_zero_neibs =        gd.getNextNumber();
		this.half_disparity =           gd.getNextNumber();
		this.half_avg_diff =            gd.getNextNumber();
		
		this.eig_use =                  gd.getNextBoolean();
		this.eig_xy_lma =               gd.getNextBoolean();
		this.eig_str_sum =              gd.getNextNumber();
		this.eig_str_sum_fpn =          gd.getNextNumber();
		this.eig_str_neib =             gd.getNextNumber();
		this.eig_str_neib_fpn =         gd.getNextNumber();
		this.eig_min_abs =              gd.getNextNumber();
		this.eig_min_rel =              gd.getNextNumber();
		this.eig_sub_frac =             gd.getNextNumber();
		this.eig_max_sqrt =             gd.getNextNumber();
		this.eig_min_sqrt =             gd.getNextNumber();
		this.eig_recenter =       (int) gd.getNextNumber();
		this.eig_sub_frac1 =            gd.getNextNumber();
		this.eig_scale_axes =           gd.getNextNumber();
		this.eig_inc_axes =             gd.getNextNumber();
		this.eig_use_neibs =            gd.getNextBoolean();
		this.eig_min_weaks =      (int) gd.getNextNumber();
		this.eig_min_strongs =    (int) gd.getNextNumber();
		this.eig_disp_diff =            gd.getNextNumber();
		this.eig_remove_neibs =         gd.getNextBoolean();
		this.eig_filt_other =           gd.getNextBoolean();
		this.eig_max_rms =              gd.getNextNumber();
		this.eig_fast2x2 =              gd.getNextBoolean();
		this.use_combo_reliable =       gd.getNextBoolean();
		this.ref_need_lma =             gd.getNextBoolean();
		this.ref_need_lma_combo =       gd.getNextBoolean();
		this.min_ref_str =              gd.getNextNumber();
		this.min_ref_str_lma =          gd.getNextNumber();
		this.min_ref_frac =             gd.getNextNumber();
		this.ref_smooth =               gd.getNextBoolean();
		ref_smooth_always =             gd.getNextBoolean();		
		this.ref_sigma =                gd.getNextNumber();
		this.ref_smooth_diff =          gd.getNextNumber();
		this.sfm_filter =               gd.getNextBoolean();
		this.sfm_minmax =               gd.getNextNumber();
		this.sfm_fracmax =              gd.getNextNumber();
		this.sfm_fracall =              gd.getNextNumber();
		this.pix_step =           (int) gd.getNextNumber();
		this.search_rad =         (int) gd.getNextNumber();
		this.maybe_sum =                gd.getNextNumber();
		this.sure_sum =                 gd.getNextNumber();
		this.maybe_avg =                gd.getNextNumber();
		this.sure_avg =                 gd.getNextNumber();
		this.max_search_rms =           gd.getNextNumber();
		this.maybe_fom =                gd.getNextNumber();
		this.sure_fom =                 gd.getNextNumber();
		this.treat_serch_fpn =          gd.getNextBoolean();
		this.use_combo_dsi =            gd.getNextBoolean();
		this.use_lma_dsi =              gd.getNextBoolean();
		this.fpn_remove =               gd.getNextBoolean();
		this.fpn_max_offset =           gd.getNextNumber();
		this.fpn_min_offset =           gd.getNextNumber();
		this.fpn_radius =               gd.getNextNumber();
		this.fpn_ignore_border =        gd.getNextBoolean();

		this.min_offset =               gd.getNextNumber();
		this.max_pull_jump =            gd.getNextNumber();
		this.max_rel_offset =           gd.getNextNumber();
		this.max_roll_deg =             gd.getNextNumber();
		this.max_zoom_diff =            gd.getNextNumber();
		
		this.fpn_skip =                 gd.getNextBoolean();
		this.fpn_rematch =              gd.getNextBoolean();
		this.refine_invert =            gd.getNextBoolean();
		this.use_precomp =              gd.getNextBoolean();
		
		this.mov_en =                   gd.getNextBoolean();
		this.mov_sigma =                gd.getNextNumber();
		this.mov_max_std =              gd.getNextNumber();
		this.mov_thresh_rel =           gd.getNextNumber();
		this.mov_thresh_abs =           gd.getNextNumber();
		this.mov_clust_max =            gd.getNextNumber();
		this.mov_grow =           (int) gd.getNextNumber();
		this.mov_max_len =        (int) gd.getNextNumber();
		
		this.mov_show =                 gd.getNextBoolean();
		this.mov_debug_level =    (int) gd.getNextNumber();
		this.adjust_atr[0] =            gd.getNextBoolean();
		this.adjust_atr[1] =            gd.getNextBoolean();
		this.adjust_atr[2] =            gd.getNextBoolean();
		this.adjust_xyz[0] =            gd.getNextBoolean();
		this.adjust_xyz[1] =            gd.getNextBoolean();
		this.adjust_xyz[2] =            gd.getNextBoolean();
		this.exit_change_atr =          gd.getNextNumber();
		this.exit_change_xyz =          gd.getNextNumber();
		this.max_cycles =         (int) gd.getNextNumber();
		this.max_LMA =            (int) gd.getNextNumber();
		this.max_rms =                  gd.getNextNumber();
		
		
		// equalization of the inter-scene correlations	
		this.eq_en =                    gd.getNextBoolean();
		this.eq_stride_hor =      (int) gd.getNextNumber();
		this.eq_stride_vert =     (int) gd.getNextNumber();
		this.eq_min_stile_weight =      gd.getNextNumber();    
		this.eq_min_stile_number =(int) gd.getNextNumber();
		this.eq_min_stile_fraction =    gd.getNextNumber();    
		this.eq_min_disparity =         gd.getNextNumber();    
		this.eq_max_disparity =         gd.getNextNumber();    
		this.eq_weight_add =            gd.getNextNumber();    
		this.eq_weight_scale =          gd.getNextNumber();    
		this.eq_level =                 gd.getNextNumber();
		
		this.mb_en =                    gd.getNextBoolean();
		this.mb_tau =                   gd.getNextNumber();    
		this.mb_max_gain =              gd.getNextNumber();    
		this.mb_max_gain_inter =        gd.getNextNumber();
		
		this.cuas_update_existing =     gd.getNextBoolean();
		this.cuas_discard_border= (int) gd.getNextNumber();
		this.cuas_max_fold =            gd.getNextNumber();
		this.cuas_min_in_row_col= (int) gd.getNextNumber();
		this.cuas_clt_variant =         gd.getNextNumber();
		this.cuas_clt_threshold =       gd.getNextNumber();
		this.cuas_clt_decrease =        gd.getNextNumber();
		this.cuas_decay_average =       gd.getNextNumber();
		this.cuas_keep_fraction =       gd.getNextNumber();
		this.cuas_step =                gd.getNextBoolean();
		this.cuas_subtract_fpn =        gd.getNextBoolean();
		this.cuas_calc_fpn =            gd.getNextBoolean();
		this.cuas_rot_period =          gd.getNextNumber();
		this.cuas_min_series =          gd.getNextNumber();
		
		this.cuas_subtract_rowcol =     gd.getNextBoolean();
		this.cuas_calc_rowcol =         gd.getNextBoolean();
		this.cuas_um_sigma =            gd.getNextNumber();
		this.cuas_max_abs_rowcol =      gd.getNextNumber();
		this.cuas_outliers_rowcol =     gd.getNextNumber();
		this.cuas_reset_first =         gd.getNextBoolean();

		this.cuas_invert_margins =(int) gd.getNextNumber();
		this.cuas_invert_iters =  (int) gd.getNextNumber();
		this.cuas_invert_tolerance =    gd.getNextNumber();
		this.cuas_invert_gap2 =   (int) gd.getNextNumber();
		
		this.cuas_w_um_sigma =          gd.getNextNumber();
		this.cuas_w_good_margins =(int) gd.getNextNumber();
		this.cuas_w_threshold =         gd.getNextNumber();
		this.cuas_w_scale_thresh =      gd.getNextNumber();
		this.cuas_w_expand_max =        gd.getNextBoolean();
		this.cuas_w_outliers =          gd.getNextNumber();
		this.cuas_w_blur =              gd.getNextNumber();
		
		this.cuas_dsi_parent =          gd.getNextBoolean();
		this.cuas_fg_strength =         gd.getNextNumber();
		this.cuas_disp_add =            gd.getNextNumber();
		this.cuas_skip_bottom =   (int) gd.getNextNumber();
		this.cuas_multi_strength =      gd.getNextNumber();
		this.cuas_reliable_str =        gd.getNextNumber();
		
		this.cuas_fat_zero =            gd.getNextNumber();
		this.cuas_cent_radius =         gd.getNextNumber();
		this.cuas_n_recenter =    (int) gd.getNextNumber();
		this.cuas_rstr =                gd.getNextNumber();
		this.cuas_smooth =              gd.getNextBoolean();   
		this.cuas_corr_pairs =    (int) gd.getNextNumber();
		this.cuas_corr_offset =   (int) gd.getNextNumber();
		this.cuas_gaussian_ra =         gd.getNextBoolean();
		this.cuas_temporal_um =   (int) gd.getNextNumber();
		this.cuas_tum_threshold =       gd.getNextNumber();
		
		this.cuas_precorr_ra =    (int) gd.getNextNumber();
		this.cuas_corr_step =     (int) gd.getNextNumber();
		
		this.cuas_half_step =           gd.getNextBoolean();
		this.cuas_max_range =     (int) gd.getNextNumber();
		this.cuas_num_cycles =    (int) gd.getNextNumber();
		
		this.cuas_speed_min =           gd.getNextNumber();
		this.cuas_speed_pref =          gd.getNextNumber();
		this.cuas_speed_boost =         gd.getNextNumber();
		
		this.cuas_lmax_flt_neglim =     gd.getNextNumber();
		this.cuas_lmax_flt_hsigma =     gd.getNextNumber();
		this.cuas_lmax_flt_lsigma =     gd.getNextNumber();
		this.cuas_lmax_flt_scale =      gd.getNextNumber();
		
		this.cuas_sky_threshold =       gd.getNextNumber();
		this.cuas_sky_offset =  IntersceneMatchParameters.StringToDoubles(gd.getNextString(), cuas_sky_offset);
		
		this.cuas_lmax_fraction =       gd.getNextNumber();
		this.cuas_lmax_hack_ridge =     gd.getNextNumber();
		this.cuas_lmax_radius =         gd.getNextNumber();
		this.cuas_lmax_zero =           gd.getNextBoolean();   

		this.cuas_centered_radius =     gd.getNextNumber();
		this.cuas_centered_blur =       gd.getNextNumber();
		this.cuas_duplicate_tolerance = gd.getNextNumber();
		
		this.cuas_target_radius =       gd.getNextNumber();
		this.cuas_target_strength =     gd.getNextNumber();
		this.cuas_target_frac = stringToDouble2d(gd.getNextString());
		this.cuas_no_border =           gd.getNextBoolean();   

		this.cuas_lma_sigma =           gd.getNextNumber();
		this.cuas_wnd_pedestal =        gd.getNextNumber();
		this.cuas_lma_r0 =              gd.getNextNumber();
		this.cuas_lma_ovrsht =          gd.getNextNumber();
		this.cuas_lma_fit_xy =          gd.getNextBoolean();   
		this.cuas_lma_fit_a =           gd.getNextBoolean();   
		this.cuas_lma_fit_c =           gd.getNextBoolean();   
		this.cuas_lma_fit_r =           gd.getNextBoolean();   
		this.cuas_lma_fit_k =           gd.getNextBoolean();   
		this.cuas_lambda =              gd.getNextNumber();
		this.cuas_lambda_good =         gd.getNextNumber();
		this.cuas_lambda_bad =          gd.getNextNumber(); 
		this.cuas_lambda_max =          gd.getNextNumber(); 
		this.cuas_rms_diff =            gd.getNextNumber();
		this.cuas_num_iter =      (int) gd.getNextNumber();
		this.cuas_lma_rms =             gd.getNextNumber();
		this.cuas_lma_arms =            gd.getNextNumber();
		this.cuas_lma_rrms =            gd.getNextNumber();
		this.cuas_lma_mina =            gd.getNextNumber();
		this.cuas_lma_pre_mina =        gd.getNextNumber();
		this.cuas_min_keep =      (int) gd.getNextNumber();
		
		this.cuas_lma_maxr =            gd.getNextNumber();
		this.cuas_lma_minr1 =           gd.getNextNumber();
		this.cuas_lma_mink =            gd.getNextNumber();
		this.cuas_lma_maxk =            gd.getNextNumber();
		this.cuas_lma_a2a =             gd.getNextNumber();
		
		this.cuas_slow_ra =             gd.getNextNumber();
		this.cuas_slow_score =          gd.getNextNumber();

		this.cuas_score_mv =            gd.getNextNumber();
		this.cuas_score_lma =           gd.getNextNumber();
		this.cuas_factor_lim =          gd.getNextNumber();
		this.cuas_factor_pow =          gd.getNextNumber();
		this.cuas_score_coeff =  IntersceneMatchParameters.StringToDoubles(gd.getNextString(), cuas_score_coeff);		
		this.cuas_center_targ =         gd.getNextBoolean();
		this.cuas_multi_targ =          gd.getNextBoolean();
		
		this.cuas_isolated =            gd.getNextBoolean();
//		this.cuas_remove_2seq =         gd.getNextBoolean();
//		this.cuas_remove_2seq_all =     gd.getNextBoolean();
		this.cuas_min_seq=        (int) gd.getNextNumber();
		this.cuas_enough_seq=     (int) gd.getNextNumber();
		this.cuas_seq_travel=           gd.getNextNumber();
		
		this.cuas_min_seq_fl=     (int) gd.getNextNumber();
		
		this.cuas_max_mismatch=         gd.getNextNumber();
		this.cuas_good_mismatch =       gd.getNextNumber();
		this.cuas_slow_fast_mismatch =  gd.getNextNumber();
		this.cuas_match_len_pwr =       gd.getNextNumber();
		
		this.cuas_fail_mismatch =       gd.getNextBoolean();
		this.cuas_ignore_mismatch =     gd.getNextBoolean();
		
		this.cuas_by_horizon =          gd.getNextBoolean();
		this.cuas_horizon=              gd.getNextNumber();

		this.cuas_mask_width =          gd.getNextNumber();
		this.cuas_mask_blur =           gd.getNextNumber();
		this.cuas_mask_round =          gd.getNextBoolean();
		this.cuas_target_type=    (int) gd.getNextNumber();
		this.cuas_known_type=     (int) gd.getNextNumber();
		   
		this.cuas_input_range =         gd.getNextNumber();
		this.cuas_scale2x =             gd.getNextBoolean();
		this.cuas_ra_background =       gd.getNextBoolean();

		this.cuas_targets =             gd.getNextBoolean();
		this.cuas_iter_show=      (int) gd.getNextNumber();
		this.cuas_2d_save_show =        gd.getNextBoolean();
		this.cuas_intermed_low =        gd.getNextBoolean();
		this.cuas_intermed_high =       gd.getNextBoolean();
		this.cuas_intermed_giga =       gd.getNextBoolean();
		this.cuas_debug_more =          gd.getNextBoolean();
		
		this.cuas_generate =            gd.getNextBoolean();
		this.cuas_generate_csv =        gd.getNextBoolean();
		
		this.cuas_save_mono =           gd.getNextBoolean();
		this.cuas_save_color =          gd.getNextBoolean();
		this.cuas_save_video =          gd.getNextBoolean();
		this.cuas_clean_video =         gd.getNextBoolean();
		
		this.cuas_annotate =            gd.getNextBoolean();
		this.cuas_annotate_uas =        gd.getNextBoolean();
		{
			String scolor =             gd.getNextString();
			long lcolor = -1;
			try {
				lcolor = Long.parseLong(scolor,16);
				this.cuas_text_color = setLongColor(lcolor);
			} catch(NumberFormatException e){
				this.cuas_text_color = setLongColor(0x00FF00);
			}
		}
		{
			String scolor =             gd.getNextString();
			long lcolor = -1;
			try {
				lcolor = Long.parseLong(scolor,16);
				this.cuas_text_bg = setLongColor(lcolor);
			} catch(NumberFormatException e){
				this.cuas_text_bg = setLongColor(0x777777);
			}
		}
		{
			String scolor =             gd.getNextString();
			long lcolor = -1;
			try {
				lcolor = Long.parseLong(scolor,16);
				this.cuas_selected_color = setLongColor(lcolor);
			} catch(NumberFormatException e){
				this.cuas_selected_color = setLongColor(0xFF0088);
			}
		}
		this.cuas_transparent =         gd.getNextBoolean();
		this.cuas_transparent_uas =     gd.getNextBoolean();
		this.cuas_font_name =           gd.getNextString();
		this.cuas_font_size =     (int) gd.getNextNumber();
		this.cuas_font_type =     (int) gd.getNextNumber();
		this.cuas_font_spacing =        gd.getNextNumber();
		this.cuas_annot_missing =       gd.getNextBoolean();
		this.cuas_px0 =           (int) gd.getNextNumber();
		this.cuas_py0 =           (int) gd.getNextNumber();
		this.cuas_az0 =                 gd.getNextNumber();
		this.cuas_el0 =                 gd.getNextNumber();
		this.cuas_show_disp =           gd.getNextBoolean();
		this.cuas_show_rng =            gd.getNextBoolean();
		this.cuas_show_inf =            gd.getNextBoolean();
		this.cuas_show_inf_gt =         gd.getNextBoolean();
		this.cuas_show_true_rng =       gd.getNextBoolean();
		
		this.cuas_smooth_omegas =       gd.getNextBoolean();
		this.cuas_rng_um =              gd.getNextBoolean();
		this.cuas_rng_um_all =          gd.getNextBoolean();
		this.cuas_rng_um_sigma =        gd.getNextNumber();
		this.cuas_rng_um2 =             gd.getNextBoolean();
		this.cuas_rng_um_weight =       gd.getNextNumber();
		this.cuas_rng_coswnd =          gd.getNextBoolean();
		this.cuas_rng_combine =   (int) gd.getNextNumber();
		
		this.cuas_rng_radius0 =         gd.getNextNumber();
		this.cuas_rng_radius =          gd.getNextNumber();
		this.cuas_rng_blur =            gd.getNextNumber();
		this.cuas_rng_fz0=              gd.getNextNumber();
		this.cuas_rng_fz =              gd.getNextNumber();
		this.cuas_rng_scale =           gd.getNextNumber();
		
		this.cuas_dual_pass =           gd.getNextBoolean();
		this.cuas_disp_only2 =          gd.getNextBoolean();
		this.cuas_lma_bypass =          gd.getNextBoolean();
		this.cuas_mcorr_sel =     (int) gd.getNextNumber();
		this.cuas_mcorr_sel_lma = (int) gd.getNextNumber();
		
		this.cuas_max_rel_rms =         gd.getNextNumber();
		this.cuas_min_strength =        gd.getNextNumber();
		this.cuas_ac_offset =           gd.getNextNumber();
		this.cuas_min_max_ac =          gd.getNextNumber();
		this.cuas_min_min_ac =          gd.getNextNumber();
		this.cuas_reset_disparity =     gd.getNextBoolean();
		this.cuas_initial_disparity =   gd.getNextNumber();
		this.cuas_infinity =            gd.getNextNumber();
		
		this.cuas_rng_img =             gd.getNextBoolean();
		this.cuas_rng_glob =            gd.getNextBoolean();
		this.cuas_glob_ends =  (int) gd.getNextNumber();
		this.cuas_rng_disp =            gd.getNextBoolean();
		this.cuas_rng_vfy =             gd.getNextBoolean();
		this.cuas_rng_niterate =  (int) gd.getNextNumber();
		this.cuas_rng_diff =            gd.getNextNumber();
		
		this.cuas_flw_levels =    (int) gd.getNextNumber();
		this.cuas_flw_len =       (int) gd.getNextNumber();
		this.cuas_flw_diff =            gd.getNextNumber();

		this.cuas_known_err=            gd.getNextNumber();
		this.cuas_tmtch_frac=           gd.getNextNumber();
		this.cuas_tmtch_ends=     (int) gd.getNextNumber();
		this.cuas_tmtch_gaps=     (int) gd.getNextNumber();
		this.cuas_tmtch_pix=            gd.getNextNumber();
		this.cuas_local_apix=           gd.getNextNumber();
		this.cuas_tmtch_apix=           gd.getNextNumber();
		this.cuas_tmtch_rpix=           gd.getNextNumber();
		this.cuas_tmtch_axv=            gd.getNextNumber();
		this.cuas_tmtch_axv_k=          gd.getNextNumber();
		this.cuas_local_diff=           gd.getNextNumber();
		this.cuas_tmtch_diff=           gd.getNextNumber();
		this.cuas_tmtch_disp=           gd.getNextNumber();
		this.cuas_tmtch_short =         gd.getNextBoolean();
		
		this.cuas_radar_range=          gd.getNextNumber();
		this.cuas_radar_height =  (int) gd.getNextNumber();
		this.cuas_radar_bottom =  (int) gd.getNextNumber();
		this.cuas_radar_top =     (int) gd.getNextNumber();
		this.cuas_radar_uas_color =     getColorFromHex(
				gd.getNextString(),            //String hex_color,
				DEFAULT_cuas_radar_uas_color); // Color  default_color) 
		this.cuas_radar_color =         getColorFromHex(
				gd.getNextString(),        //String hex_color,
				DEFAULT_cuas_radar_color); // Color  default_color) 
		this.cuas_radar_radius =        gd.getNextNumber();
		this.cuas_radar_uas_rad =       gd.getNextNumber();
		// Radar grid generation
		this.cuas_grid_ring =           gd.getNextNumber();
		this.cuas_grid_line =           gd.getNextNumber();
		this.cuas_grid_annot =          gd.getNextBoolean();
		
		this.cuas_grid_color =          getColorFromHex(
				gd.getNextString(),       //String hex_color,
				DEFAULT_cuas_grid_color); // Color  default_color) 
		this.cuas_grid_font =     (int) gd.getNextNumber();
		this.cuas_grid_az_top =   (int) gd.getNextNumber();
		this.cuas_grid_sides =    (int) gd.getNextNumber();
		
		for (int i = 0; i < CuasMotion.ANNOT_LABELS.length; i++) {
			String s = gd.getNextString();
			CuasMotion.setCuasAnnots(
					s,                // String annot,
					i,                // int    indx,
					cuas_annot_sel);  // int [] annots)
		}		

		this.cuas_debug =               gd.getNextBoolean();
		this.cuas_dbg_rng_seq =   (int) gd.getNextNumber();
		this.cuas_dbg_rng_tgt =   (int) gd.getNextNumber();
		this.cuas_dbg_show_mode = (int) gd.getNextNumber();
		
		this.cuas_step_debug =          gd.getNextBoolean();
		this.cuas_save_stats =          gd.getNextBoolean();
		this.cuas_target_debug =        gd.getNextBoolean();
		this.cuas_target_score =        gd.getNextBoolean();
		this.cuas_target_color =        gd.getNextBoolean();
		this.cuas_log_ranging =         gd.getNextBoolean();

		this.cuas_overwrite =           gd.getNextBoolean();
		this.cuas_num_orient =    (int) gd.getNextNumber();
		this.cuas_num_accum =     (int) gd.getNextNumber();
		
		this.cuas_reuse_targets =       gd.getNextBoolean();
		this.cuas_reuse_path =          gd.getNextString();
		this.cuas_reuse_disparity =     gd.getNextBoolean();
		this.cuas_reuse_globals =       gd.getNextBoolean();
		this.cuas_max_disp_diff =       gd.getNextNumber();
		this.cuas_min_disp_str =        gd.getNextNumber();
		this.cuas_rng_limit =           gd.getNextNumber();
		
		this.mb_gain_index_pose = (int) gd.getNextNumber();
		this.mb_gain_index_depth =(int) gd.getNextNumber();
		
		this.mb_ers_index =       (int) gd.getNextNumber();
		this.mb_ers_y_index =     (int) gd.getNextNumber();
		this.mb_ers_r_index =     (int) gd.getNextNumber();
		this.mb_all_index =       (int) gd.getNextNumber();
		this.mb_pref_orient =           gd.getNextBoolean();
		this.lma_use_Z =                gd.getNextBoolean();
		this.lma_use_R =                gd.getNextBoolean();
		
		this.configured_lma =           gd.getNextBoolean();
		this.freeze_xy_pull =           gd.getNextBoolean();
		this.copy_pull_current =        gd.getNextBoolean();
		this.restore_imu =              gd.getNextBoolean();
		this.lpf_xy =                   gd.getNextBoolean();
		this.readjust_xy_ims =          gd.getNextBoolean();
		this.reg_weight_xy =            gd.getNextNumber();
		
		this.quat_lambda =              gd.getNextNumber();
		this.quat_lambda_scale_good =   gd.getNextNumber();
		this.quat_lambda_scale_bad =    gd.getNextNumber();
		this.quat_lambda_max =          gd.getNextNumber();
		this.quat_rms_diff =            gd.getNextNumber();
		this.quat_num_iter =      (int) gd.getNextNumber();
		this.quat_reg_w =               gd.getNextNumber();
		
		this.quat_max_ratio =           gd.getNextNumber();
		this.quat_max_change =          gd.getNextNumber();
		this.quat_min_transl =          gd.getNextNumber();
		this.quat_min_rot =             gd.getNextNumber();
		this.quat_min_lin =             gd.getNextNumber();

		if (stereo_views.length > 0) {
			int i =                     gd.getNextChoiceIndex();
			if (i > 0) {
				removeStereoView(i-1);
			}
		}
		String s =                      gd.getNextString();
		addStereoView(s, true);
		this.stereo_merge =             gd.getNextBoolean();
		this.stereo_gap =         (int) gd.getNextNumber();
		this.stereo_intereye =          gd.getNextNumber();
		this.stereo_phone_width =       gd.getNextNumber();
		this.extra_hor_tile =     (int) gd.getNextNumber();
		this.extra_vert_tile =    (int) gd.getNextNumber();
		this.crop_3d =                  gd.getNextBoolean();
		this.sensor_mask =        (int) gd.getNextNumber();
		this.merge_all =                gd.getNextBoolean();
		this.video_fps =                gd.getNextNumber();
		this.sensor_fps =               gd.getNextNumber();
		this.mode_avi =                 gd.getNextChoiceIndex();
		this.avi_JPEG_quality =   (int) gd.getNextNumber();
		this.run_ffmpeg =               gd.getNextBoolean();
		this.video_ext=                 gd.getNextString();	
		this.video_codec=               gd.getNextString();	
		this.video_crf =          (int) gd.getNextNumber();
		this.video_bitrate_m =          gd.getNextNumber();
		this.remove_avi =               gd.getNextBoolean();
		this.video_codec_combo=         gd.getNextString();	
		this.video_crf_combo =    (int) gd.getNextNumber();
		this.add_average =              gd.getNextBoolean();
		this.calculate_average =        gd.getNextBoolean();
		this.subtract_average =         gd.getNextBoolean();
		this.running_average =    (int) gd.getNextNumber();
		this.extract_center_orientation=gd.getNextBoolean();
		this.um_mono =                  gd.getNextBoolean();
		this.um_sigma =                 gd.getNextNumber();
		this.um_weight =                gd.getNextNumber();
		this.um_mono_linear =           gd.getNextBoolean();
		this.mono_fixed =               gd.getNextBoolean();
		this.mono_range =               gd.getNextNumber();
		
		this.anaglyth_en =              gd.getNextBoolean();
		{
			String scolor =             gd.getNextString();
			long lcolor = -1;
			try {
				lcolor = Long.parseLong(scolor,16);
				this.anaglyph_left = setLongColor(lcolor);
			} catch(NumberFormatException e){
				this.anaglyph_left = anaglyph_left_default;
			}
		}

		{
			String scolor =             gd.getNextString();
			long lcolor = -1;
			try {
				lcolor = Long.parseLong(scolor,16);
				this.anaglyph_right = setLongColor(lcolor);
			} catch(NumberFormatException e){
				this.anaglyph_right = anaglyph_right_default;
			}
		}
		
		this.annotate_color =           gd.getNextBoolean();
		this.annotate_mono =            gd.getNextBoolean();
		{
			String scolor =             gd.getNextString();
			long lcolor = -1;
			try {
				lcolor = Long.parseLong(scolor,16);
				this.annotate_color_color = setLongColor(lcolor);
			} catch(NumberFormatException e){
				this.annotate_color_color = null;
			}
		}
		{
			String scolor =             gd.getNextString();
			long lcolor = -1;
			try {
				lcolor = Long.parseLong(scolor,16);
				this.annotate_color_mono = setLongColor(lcolor);
			} catch(NumberFormatException e){
				this.annotate_color_mono = null;
			}
		}
		this.annotate_transparent_mono= gd.getNextBoolean();
		
// vegetation
		terr_model_path =          gd.getNextString();
		terr_model_state =         gd.getNextString();
		terr_segments_dir =        gd.getNextString();
		terr_segments_suffix =     gd.getNextString();
		terr_par_dir =             gd.getNextString();
		terr_par_file =            gd.getNextString();
		terr_par_restore =         gd.getNextBoolean(); // boolean
		terr_step_restore =  (int) gd.getNextNumber();  // int
		
		terr_um_en =               gd.getNextBoolean(); // boolean
		terr_um_sigma =            gd.getNextNumber();  // double 
		terr_um_weight =           gd.getNextNumber();  // double 
		terr_nan_tolerance =       gd.getNextNumber();  // double 
		terr_nan_grow =      (int) gd.getNextNumber();  // int
		
		terr_shrink_veget =  (int) gd.getNextNumber();  // int
		terr_shrink_terrain =(int) gd.getNextNumber();  // int
		terr_vegetation_over =     gd.getNextNumber();  // double
		terr_filter_veget =  (int) gd.getNextNumber();  // int

		terr_veget_grow =    (int) gd.getNextNumber();  // int

		terr_tile_woi =            gd.getNextBoolean(); // boolean
		terr_continue =            gd.getNextBoolean(); // boolean
		terr_woi_enclos =          stringToRectangle(gd.getNextString());// Rectangle
		terr_woi_step =            stringToRectangle(gd.getNextString());// Rectangle
		terr_woi_last =            stringToRectangle(gd.getNextString());// Rectangle
		terr_skip_exist =          gd.getNextBoolean();// boolean
		
		terr_fit_terr =            gd.getNextBoolean();// boolean
		terr_fit_veget =           gd.getNextBoolean();// boolean
		terr_fit_alpha =           gd.getNextBoolean();// boolean
		terr_fit_elevations =      gd.getNextBoolean();// boolean
		terr_fit_terr_elev =       gd.getNextBoolean();// boolean
		terr_fit_terr_elev_pix =   gd.getNextBoolean();// boolean
		terr_fit_scenes =          gd.getNextBoolean();// boolean
		for (int i = 0; i < terr_fits_disable.length; i++) {
			terr_fits_disable[i] = StringToBooleans(gd.getNextString());// booleans
		}
		for (int i = 0; i < terr_only_fits_disable.length; i++) {
			terr_only_fits_disable[i] = StringToBooleans(gd.getNextString());// booleans
		}
		terr_max_warp =            gd.getNextNumber();// double 
		terr_max_elevation = (int) gd.getNextNumber();// int
		terr_max_elev_terr = (int) gd.getNextNumber();// int
		terr_max_elev_chng =       gd.getNextNumber();// double 
		
		terr_min_scenes =    (int) gd.getNextNumber();// int
		terr_min_samples_scene=(int) gd.getNextNumber();// int
		terr_min_total_scenes=(int) gd.getNextNumber();// int
		terr_min_pixels =    (int) gd.getNextNumber();// int
		terr_warm_veget =          gd.getNextBoolean();// boolean
		terr_warmest =             gd.getNextNumber();// double 
		terr_initial_split =       gd.getNextNumber();// double 
		terr_min_split_frac =      gd.getNextNumber();// double 
		terr_difference =          gd.getNextNumber();// double
		terr_pull_cold =           gd.getNextNumber();// double 
		terr_elevation_radius =    gd.getNextNumber();// double 
		terr_terr_elev_radius =    gd.getNextNumber();// double 
		terr_elev_radius_extra =   gd.getNextNumber();// double 
		
		terr_alpha_contrast =      gd.getNextNumber();// double 
		terr_alpha_dflt =          gd.getNextNumber();// double
		
        terr_alpha_sigma =         gd.getNextNumber();// double
        terr_alpha_init_min =      gd.getNextNumber();// double
        terr_alpha_init_max =      gd.getNextNumber();// double
        terr_alpha_init_offs =     gd.getNextNumber();// double
		
		terr_alpha_loss =          gd.getNextNumber();// double 
		terr_alpha_loss_lin =      gd.getNextNumber();// double 
		terr_alpha_offset =        gd.getNextNumber();// double 
		terr_alpha_0offset =       gd.getNextNumber();// double 
		terr_alpha_min_veg =       gd.getNextNumber();// double
		terr_alpha_max_terrain =   gd.getNextNumber();// double
		terr_alpha_pull_pwr =      gd.getNextNumber();// double
		
		terr_alpha_lpf =           gd.getNextNumber();// double 
		terr_alpha_lpf_border =    gd.getNextNumber();// double 
		terr_alpha_piece_linear =  gd.getNextBoolean();// boolean
		terr_alpha_scale_avg =     gd.getNextNumber();// double 
		terr_alpha_push =          gd.getNextNumber();// double 
		terr_alpha_push_neutral =  gd.getNextNumber();// double 
		terr_alpha_weight_center = gd.getNextNumber();// double 
		terr_en_holes =            gd.getNextBoolean();// boolean
		terr_alpha_mm_hole =       gd.getNextNumber();// double
		terr_alpha_diff_hole =     gd.getNextNumber();// double
		
		terr_terr_lpf =            gd.getNextNumber();// double 
		terr_veget_lpf =           gd.getNextNumber();// double
		terr_elev_lpf =            gd.getNextNumber();// double
		terr_terr_elev_lpf =       gd.getNextNumber();// double
		terr_terr_pull0 =          gd.getNextNumber();// double 
		terr_terr_pull_up =        gd.getNextNumber();// double 
		terr_terr_pull_avg =       gd.getNextNumber();// double 
		terr_veget_pull0 =         gd.getNextNumber();// double
		terr_veget_pull_low_alpha= gd.getNextNumber();// double
		terr_elev_pull0 =          gd.getNextNumber();// double
		terr_terr_elev_pull0 =     gd.getNextNumber();// double
		terr_elev_alpha_en =       gd.getNextBoolean();// boolean
		terr_elev_alpha =          gd.getNextNumber();// double
		terr_elev_alpha_pwr =      gd.getNextNumber();// double
		terr_low_veget =           gd.getNextNumber();// double
		terr_scenes_pull0 =        gd.getNextNumber();// double

		// scaling elevation losses for high elevations (decrease pull and/or lpf for high elevations)
		terr_elev_scale_thresh =   gd.getNextNumber();// double 
		terr_elev_scale_pull =     gd.getNextBoolean();// boolean
		terr_elev_scale_lpf =      gd.getNextBoolean();// boolean
		// tree-top removal
		terr_ttop_en =             gd.getNextBoolean();// boolean
		terr_ttop_gb =             gd.getNextNumber();// double 
		terr_ttop_min =            gd.getNextNumber();// double 
		terr_ttop_rel_lev =        gd.getNextNumber();// double 
		terr_ttop_rel_rad =        gd.getNextNumber();// double 
		terr_ttop_frac =           gd.getNextNumber();// double 
		terr_ttop_rem_rad =        gd.getNextNumber();// double 
		
		terr_boost_parallax =      gd.getNextNumber();// double 
		terr_max_parallax =        gd.getNextNumber();// double 
		terr_hifreq_weight =       gd.getNextNumber();// double
		terr_terr_corr =           gd.getNextNumber();// double
		terr_reg_weights =         gd.getNextNumber();// double 
		terr_lambda =              gd.getNextNumber();// double 
		terr_lambda_scale_good =   gd.getNextNumber();// double 
		terr_lambda_scale_bad =    gd.getNextNumber();// double 
		terr_lambda_max =          gd.getNextNumber();// double 
		terr_rms_diff =            gd.getNextNumber();// double 
		terr_last_series =   (int) gd.getNextNumber();// int    
		terr_num_iters =           StringToInts(gd.getNextString());
		
///		terr_only_special =        gd.getNextBoolean();// boolean
///		terr_only_pix =            gd.getNextBoolean();// boolean
		terr_only_series =   (int) gd.getNextNumber();// int    
		terr_only_num_iters =      StringToInts(gd.getNextString());
		terr_cholesky =            gd.getNextChoiceIndex();
		terr_recalc_weights =      StringToBooleans(gd.getNextString());// booleans
		terr_recalc_opaque =       gd.getNextNumber(); // double 
		terr_recalc_pedestal =     gd.getNextNumber(); // double 
		terr_recalc_frac =         gd.getNextNumber(); // double 
		terr_recalc_dist =         gd.getNextNumber(); // double 
		terr_recalc_pwr =          gd.getNextNumber(); // double 
		terr_recalc_gb =           gd.getNextNumber(); // double 
		terr_recalc_boost =        gd.getNextNumber(); // double 
		terr_recalc_average =      gd.getNextBoolean();// boolean
		
		terr_crop =                gd.getNextBoolean();// boolean
		terr_keep_partial =        gd.getNextBoolean();// boolean
		terr_border_width =  (int) gd.getNextNumber(); // int    
		terr_render_open =         gd.getNextBoolean();// boolean
		terr_render_no_alpha =     gd.getNextBoolean();// boolean
		terr_alpha_min =           gd.getNextNumber(); // double 
		terr_alpha_max =           gd.getNextNumber(); // double 
		terr_weight_opaque =       gd.getNextNumber(); // double
		terr_boost_render =        gd.getNextNumber(); // double
		terr_max_render =          gd.getNextNumber(); // double
		terr_num_exaggerate = (int)gd.getNextNumber(); // int    
		terr_rebuild_elev =        gd.getNextBoolean();// boolean
		terr_elev_grow =      (int)gd.getNextNumber(); // int    
		terr_threshold_terrain =   gd.getNextNumber();// double 
		terr_min_max_terrain =     gd.getNextNumber();// double 
		terr_min_terrain =         gd.getNextNumber();// double 
		terr_min_vegetation =      gd.getNextNumber();// double 
		terr_debug =         (int) gd.getNextNumber(); // int
		terr_debug_path =          gd.getNextString(); // String
		terr_debug_length =  (int) gd.getNextNumber(); // int
		terr_debug_improved =      gd.getNextBoolean();// boolean
		terr_debug_worsened =      gd.getNextBoolean();// boolean
// end of vegetation		
// Synthesis		
		synth_directory =          gd.getNextString(); // String
		synth_scene =              gd.getNextString(); // String
		synth_model =              gd.getNextString(); // String
		synth_segments =           gd.getNextString(); // String
		synth_render =             gd.getNextString(); // String
		synth_crop_real =          gd.getNextBoolean();// boolean
		synth_scene_offs =         gd.getNextString(); // String
		synth_add_offs =           gd.getNextBoolean();// boolean
		
		if (this.weight_zero_neibs > 1.0) this.weight_zero_neibs = 1.0;
		return;
	}
	
	public void setProperties(String prefix,Properties properties){
		properties.setProperty(prefix+"ims_use",                       this.ims_use + "");                     // boolean
		properties.setProperty(prefix+"ims_rebuild",                   this.ims_rebuild + "");                 // boolean
		properties.setProperty(prefix+"ims_offset",                    this.ims_offset+"");                    // double
		properties.setProperty(prefix+"gmt_plus",                      this.gmt_plus+"");                      // double
		properties.setProperty(prefix+"ims_ortho",                     IntersceneMatchParameters.doublesToString(this.ims_ortho));		
		properties.setProperty(prefix+"ims_mount_atr",                 IntersceneMatchParameters.doublesToString(this.ims_mount_atr));		
		properties.setProperty(prefix+"ims_mount_xyz",                 IntersceneMatchParameters.doublesToString(this.ims_mount_xyz));
		properties.setProperty(prefix+"pimu_gyr_offs",                 IntersceneMatchParameters.doublesToString(this.pimu_gyr_offs));
		properties.setProperty(prefix+"pimu_acc_offs",                 IntersceneMatchParameters.doublesToString(this.pimu_acc_offs));
		properties.setProperty(prefix+"ims_scale_xyz",                 IntersceneMatchParameters.doublesToString(this.ims_scale_xyz));
		properties.setProperty(prefix+"ims_scale_atr",                 IntersceneMatchParameters.doublesToString(this.ims_scale_atr));		
		properties.setProperty(prefix+"fmg_initial_en",                this.fmg_initial_en + "");              // boolean
		properties.setProperty(prefix+"fmg_reorient_en",               this.fmg_reorient_en + "");             // boolean
		properties.setProperty(prefix+"fmg_distance",                  this.fmg_distance+"");                  // double
		properties.setProperty(prefix+"fmg_max_quad",                  this.fmg_max_quad+"");                  // double
		properties.setProperty(prefix+"fmg_rectilinear",               this.fmg_rectilinear + "");             // boolean
		properties.setProperty(prefix+"sfm_use",                       this.sfm_use + "");                     // boolean
		properties.setProperty(prefix+"sfm_min_base",                  this.sfm_min_base+"");                  // double
		properties.setProperty(prefix+"sfm_min_gain",                  this.sfm_min_gain+"");                  // double
		properties.setProperty(prefix+"sfm_min_frac",                  this.sfm_min_frac+"");                  // double
		properties.setProperty(prefix+"sfm_num_pairs",                 this.sfm_num_pairs+"");                 // int
		properties.setProperty(prefix+"sfp_tolerance",                 this.sfp_tolerance+"");                 // double
		properties.setProperty(prefix+"sfm_readjust",                  this.sfm_readjust+"");                  // int
		properties.setProperty(prefix+"sfm_prev_frac",                 this.sfm_prev_frac+"");                 // double
		properties.setProperty(prefix+"sfm_same_weight",               this.sfm_same_weight+"");               // double
		properties.setProperty(prefix+"sfm_shrink",                    this.sfm_shrink+"");                    // int
		properties.setProperty(prefix+"sfm_fade_sigma",                this.sfm_fade_sigma+"");                // double
		properties.setProperty(prefix+"sfm_min_str1",                  this.sfm_min_str1+"");                  // double
		properties.setProperty(prefix+"sfm_min_str16",                 this.sfm_min_str16+"");                 // double
		properties.setProperty(prefix+"sfm_use_neibs",                 this.sfm_use_neibs+"");                 // boolean
		properties.setProperty(prefix+"sfm_neib_too_str1",             this.sfm_neib_too_str1+"");             // double
		properties.setProperty(prefix+"sfm_neib_too_str16",            this.sfm_neib_too_str16+"");            // double
		properties.setProperty(prefix+"sfm_neib_str1",                 this.sfm_neib_str1 +"");                // double
		properties.setProperty(prefix+"sfm_neib_str16",                this.sfm_neib_str16 +"");               // double
		properties.setProperty(prefix+"sfm_extrap_steps",              this.sfm_extrap_steps+"");              // int
		properties.setProperty(prefix+"sfm_extrap_radius",             this.sfm_extrap_radius+"");             // int
		properties.setProperty(prefix+"sfm_average_neibs",             this.sfm_average_neibs + "");           // boolean
		properties.setProperty(prefix+"sfm_fill_weak",                 this.sfm_fill_weak + "");               // boolean
		properties.setProperty(prefix+"sfm_extrapolate",               this.sfm_extrapolate + "");             // boolean
		
		properties.setProperty(prefix+"sfm_max_dip",                   this.sfm_max_dip +"");                  // double
		properties.setProperty(prefix+"sfm_max_bump",                  this.sfm_max_bump +"");                 // double
		properties.setProperty(prefix+"sfm_filter_nan",                this.sfm_filter_nan + "");              // boolean
		properties.setProperty(prefix+"sfm_only",                      this.sfm_only + "");                    // boolean
		
		properties.setProperty(prefix+"sfm_save_seq",                  this.sfm_save_seq + "");                // boolean
		properties.setProperty(prefix+"sfm_show_seq",                  this.sfm_show_seq + "");                // boolean
		properties.setProperty(prefix+"sfm_show_corr_ind",             this.sfm_show_corr_ind + "");           // boolean
		properties.setProperty(prefix+"sfm_show_corr",                 this.sfm_show_corr + "");               // boolean
		properties.setProperty(prefix+"omtch_img_set",                 this.omtch_img_set + "");              // String

		properties.setProperty(prefix+"rln_gpu_width",                 this.rln_gpu_width+"");                 // int
		properties.setProperty(prefix+"rln_gpu_height",                this.rln_gpu_height+"");                // int
		properties.setProperty(prefix+"rln_fat_zero",                  this.rln_fat_zero +"");                 // double
		properties.setProperty(prefix+"rln_use_neibs",                 this.rln_use_neibs + "");               // boolean
		properties.setProperty(prefix+"rln_neibs_fill",                this.rln_neibs_fill + "");              // boolean
		properties.setProperty(prefix+"rln_neib_radius",               this.rln_neib_radius +"");              // double
		properties.setProperty(prefix+"rln_radius_frac",               this.rln_radius_frac +"");              // double
		properties.setProperty(prefix+"rln_cent_radius",               this.rln_cent_radius +"");              // double
		properties.setProperty(prefix+"rln_n_recenter",                this.rln_n_recenter+"");                // int
		properties.setProperty(prefix+"rln_sngl_rstr",                 this.rln_sngl_rstr +"");                // double
		properties.setProperty(prefix+"rln_neib_rstr",                 this.rln_neib_rstr +"");                // double

		properties.setProperty(prefix+"pmtch_use_affine",              this.pmtch_use_affine +"");             // boolean
		properties.setProperty(prefix+"pmtch_frac_remove",             this.pmtch_frac_remove +"");            // double
		properties.setProperty(prefix+"pmtch_metric_err",              this.pmtch_metric_err +"");             // double
		properties.setProperty(prefix+"pmtch_max_std",                 this.pmtch_max_std +"");                // double
		properties.setProperty(prefix+"pmtch_min_std_rad",             this.pmtch_min_std_rad +"");            // double
		properties.setProperty(prefix+"pmtch_ignore_rms",              this.pmtch_ignore_rms +"");             // boolean
		properties.setProperty(prefix+"pmtch_num_iter",                this.pmtch_num_iter +"");               // int
		
		properties.setProperty(prefix+"pmtch_pull_skew",               this.pmtch_pull_skew +   "");           // double
		properties.setProperty(prefix+"pmtch_pull_tilt",               this.pmtch_pull_tilt +   "");           // double
		properties.setProperty(prefix+"pmtch_pull_scale",              this.pmtch_pull_scale +   "");          // double

		properties.setProperty(prefix+"pmtch_cent_rad",                this.pmtch_cent_rad +   "");            // double
		properties.setProperty(prefix+"pmtch_max_cent_rad",            this.pmtch_max_cent_rad +   "");        // double
		properties.setProperty(prefix+"pmtch_cent_fill",               this.pmtch_cent_fill +  "");            // double
		properties.setProperty(prefix+"pmtch_cent_final",              this.pmtch_cent_final + "");            // double
		properties.setProperty(prefix+"pmtch_ease_nosfm",              this.pmtch_ease_nosfm + "");            // double
		
		properties.setProperty(prefix+"ospir_existing",                this.ospir_existing +"");               // boolean
		properties.setProperty(prefix+"ospir_invert",                  this.ospir_invert +"");                 // boolean
		properties.setProperty(prefix+"ospir_step",                    this.ospir_step +"");                   // double
		properties.setProperty(prefix+"ospir_range",                   this.ospir_range +"");                  // double
		properties.setProperty(prefix+"ospir_double",                  this.ospir_double +"");                 // double
		properties.setProperty(prefix+"ospir_good_rms",                this.ospir_good_rms +"");               // double
		properties.setProperty(prefix+"ospir_max_rms",                 this.ospir_max_rms +"");                // double
		properties.setProperty(prefix+"ospir_overlap",                 this.ospir_overlap +"");                // int
		properties.setProperty(prefix+"ospir_num_iter",                this.ospir_num_iter +"");               // int
		properties.setProperty(prefix+"ospir_rms_iter_0",              this.ospir_rms_iter[0] +"");            // double
		properties.setProperty(prefix+"ospir_rms_iter_1",              this.ospir_rms_iter[1] +"");            // double
		
		properties.setProperty(prefix+"ospir_ignore_rms",              this.ospir_ignore_rms +"");             // boolean  
		properties.setProperty(prefix+"ospir_debug",                   this.ospir_debug +"");                  // int
		properties.setProperty(prefix+"ospir_augment",                 this.ospir_augment +"");                // boolean  
		
		properties.setProperty(prefix+"pwise_zoom",                    this.pwise_zoom + "");                  // int
		properties.setProperty(prefix+"pwise_overlap",                 this.pwise_overlap + "");               // double
		properties.setProperty(prefix+"pwise_max_rms",                 this.pwise_max_rms + "");               // double
				
		properties.setProperty(prefix+"pwise_skip_exist",              this.pwise_skip_exist + "");            // boolean
		properties.setProperty(prefix+"pwise_refine_exist",            this.pwise_refine_exist + "");          // boolean
		properties.setProperty(prefix+"pwise_delete_fail",             this.pwise_delete_fail + "");           // boolean
		properties.setProperty(prefix+"pwise_gen_inverse",             this.pwise_gen_inverse + "");           // boolean
		properties.setProperty(prefix+"pwise_save_each",               this.pwise_save_each + "");             // boolean
		properties.setProperty(prefix+"pwise_log_append",              this.pwise_log_append + "");            // boolean
		properties.setProperty(prefix+"pwise_log_path",                this.pwise_log_path + "");              // String
		properties.setProperty(prefix+"pwise_debug",                   this.pwise_debug + "");                 // int
		
		properties.setProperty(prefix+"alt_overwrite",                 this.alt_overwrite + "");               // boolean
		properties.setProperty(prefix+"alt_pairwise",                  this.alt_pairwise + "");                // boolean
		properties.setProperty(prefix+"alt_sigma",                     this.alt_sigma + "");                   // double
		properties.setProperty(prefix+"alt_abs_outliers",              this.alt_abs_outliers + "");            // double
		properties.setProperty(prefix+"alt_outliers",                  this.alt_outliers + "");                // double
		properties.setProperty(prefix+"alt_refine",                    this.alt_refine + "");                  // int
		
		properties.setProperty(prefix+"alt_weight_rot",                this.alt_weight_rot + "");              // double
		properties.setProperty(prefix+"alt_weight_tilt",               this.alt_weight_tilt + "");             // double
		properties.setProperty(prefix+"alt_weight_scale",              this.alt_weight_scale + "");            // double
		properties.setProperty(prefix+"alt_pull",                      this.alt_pull + "");                    // double
		properties.setProperty(prefix+"alt_pull_rots",                 this.alt_pull_rots + "");               // double
		properties.setProperty(prefix+"alt_pull_tilts  ",              this.alt_pull_tilts   + "");            // double
		properties.setProperty(prefix+"alt_pull_scales",               this.alt_pull_scales + "");             // double
		
		properties.setProperty(prefix+"pmap_move_only",                this.pmap_move_only + "");              // boolean
		properties.setProperty(prefix+"pmap_ignore_affines",           this.pmap_ignore_affines + "");         // boolean
		properties.setProperty(prefix+"pmap_use_inv",                  this.pmap_use_inv + "");                // boolean
		properties.setProperty(prefix+"pmap_skew_pull",                this.pmap_skew_pull + "");              // double
		properties.setProperty(prefix+"pmap_tilt_pull",                this.pmap_tilt_pull + "");              // double
		properties.setProperty(prefix+"pmap_scale_pull",               this.pmap_scale_pull + "");             // double
		properties.setProperty(prefix+"pmap_position_pull",            this.pmap_position_pull + "");          // double
		properties.setProperty(prefix+"pmap_overlap_pow",              this.pmap_overlap_pow + "");            // double
		properties.setProperty(prefix+"pmap_rms_diff",                 this.pmap_rms_diff + "");               // double
		properties.setProperty(prefix+"pmap_num_iter",                 this.pmap_num_iter + "");               // int
		properties.setProperty(prefix+"pmap_show_result",              this.pmap_show_result + "");            // boolean
		properties.setProperty(prefix+"pmap_debugLevel",               this.pmap_debugLevel + "");             // int

		properties.setProperty(prefix+"pequ_ignore_equalize",          this.pequ_ignore_equalize + "");        // boolean
		properties.setProperty(prefix+"pequ_use_inv",                  this.pequ_use_inv + "");                // boolean
		properties.setProperty(prefix+"pequ_scale_weight",             this.pequ_scale_weight + "");           // double
		properties.setProperty(prefix+"pequ_pull_weight",              this.pequ_pull_weight + "");            // double
		properties.setProperty(prefix+"pequ_half_weight_sec",          this.pequ_half_weight_sec + "");        // double
		properties.setProperty(prefix+"pequ_min_weight_sec ",          this.pequ_min_weight_sec  + "");        // double
		properties.setProperty(prefix+"pequ_overlap_pow",              this.pequ_overlap_pow + "");            // double
		properties.setProperty(prefix+"pequ_rms_diff",                 this.pequ_rms_diff + "");               // double
		properties.setProperty(prefix+"pequ_num_iter",                 this.pequ_num_iter + "");               // int
		properties.setProperty(prefix+"pequ_debugLevel",               this.pequ_debugLevel + "");             // int
		
		properties.setProperty(prefix+"flt_list",                      this.flt_list + "");                    // boolean
		properties.setProperty(prefix+"flt_undef_only",                this.flt_undef_only + "");              // boolean
		properties.setProperty(prefix+"flt_min_overlap",               this.flt_min_overlap + "");             // double
		properties.setProperty(prefix+"flt_max_overlap",               this.flt_max_overlap + "");             // double
		properties.setProperty(prefix+"flt_min_rms",                   this.flt_min_rms + "");                 // double
		properties.setProperty(prefix+"flt_max_rms",                   this.flt_max_rms + "");                 // double
		properties.setProperty(prefix+"flt_nan_rms",                   this.flt_nan_rms + "");                 // boolean
		properties.setProperty(prefix+"flt_filt_zoom",                 this.flt_filt_zoom + "");               // boolean
		properties.setProperty(prefix+"flt_min_zoom",                  this.flt_min_zoom + "");                // int
		properties.setProperty(prefix+"flt_max_zoom",                  this.flt_max_zoom + "");                // int
		properties.setProperty(prefix+"flt_min_sfm",                   this.flt_min_sfm + "");                 // double
		properties.setProperty(prefix+"flt_max_sfm",                   this.flt_max_sfm + "");                 // double
		properties.setProperty(prefix+"flt_alt",                       this.flt_alt + "");                     // int
		properties.setProperty(prefix+"flt_orient",                    this.flt_orient + "");                  // int
		properties.setProperty(prefix+"flt_show_names",                this.flt_show_names + "");              // boolean
		properties.setProperty(prefix+"flt_show_overlaps",             this.flt_show_overlaps + "");           // boolean
		properties.setProperty(prefix+"flt_show_rms",                  this.flt_show_rms + "");                // boolean
		properties.setProperty(prefix+"flt_show_zoom",                 this.flt_show_zoom + "");               // boolean
		properties.setProperty(prefix+"flt_show_alt",                  this.flt_show_alt + "");                // boolean
		
		properties.setProperty(prefix+"patt_save_top",                 this.patt_save_top + "");               // String
		properties.setProperty(prefix+"patt_save_subdir",              this.patt_save_subdir + "");            // String

		properties.setProperty(prefix+"center_reference",              this.center_reference + "");            // boolean
		properties.setProperty(prefix+"lock_position",                 this.lock_position + "");               // boolean
		properties.setProperty(prefix+"cuas_rotation",                 this.cuas_rotation + "");               // boolean
		properties.setProperty(prefix+"manual_correction",             this.manual_correction + "");           // boolean
		properties.setProperty(prefix+"overlap_sequences",             this.overlap_sequences + "");           // boolean
		properties.setProperty(prefix+"reset_photometric",             this.reset_photometric + "");           // boolean
		properties.setProperty(prefix+"force_ref_dsi",                 this.force_ref_dsi + "");               // boolean
		properties.setProperty(prefix+"force_orientations",            this.force_orientations + "");          // boolean
		properties.setProperty(prefix+"run_ly",                        this.run_ly + "");                      // boolean
		properties.setProperty(prefix+"run_ly_mode",                   this.run_ly_mode+"");                   // int
		properties.setProperty(prefix+"run_ly_lma",                    this.run_ly_lma + "");                  // boolean
		properties.setProperty(prefix+"min_num_orient",                this.min_num_orient+"");                // int
		properties.setProperty(prefix+"min_num_interscene",            this.min_num_interscene+"");            // int
		properties.setProperty(prefix+"adjust_imu_orient",             this.adjust_imu_orient+"");             // boolean
		properties.setProperty(prefix+"apply_imu_orient",              this.apply_imu_orient+"");              // boolean
		properties.setProperty(prefix+"orient_by_move",                this.orient_by_move+"");                // boolean
		properties.setProperty(prefix+"orient_by_rot",                 this.orient_by_rot+"");                 // boolean
		properties.setProperty(prefix+"orient_combo",                  this.orient_combo+"");                  // boolean
		properties.setProperty(prefix+"adjust_gyro",                   this.adjust_gyro+"");                   // boolean
		properties.setProperty(prefix+"apply_gyro",                    this.apply_gyro+"");                    // boolean
		properties.setProperty(prefix+"adjust_accl",                   this.adjust_accl+"");                   // boolean
		properties.setProperty(prefix+"apply_accl",                    this.apply_accl+"");                    // boolean
		properties.setProperty(prefix+"calc_quat_corr",                this.calc_quat_corr+"");                // boolean
		properties.setProperty(prefix+"apply_quat_corr",               this.apply_quat_corr+"");               // boolean
		properties.setProperty(prefix+"use_quat_corr",                 this.use_quat_corr+"");                 // boolean
		properties.setProperty(prefix+"inertial_only",                 this.inertial_only+"");                 // boolean
		properties.setProperty(prefix+"generate_egomotion",            this.generate_egomotion+"");            // boolean
		properties.setProperty(prefix+"generate_mapped",               this.generate_mapped+"");               // boolean
		properties.setProperty(prefix+"reuse_video",                   this.reuse_video+"");                   // boolean
		properties.setProperty(prefix+"save_mapped_color",             this.save_mapped_color+"");             // boolean
		properties.setProperty(prefix+"save_mapped_mono",              this.save_mapped_mono+"");              // boolean
		properties.setProperty(prefix+"gen_avi_color",                 this.gen_avi_color+"");       // boolean
		properties.setProperty(prefix+"gen_avi_mono",                  this.gen_avi_mono+"");        // boolean
		properties.setProperty(prefix+"show_mapped_color",    this.show_mapped_color+"");   // boolean
		properties.setProperty(prefix+"show_mapped_mono",     this.show_mapped_mono+"");    // boolean
		
		properties.setProperty(prefix+"generate_raw",                  this.generate_raw+"");                  // boolean
		properties.setProperty(prefix+"generate_inf",                  this.generate_inf+"");                  // boolean
		properties.setProperty(prefix+"generate_fg",                   this.generate_fg+"");                   // boolean
		properties.setProperty(prefix+"generate_bg",                   this.generate_bg+"");                   // boolean
		properties.setProperty(prefix+"generate_stereo",               this.generate_stereo+"");               // boolean
		/*
		properties.setProperty(prefix+"stereo_bases_num",              this.stereo_bases.length+"");           // int
		for (int i = 0; i < this.stereo_bases.length; i++) {
			properties.setProperty(prefix+"stereo_bases_"+i,           this.stereo_bases[i]+"");               // double
			properties.setProperty(prefix+"generate_stereo_var_"+i,    this.generate_stereo_var[i]+"");        // boolean
		}
	   */
		properties.setProperty(prefix+"stereo_views_num",                  this.stereo_views.length+"");           // int
		for (int i = 0; i < this.stereo_views.length; i++) {
			properties.setProperty(prefix+"stereo_views_"+i,           doublesToString(this.stereo_views[i],"%.0f")); // String
			properties.setProperty(prefix+"generate_stereo_var_"+i,    this.generate_stereo_var[i]+"");        // boolean
		}
		
		
		
		properties.setProperty(prefix+"export_images",                 this.export_images + "");               // boolean
		properties.setProperty(prefix+"show_images",                   this.show_images + "");                 // boolean
		properties.setProperty(prefix+"show_images_bgfg",              this.show_images_bgfg + "");            // boolean
		properties.setProperty(prefix+"show_images_mono",              this.show_images_mono + "");            // boolean
		properties.setProperty(prefix+"export_ranges",                 this.export_ranges + "");               // boolean
		properties.setProperty(prefix+"debug_ranges",                  this.debug_ranges + "");                // boolean
		properties.setProperty(prefix+"show_ranges",                   this.show_ranges + "");                 // boolean
		properties.setProperty(prefix+"export_ml_files",               this.export_ml_files + "");             // boolean
		
		properties.setProperty(prefix+"show_color_nan",                this.show_color_nan + "");              // boolean
		properties.setProperty(prefix+"show_mono_nan",                 this.show_mono_nan + "");               // boolean
		properties.setProperty(prefix+"scale_extrap_atr",              this.scale_extrap_atr+"");              // double
		properties.setProperty(prefix+"scale_extrap_xyz",              this.scale_extrap_xyz+"");              // double
		properties.setProperty(prefix+"avg_len",                       this.avg_len+"");                       // int
		properties.setProperty(prefix+"min_num_scenes",                this.min_num_scenes+"");                // int
		properties.setProperty(prefix+"max_num_scenes",                this.max_num_scenes+"");                // int
		properties.setProperty(prefix+"blur_egomotion",                this.blur_egomotion+"");                // double
		properties.setProperty(prefix+"range_disparity_offset",        this.range_disparity_offset+"");        // double
		properties.setProperty(prefix+"range_min_strength",            this.range_min_strength+"");            // double
		properties.setProperty(prefix+"range_max",                     this.range_max+"");                     // double
		properties.setProperty(prefix+"export3d",                      this.export3d+"");                      // boolean
		properties.setProperty(prefix+"export3dterrain",               this.export3dterrain+"");               // boolean
		
		properties.setProperty(prefix+"export_CT",                     this.export_CT+"");                     // boolean
		properties.setProperty(prefix+"ct_min",                        this.ct_min+"");                        // double
		properties.setProperty(prefix+"ct_max",                        this.ct_max+"");                        // double
		properties.setProperty(prefix+"ct_step",                       this.ct_step+"");                       // double
		properties.setProperty(prefix+"ct_expand",                     this.ct_expand+"");                     // int
		
//		properties.setProperty(prefix+"terr_enable",                   this.terr_enable+"");                   // boolean
		properties.setProperty(prefix+"terr_rmse_above",               this.terr_rmse_above+"");               // double
		properties.setProperty(prefix+"terr_rmse_below",               this.terr_rmse_below+"");               // double
		properties.setProperty(prefix+"terr_num_refine",               this.terr_num_refine+"");               // int
		properties.setProperty(prefix+"terr_frac_above",               this.terr_frac_above+"");               // double
		properties.setProperty(prefix+"terr_frac_below",               this.terr_frac_below+"");               // double
		properties.setProperty(prefix+"terr_rthrsh_abv",               this.terr_rthrsh_abv+"");               // double
		properties.setProperty(prefix+"terr_flat_brd",                 this.terr_flat_brd+"");                 // boolean
		properties.setProperty(prefix+"terr_expand",                   this.terr_expand+"");                   // int
		properties.setProperty(prefix+"terr_save",                     this.terr_save+"");                     // boolean
		
		properties.setProperty(prefix+"scene_is_ref_test",             this.scene_is_ref_test+"");   // boolean
		properties.setProperty(prefix+"render_ref",                    this.render_ref+"");          // boolean
		properties.setProperty(prefix+"render_scene",                  this.render_scene+"");        // boolean
		properties.setProperty(prefix+"toRGB",                         this.toRGB+"");               // boolean
		properties.setProperty(prefix+"show_2d_correlations",          this.show_2d_correlations+"");// boolean
		properties.setProperty(prefix+"show_motion_vectors",           this.show_motion_vectors+""); // boolean
		properties.setProperty(prefix+"debug_level",                   this.debug_level+"");         // int
		properties.setProperty(prefix+"test_ers",                      this.test_ers+"");            // boolean
		properties.setProperty(prefix+"test_ers0",                     this.test_ers0+"");           // int
		properties.setProperty(prefix+"test_ers1",                     this.test_ers1+"");           // int
	
		properties.setProperty(prefix+"num_bottom",                    this.num_bottom+"");                    // int
		properties.setProperty(prefix+"num_passes",                    this.num_passes+"");                    // int
		properties.setProperty(prefix+"max_change",                    this.max_change+"");                    // double
		properties.setProperty(prefix+"min_disparity",                 this.min_disparity+"");                 // double
		properties.setProperty(prefix+"max_sym_disparity",             this.max_sym_disparity+"");             // double
		properties.setProperty(prefix+"min_strength_lma",              this.min_strength_lma+"");          // double
		properties.setProperty(prefix+"min_strength_replace",          this.min_strength_replace+"");          // double
		properties.setProperty(prefix+"min_strength_blur",             this.min_strength_blur+"");             // double
		properties.setProperty(prefix+"sigma",                         this.sigma+"");                         // double
		properties.setProperty(prefix+"num_blur",                      this.num_blur+"");                      // int
		properties.setProperty(prefix+"disparity_corr",                this.disparity_corr+"");                // double
		properties.setProperty(prefix+"outliers_nth_fromextrem",       this.outliers_nth_fromextrem+"");       // int
		properties.setProperty(prefix+"outliers_tolerance_absolute",   this.outliers_tolerance_absolute+"");   // double
		properties.setProperty(prefix+"outliers_tolerance_relative",   this.outliers_tolerance_relative+"");   // double
		properties.setProperty(prefix+"outliers_max_iter",             this.outliers_max_iter+"");             // int
		properties.setProperty(prefix+"outliers_max_strength2",        this.outliers_max_strength2+"");        // double
		properties.setProperty(prefix+"outliers_nth_fromextrem2",      this.outliers_nth_fromextrem2+"");      // int
		properties.setProperty(prefix+"outliers_tolerance_absolute2",  this.outliers_tolerance_absolute2+"");  // double
		properties.setProperty(prefix+"outliers_tolerance_relative2",  this.outliers_tolerance_relative2+"");  // double
		properties.setProperty(prefix+"outliers_lma_max_strength",     this.outliers_lma_max_strength+"");     // double
		properties.setProperty(prefix+"outliers_max_strength",         this.outliers_max_strength+"");         // double
		properties.setProperty(prefix+"outliers_from_lma_max_strength",this.outliers_from_lma_max_strength+"");// double
		properties.setProperty(prefix+"search_radius",                 this.search_radius+"");                 // int
		properties.setProperty(prefix+"remove_no_lma_neib",            this.remove_no_lma_neib+"");            // boolean
		properties.setProperty(prefix+"diff_from_lma_pos",             this.diff_from_lma_pos+"");             // double
		properties.setProperty(prefix+"diff_from_lma_neg",             this.diff_from_lma_neg+"");             // double
		properties.setProperty(prefix+"outliers_lma_nth_fromextrem",   this.outliers_lma_nth_fromextrem+"");   // int
		properties.setProperty(prefix+"filter_margin",                 this.filter_margin+"");                 // int
		properties.setProperty(prefix+"weak_tolerance_absolute",       this.weak_tolerance_absolute+"");       // double
		properties.setProperty(prefix+"weak_tolerance_relative",       this.weak_tolerance_relative+"");       // double
		properties.setProperty(prefix+"weak_min_neibs",                this.weak_min_neibs+"");                // int
		properties.setProperty(prefix+"strong_strength",               this.strong_strength+"");               // double
		properties.setProperty(prefix+"weak_strength",                 this.weak_strength+"");                 // double

		properties.setProperty(prefix+"sky_extract",          this.sky_extract+"");         // boolean
		properties.setProperty(prefix+"sky_recalc",           this.sky_recalc+"");          // boolean
		properties.setProperty(prefix+"sky_highest_min",      this.sky_highest_min+"");     // double
		properties.setProperty(prefix+"cold_frac",            this.cold_frac+"");           // double
		properties.setProperty(prefix+"hot_frac",             this.hot_frac+"");            // double
		properties.setProperty(prefix+"cold_scale",           this.cold_scale+"");          // double
		properties.setProperty(prefix+"sky_seed",             this.sky_seed+"");            // double
		properties.setProperty(prefix+"lma_seed",             this.lma_seed+"");            // double
		properties.setProperty(prefix+"seed_temp",            this.seed_temp+"");           // double
		properties.setProperty(prefix+"sky_shrink",           this.sky_shrink+"");          // int
		properties.setProperty(prefix+"seed_rows",            this.seed_rows+"");           // int
		properties.setProperty(prefix+"max_disparity",        this.max_disparity+"");       // double
		properties.setProperty(prefix+"max_disparity_strength",this.max_disparity_strength+"");// double
		properties.setProperty(prefix+"sky_lim",              this.sky_lim+"");             // double
		properties.setProperty(prefix+"lim_temp",             this.lim_temp+"");            // double
		properties.setProperty(prefix+"sky_expand_extra",     this.sky_expand_extra+"");    // int
		properties.setProperty(prefix+"sky_bottleneck",       this.sky_bottleneck+"");      // int
		properties.setProperty(prefix+"sky_reexpand_extra",   this.sky_reexpand_extra+"");  // int
		properties.setProperty(prefix+"min_strength",         this.min_strength+"");        // double
		properties.setProperty(prefix+"lowest_sky_row",       this.lowest_sky_row+"");      // int
		properties.setProperty(prefix+"sky_bottom_override",  this.sky_bottom_override+""); // double
		properties.setProperty(prefix+"sky_override_shrink",  this.sky_override_shrink+""); // int
		
		properties.setProperty(prefix+"clouds_en",            this.clouds_en+"");           // boolean
		properties.setProperty(prefix+"clouds_fom",           this.clouds_fom+"");          // double
		properties.setProperty(prefix+"clouds_spread",        this.clouds_spread+"");       // double
		properties.setProperty(prefix+"clouds_disparity",     this.clouds_disparity+"");    // double
		properties.setProperty(prefix+"clouds_weak",          this.clouds_weak+"");         // double
		properties.setProperty(prefix+"clouds_strength",      this.clouds_strength+"");     // double
		properties.setProperty(prefix+"clouds_not_strength",  this.clouds_not_strength+""); // double
		properties.setProperty(prefix+"clouds_strong",        this.clouds_strong+"");       // boolean
		
		properties.setProperty(prefix+"wall_en",              this.wall_en+"");             // boolean
		properties.setProperty(prefix+"wall_dflt",            this.wall_dflt+"");           // boolean
		properties.setProperty(prefix+"wall_str",             this.wall_str+"");            // double
		properties.setProperty(prefix+"wall_far",             this.wall_far+"");            // double
		properties.setProperty(prefix+"wall_near",            this.wall_near+"");           // double

		properties.setProperty(prefix+"treeline_en",          this.treeline_en+"");         // boolean
		properties.setProperty(prefix+"treeline_wide",        this.treeline_wide+"");       // boolean
		properties.setProperty(prefix+"treeline_height",      this.treeline_height+"");     // int
		properties.setProperty(prefix+"treeline_width",       this.treeline_width+"");      // int
		properties.setProperty(prefix+"treeline_lim_high",    this.treeline_lim_high+"");   // boolean
		properties.setProperty(prefix+"treeline_str",         this.treeline_str+"");        // double
		properties.setProperty(prefix+"treeline_far",         this.treeline_far+"");        // double
		properties.setProperty(prefix+"treeline_near",        this.treeline_near+"");       // double
		properties.setProperty(prefix+"treeline_fg_str",      this.treeline_fg_str+"");     // double
		properties.setProperty(prefix+"treeline_fg_far",      this.treeline_fg_far+"");     // double
		properties.setProperty(prefix+"treeline_fg_near",     this.treeline_fg_near+"");    // double
		
		properties.setProperty(prefix+"indoors_en",           this.indoors_en+"");          // boolean
		properties.setProperty(prefix+"indoors_str",          this.indoors_str+"");         // double
		properties.setProperty(prefix+"indoors_disp",         this.indoors_disp+"");        // double
		properties.setProperty(prefix+"indoors_min_out",      this.indoors_min_out+"");     // int
		
		properties.setProperty(prefix+"disp_boost_min",       this.disp_boost_min+"");      // double
		properties.setProperty(prefix+"disp_boost_diff",      this.disp_boost_diff+"");     // double
		properties.setProperty(prefix+"disp_boost_neibs",     this.disp_boost_neibs+"");    // int
		properties.setProperty(prefix+"disp_boost_amount",    this.disp_boost_amount+"");   // double
		properties.setProperty(prefix+"scale_combo_strength", this.scale_combo_strength+"");// double
		properties.setProperty(prefix+"save_debug_images",    this.save_debug_images+"");   // boolean
		
		properties.setProperty(prefix+"margin",               this.margin+"");              // int
		properties.setProperty(prefix+"sensor_mask_inter",    this.sensor_mask_inter+"");   // int
		properties.setProperty(prefix+"use_partial",          this.use_partial+"");         // boolean
		properties.setProperty(prefix+"run_poly",             this.run_poly+"");            // boolean
		properties.setProperty(prefix+"centroid_radius",      this.centroid_radius+"");     // double
		properties.setProperty(prefix+"n_recenter",           this.n_recenter+"");          // int
		properties.setProperty(prefix+"td_weight",            this.td_weight+"");           // double
		properties.setProperty(prefix+"td_neib_weight",       this.td_neib_weight+"");      // double
		properties.setProperty(prefix+"pd_weight",            this.pd_weight+"");           // double
		properties.setProperty(prefix+"td_nopd_only",         this.td_nopd_only+"");        // boolean
		properties.setProperty(prefix+"neib_notd_only",       this.neib_notd_only+"");      // boolean
		
		properties.setProperty(prefix+"use_neibs",            this.use_neibs+"");           // boolean
		properties.setProperty(prefix+"neibs_nofpn_only",     this.neibs_nofpn_only+"");    // boolean
		properties.setProperty(prefix+"neibs_nofpn_init",     this.neibs_nofpn_init+"");    // boolean
		properties.setProperty(prefix+"redo_both",            this.redo_both+"");           // boolean
		properties.setProperty(prefix+"min_num_neibs",        this.min_num_neibs+"");       // int
        properties.setProperty(prefix+"scale_neibs_pd",       this.scale_neibs_pd+"");      // double
		properties.setProperty(prefix+"scale_neibs_td",       this.scale_neibs_td+"");      // double
		properties.setProperty(prefix+"scale_avg_weight",     this.scale_avg_weight+"");    // double
		
		properties.setProperty(prefix+"min_str",              this.min_str+"");             // double
		properties.setProperty(prefix+"min_str_fpn",          this.min_str_fpn+"");         // double
		properties.setProperty(prefix+"min_str_sum",          this.min_str_sum+"");         // double
		properties.setProperty(prefix+"min_str_sum_fpn",      this.min_str_sum_fpn+"");     // double
		properties.setProperty(prefix+"min_str_neib",         this.min_str_neib+"");        // double
		properties.setProperty(prefix+"min_str_neib_fpn",     this.min_str_neib_fpn+"");    // double
		
		properties.setProperty(prefix+"min_neibs",            this.min_neibs+"");           // int
		properties.setProperty(prefix+"weight_zero_neibs",    this.weight_zero_neibs+"");   // double
		properties.setProperty(prefix+"half_disparity",       this.half_disparity+"");      // double
		properties.setProperty(prefix+"half_avg_diff",        this.half_avg_diff+"");       // double
		
		properties.setProperty(prefix+"eig_use",              this.eig_use+"");             // boolean
		properties.setProperty(prefix+"eig_xy_lma",           this.eig_xy_lma+"");          // boolean
		properties.setProperty(prefix+"eig_str_sum",          this.eig_str_sum+"");         // double
		properties.setProperty(prefix+"eig_str_sum_fpn",      this.eig_str_sum_fpn+"");     // double
		properties.setProperty(prefix+"eig_str_neib",         this.eig_str_neib+"");        // double
		properties.setProperty(prefix+"eig_str_neib_fpn",     this.eig_str_neib_fpn+"");    // double
		properties.setProperty(prefix+"eig_min_abs",          this.eig_min_abs+"");         // double
		properties.setProperty(prefix+"eig_min_rel",          this.eig_min_rel+"");         // double
		properties.setProperty(prefix+"eig_sub_frac",         this.eig_sub_frac+"");        // double
		properties.setProperty(prefix+"eig_max_sqrt",         this.eig_max_sqrt+"");        // double
		properties.setProperty(prefix+"eig_min_sqrt",         this.eig_min_sqrt+"");        // double
		
		properties.setProperty(prefix+"eig_recenter",         this.eig_recenter+"");        // int
		properties.setProperty(prefix+"eig_sub_frac1",        this.eig_sub_frac1+"");       // double
		properties.setProperty(prefix+"eig_scale_axes",       this.eig_scale_axes+"");      // double
		properties.setProperty(prefix+"eig_inc_axes",         this.eig_inc_axes+"");        // double
		
		properties.setProperty(prefix+"eig_use_neibs",        this.eig_use_neibs+"");       // boolean
		properties.setProperty(prefix+"eig_min_weaks",        this.eig_min_weaks+"");       // int
		properties.setProperty(prefix+"eig_min_strongs",      this.eig_min_strongs+"");     // int
		properties.setProperty(prefix+"eig_disp_diff",        this.eig_disp_diff+"");       // double
		
		properties.setProperty(prefix+"eig_remove_neibs",     this.eig_remove_neibs+"");    // boolean
		properties.setProperty(prefix+"eig_filt_other",       this.eig_filt_other+"");      // boolean
		properties.setProperty(prefix+"eig_max_rms",          this.eig_max_rms+"");         // double
		properties.setProperty(prefix+"eig_fast2x2",          this.eig_fast2x2+"");         // boolean
		
		properties.setProperty(prefix+"use_combo_reliable",   this.use_combo_reliable+"");  // boolean
		properties.setProperty(prefix+"ref_need_lma",         this.ref_need_lma+"");        // boolean
		properties.setProperty(prefix+"ref_need_lma_combo",   this.ref_need_lma_combo+"");  // boolean
		properties.setProperty(prefix+"min_ref_str",          this.min_ref_str+"");         // double
		properties.setProperty(prefix+"min_ref_str_lma",      this.min_ref_str_lma+"");     // double
		properties.setProperty(prefix+"min_ref_frac",         this.min_ref_frac+"");        // double
		
		properties.setProperty(prefix+"ref_smooth",           this.ref_smooth+"");          // boolean
		properties.setProperty(prefix+"ref_smooth_always",    this.ref_smooth_always+"");   // boolean
		properties.setProperty(prefix+"ref_sigma",            this.ref_sigma+"");           // double
		properties.setProperty(prefix+"ref_smooth_diff",      this.ref_smooth_diff+"");     // double
		
		properties.setProperty(prefix+"sfm_filter",           this.sfm_filter+"");          // boolean
		properties.setProperty(prefix+"sfm_minmax",           this.sfm_minmax+"");          // double
		properties.setProperty(prefix+"sfm_fracmax",          this.sfm_fracmax+"");         // double
		properties.setProperty(prefix+"sfm_fracall",          this.sfm_fracall+"");         // double
		
		
		properties.setProperty(prefix+"pix_step",             this.pix_step+"");            // int
		properties.setProperty(prefix+"search_rad",           this.search_rad+"");          // int
		properties.setProperty(prefix+"maybe_sum",            this.maybe_sum+"");           // double
		properties.setProperty(prefix+"sure_sum",             this.sure_sum+"");            // double
		properties.setProperty(prefix+"maybe_avg",            this.maybe_avg+"");           // double
		properties.setProperty(prefix+"sure_avg",             this.sure_avg+"");            // double
		properties.setProperty(prefix+"max_search_rms",       this.max_search_rms+"");      // double
		properties.setProperty(prefix+"maybe_fom",            this.maybe_fom+"");           // double
		properties.setProperty(prefix+"sure_fom",             this.sure_fom+"");            // double
		properties.setProperty(prefix+"treat_serch_fpn",        this.treat_serch_fpn+"");   // boolean
		properties.setProperty(prefix+"use_combo_dsi",        this.use_combo_dsi+"");       // boolean
		properties.setProperty(prefix+"use_lma_dsi",          this.use_lma_dsi+"");         // boolean
		properties.setProperty(prefix+"fpn_remove",           this.fpn_remove+"");          // boolean
		properties.setProperty(prefix+"fpn_max_offset",       this.fpn_max_offset+"");      // double
		properties.setProperty(prefix+"fpn_min_offset",       this.fpn_min_offset+"");      // double
		properties.setProperty(prefix+"fpn_radius",           this.fpn_radius+"");          // double
		properties.setProperty(prefix+"fpn_ignore_border",    this.fpn_ignore_border+"");   // boolean
		
		properties.setProperty(prefix+"min_offset",           this.min_offset+"");          // double
		properties.setProperty(prefix+"max_pull_jump",        this.max_pull_jump+"");       // double
		properties.setProperty(prefix+"max_rel_offset",       this.max_rel_offset+"");      // double
		properties.setProperty(prefix+"max_roll_deg",         this.max_roll_deg+"");        // double
		properties.setProperty(prefix+"max_zoom_diff",        this.max_zoom_diff+"");       // double
		properties.setProperty(prefix+"fpn_skip",             this.fpn_skip+"");            // boolean
		properties.setProperty(prefix+"fpn_rematch",          this.fpn_rematch+"");         // boolean
		properties.setProperty(prefix+"refine_invert",        this.refine_invert+"");       // boolean
		properties.setProperty(prefix+"use_precomp",          this.use_precomp+"");         // boolean

		properties.setProperty(prefix+"mov_en",               this.mov_en+"");              // boolean
		properties.setProperty(prefix+"mov_sigma",            this.mov_sigma+"");           // double
		properties.setProperty(prefix+"mov_max_std",          this.mov_max_std+"");         // double
		properties.setProperty(prefix+"mov_thresh_rel",       this.mov_thresh_rel+"");      // double
		properties.setProperty(prefix+"mov_thresh_abs",       this.mov_thresh_abs+"");      // double
		properties.setProperty(prefix+"mov_clust_max",        this.mov_clust_max+"");       // double
		properties.setProperty(prefix+"mov_grow",             this.mov_grow+"");            // int
		properties.setProperty(prefix+"mov_max_len",          this.mov_max_len+"");         // int
		
		properties.setProperty(prefix+"mov_show",             this.mov_show+"");            // boolean
		properties.setProperty(prefix+"mov_debug_level",      this.mov_debug_level+"");     // int
		properties.setProperty(prefix+"adjust_atr_0",         this.adjust_atr[0]+"");       // boolean
		properties.setProperty(prefix+"adjust_atr_1",         this.adjust_atr[1]+"");       // boolean
		properties.setProperty(prefix+"adjust_atr_2",         this.adjust_atr[2]+"");       // boolean
		properties.setProperty(prefix+"adjust_xyz_0",         this.adjust_xyz[0]+"");       // boolean
		properties.setProperty(prefix+"adjust_xyz_1",         this.adjust_xyz[1]+"");       // boolean
		properties.setProperty(prefix+"adjust_xyz_2",         this.adjust_xyz[2]+"");       // boolean
		properties.setProperty(prefix+"exit_change_atr",      this.exit_change_atr+"");     // double
		properties.setProperty(prefix+"exit_change_xyz",      this.exit_change_xyz+"");     // double
		properties.setProperty(prefix+"max_cycles",           this.max_cycles+"");          // int
		properties.setProperty(prefix+"max_LMA",              this.max_LMA+"");             // int
		properties.setProperty(prefix+"max_rms",              this.max_rms+"");             // double
	
		properties.setProperty(prefix+"eq_en",                this.eq_en+"");               // boolean
		properties.setProperty(prefix+"eq_stride_hor",        this.eq_stride_hor+"");       // int
		properties.setProperty(prefix+"eq_stride_vert",       this.eq_stride_vert+"");      // int
		properties.setProperty(prefix+"eq_min_stile_weight",  this.eq_min_stile_weight+""); // double
		properties.setProperty(prefix+"eq_min_stile_number",  this.eq_min_stile_number+""); // int
		properties.setProperty(prefix+"eq_min_stile_fraction",this.eq_min_stile_fraction+"");// double
		properties.setProperty(prefix+"eq_min_disparity",     this.eq_min_disparity+"");    // double
		properties.setProperty(prefix+"eq_max_disparity",     this.eq_max_disparity+"");    // double
		properties.setProperty(prefix+"eq_weight_add",        this.eq_weight_add+"");       // double
		properties.setProperty(prefix+"eq_weight_scale",      this.eq_weight_scale+"");     // double
		properties.setProperty(prefix+"eq_level",             this.eq_level+"");            // double

		properties.setProperty(prefix+"mb_en",                this.mb_en+"");               // boolean
		properties.setProperty(prefix+"mb_tau",               this.mb_tau+"");              // double
		properties.setProperty(prefix+"mb_max_gain",          this.mb_max_gain+"");         // double
		properties.setProperty(prefix+"mb_max_gain_inter",    this.mb_max_gain_inter+"");   // double
		
		properties.setProperty(prefix+"cuas_update_existing", this.cuas_update_existing+"");// boolean
		properties.setProperty(prefix+"cuas_discard_border",  this.cuas_discard_border+""); // int
		properties.setProperty(prefix+"cuas_max_fold",        this.cuas_max_fold+"");       // double
		properties.setProperty(prefix+"cuas_min_in_row_col",  this.cuas_min_in_row_col+""); // int
		properties.setProperty(prefix+"cuas_clt_variant",     this.cuas_clt_variant+"");    // double
		properties.setProperty(prefix+"cuas_clt_threshold",   this.cuas_clt_threshold+"");  // double
		properties.setProperty(prefix+"cuas_clt_decrease",    this.cuas_clt_decrease+"");   // double
		properties.setProperty(prefix+"cuas_decay_average",   this.cuas_decay_average+"");  // double
		properties.setProperty(prefix+"cuas_keep_fraction",   this.cuas_keep_fraction+"");  // double
		properties.setProperty(prefix+"cuas_step",            this.cuas_step+"");           // boolean

		properties.setProperty(prefix+"cuas_subtract_fpn",    this.cuas_subtract_fpn+"");   // boolean
		properties.setProperty(prefix+"cuas_calc_fpn",        this.cuas_calc_fpn+"");       // boolean
		properties.setProperty(prefix+"cuas_rot_period",      this.cuas_rot_period+"");     // double
		properties.setProperty(prefix+"cuas_min_series",      this.cuas_min_series+"");     // double
		
		properties.setProperty(prefix+"cuas_subtract_rowcol", this.cuas_subtract_rowcol+"");// boolean
		properties.setProperty(prefix+"cuas_calc_rowcol",     this.cuas_calc_rowcol+"");    // boolean
		properties.setProperty(prefix+"cuas_um_sigma",        this.cuas_um_sigma+"");       // double
		properties.setProperty(prefix+"cuas_max_abs_rowcol",  this.cuas_max_abs_rowcol+""); // double
		properties.setProperty(prefix+"cuas_outliers_rowcol", this.cuas_outliers_rowcol+"");// double
		properties.setProperty(prefix+"cuas_reset_first",     this.cuas_reset_first+"");    // boolean

		properties.setProperty(prefix+"cuas_invert_margins",  this.cuas_invert_margins+""); // int
		properties.setProperty(prefix+"cuas_invert_iters",    this.cuas_invert_iters+"");   // int
		properties.setProperty(prefix+"cuas_invert_tolerance",this.cuas_invert_tolerance+"");// double
		properties.setProperty(prefix+"cuas_invert_gap2",     this.cuas_invert_gap2+"");    // int
		
		properties.setProperty(prefix+"cuas_w_um_sigma",      this.cuas_w_um_sigma+"");     // double
		properties.setProperty(prefix+"cuas_w_good_margins",  this.cuas_w_good_margins+""); // int
		properties.setProperty(prefix+"cuas_w_threshold",     this.cuas_w_threshold+"");    // double
		properties.setProperty(prefix+"cuas_w_scale_thresh",  this.cuas_w_scale_thresh+""); // double
		properties.setProperty(prefix+"cuas_w_expand_max",    this.cuas_w_expand_max+"");   // boolean
		properties.setProperty(prefix+"cuas_w_outliers",      this.cuas_w_outliers+"");     // double
		properties.setProperty(prefix+"cuas_w_blur",          this.cuas_w_blur+"");         // double
		
		properties.setProperty(prefix+"cuas_dsi_parent",      this.cuas_dsi_parent+"");     // boolean
		properties.setProperty(prefix+"cuas_fg_strength",     this.cuas_fg_strength+"");    // double
		properties.setProperty(prefix+"cuas_disp_add",        this.cuas_disp_add+"");       // double
		properties.setProperty(prefix+"cuas_skip_bottom",     this.cuas_skip_bottom+"");    // int
		properties.setProperty(prefix+"cuas_multi_strength",  this.cuas_multi_strength+""); // double
		properties.setProperty(prefix+"cuas_reliable_str",    this.cuas_reliable_str+"");   // double
		
		properties.setProperty(prefix+"cuas_fat_zero",        this.cuas_fat_zero+"");       // double
		properties.setProperty(prefix+"cuas_cent_radius",     this.cuas_cent_radius+"");    // double
		properties.setProperty(prefix+"cuas_n_recenter",      this.cuas_n_recenter+"");     // int
		properties.setProperty(prefix+"cuas_rstr",            this.cuas_rstr+"");           // double
        properties.setProperty(prefix+"cuas_smooth",          this.cuas_smooth+"");         // boolean
        properties.setProperty(prefix+"cuas_corr_pairs",      this.cuas_corr_pairs+"");     // int
		properties.setProperty(prefix+"cuas_corr_offset",     this.cuas_corr_offset+"");    // int
        properties.setProperty(prefix+"cuas_gaussian_ra",     this.cuas_gaussian_ra+"");    // boolean
		properties.setProperty(prefix+"cuas_temporal_um",     this.cuas_temporal_um+"");    // int
		properties.setProperty(prefix+"cuas_tum_threshold",   this.cuas_tum_threshold+"");  // double
		
		properties.setProperty(prefix+"cuas_precorr_ra",      this.cuas_precorr_ra+"");     // int
		properties.setProperty(prefix+"cuas_corr_step",       this.cuas_corr_step+"");      // int
		
        properties.setProperty(prefix+"cuas_half_step",       this.cuas_half_step+"");      // boolean
		properties.setProperty(prefix+"cuas_max_range",       this.cuas_max_range+"");      // int
		properties.setProperty(prefix+"cuas_num_cycles",      this.cuas_num_cycles+"");     // int

		properties.setProperty(prefix+"cuas_speed_min",       this.cuas_speed_min+"");      // double
		properties.setProperty(prefix+"cuas_speed_pref",      this.cuas_speed_pref+"");     // double
		properties.setProperty(prefix+"cuas_speed_boost",     this.cuas_speed_boost+"");    // double
		properties.setProperty(prefix+"cuas_lmax_flt_neglim", this.cuas_lmax_flt_neglim+"");// double
		properties.setProperty(prefix+"cuas_lmax_flt_hsigma", this.cuas_lmax_flt_hsigma+"");// double
		properties.setProperty(prefix+"cuas_lmax_flt_lsigma", this.cuas_lmax_flt_lsigma+"");// double
		properties.setProperty(prefix+"cuas_lmax_flt_scale",  this.cuas_lmax_flt_scale+""); // double
		properties.setProperty(prefix+"cuas_sky_threshold",   this.cuas_sky_threshold+"");  // double
  		properties.setProperty(prefix+"cuas_sky_offset",   IntersceneMatchParameters.doublesToString(this.cuas_sky_offset));		
		properties.setProperty(prefix+"cuas_lmax_fraction",   this.cuas_lmax_fraction+"");  // double
		properties.setProperty(prefix+"cuas_lmax_hack_ridge", this.cuas_lmax_hack_ridge+"");// double
		properties.setProperty(prefix+"cuas_lmax_radius",     this.cuas_lmax_radius+"");    // double
        properties.setProperty(prefix+"cuas_lmax_zero",       this.cuas_lmax_zero+"");      // boolean

		properties.setProperty(prefix+"cuas_centered_radius", this.cuas_centered_radius+"");// double
		properties.setProperty(prefix+"cuas_centered_blur",   this.cuas_centered_blur+"");  // double
		properties.setProperty(prefix+"cuas_duplicate_tolerance",this.cuas_duplicate_tolerance+"");  // double
        
		properties.setProperty(prefix+"cuas_target_radius",   this.cuas_target_radius+"");  // double
		properties.setProperty(prefix+"cuas_target_strength", this.cuas_target_strength+"");// double
		properties.setProperty(prefix+"cuas_target_frac", double2dToString(cuas_target_frac)+"");  // double[][]
        properties.setProperty(prefix+"cuas_no_border",       this.cuas_no_border+"");      // boolean
		
		properties.setProperty(prefix+"cuas_lma_sigma",       this.cuas_lma_sigma+"");      // double
		properties.setProperty(prefix+"cuas_wnd_pedestal",    this.cuas_wnd_pedestal+"");      // double
		properties.setProperty(prefix+"cuas_lma_r0",          this.cuas_lma_r0+"");         // double
		properties.setProperty(prefix+"cuas_lma_ovrsht",      this.cuas_lma_ovrsht+"");     // double
        properties.setProperty(prefix+"cuas_lma_fit_xy",      this.cuas_lma_fit_xy+"");     // boolean
        properties.setProperty(prefix+"cuas_lma_fit_a",       this.cuas_lma_fit_a+"");      // boolean
        properties.setProperty(prefix+"cuas_lma_fit_c",       this.cuas_lma_fit_c+"");      // boolean
        properties.setProperty(prefix+"cuas_lma_fit_r",       this.cuas_lma_fit_r+"");      // boolean
        properties.setProperty(prefix+"cuas_lma_fit_k",       this.cuas_lma_fit_k+"");      // boolean
		properties.setProperty(prefix+"cuas_lambda",          this.cuas_lambda+"");         // double
		properties.setProperty(prefix+"cuas_lambda_good",     this.cuas_lambda_good+"");    // double
		properties.setProperty(prefix+"cuas_lambda_bad",      this.cuas_lambda_bad+"");     // double
		properties.setProperty(prefix+"cuas_lambda_max",      this.cuas_lambda_max+"");     // double
		properties.setProperty(prefix+"cuas_rms_diff",        this.cuas_rms_diff+"");       // double
		properties.setProperty(prefix+"cuas_num_iter",        this.cuas_num_iter+"");       // int
		properties.setProperty(prefix+"cuas_lma_rms",         this.cuas_lma_rms+"");        // double
		properties.setProperty(prefix+"cuas_lma_arms",        this.cuas_lma_arms+"");       // double
		properties.setProperty(prefix+"cuas_lma_rrms",        this.cuas_lma_rrms+"");       // double
		properties.setProperty(prefix+"cuas_lma_mina",        this.cuas_lma_mina+"");       // double
		properties.setProperty(prefix+"cuas_lma_pre_mina",    this.cuas_lma_pre_mina+"");   // double
		properties.setProperty(prefix+"cuas_min_keep",        this.cuas_min_keep+"");       // int
		
		properties.setProperty(prefix+"cuas_lma_maxr",        this.cuas_lma_maxr+"");       // double
		properties.setProperty(prefix+"cuas_lma_minr1",       this.cuas_lma_minr1+"");      // double
		properties.setProperty(prefix+"cuas_lma_mink",        this.cuas_lma_mink+"");       // double
		properties.setProperty(prefix+"cuas_lma_maxk",        this.cuas_lma_maxk+"");       // double
		properties.setProperty(prefix+"cuas_lma_a2a",         this.cuas_lma_a2a+"");        // double
		
		properties.setProperty(prefix+"cuas_slow_ra",         this.cuas_slow_ra+"");        // double
		properties.setProperty(prefix+"cuas_slow_score",      this.cuas_slow_score+"");     // double

		properties.setProperty(prefix+"cuas_score_mv",        this.cuas_score_mv+"");       // double
		properties.setProperty(prefix+"cuas_score_lma",       this.cuas_score_lma+"");      // double
		properties.setProperty(prefix+"cuas_factor_lim",      this.cuas_factor_lim+"");     // double
		properties.setProperty(prefix+"cuas_factor_pow",      this.cuas_factor_pow+"");     // double
		properties.setProperty(prefix+"cuas_score_coeff",     IntersceneMatchParameters.doublesToString(this.cuas_score_coeff));		
		properties.setProperty(prefix+"cuas_center_targ",     this.cuas_center_targ+"");    // boolean
		properties.setProperty(prefix+"cuas_multi_targ",      this.cuas_multi_targ+"");     // boolean
		
		
		properties.setProperty(prefix+"cuas_isolated",        this.cuas_isolated+"");       // boolean
//		properties.setProperty(prefix+"cuas_remove_2seq",     this.cuas_remove_2seq+"");    // boolean
//		properties.setProperty(prefix+"cuas_remove_2seq_all", this.cuas_remove_2seq_all+"");// boolean
		properties.setProperty(prefix+"cuas_min_seq",         this.cuas_min_seq+"");        // int
		properties.setProperty(prefix+"cuas_enough_seq",      this.cuas_enough_seq+"");     // int
		properties.setProperty(prefix+"cuas_seq_travel",      this.cuas_seq_travel+"");     // double
		properties.setProperty(prefix+"cuas_min_seq_fl",      this.cuas_min_seq_fl+"");     // int
		
		properties.setProperty(prefix+"cuas_max_mismatch",    this.cuas_max_mismatch+"");   // double
		properties.setProperty(prefix+"cuas_good_mismatch",   this.cuas_good_mismatch+"");  // double
		properties.setProperty(prefix+"cuas_slow_fast_mismatch",this.cuas_slow_fast_mismatch+"");// double
		properties.setProperty(prefix+"cuas_match_len_pwr",   this.cuas_match_len_pwr+"");  // double
		
		properties.setProperty(prefix+"cuas_fail_mismatch",   this.cuas_fail_mismatch+"");  // boolean
		properties.setProperty(prefix+"cuas_ignore_mismatch", this.cuas_ignore_mismatch+"");// boolean
		properties.setProperty(prefix+"cuas_by_horizon",      this.cuas_by_horizon+"");     // boolean
		properties.setProperty(prefix+"cuas_horizon",         this.cuas_horizon+"");        // double
		
		properties.setProperty(prefix+"cuas_mask_width",      this.cuas_mask_width+"");     // double
		properties.setProperty(prefix+"cuas_mask_blur",       this.cuas_mask_blur+"");      // double
		properties.setProperty(prefix+"cuas_mask_round",      this.cuas_mask_round+"");     // boolean
		properties.setProperty(prefix+"cuas_target_type",     this.cuas_target_type+"");    // int
		properties.setProperty(prefix+"cuas_known_type",      this.cuas_known_type+"");     // int
		
		properties.setProperty(prefix+"cuas_input_range",     this.cuas_input_range+"");    // double
		properties.setProperty(prefix+"cuas_scale2x",         this.cuas_scale2x+"");        // boolean
		properties.setProperty(prefix+"cuas_ra_background",   this.cuas_ra_background+"");  // boolean

		properties.setProperty(prefix+"cuas_targets",         this.cuas_targets+"");        // boolean
		properties.setProperty(prefix+"cuas_iter_show",       this.cuas_iter_show+"");      // int
		properties.setProperty(prefix+"cuas_2d_save_show",    this.cuas_2d_save_show+"");   // boolean
		properties.setProperty(prefix+"cuas_intermed_low",    this.cuas_intermed_low+"");   // boolean
		properties.setProperty(prefix+"cuas_intermed_high",   this.cuas_intermed_high+"");  // boolean
		properties.setProperty(prefix+"cuas_intermed_giga",   this.cuas_intermed_giga+"");  // boolean
		properties.setProperty(prefix+"cuas_debug_more",      this.cuas_debug_more+"");     // boolean
		
		properties.setProperty(prefix+"cuas_generate",        this.cuas_generate+"");       // boolean
		properties.setProperty(prefix+"cuas_generate_csv",    this.cuas_generate_csv+"");   // boolean
		properties.setProperty(prefix+"cuas_save_mono",       this.cuas_save_mono+"");      // boolean
		properties.setProperty(prefix+"cuas_save_color",      this.cuas_save_color+"");     // boolean
		properties.setProperty(prefix+"cuas_save_video",      this.cuas_save_video+"");     // boolean
		properties.setProperty(prefix+"cuas_clean_video",     this.cuas_clean_video+"");    // boolean
		
		
		properties.setProperty(prefix+"cuas_annotate",        this.cuas_annotate+"");       // boolean
		properties.setProperty(prefix+"cuas_annotate_uas",    this.cuas_annotate_uas+"");   // boolean
		{
			long lcolor_annotate =        (cuas_text_color == null) ? 0x00FF00 : getLongColor(this.cuas_text_color);
			properties.setProperty(prefix+"cuas_text_color", lcolor_annotate+"");
		}
		{
			long lcolor_annotate =        (cuas_text_bg == null) ? 0x777777 : getLongColor(this.cuas_text_bg);
			properties.setProperty(prefix+"cuas_text_bg", lcolor_annotate+"");
		}
		{
			long lcolor_annotate =        (cuas_selected_color == null) ? 0xFF0088 : getLongColor(this.cuas_selected_color);
			properties.setProperty(prefix+"cuas_selected_color", lcolor_annotate+"");
		}

		properties.setProperty(prefix+"cuas_transparent",     this.cuas_transparent+"");    // boolean
		properties.setProperty(prefix+"cuas_transparent_uas", this.cuas_transparent_uas+"");    // boolean
		properties.setProperty(prefix+"cuas_font_name",       this.cuas_font_name+"");      // String
		properties.setProperty(prefix+"cuas_font_size",       this.cuas_font_size+"");      // int
		properties.setProperty(prefix+"cuas_font_type",       this.cuas_font_type+"");      // int
		properties.setProperty(prefix+"cuas_font_spacing",    this.cuas_font_spacing+"");   // double
		properties.setProperty(prefix+"cuas_annot_missing",   this.cuas_annot_missing+"");  // boolean
		properties.setProperty(prefix+"cuas_px0",             this.cuas_px0+"");            // int
		properties.setProperty(prefix+"cuas_py0",             this.cuas_py0+"");            // int
		properties.setProperty(prefix+"cuas_az0",             this.cuas_az0+"");            // double
		properties.setProperty(prefix+"cuas_el0",             this.cuas_el0+"");            // double
		properties.setProperty(prefix+"cuas_show_disp",       this.cuas_show_disp+"");      // boolean
		properties.setProperty(prefix+"cuas_show_rng",        this.cuas_show_rng+"");       // boolean
		properties.setProperty(prefix+"cuas_show_inf",        this.cuas_show_inf+"");       // boolean
		properties.setProperty(prefix+"cuas_show_inf_gt",     this.cuas_show_inf_gt+"");    // boolean
		properties.setProperty(prefix+"cuas_show_true_rng",   this.cuas_show_true_rng+"");  // boolean
		
		properties.setProperty(prefix+"cuas_smooth_omegas",   this.cuas_smooth_omegas+"");  // boolean
		properties.setProperty(prefix+"cuas_rng_um",          this.cuas_rng_um+"");         // boolean
		properties.setProperty(prefix+"cuas_rng_um_all",      this.cuas_rng_um_all+"");     // boolean
		properties.setProperty(prefix+"cuas_rng_um_sigma",    this.cuas_rng_um_sigma+"");   // double
		properties.setProperty(prefix+"cuas_rng_um2",         this.cuas_rng_um2+"");        // boolean
		properties.setProperty(prefix+"cuas_rng_um_weight",   this.cuas_rng_um_weight+"");  // double
		properties.setProperty(prefix+"cuas_rng_coswnd",      this.cuas_rng_coswnd+"");     // boolean
		
		properties.setProperty(prefix+"cuas_rng_combine",     this.cuas_rng_combine+"");    // int
		
		properties.setProperty(prefix+"cuas_rng_radius0",     this.cuas_rng_radius0+"");    // double
		properties.setProperty(prefix+"cuas_rng_radius",      this.cuas_rng_radius+"");     // double
		properties.setProperty(prefix+"cuas_rng_blur",        this.cuas_rng_blur+"");       // double
		properties.setProperty(prefix+"cuas_rng_fz0",         this.cuas_rng_fz0+"");        // double
		properties.setProperty(prefix+"cuas_rng_fz",          this.cuas_rng_fz+"");         // double
		properties.setProperty(prefix+"cuas_rng_scale",       this.cuas_rng_scale+"");      // double
		
		properties.setProperty(prefix+"cuas_dual_pass",       this.cuas_dual_pass+"");      // boolean
		properties.setProperty(prefix+"cuas_disp_only2",      this.cuas_disp_only2+"");     // boolean
		properties.setProperty(prefix+"cuas_lma_bypass",      this.cuas_lma_bypass+"");     // boolean
		properties.setProperty(prefix+"cuas_mcorr_sel",       this.cuas_mcorr_sel+"");      // int
		properties.setProperty(prefix+"cuas_mcorr_sel_lma",   this.cuas_mcorr_sel_lma+"");  // int
		
		properties.setProperty(prefix+"cuas_max_rel_rms",     this.cuas_max_rel_rms+"");    // double
		properties.setProperty(prefix+"cuas_min_strength",    this.cuas_min_strength+"");   // double
		properties.setProperty(prefix+"cuas_ac_offset",       this.cuas_ac_offset+"");      // double
		properties.setProperty(prefix+"cuas_min_max_ac",      this.cuas_min_max_ac+"");     // double
		properties.setProperty(prefix+"cuas_min_min_ac",      this.cuas_min_min_ac+"");     // double
		properties.setProperty(prefix+"cuas_reset_disparity", this.cuas_reset_disparity+"");// boolean
		properties.setProperty(prefix+"cuas_initial_disparity",this.cuas_initial_disparity+"");// double
		properties.setProperty(prefix+"cuas_infinity",        this.cuas_infinity+"");       // double
		
		properties.setProperty(prefix+"cuas_rng_img",         this.cuas_rng_img+"");        // boolean
		properties.setProperty(prefix+"cuas_rng_glob",        this.cuas_rng_glob+"");       // boolean
		properties.setProperty(prefix+"cuas_glob_ends",       this.cuas_glob_ends+"");      // int
		properties.setProperty(prefix+"cuas_rng_disp",        this.cuas_rng_disp+"");       // boolean
		properties.setProperty(prefix+"cuas_rng_vfy",         this.cuas_rng_vfy+"");        // boolean
		properties.setProperty(prefix+"cuas_rng_niterate",    this.cuas_rng_niterate+"");   // int
		properties.setProperty(prefix+"cuas_rng_diff",        this.cuas_rng_diff+"");       // double

		properties.setProperty(prefix+"cuas_flw_levels",      this.cuas_flw_levels+"");     // int
		properties.setProperty(prefix+"cuas_flw_len",         this.cuas_flw_len+"");        // int
		properties.setProperty(prefix+"cuas_flw_diff",        this.cuas_flw_diff+"");       // double

		properties.setProperty(prefix+"cuas_known_err",       this.cuas_known_err+"");      // double
		properties.setProperty(prefix+"cuas_tmtch_frac",      this.cuas_tmtch_frac+"");     // double
		properties.setProperty(prefix+"cuas_tmtch_ends",      this.cuas_tmtch_ends+"");     // int
		properties.setProperty(prefix+"cuas_tmtch_gaps",      this.cuas_tmtch_gaps+"");     // int
		properties.setProperty(prefix+"cuas_tmtch_pix",       this.cuas_tmtch_pix+"");      // double
		properties.setProperty(prefix+"cuas_local_apix",      this.cuas_local_apix+"");     // double
		properties.setProperty(prefix+"cuas_tmtch_apix",      this.cuas_tmtch_apix+"");     // double
		properties.setProperty(prefix+"cuas_tmtch_rpix",      this.cuas_tmtch_rpix+"");     // double
		properties.setProperty(prefix+"cuas_tmtch_axv",       this.cuas_tmtch_axv+"");      // double
		properties.setProperty(prefix+"cuas_tmtch_axv_k",     this.cuas_tmtch_axv_k+"");    // double
		properties.setProperty(prefix+"cuas_local_diff",      this.cuas_local_diff+"");     // double
		properties.setProperty(prefix+"cuas_tmtch_diff",      this.cuas_tmtch_diff+"");     // double
		properties.setProperty(prefix+"cuas_tmtch_disp",      this.cuas_tmtch_disp+"");     // double
		properties.setProperty(prefix+"cuas_tmtch_short",     this.cuas_tmtch_short+"");    // boolean
		
		properties.setProperty(prefix+"cuas_radar_range",     this.cuas_radar_range+"");    // double
		
		properties.setProperty(prefix+"cuas_radar_height",    this.cuas_radar_height+"");   // int
		properties.setProperty(prefix+"cuas_radar_bottom",    this.cuas_radar_bottom+"");   // int
		properties.setProperty(prefix+"cuas_radar_top",       this.cuas_radar_top+"");      // int
		properties.setProperty(prefix+"cuas_radar_uas_color", getStringColor(this.cuas_radar_uas_color, DEFAULT_cuas_radar_uas_color)); // Color
		properties.setProperty(prefix+"cuas_radar_color",     getStringColor(this.cuas_radar_color,     DEFAULT_cuas_radar_color));     // Color
		properties.setProperty(prefix+"cuas_radar_radius",    this.cuas_radar_radius+"");   // double
		properties.setProperty(prefix+"cuas_radar_uas_rad",   this.cuas_radar_uas_rad+"");  // double
		properties.setProperty(prefix+"cuas_grid_ring",       this.cuas_grid_ring+"");      // double
		properties.setProperty(prefix+"cuas_grid_line",       this.cuas_grid_line+"");      // double
		properties.setProperty(prefix+"cuas_grid_annot",      this.cuas_grid_annot+"");     // boolean
		properties.setProperty(prefix+"cuas_grid_color",      getStringColor(this.cuas_grid_color,      DEFAULT_cuas_grid_color));      // Color
		properties.setProperty(prefix+"cuas_grid_font",       this.cuas_grid_font+"");      // int
		properties.setProperty(prefix+"cuas_grid_az_top",     this.cuas_grid_az_top+"");    // int
		properties.setProperty(prefix+"cuas_grid_sides",      this.cuas_grid_sides+"");     // int
  		properties.setProperty(prefix+"cuas_annot_sel",   IntersceneMatchParameters.intsToString(this.cuas_annot_sel));		
		properties.setProperty(prefix+"cuas_debug",           this.cuas_debug+"");          // boolean
		properties.setProperty(prefix+"cuas_dbg_rng_seq",     this.cuas_dbg_rng_seq+"");    // int
		properties.setProperty(prefix+"cuas_dbg_rng_tgt",     this.cuas_dbg_rng_tgt+"");    // int
		properties.setProperty(prefix+"cuas_dbg_show_mode",   this.cuas_dbg_show_mode+"");  // int
		
		properties.setProperty(prefix+"cuas_step_debug",      this.cuas_step_debug+"");     // boolean
		properties.setProperty(prefix+"cuas_save_stats",      this.cuas_save_stats+"");     // boolean
		properties.setProperty(prefix+"cuas_target_debug",    this.cuas_target_debug+"");   // boolean
		properties.setProperty(prefix+"cuas_target_score",    this.cuas_target_score+"");   // boolean
		properties.setProperty(prefix+"cuas_target_color",    this.cuas_target_color+"");   // boolean
		properties.setProperty(prefix+"cuas_log_ranging",     this.cuas_log_ranging+"");    // boolean
		
		properties.setProperty(prefix+"cuas_overwrite",       this.cuas_overwrite+"");      // boolean
		properties.setProperty(prefix+"cuas_num_orient",      this.cuas_num_orient+"");     // int
		properties.setProperty(prefix+"cuas_num_accum",       this.cuas_num_accum+"");      // int
		
		properties.setProperty(prefix+"cuas_reuse_targets",   this.cuas_reuse_targets+"");  // boolean
		properties.setProperty(prefix+"cuas_reuse_path",      this.cuas_reuse_path+"");     // String
		
		properties.setProperty(prefix+"cuas_reuse_disparity", this.cuas_reuse_disparity+"");// boolean
		properties.setProperty(prefix+"cuas_reuse_globals",   this.cuas_reuse_globals+"");  // boolean
		properties.setProperty(prefix+"cuas_max_disp_diff",   this.cuas_max_disp_diff+"");  // double
		properties.setProperty(prefix+"cuas_min_disp_str",    this.cuas_min_disp_str+"");   // double
		properties.setProperty(prefix+"cuas_rng_limit",       this.cuas_rng_limit+"");      // double
		
		properties.setProperty(prefix+"mb_gain_index_pose",   this.mb_gain_index_pose+"");  // int
		properties.setProperty(prefix+"mb_gain_index_depth",  this.mb_gain_index_depth+""); // int

		properties.setProperty(prefix+"mb_ers_index",         this.mb_ers_index+"");        // int
		properties.setProperty(prefix+"mb_ers_y_index",       this.mb_ers_y_index+"");      // int
		properties.setProperty(prefix+"mb_ers_r_index",       this.mb_ers_r_index+"");      // int
		properties.setProperty(prefix+"mb_all_index",         this.mb_all_index+"");        // int
		properties.setProperty(prefix+"mb_pref_orient",       this.mb_pref_orient+"");      // boolean
		properties.setProperty(prefix+"lma_use_Z",            this.lma_use_Z+"");           // boolean
		properties.setProperty(prefix+"lma_use_R",            this.lma_use_R+"");           // boolean
		
		properties.setProperty(prefix+"configured_lma",       this.configured_lma+"");      // boolean
		properties.setProperty(prefix+"freeze_xy_pull",       this.freeze_xy_pull+"");      // boolean
		properties.setProperty(prefix+"copy_pull_current",    this.copy_pull_current+"");   // boolean
		properties.setProperty(prefix+"restore_imu",          this.restore_imu+"");         // boolean
		properties.setProperty(prefix+"lpf_xy",               this.lpf_xy+"");              // boolean
		properties.setProperty(prefix+"readjust_xy_ims",      this.readjust_xy_ims+"");     // boolean
		properties.setProperty(prefix+"reg_weight_xy",        this.reg_weight_xy+"");       // double
		
		properties.setProperty(prefix+"quat_lambda",          this.quat_lambda+"");         // double
		properties.setProperty(prefix+"quat_lambda_scale_good",this.quat_lambda_scale_good+"");// double
		properties.setProperty(prefix+"quat_lambda_scale_bad",this.quat_lambda_scale_bad+"");// double
		properties.setProperty(prefix+"quat_lambda_max",      this.quat_lambda_max+"");     // double
		properties.setProperty(prefix+"quat_rms_diff",        this.quat_rms_diff+"");       // double
		properties.setProperty(prefix+"quat_num_iter",        this.quat_num_iter+"");       // int
		properties.setProperty(prefix+"quat_reg_w",           this.quat_reg_w+"");          // double
		
		properties.setProperty(prefix+"quat_max_ratio",       this.quat_max_ratio+"");      // double
		properties.setProperty(prefix+"quat_max_change",      this.quat_max_change+"");     // double
		properties.setProperty(prefix+"quat_min_transl",      this.quat_min_transl+"");     // double
		properties.setProperty(prefix+"quat_min_rot",         this.quat_min_rot+"");        // double
		properties.setProperty(prefix+"quat_min_lin",         this.quat_min_lin+"");        // double
		
		properties.setProperty(prefix+"stereo_merge",         this.stereo_merge+"");        // boolean
		properties.setProperty(prefix+"stereo_gap",           this.stereo_gap+"");          // int
		properties.setProperty(prefix+"stereo_intereye",      this.stereo_intereye+"");     // double
		properties.setProperty(prefix+"stereo_phone_width",   this.stereo_phone_width+"");  // double
		properties.setProperty(prefix+"extra_hor_tile",       this.extra_hor_tile+"");      // int
		properties.setProperty(prefix+"extra_vert_tile",      this.extra_vert_tile+"");     // int
		properties.setProperty(prefix+"crop_3d",              this.crop_3d+"");             // boolean
		properties.setProperty(prefix+"sensor_mask",          this.sensor_mask+"");         // int
		properties.setProperty(prefix+"merge_all",            this.merge_all+"");           // boolean
		properties.setProperty(prefix+"video_fps",            this.video_fps+"");           // double
		properties.setProperty(prefix+"sensor_fps",           this.sensor_fps+"");          // double
		properties.setProperty(prefix+"mode_avi",             this.mode_avi+"");            // int
		properties.setProperty(prefix+"avi_JPEG_quality",     this.avi_JPEG_quality+"");    // int
		properties.setProperty(prefix+"run_ffmpeg",           this.run_ffmpeg+"");          // boolean
		properties.setProperty(prefix+"video_ext",            this.video_ext+"");           // String
		properties.setProperty(prefix+"video_codec",          this.video_codec+"");         // String
		properties.setProperty(prefix+"video_crf",            this.video_crf+"");           // int
		properties.setProperty(prefix+"video_bitrate_m",      this.video_bitrate_m+"");          // double
		properties.setProperty(prefix+"remove_avi",           this.remove_avi+"");          // boolean
		properties.setProperty(prefix+"video_codec_combo",    this.video_codec_combo+"");   // String
		properties.setProperty(prefix+"video_crf_combo",      this.video_crf_combo+"");     // int
		properties.setProperty(prefix+"add_average",          this.add_average+"");         // boolean
		properties.setProperty(prefix+"calculate_average",    this.calculate_average+"");   // boolean
		properties.setProperty(prefix+"subtract_average",     this.subtract_average+"");    // boolean
		properties.setProperty(prefix+"running_average",      this.running_average+"");     // int
		
		properties.setProperty(prefix+"extract_center_orientation",this.extract_center_orientation+"");// boolean
		properties.setProperty(prefix+"um_mono",              this.um_mono+"");             // boolean
		properties.setProperty(prefix+"um_sigma",             this.um_sigma+"");            // double
		properties.setProperty(prefix+"um_weight",            this.um_weight+"");           // double
		properties.setProperty(prefix+"um_mono_linear",       this.um_mono_linear+"");      // boolean
		properties.setProperty(prefix+"mono_fixed",           this.mono_fixed+"");          // boolean
		properties.setProperty(prefix+"mono_range",           this.mono_range+"");          // double
		properties.setProperty(prefix+"anaglyth_en",          this.anaglyth_en+"");         // boolean
		properties.setProperty(prefix+"anaglyph_left",        getLongColor(anaglyph_left)+""); // Color
		properties.setProperty(prefix+"anaglyph_right",       getLongColor(anaglyph_right)+""); // Color
		properties.setProperty(prefix+"annotate_color",       this.annotate_color+"");      // boolean
		properties.setProperty(prefix+"annotate_mono",        this.annotate_mono+"");       // boolean
		{
			long lcolor_annotate =        (annotate_color_color == null) ? -1 : getLongColor(annotate_color_color);
			properties.setProperty(prefix+"annotate_color_color",              lcolor_annotate+"");
		}
		{
			long lcolor_annotate =        (annotate_color_mono == null) ? -1 : getLongColor(annotate_color_mono);
			properties.setProperty(prefix+"annotate_color_mono",              lcolor_annotate+"");
		}
		properties.setProperty(prefix+"annotate_transparent_mono",this.annotate_transparent_mono+"");// boolean
		
		// vegetation
		properties.setProperty(prefix+"terr_model_path",             terr_model_path+"");         // String
		properties.setProperty(prefix+"terr_model_state",            terr_model_state+"");        // String
		properties.setProperty(prefix+"terr_segments_dir",           terr_segments_dir+"");       // String
		properties.setProperty(prefix+"terr_segments_suffix",        terr_segments_suffix+"");    // String
		
		properties.setProperty(prefix+"terr_par_dir",                terr_par_dir+"");            // String
		properties.setProperty(prefix+"terr_par_file",               terr_par_file+"");           // String
		properties.setProperty(prefix+"terr_par_restore",            terr_par_restore+"");        // boolean
		properties.setProperty(prefix+"terr_step_restore",           terr_step_restore+"");       // int
		
		properties.setProperty(prefix+"terr_um_en",                  terr_um_en+"");              // boolean
		properties.setProperty(prefix+"terr_um_sigma",               terr_um_sigma+"");           // double 
		properties.setProperty(prefix+"terr_um_weight",              terr_um_weight+"");          // double 
		properties.setProperty(prefix+"terr_nan_tolerance",          terr_nan_tolerance+"");      // double 
		properties.setProperty(prefix+"terr_nan_grow",               terr_nan_grow+"");           // int
		
		properties.setProperty(prefix+"terr_shrink_veget",           terr_shrink_veget+"");       // int
		properties.setProperty(prefix+"terr_shrink_terrain",         terr_shrink_terrain+"");     // int
		properties.setProperty(prefix+"terr_vegetation_over",        terr_vegetation_over+"");     // double 
		properties.setProperty(prefix+"terr_filter_veget",           terr_filter_veget+"");       // int
		properties.setProperty(prefix+"terr_veget_grow",             terr_veget_grow+"");       // int
		
		properties.setProperty(prefix+"terr_nan_tolerance",          terr_nan_tolerance+"");      // double 
		properties.setProperty(prefix+"terr_nan_grow",               terr_nan_grow+"");           // int

		properties.setProperty(prefix+"terr_tile_woi",               terr_tile_woi+"");           // boolean
		properties.setProperty(prefix+"terr_continue",               terr_continue+"");           // boolean
		properties.setProperty(prefix+"terr_woi_enclos", rectangleToString(terr_woi_enclos)+"");  // Rectangle
		properties.setProperty(prefix+"terr_woi_step",   rectangleToString(terr_woi_step)+"");    // Rectangle
		properties.setProperty(prefix+"terr_woi_last",   rectangleToString(terr_woi_last)+"");    // Rectangle
		properties.setProperty(prefix+"terr_skip_exist",             terr_skip_exist+"");         // boolean
		
		properties.setProperty(prefix+"terr_fit_terr",               terr_fit_terr+"");           // boolean
		properties.setProperty(prefix+"terr_fit_veget",              terr_fit_veget+"");          // boolean
		properties.setProperty(prefix+"terr_fit_alpha",              terr_fit_alpha+"");          // boolean
		properties.setProperty(prefix+"terr_fit_elevations",         terr_fit_elevations+"");     // boolean
		properties.setProperty(prefix+"terr_fit_terr_elev",          terr_fit_terr_elev+"");      // boolean
		properties.setProperty(prefix+"terr_fit_terr_elev_pix",      terr_fit_terr_elev_pix+"");  // boolean
		properties.setProperty(prefix+"terr_fit_scenes",             terr_fit_scenes+"");         // boolean
		
		for (int i = 0; i < terr_fits_disable.length; i++) {
			String prop_name = prefix+"terr_fits_disable_"+VegetationLMA.TVAO_NAMES[i];
			properties.setProperty(prop_name,                        booleansToString(terr_fits_disable[i],2));     // boolean
		}
		
		for (int i = 0; i < terr_only_fits_disable.length; i++) {
			String prop_name = prefix+"terr_only_fits_disable_"+VegetationLMA.TVAO_NAMES[i];
			properties.setProperty(prop_name,                        booleansToString(terr_only_fits_disable[i],2));     // boolean
		}
		
		
		properties.setProperty(prefix+"terr_max_warp",               terr_max_warp+"");           // double 
		properties.setProperty(prefix+"terr_max_elevation",          terr_max_elevation+"");      // int
		properties.setProperty(prefix+"terr_max_elev_terr",          terr_max_elev_terr+"");      // int
		properties.setProperty(prefix+"terr_max_elev_chng",          terr_max_elev_chng+"");      // double 
		
		properties.setProperty(prefix+"terr_min_scenes",             terr_min_scenes+"");         // int
		properties.setProperty(prefix+"terr_min_samples_scene",      terr_min_samples_scene+"");  // int
		properties.setProperty(prefix+"terr_min_total_scenes",       terr_min_total_scenes+"");   // int
		properties.setProperty(prefix+"terr_min_pixels",             terr_min_pixels+"");         // int
		properties.setProperty(prefix+"terr_warm_veget",             terr_warm_veget+"");         // boolean
		properties.setProperty(prefix+"terr_warmest",                terr_warmest+"");            // double 
		properties.setProperty(prefix+"terr_initial_split",          terr_initial_split+"");      // double 
		properties.setProperty(prefix+"terr_min_split_frac",         terr_min_split_frac+"");     // double 
		properties.setProperty(prefix+"terr_difference",             terr_difference+"");         // double 
				
		properties.setProperty(prefix+"terr_pull_cold",              terr_pull_cold+"");          // double
		properties.setProperty(prefix+"terr_elevation_radius",       terr_elevation_radius+"");   // double
		properties.setProperty(prefix+"terr_terr_elev_radius",       terr_terr_elev_radius+"");   // double
		properties.setProperty(prefix+"terr_elev_radius_extra",      terr_elev_radius_extra+"");  // double
		
		
		properties.setProperty(prefix+"terr_alpha_contrast",         terr_alpha_contrast+"");     // double 
		properties.setProperty(prefix+"terr_alpha_dflt",             terr_alpha_dflt+"");         // double
		
		properties.setProperty(prefix+"terr_alpha_sigma",            terr_alpha_sigma+"");        // double
		properties.setProperty(prefix+"terr_alpha_init_min",         terr_alpha_init_min+"");     // double
		properties.setProperty(prefix+"terr_alpha_init_max",         terr_alpha_init_max+"");     // double
		properties.setProperty(prefix+"terr_alpha_init_offs",        terr_alpha_init_offs+"");    // double
		
		properties.setProperty(prefix+"terr_alpha_loss",             terr_alpha_loss+"");         // double 
		properties.setProperty(prefix+"terr_alpha_loss_lin",         terr_alpha_loss_lin+"");     // double 
		properties.setProperty(prefix+"terr_alpha_offset",           terr_alpha_offset+"");       // double 
		properties.setProperty(prefix+"terr_alpha_0offset",          terr_alpha_0offset+"");      // double 
		properties.setProperty(prefix+"terr_alpha_min_veg",          terr_alpha_min_veg+"");      // double 
		properties.setProperty(prefix+"terr_alpha_max_terrain",      terr_alpha_max_terrain+"");  // double 
		properties.setProperty(prefix+"terr_alpha_pull_pwr",         terr_alpha_pull_pwr+"");     // double 
		properties.setProperty(prefix+"terr_alpha_lpf",              terr_alpha_lpf+"");          // double 
		properties.setProperty(prefix+"terr_alpha_lpf_border",       terr_alpha_lpf_border+"");   // double 
		properties.setProperty(prefix+"terr_alpha_piece_linear",     terr_alpha_piece_linear+""); // boolean
		properties.setProperty(prefix+"terr_alpha_scale_avg",        terr_alpha_scale_avg+"");    // double 
		properties.setProperty(prefix+"terr_alpha_push",             terr_alpha_push+"");         // double 
		properties.setProperty(prefix+"terr_alpha_push_neutral",     terr_alpha_push_neutral+""); // double 
		properties.setProperty(prefix+"terr_alpha_weight_center",    terr_alpha_weight_center+"");// double 
		properties.setProperty(prefix+"terr_en_holes",               terr_en_holes+"");           // boolean
		properties.setProperty(prefix+"terr_alpha_mm_hole",          terr_alpha_mm_hole+"");      // double 
		properties.setProperty(prefix+"terr_alpha_diff_hole",        terr_alpha_diff_hole+"");    // double 
		
		properties.setProperty(prefix+"terr_terr_lpf",               terr_terr_lpf+"");           // double 
		properties.setProperty(prefix+"terr_veget_lpf",              terr_veget_lpf+"");          // double 
		properties.setProperty(prefix+"terr_elev_lpf",               terr_elev_lpf+"");           // double 
		properties.setProperty(prefix+"terr_terr_elev_lpf",          terr_terr_elev_lpf+"");      // double 
		properties.setProperty(prefix+"terr_terr_pull0",             terr_terr_pull0+"");         // double 
		properties.setProperty(prefix+"terr_terr_pull_up",           terr_terr_pull_up+"");       // double 
		properties.setProperty(prefix+"terr_terr_pull_avg",          terr_terr_pull_avg+"");      // double 
		properties.setProperty(prefix+"terr_veget_pull0",            terr_veget_pull0+"");        // double
		properties.setProperty(prefix+"terr_veget_pull_low_alpha",   terr_veget_pull_low_alpha+"");// double
		properties.setProperty(prefix+"terr_elev_pull0",             terr_elev_pull0+"");         // double
		properties.setProperty(prefix+"terr_terr_elev_pull0",        terr_terr_elev_pull0+"");    // double
		
		properties.setProperty(prefix+"terr_elev_alpha_en",          terr_elev_alpha_en+"");      // boolean
		properties.setProperty(prefix+"terr_elev_alpha",             terr_elev_alpha+"");         // double
		properties.setProperty(prefix+"terr_elev_alpha_pwr",         terr_elev_alpha_pwr+"");     // double
		properties.setProperty(prefix+"terr_low_veget",              terr_low_veget+"");          // double
		properties.setProperty(prefix+"terr_scenes_pull0",           terr_scenes_pull0+"");       // double

		// scaling elevation losses for high elevations (decrease pull and/or lpf for high elevations)
		properties.setProperty(prefix+"terr_elev_scale_thresh",      terr_elev_scale_thresh+"");  // double 
		properties.setProperty(prefix+"terr_elev_scale_pull",        terr_elev_scale_pull+"");    // boolean
		properties.setProperty(prefix+"terr_elev_scale_lpf",         terr_elev_scale_lpf+"");     // boolean
		// tree-top removal
		properties.setProperty(prefix+"terr_ttop_en",                terr_ttop_en+"");            // boolean
		properties.setProperty(prefix+"terr_ttop_gb",                terr_ttop_gb+"");            // double 
		properties.setProperty(prefix+"terr_ttop_min",               terr_ttop_min+"");           // double 
		properties.setProperty(prefix+"terr_ttop_rel_lev",           terr_ttop_rel_lev+"");       // double 
		properties.setProperty(prefix+"terr_ttop_rel_rad",           terr_ttop_rel_rad+"");       // double 
		properties.setProperty(prefix+"terr_ttop_frac",              terr_ttop_frac+"");          // double 
		properties.setProperty(prefix+"terr_ttop_rem_rad",           terr_ttop_rem_rad+"");       // double 


		properties.setProperty(prefix+"terr_boost_parallax",         terr_boost_parallax+"");     // double 
		properties.setProperty(prefix+"terr_max_parallax",           terr_max_parallax+"");       // double 
		properties.setProperty(prefix+"terr_hifreq_weight",          terr_hifreq_weight+"");      // double
		properties.setProperty(prefix+"terr_terr_corr",              terr_terr_corr+"");          // double
		properties.setProperty(prefix+"terr_reg_weights",            terr_reg_weights+"");        // double 
		properties.setProperty(prefix+"terr_lambda",                 terr_lambda+"");             // double 
		properties.setProperty(prefix+"terr_lambda_scale_good",      terr_lambda_scale_good+"");  // double 
		properties.setProperty(prefix+"terr_lambda_scale_bad",       terr_lambda_scale_bad+"");   // double 
		properties.setProperty(prefix+"terr_lambda_max",             terr_lambda_max+"");         // double 
		properties.setProperty(prefix+"terr_rms_diff",               terr_rms_diff+"");           // double 
		properties.setProperty(prefix+"terr_last_series",            terr_last_series+"");        // int
		properties.setProperty(prefix+"terr_num_iters",              intsToString(terr_num_iters)+"");  // int []
		
///		properties.setProperty(prefix+"terr_only_special",           terr_only_special+"");     // boolean
///		properties.setProperty(prefix+"terr_only_pix",               terr_only_pix+"");     // boolean
		properties.setProperty(prefix+"terr_only_series",            terr_only_series+"");       // int    
		properties.setProperty(prefix+"terr_only_num_iters",         intsToString(terr_only_num_iters)+"");  // int []
		properties.setProperty(prefix+"terr_cholesky",               terr_cholesky+"");           // int
		properties.setProperty(prefix+"terr_recalc_weights",         booleansToString(terr_recalc_weights,2));     // boolean
		properties.setProperty(prefix+"terr_recalc_opaque",          terr_recalc_opaque+"");      // double 
		properties.setProperty(prefix+"terr_recalc_pedestal",        terr_recalc_pedestal+"");    // double 
		properties.setProperty(prefix+"terr_recalc_frac",            terr_recalc_frac+"");        // double 
		properties.setProperty(prefix+"terr_recalc_dist",            terr_recalc_dist+"");        // double 
		properties.setProperty(prefix+"terr_recalc_pwr",             terr_recalc_pwr+"");         // double 
		properties.setProperty(prefix+"terr_recalc_gb",              terr_recalc_gb+"");          // double 
		properties.setProperty(prefix+"terr_recalc_boost",           terr_recalc_boost+"");       // double 
		properties.setProperty(prefix+"terr_recalc_average",         terr_recalc_average+"");     // boolean

		properties.setProperty(prefix+"terr_crop",                   terr_crop+"");               // boolean
		properties.setProperty(prefix+"terr_keep_partial",           terr_keep_partial+"");       // boolean
		properties.setProperty(prefix+"terr_border_width",           terr_border_width+"");       // int    
		properties.setProperty(prefix+"terr_render_open",            terr_render_open+"");        // boolean
		properties.setProperty(prefix+"terr_render_no_alpha",        terr_render_no_alpha+"");    // boolean
		properties.setProperty(prefix+"terr_alpha_min",              terr_alpha_min+"");          // double 
		properties.setProperty(prefix+"terr_alpha_max",              terr_alpha_max+"");          // double 
		properties.setProperty(prefix+"terr_weight_opaque",          terr_weight_opaque+"");      // double 
		properties.setProperty(prefix+"terr_boost_render",           terr_boost_render+"");       // double 
		properties.setProperty(prefix+"terr_max_render",             terr_max_render+"");         // double 
		properties.setProperty(prefix+"terr_num_exaggerate",         terr_num_exaggerate+"");     // int    
		properties.setProperty(prefix+"terr_rebuild_elev",           terr_rebuild_elev+"");    // boolean
		properties.setProperty(prefix+"terr_elev_grow",              terr_elev_grow+"");     // int    
		properties.setProperty(prefix+"terr_threshold_terrain",      terr_threshold_terrain+"");  // double 
		properties.setProperty(prefix+"terr_min_max_terrain",        terr_min_max_terrain+"");    // double 
		properties.setProperty(prefix+"terr_min_terrain",            terr_min_terrain+"");        // double 
		properties.setProperty(prefix+"terr_min_vegetation",         terr_min_vegetation+"");     // double 
		properties.setProperty(prefix+"terr_debug",                  terr_debug+"");              // int
		properties.setProperty(prefix+"terr_debug_path",             terr_debug_path+"");         // String
		properties.setProperty(prefix+"terr_debug_length",           terr_debug_length+"");       // int
		properties.setProperty(prefix+"terr_debug_improved",         terr_debug_improved+"");     // boolean
		properties.setProperty(prefix+"terr_debug_worsened",         terr_debug_worsened+"");     // boolean
		// end of vegetation
		// Synthesis		
		properties.setProperty(prefix+"synth_directory",             synth_directory+"");         // String
		properties.setProperty(prefix+"synth_scene",                 synth_scene+"");             // String
		properties.setProperty(prefix+"synth_model",                 synth_model+"");             // String
		properties.setProperty(prefix+"synth_segments",              synth_segments+"");          // String
		properties.setProperty(prefix+"synth_render",                synth_render+"");            // String
		properties.setProperty(prefix+"synth_crop_real",             synth_crop_real+"");         // boolean
		properties.setProperty(prefix+"synth_scene_offs",            synth_scene_offs+"");        // String
		properties.setProperty(prefix+"synth_add_offs",              synth_add_offs+"");          // boolean
	}
	
	public void getProperties(String prefix,Properties properties){
		if (properties.getProperty(prefix+"ims_use")!=null)                       this.ims_use=Boolean.parseBoolean(properties.getProperty(prefix+"ims_use"));		
		if (properties.getProperty(prefix+"ims_rebuild")!=null)                   this.ims_rebuild=Boolean.parseBoolean(properties.getProperty(prefix+"ims_rebuild"));		
		if (properties.getProperty(prefix+"ims_offset")!=null)                    this.ims_offset=Double.parseDouble(properties.getProperty(prefix+"ims_offset"));
		if (properties.getProperty(prefix+"gmt_plus")!=null)                      this.gmt_plus=Double.parseDouble(properties.getProperty(prefix+"gmt_plus"));
		if (properties.getProperty(prefix+"ims_ortho")!=null) 	       this.ims_ortho= IntersceneMatchParameters.StringToDoubles(properties.getProperty(prefix+"ims_ortho"),4);
		if (properties.getProperty(prefix+"ims_mount_atr")!=null) 	   this.ims_mount_atr= IntersceneMatchParameters.StringToDoubles(properties.getProperty(prefix+"ims_mount_atr"),3);
		if (properties.getProperty(prefix+"ims_mount_xyz")!=null) 	   this.ims_mount_xyz= IntersceneMatchParameters.StringToDoubles(properties.getProperty(prefix+"ims_mount_xyz"),3);
		if (properties.getProperty(prefix+"pimu_gyr_offs")!=null) 	   this.pimu_gyr_offs= IntersceneMatchParameters.StringToDoubles(properties.getProperty(prefix+"pimu_gyr_offs"),3);
		if (properties.getProperty(prefix+"pimu_acc_offs")!=null) 	   this.pimu_acc_offs= IntersceneMatchParameters.StringToDoubles(properties.getProperty(prefix+"pimu_acc_offs"),3);
		if (properties.getProperty(prefix+"ims_scale_xyz")!=null) 	   this.ims_scale_xyz= IntersceneMatchParameters.StringToDoubles(properties.getProperty(prefix+"ims_scale_xyz"),3);
		if (properties.getProperty(prefix+"ims_scale_atr")!=null) 	   this.ims_scale_atr= IntersceneMatchParameters.StringToDoubles(properties.getProperty(prefix+"ims_scale_atr"),3);
		if (properties.getProperty(prefix+"fmg_initial_en")!=null)     this.fmg_initial_en=Boolean.parseBoolean(properties.getProperty(prefix+"fmg_initial_en"));		
		if (properties.getProperty(prefix+"fmg_reorient_en")!=null)    this.fmg_reorient_en=Boolean.parseBoolean(properties.getProperty(prefix+"fmg_reorient_en"));		
		if (properties.getProperty(prefix+"fmg_distance")!=null)       this.fmg_distance=Double.parseDouble(properties.getProperty(prefix+"fmg_distance"));
		if (properties.getProperty(prefix+"fmg_max_quad")!=null)       this.fmg_max_quad=Double.parseDouble(properties.getProperty(prefix+"fmg_max_quad"));
		if (properties.getProperty(prefix+"fmg_rectilinear")!=null)    this.fmg_rectilinear=Boolean.parseBoolean(properties.getProperty(prefix+"fmg_rectilinear"));		
		if (properties.getProperty(prefix+"sfm_use")!=null)            this.sfm_use=Boolean.parseBoolean(properties.getProperty(prefix+"sfm_use"));		
		if (properties.getProperty(prefix+"sfm_min_base")!=null)       this.sfm_min_base=Double.parseDouble(properties.getProperty(prefix+"sfm_min_base"));
		if (properties.getProperty(prefix+"sfm_min_gain")!=null)       this.sfm_min_gain=Double.parseDouble(properties.getProperty(prefix+"sfm_min_gain"));
		if (properties.getProperty(prefix+"sfm_min_frac")!=null)       this.sfm_min_frac=Double.parseDouble(properties.getProperty(prefix+"sfm_min_frac"));
		if (properties.getProperty(prefix+"sfm_num_pairs")!=null)      this.sfm_num_pairs=Integer.parseInt(properties.getProperty(prefix+"sfm_num_pairs"));
		if (properties.getProperty(prefix+"sfp_tolerance")!=null)      this.sfp_tolerance=Double.parseDouble(properties.getProperty(prefix+"sfp_tolerance"));
		if (properties.getProperty(prefix+"sfm_readjust")!=null)       this.sfm_readjust=Integer.parseInt(properties.getProperty(prefix+"sfm_readjust"));
		if (properties.getProperty(prefix+"sfm_prev_frac")!=null)      this.sfm_prev_frac=Double.parseDouble(properties.getProperty(prefix+"sfm_prev_frac"));
		if (properties.getProperty(prefix+"sfm_same_weight")!=null)    this.sfm_same_weight=Double.parseDouble(properties.getProperty(prefix+"sfm_same_weight"));
		if (properties.getProperty(prefix+"sfm_shrink")!=null)         this.sfm_shrink=Integer.parseInt(properties.getProperty(prefix+"sfm_shrink"));
		if (properties.getProperty(prefix+"sfm_fade_sigma")!=null)     this.sfm_fade_sigma=Double.parseDouble(properties.getProperty(prefix+"sfm_fade_sigma"));
		if (properties.getProperty(prefix+"sfm_min_str1")!=null)       this.sfm_min_str1=Double.parseDouble(properties.getProperty(prefix+"sfm_min_str1"));
		if (properties.getProperty(prefix+"sfm_min_str16")!=null)      this.sfm_min_str16=Double.parseDouble(properties.getProperty(prefix+"sfm_min_str16"));
		if (properties.getProperty(prefix+"sfm_use_neibs")!=null)      this.sfm_use_neibs=Boolean.parseBoolean(properties.getProperty(prefix+"sfm_use_neibs"));
		if (properties.getProperty(prefix+"sfm_neib_too_str1")!=null)  this.sfm_neib_too_str1=Double.parseDouble(properties.getProperty(prefix+"sfm_neib_too_str1"));
		if (properties.getProperty(prefix+"sfm_neib_too_str16")!=null) this.sfm_neib_too_str16=Double.parseDouble(properties.getProperty(prefix+"sfm_neib_too_str16"));
		if (properties.getProperty(prefix+"sfm_neib_str1")!=null)      this.sfm_neib_str1=Double.parseDouble(properties.getProperty(prefix+"sfm_neib_str1"));
		if (properties.getProperty(prefix+"sfm_neib_str16")!=null)     this.sfm_neib_str16=Double.parseDouble(properties.getProperty(prefix+"sfm_neib_str16"));
		if (properties.getProperty(prefix+"sfm_extrap_steps")!=null)   this.sfm_extrap_steps=Integer.parseInt(properties.getProperty(prefix+"sfm_extrap_steps"));
		if (properties.getProperty(prefix+"sfm_extrap_radius")!=null)  this.sfm_extrap_radius=Integer.parseInt(properties.getProperty(prefix+"sfm_extrap_radius"));
		if (properties.getProperty(prefix+"sfm_average_neibs")!=null)  this.sfm_average_neibs=Boolean.parseBoolean(properties.getProperty(prefix+"sfm_average_neibs"));		
		if (properties.getProperty(prefix+"sfm_fill_weak")!=null)      this.sfm_fill_weak=Boolean.parseBoolean(properties.getProperty(prefix+"sfm_fill_weak"));		
		if (properties.getProperty(prefix+"sfm_extrapolate")!=null)    this.sfm_extrapolate=Boolean.parseBoolean(properties.getProperty(prefix+"sfm_extrapolate"));
		
		if (properties.getProperty(prefix+"sfm_max_dip")!=null)        this.sfm_max_dip=Double.parseDouble(properties.getProperty(prefix+"sfm_max_dip"));
		if (properties.getProperty(prefix+"sfm_max_bump")!=null)       this.sfm_max_bump=Double.parseDouble(properties.getProperty(prefix+"sfm_max_bump"));
		if (properties.getProperty(prefix+"sfm_filter_nan")!=null)     this.sfm_filter_nan=Boolean.parseBoolean(properties.getProperty(prefix+"sfm_filter_nan"));		
		if (properties.getProperty(prefix+"sfm_only")!=null)           this.sfm_only=Boolean.parseBoolean(properties.getProperty(prefix+"sfm_only"));		
		
		if (properties.getProperty(prefix+"sfm_save_seq")!=null)       this.sfm_save_seq=Boolean.parseBoolean(properties.getProperty(prefix+"sfm_save_seq"));		
		if (properties.getProperty(prefix+"sfm_show_seq")!=null)       this.sfm_show_seq=Boolean.parseBoolean(properties.getProperty(prefix+"sfm_show_seq"));		
		if (properties.getProperty(prefix+"sfm_show_corr_ind")!=null)  this.sfm_show_corr_ind=Boolean.parseBoolean(properties.getProperty(prefix+"sfm_show_corr_ind"));		
		if (properties.getProperty(prefix+"sfm_show_corr")!=null)      this.sfm_show_corr=Boolean.parseBoolean(properties.getProperty(prefix+"sfm_show_corr"));
		
		if (properties.getProperty(prefix+"omtch_img_set")!=null)     this.omtch_img_set=(String) properties.getProperty(prefix+                "omtch_img_set");
		
		if (properties.getProperty(prefix+"rln_gpu_width")!=null)      this.rln_gpu_width=Integer.parseInt(properties.getProperty(prefix+"rln_gpu_width"));
		if (properties.getProperty(prefix+"rln_gpu_height")!=null)     this.rln_gpu_height=Integer.parseInt(properties.getProperty(prefix+"rln_gpu_height"));
		if (properties.getProperty(prefix+"rln_fat_zero")!=null)       this.rln_fat_zero=Double.parseDouble(properties.getProperty(prefix+"rln_fat_zero"));
		if (properties.getProperty(prefix+"rln_use_neibs")!=null)      this.rln_use_neibs=Boolean.parseBoolean(properties.getProperty(prefix+"rln_use_neibs"));		
		if (properties.getProperty(prefix+"rln_neibs_fill")!=null)     this.rln_neibs_fill=Boolean.parseBoolean(properties.getProperty(prefix+"rln_neibs_fill"));		
		if (properties.getProperty(prefix+"rln_neib_radius")!=null)    this.rln_neib_radius=Double.parseDouble(properties.getProperty(prefix+"rln_neib_radius"));
		if (properties.getProperty(prefix+"rln_radius_frac")!=null)    this.rln_radius_frac=Double.parseDouble(properties.getProperty(prefix+"rln_radius_frac"));
		if (properties.getProperty(prefix+"rln_cent_radius")!=null)    this.rln_cent_radius=Double.parseDouble(properties.getProperty(prefix+"rln_cent_radius"));
		if (properties.getProperty(prefix+"rln_n_recenter")!=null)     this.rln_n_recenter=Integer.parseInt(properties.getProperty(prefix+"rln_n_recenter"));
		if (properties.getProperty(prefix+"rln_sngl_rstr")!=null)      this.rln_sngl_rstr=Double.parseDouble(properties.getProperty(prefix+"rln_sngl_rstr"));
		if (properties.getProperty(prefix+"rln_neib_rstr")!=null)      this.rln_neib_rstr=Double.parseDouble(properties.getProperty(prefix+"rln_neib_rstr"));

		if (properties.getProperty(prefix+"pmtch_use_affine")!=null)   this.pmtch_use_affine= Boolean.parseBoolean(properties.getProperty(prefix+"pmtch_use_affine"));		
   		if (properties.getProperty(prefix+"pmtch_frac_remove")!=null)  this.pmtch_frac_remove=Double.parseDouble(properties.getProperty(prefix+  "pmtch_frac_remove"));
		if (properties.getProperty(prefix+"pmtch_metric_err")!=null)   this.pmtch_metric_err= Double.parseDouble(properties.getProperty(prefix+  "pmtch_metric_err"));
		if (properties.getProperty(prefix+"pmtch_max_std")!=null)      this.pmtch_max_std=    Double.parseDouble(properties.getProperty(prefix+  "pmtch_max_std"));
		if (properties.getProperty(prefix+"pmtch_min_std_rad")!=null)  this.pmtch_min_std_rad=Double.parseDouble(properties.getProperty(prefix+  "pmtch_min_std_rad"));
		if (properties.getProperty(prefix+"pmtch_ignore_rms")!=null)   this.pmtch_ignore_rms= Boolean.parseBoolean(properties.getProperty(prefix+"pmtch_ignore_rms"));		
		if (properties.getProperty(prefix+"pmtch_num_iter")!=null)     this.pmtch_num_iter=   Integer.parseInt(properties.getProperty(prefix+    "pmtch_num_iter"));
		
		if (properties.getProperty(prefix+"pmtch_pull_skew")!=null)    this.pmtch_pull_skew=  Double.parseDouble(properties.getProperty(prefix+  "pmtch_pull_skew"));
		if (properties.getProperty(prefix+"pmtch_pull_tilt")!=null)    this.pmtch_pull_tilt=  Double.parseDouble(properties.getProperty(prefix+  "pmtch_pull_tilt"));
		if (properties.getProperty(prefix+"pmtch_pull_scale")!=null)   this.pmtch_pull_scale= Double.parseDouble(properties.getProperty(prefix+  "pmtch_pull_scale"));
		
		if (properties.getProperty(prefix+"pmtch_cent_rad"   )!=null)  this.pmtch_cent_rad=   Double.parseDouble(properties.getProperty(prefix+  "pmtch_cent_rad"));
		if (properties.getProperty(prefix+"pmtch_max_cent_rad")!=null) this.pmtch_max_cent_rad=Double.parseDouble(properties.getProperty(prefix+  "pmtch_max_cent_rad"));
		if (properties.getProperty(prefix+"pmtch_cent_fill"  )!=null)  this.pmtch_cent_fill=  Double.parseDouble(properties.getProperty(prefix+  "pmtch_cent_fill"));
		if (properties.getProperty(prefix+"pmtch_cent_final" )!=null)  this.pmtch_cent_final= Double.parseDouble(properties.getProperty(prefix+  "pmtch_cent_final"));
		if (properties.getProperty(prefix+"pmtch_ease_nosfm" )!=null)  this.pmtch_ease_nosfm= Double.parseDouble(properties.getProperty(prefix+  "pmtch_ease_nosfm"));
		
		if (properties.getProperty(prefix+"ospir_existing")!=null)     this.ospir_existing=Boolean.parseBoolean(properties.getProperty(prefix+  "ospir_existing"));		
		if (properties.getProperty(prefix+"ospir_invert")!=null)       this.ospir_invert=Boolean.parseBoolean(properties.getProperty(prefix+    "ospir_invert"));		
		if (properties.getProperty(prefix+"ospir_step")!=null)         this.ospir_step=Double.parseDouble(properties.getProperty(prefix+        "ospir_step"));
		if (properties.getProperty(prefix+"ospir_range")!=null)        this.ospir_range=Double.parseDouble(properties.getProperty(prefix+       "ospir_range"));
		if (properties.getProperty(prefix+"ospir_double")!=null)       this.ospir_double=Double.parseDouble(properties.getProperty(prefix+      "ospir_double"));
		if (properties.getProperty(prefix+"ospir_good_rms")!=null)     this.ospir_good_rms=Double.parseDouble(properties.getProperty(prefix+    "ospir_good_rms"));
		if (properties.getProperty(prefix+"ospir_max_rms")!=null)      this.ospir_max_rms=Double.parseDouble(properties.getProperty(prefix+     "ospir_max_rms"));
		if (properties.getProperty(prefix+"ospir_overlap")!=null)      this.ospir_overlap=Integer.parseInt(properties.getProperty(prefix+       "ospir_overlap"));
		if (properties.getProperty(prefix+"ospir_num_iter")!=null)     this.ospir_num_iter=Integer.parseInt(properties.getProperty(prefix+      "ospir_num_iter"));
		if (properties.getProperty(prefix+"ospir_rms_iter_0")!=null)   this.ospir_rms_iter[0]=Double.parseDouble(properties.getProperty(prefix+ "ospir_rms_iter_0"));
		if (properties.getProperty(prefix+"ospir_rms_iter_1")!=null)   this.ospir_rms_iter[1]=Double.parseDouble(properties.getProperty(prefix+ "ospir_rms_iter_1"));
		if (properties.getProperty(prefix+"ospir_ignore_rms")!=null)   this.ospir_ignore_rms=Boolean.parseBoolean(properties.getProperty(prefix+"ospir_ignore_rms"));		
		if (properties.getProperty(prefix+"ospir_debug")!=null)        this.ospir_debug=Integer.parseInt(properties.getProperty(prefix+         "ospir_debug"));
		if (properties.getProperty(prefix+"ospir_augment")!=null)      this.ospir_augment=Boolean.parseBoolean(properties.getProperty(prefix+    "ospir_augment"));		

		if (properties.getProperty(prefix+"pwise_zoom")!=null)         this.pwise_zoom=Integer.parseInt(properties.getProperty(prefix+            "pwise_zoom"));
		if (properties.getProperty(prefix+"pwise_overlap")!=null)      this.pwise_overlap=Double.parseDouble(properties.getProperty(prefix+       "pwise_overlap"));
		if (properties.getProperty(prefix+"pwise_max_rms")!=null)      this.pwise_max_rms=Double.parseDouble(properties.getProperty(prefix+       "pwise_max_rms"));
		
		if (properties.getProperty(prefix+"pwise_skip_exist")!=null)   this.pwise_skip_exist=Boolean.parseBoolean(properties.getProperty(prefix+  "pwise_skip_exist"));		
		if (properties.getProperty(prefix+"pwise_refine_exist")!=null) this.pwise_refine_exist=Boolean.parseBoolean(properties.getProperty(prefix+"pwise_refine_exist"));		
		if (properties.getProperty(prefix+"pwise_delete_fail")!=null)  this.pwise_delete_fail=Boolean.parseBoolean(properties.getProperty(prefix+ "pwise_delete_fail"));		
		if (properties.getProperty(prefix+"pwise_gen_inverse")!=null)  this.pwise_gen_inverse=Boolean.parseBoolean(properties.getProperty(prefix+ "pwise_gen_inverse"));		
		if (properties.getProperty(prefix+"pwise_save_each")!=null)    this.pwise_save_each=Boolean.parseBoolean(properties.getProperty(prefix+   "pwise_save_each"));		
		if (properties.getProperty(prefix+"pwise_log_append")!=null)   this.pwise_log_append=Boolean.parseBoolean(properties.getProperty(prefix+  "pwise_log_append"));		
		if (properties.getProperty(prefix+"pwise_log_path")!=null)     this.pwise_log_path=(String) properties.getProperty(prefix+                "pwise_log_path");
		if (properties.getProperty(prefix+"pwise_debug")!=null)        this.pwise_debug=Integer.parseInt(properties.getProperty(prefix+           "pwise_debug"));
	
		if (properties.getProperty(prefix+"alt_overwrite")!=null)      this.alt_overwrite=Boolean.parseBoolean(properties.getProperty(prefix+     "alt_overwrite"));		
		if (properties.getProperty(prefix+"alt_pairwise")!=null)       this.alt_pairwise=Boolean.parseBoolean(properties.getProperty(prefix+      "alt_pairwise"));
		if (properties.getProperty(prefix+"alt_sigma")!=null)          this.alt_sigma=Double.parseDouble(properties.getProperty(prefix+           "alt_sigma"));
		if (properties.getProperty(prefix+"alt_abs_outliers")!=null)   this.alt_abs_outliers=Double.parseDouble(properties.getProperty(prefix+    "alt_abs_outliers"));
		if (properties.getProperty(prefix+"alt_outliers")!=null)       this.alt_outliers=Double.parseDouble(properties.getProperty(prefix+        "alt_outliers"));
		if (properties.getProperty(prefix+"alt_refine")!=null)         this.alt_refine=Integer.parseInt(properties.getProperty(prefix+            "alt_refine"));

		if (properties.getProperty(prefix+"alt_weight_rot")!=null)     this.alt_weight_rot=Double.parseDouble(properties.getProperty(prefix+      "alt_weight_rot"));
		if (properties.getProperty(prefix+"alt_weight_tilt")!=null)    this.alt_weight_tilt=Double.parseDouble(properties.getProperty(prefix+     "alt_weight_tilt"));
		if (properties.getProperty(prefix+"alt_weight_scale")!=null)   this.alt_weight_scale=Double.parseDouble(properties.getProperty(prefix+    "alt_weight_scale"));
		if (properties.getProperty(prefix+"alt_pull")!=null)           this.alt_pull=Double.parseDouble(properties.getProperty(prefix+            "alt_pull"));
		if (properties.getProperty(prefix+"alt_pull_rots")!=null)      this.alt_pull_rots=Double.parseDouble(properties.getProperty(prefix+       "alt_pull_rots"));
		if (properties.getProperty(prefix+"alt_pull_tilts  ")!=null)   this.alt_pull_tilts  =Double.parseDouble(properties.getProperty(prefix+    "alt_pull_tilts  "));
		if (properties.getProperty(prefix+"alt_pull_scales")!=null)    this.alt_pull_scales=Double.parseDouble(properties.getProperty(prefix+     "alt_pull_scales"));
		
		if (properties.getProperty(prefix+"pmap_move_only")!=null)     this.pmap_move_only=Boolean.parseBoolean(properties.getProperty(prefix+     "pmap_move_only"));		
		if (properties.getProperty(prefix+"pmap_ignore_affines")!=null)this.pmap_ignore_affines=Boolean.parseBoolean(properties.getProperty(prefix+"pmap_ignore_affines"));		
		if (properties.getProperty(prefix+"pmap_use_inv")!=null)       this.pmap_use_inv=Boolean.parseBoolean(properties.getProperty(prefix+       "pmap_use_inv"));		
		if (properties.getProperty(prefix+"pmap_skew_pull")!=null)     this.pmap_skew_pull=Double.parseDouble(properties.getProperty(prefix+       "pmap_skew_pull"));
		if (properties.getProperty(prefix+"pmap_tilt_pull")!=null)     this.pmap_tilt_pull=Double.parseDouble(properties.getProperty(prefix+       "pmap_tilt_pull"));
		if (properties.getProperty(prefix+"pmap_scale_pull")!=null)    this.pmap_scale_pull=Double.parseDouble(properties.getProperty(prefix+      "pmap_scale_pull"));
		if (properties.getProperty(prefix+"pmap_position_pull")!=null) this.pmap_position_pull=Double.parseDouble(properties.getProperty(prefix+   "pmap_position_pull"));
		if (properties.getProperty(prefix+"pmap_overlap_pow")!=null)   this.pmap_overlap_pow=Double.parseDouble(properties.getProperty(prefix+     "pmap_overlap_pow"));
		if (properties.getProperty(prefix+"pmap_rms_diff")!=null)      this.pmap_rms_diff=Double.parseDouble(properties.getProperty(prefix+        "pmap_rms_diff"));
		if (properties.getProperty(prefix+"pmap_num_iter")!=null)      this.pmap_num_iter=Integer.parseInt(properties.getProperty(prefix+          "pmap_num_iter"));
		if (properties.getProperty(prefix+"pmap_show_result")!=null)   this.pmap_show_result=Boolean.parseBoolean(properties.getProperty(prefix+   "pmap_show_result"));		
		if (properties.getProperty(prefix+"pmap_debugLevel")!=null)    this.pmap_debugLevel=Integer.parseInt(properties.getProperty(prefix+        "pmap_debugLevel"));
                                       
		if (properties.getProperty(prefix+"pequ_ignore_equalize")!=null)this.pequ_ignore_equalize=Boolean.parseBoolean(properties.getProperty(prefix+"pequ_ignore_equalize"));		
		if (properties.getProperty(prefix+"pequ_use_inv")!=null)        this.pequ_use_inv=Boolean.parseBoolean(properties.getProperty(prefix+        "pequ_use_inv"));		
		if (properties.getProperty(prefix+"pequ_scale_weight")!=null)   this.pequ_scale_weight=Double.parseDouble(properties.getProperty(prefix+     "pequ_scale_weight"));
		if (properties.getProperty(prefix+"pequ_pull_weight")!=null)    this.pequ_pull_weight=Double.parseDouble(properties.getProperty(prefix+      "pequ_pull_weight"));
		if (properties.getProperty(prefix+"pequ_half_weight_sec")!=null)this.pequ_half_weight_sec=Double.parseDouble(properties.getProperty(prefix+  "pequ_half_weight_sec"));
		if (properties.getProperty(prefix+"pequ_min_weight_sec ")!=null)this.pequ_min_weight_sec =Double.parseDouble(properties.getProperty(prefix+  "pequ_min_weight_sec "));
		if (properties.getProperty(prefix+"pequ_overlap_pow")!=null)    this.pequ_overlap_pow=Double.parseDouble(properties.getProperty(prefix+      "pequ_overlap_pow"));
		if (properties.getProperty(prefix+"pequ_rms_diff")!=null)       this.pequ_rms_diff=Double.parseDouble(properties.getProperty(prefix+         "pequ_rms_diff"));
		if (properties.getProperty(prefix+"pequ_num_iter")!=null)       this.pequ_num_iter=Integer.parseInt(properties.getProperty(prefix+           "pequ_num_iter"));
		if (properties.getProperty(prefix+"pequ_debugLevel")!=null)     this.pequ_debugLevel=Integer.parseInt(properties.getProperty(prefix+         "pequ_debugLevel"));
		
		if (properties.getProperty(prefix+"flt_list")!=         null)   this.flt_list=         Boolean.parseBoolean(properties.getProperty(prefix+"flt_list"));		
		if (properties.getProperty(prefix+"flt_undef_only")!=   null)   this.flt_undef_only=   Boolean.parseBoolean(properties.getProperty(prefix+"flt_undef_only"));		
		if (properties.getProperty(prefix+"flt_min_overlap")!=  null)   this.flt_min_overlap=  Double.parseDouble(properties.getProperty(prefix+  "flt_min_overlap"));
		if (properties.getProperty(prefix+"flt_max_overlapc")!= null)   this.flt_max_overlap=  Double.parseDouble(properties.getProperty(prefix+  "flt_max_overlap"));
		if (properties.getProperty(prefix+"flt_min_rms")!=      null)   this.flt_min_rms=      Double.parseDouble(properties.getProperty(prefix+  "flt_min_rms"));
		if (properties.getProperty(prefix+"flt_max_rms")!=      null)   this.flt_max_rms=      Double.parseDouble(properties.getProperty(prefix+  "flt_max_rms"));
		if (properties.getProperty(prefix+"flt_nan_rms")!=      null)   this.flt_nan_rms=      Boolean.parseBoolean(properties.getProperty(prefix+"flt_nan_rms"));
		if (properties.getProperty(prefix+"flt_filt_zoom")!=    null)   this.flt_filt_zoom=    Boolean.parseBoolean(properties.getProperty(prefix+"flt_filt_zoom"));
		if (properties.getProperty(prefix+"flt_min_zoom")!=     null)   this.flt_min_zoom=     Integer.parseInt(properties.getProperty(prefix+    "flt_min_zoom"));
		if (properties.getProperty(prefix+"flt_max_zoom")!=     null)   this.flt_max_zoom=     Integer.parseInt(properties.getProperty(prefix+    "flt_max_zoom"));
		if (properties.getProperty(prefix+"flt_min_sfm")!=      null)   this.flt_min_sfm=      Double.parseDouble(properties.getProperty(prefix+  "flt_min_sfm"));
		if (properties.getProperty(prefix+"flt_max_sfm")!=      null)   this.flt_max_sfm=      Double.parseDouble(properties.getProperty(prefix+  "flt_max_sfm"));
		if (properties.getProperty(prefix+"flt_alt")!=          null)   this.flt_alt=          Integer.parseInt(properties.getProperty(prefix+    "flt_alt"));
		if (properties.getProperty(prefix+"flt_orient")!=       null)   this.flt_orient=       Integer.parseInt(properties.getProperty(prefix+    "flt_orient"));
		if (properties.getProperty(prefix+"flt_show_names")!=   null)   this.flt_show_names=   Boolean.parseBoolean(properties.getProperty(prefix+"flt_show_names"));		
		if (properties.getProperty(prefix+"flt_show_overlaps")!=null)   this.flt_show_overlaps=Boolean.parseBoolean(properties.getProperty(prefix+"flt_show_overlaps"));		
		if (properties.getProperty(prefix+"flt_show_rms")!=     null)   this.flt_show_rms=     Boolean.parseBoolean(properties.getProperty(prefix+"flt_show_rms"));		
		if (properties.getProperty(prefix+"flt_show_zoom")!=    null)   this.flt_show_zoom=    Boolean.parseBoolean(properties.getProperty(prefix+"flt_show_zoom"));
		if (properties.getProperty(prefix+"flt_show_alt")!=     null)   this.flt_show_alt=     Boolean.parseBoolean(properties.getProperty(prefix+"flt_show_alt"));
		if (properties.getProperty(prefix+"patt_save_top")!=null)       this.patt_save_top=(String) properties.getProperty(prefix+                "patt_save_top");
		if (properties.getProperty(prefix+"patt_save_subdir")!=null)    this.patt_save_subdir=(String) properties.getProperty(prefix+             "patt_save_subdir");
		
		if (properties.getProperty(prefix+"center_reference")!=null)              this.center_reference=Boolean.parseBoolean(properties.getProperty(prefix+"center_reference"));		
		if (properties.getProperty(prefix+"lock_position")!=null)                 this.lock_position=Boolean.parseBoolean(properties.getProperty(prefix+"lock_position"));		
		if (properties.getProperty(prefix+"cuas_rotation")!=null)                 this.cuas_rotation=Boolean.parseBoolean(properties.getProperty(prefix+"cuas_rotation"));
		this.lock_position |= this.cuas_rotation; 
		if (properties.getProperty(prefix+"manual_correction")!=null)             this.manual_correction=Boolean.parseBoolean(properties.getProperty(prefix+"manual_correction"));		
		if (properties.getProperty(prefix+"overlap_sequences")!=null)             this.overlap_sequences=Boolean.parseBoolean(properties.getProperty(prefix+"overlap_sequences"));		
		if (properties.getProperty(prefix+"reset_photometric")!=null)             this.reset_photometric=Boolean.parseBoolean(properties.getProperty(prefix+"reset_photometric"));		
		if (properties.getProperty(prefix+"force_ref_dsi")!=null)                 this.force_ref_dsi=Boolean.parseBoolean(properties.getProperty(prefix+"force_ref_dsi"));		
		if (properties.getProperty(prefix+"force_orientations")!=null)            this.force_orientations=Boolean.parseBoolean(properties.getProperty(prefix+"force_orientations"));		
		if (properties.getProperty(prefix+"run_ly")!=null)                        this.run_ly=Boolean.parseBoolean(properties.getProperty(prefix+"run_ly"));		
		if (properties.getProperty(prefix+"run_ly_mode")!=null)                   this.run_ly_mode=Integer.parseInt(properties.getProperty(prefix+"run_ly_mode"));
		if (properties.getProperty(prefix+"run_ly_lma")!=null)                    this.run_ly_lma=Boolean.parseBoolean(properties.getProperty(prefix+"run_ly_lma"));		
		if (properties.getProperty(prefix+"min_num_orient")!=null)                this.min_num_orient=Integer.parseInt(properties.getProperty(prefix+"min_num_orient"));
		if (properties.getProperty(prefix+"min_num_interscene")!=null)            this.min_num_interscene=Integer.parseInt(properties.getProperty(prefix+"min_num_interscene"));
		if (properties.getProperty(prefix+"adjust_imu_orient")!=null)             this.adjust_imu_orient=Boolean.parseBoolean(properties.getProperty(prefix+"adjust_imu_orient"));
		if (properties.getProperty(prefix+"apply_imu_orient")!=null)              this.apply_imu_orient=Boolean.parseBoolean(properties.getProperty(prefix+"apply_imu_orient"));
		if (properties.getProperty(prefix+"orient_by_move")!=null)                this.orient_by_move=Boolean.parseBoolean(properties.getProperty(prefix+"orient_by_move"));
		if (properties.getProperty(prefix+"orient_by_rot")!=null)                 this.orient_by_rot=Boolean.parseBoolean(properties.getProperty(prefix+"orient_by_rot"));
		if (properties.getProperty(prefix+"orient_combo")!=null)                  this.orient_combo=Boolean.parseBoolean(properties.getProperty(prefix+"orient_combo"));
		if (properties.getProperty(prefix+"adjust_gyro")!=null)                   this.adjust_gyro=Boolean.parseBoolean(properties.getProperty(prefix+"adjust_gyro"));
		if (properties.getProperty(prefix+"apply_gyro")!=null)                    this.apply_gyro=Boolean.parseBoolean(properties.getProperty(prefix+"apply_gyro"));
		if (properties.getProperty(prefix+"adjust_accl")!=null)                   this.adjust_accl=Boolean.parseBoolean(properties.getProperty(prefix+"adjust_accl"));
		if (properties.getProperty(prefix+"apply_accl")!=null)                    this.apply_accl=Boolean.parseBoolean(properties.getProperty(prefix+"apply_accl"));
		if (properties.getProperty(prefix+"calc_quat_corr")!=null)                this.calc_quat_corr=Boolean.parseBoolean(properties.getProperty(prefix+"calc_quat_corr"));
		if (properties.getProperty(prefix+"apply_quat_corr")!=null)               this.apply_quat_corr=Boolean.parseBoolean(properties.getProperty(prefix+"apply_quat_corr"));
		if (properties.getProperty(prefix+"use_quat_corr")!=null)                 this.use_quat_corr=Boolean.parseBoolean(properties.getProperty(prefix+"use_quat_corr"));
		if (properties.getProperty(prefix+"inertial_only")!=null)                 this.inertial_only=Boolean.parseBoolean(properties.getProperty(prefix+"inertial_only"));
		if (properties.getProperty(prefix+"generate_egomotion")!=null)            this.generate_egomotion=Boolean.parseBoolean(properties.getProperty(prefix+"generate_egomotion"));
		if (properties.getProperty(prefix+"generate_mapped")!=null)               this.generate_mapped=Boolean.parseBoolean(properties.getProperty(prefix+"generate_mapped"));
		if (properties.getProperty(prefix+"reuse_video")!=null)                   this.reuse_video=Boolean.parseBoolean(properties.getProperty(prefix+"reuse_video"));
		
		if (properties.getProperty(prefix+"save_mapped_color")!=null)             this.save_mapped_color=Boolean.parseBoolean(properties.getProperty(prefix+"save_mapped_color"));
		if (properties.getProperty(prefix+"save_mapped_mono")!=null)              this.save_mapped_mono=Boolean.parseBoolean(properties.getProperty(prefix+"save_mapped_mono"));
		if (properties.getProperty(prefix+"gen_avi_color")!=null)                 this.gen_avi_color=Boolean.parseBoolean(properties.getProperty(prefix+"gen_avi_color"));
		if (properties.getProperty(prefix+"gen_avi_mono")!=null)                  this.gen_avi_mono=Boolean.parseBoolean(properties.getProperty(prefix+"gen_avi_mono"));
		if (properties.getProperty(prefix+"show_mapped_color")!=null)    this.show_mapped_color=Boolean.parseBoolean(properties.getProperty(prefix+"show_mapped_color"));
		if (properties.getProperty(prefix+"show_mapped_mono")!=null)     this.show_mapped_mono=Boolean.parseBoolean(properties.getProperty(prefix+"show_mapped_mono"));
		
		if (properties.getProperty(prefix+"generate_raw")!=null)                  this.generate_raw=Boolean.parseBoolean(properties.getProperty(prefix+"generate_raw"));
		if (properties.getProperty(prefix+"generate_inf")!=null)                  this.generate_inf=Boolean.parseBoolean(properties.getProperty(prefix+"generate_inf"));
		if (properties.getProperty(prefix+"generate_fg")!=null)                   this.generate_fg=Boolean.parseBoolean(properties.getProperty(prefix+"generate_fg"));
		if (properties.getProperty(prefix+"generate_bg")!=null)                   this.generate_bg=Boolean.parseBoolean(properties.getProperty(prefix+"generate_bg"));
		
		if (properties.getProperty(prefix+"generate_stereo")!=null)               this.generate_stereo=Boolean.parseBoolean(properties.getProperty(prefix+"generate_stereo"));
		if (properties.getProperty(prefix+"stereo_views_num")!=null) {
			 int stereo_views_num=Integer.parseInt(properties.getProperty(prefix+"stereo_views_num"));
			 this.stereo_views =        new double[stereo_views_num][];
			 this.generate_stereo_var = new boolean[stereo_views_num];
			 for (int i = 0; i < stereo_views_num; i++) {
				 if (properties.getProperty(prefix+"stereo_views_"+i)!=null) {
					 this.stereo_views[i]=StringToDoubles(properties.getProperty(prefix+"stereo_views_"+i),3);
				 }
				 if (properties.getProperty(prefix+"generate_stereo_var_"+i)!=null) {
					 this.generate_stereo_var[i]=Boolean.parseBoolean(properties.getProperty(prefix+"generate_stereo_var_"+i));
				 }
			 }
			 orderStereoViews(); 
		}
		
		if (properties.getProperty(prefix+"export_images")!=null)        this.export_images=Boolean.parseBoolean(properties.getProperty(prefix+"export_images"));		
		if (properties.getProperty(prefix+"show_images")!=null)          this.show_images=Boolean.parseBoolean(properties.getProperty(prefix+"show_images"));		
		if (properties.getProperty(prefix+"show_images_bgfg")!=null)     this.show_images_bgfg=Boolean.parseBoolean(properties.getProperty(prefix+"show_images_bgfg"));		
		if (properties.getProperty(prefix+"show_images_mono")!=null)     this.show_images_mono=Boolean.parseBoolean(properties.getProperty(prefix+"show_images_mono"));		
		if (properties.getProperty(prefix+"export_ranges")!=null)        this.export_ranges=Boolean.parseBoolean(properties.getProperty(prefix+"export_ranges"));
		if (properties.getProperty(prefix+"debug_ranges")!=null)         this.debug_ranges=Boolean.parseBoolean(properties.getProperty(prefix+"debug_ranges"));
		if (properties.getProperty(prefix+"show_ranges")!=null)          this.show_ranges=Boolean.parseBoolean(properties.getProperty(prefix+"show_ranges"));
		if (properties.getProperty(prefix+"export_ml_files")!=null)      this.export_ml_files=Boolean.parseBoolean(properties.getProperty(prefix+"export_ml_files"));

		if (properties.getProperty(prefix+"show_color_nan")!=null)       this.show_color_nan=Boolean.parseBoolean(properties.getProperty(prefix+"show_color_nan"));		
		if (properties.getProperty(prefix+"show_mono_nan")!=null)        this.show_mono_nan=Boolean.parseBoolean(properties.getProperty(prefix+"show_mono_nan"));

		if (properties.getProperty(prefix+"scale_extrap_atr")!=null)     this.scale_extrap_atr=Double.parseDouble(properties.getProperty(prefix+"scale_extrap_atr"));
		if (properties.getProperty(prefix+"scale_extrap_xyz")!=null)     this.scale_extrap_xyz=Double.parseDouble(properties.getProperty(prefix+"scale_extrap_xyz"));
		
		if (properties.getProperty(prefix+"avg_len")!=null)              this.avg_len=Integer.parseInt(properties.getProperty(prefix+"avg_len"));
		if (properties.getProperty(prefix+"min_num_scenes")!=null)       this.min_num_scenes=Integer.parseInt(properties.getProperty(prefix+"min_num_scenes"));
		if (properties.getProperty(prefix+"max_num_scenes")!=null)       this.max_num_scenes=Integer.parseInt(properties.getProperty(prefix+"max_num_scenes"));
		if (properties.getProperty(prefix+"blur_egomotion")!=null)       this.blur_egomotion=Double.parseDouble(properties.getProperty(prefix+"blur_egomotion"));
		if (properties.getProperty(prefix+"range_disparity_offset")!=null)this.range_disparity_offset=Double.parseDouble(properties.getProperty(prefix+"range_disparity_offset"));
		if (properties.getProperty(prefix+"range_min_strength")!=null)   this.range_min_strength=Double.parseDouble(properties.getProperty(prefix+"range_min_strength"));
		if (properties.getProperty(prefix+"range_max")!=null)            this.range_max=Double.parseDouble(properties.getProperty(prefix+"range_max"));

		if (properties.getProperty(prefix+"export3d")!=null)             this.export3d=Boolean.parseBoolean(properties.getProperty(prefix+"export3d"));		
		if (properties.getProperty(prefix+"export3dterrain")!=null)      this.export3dterrain=Boolean.parseBoolean(properties.getProperty(prefix+"export3dterrain"));		
		
		if (properties.getProperty(prefix+"export_CT")!=null)            this.export_CT=Boolean.parseBoolean(properties.getProperty(prefix+"export_CT"));		
		if (properties.getProperty(prefix+"ct_min")!=null)               this.ct_min=Double.parseDouble(properties.getProperty(prefix+"ct_min"));
		if (properties.getProperty(prefix+"ct_max")!=null)               this.ct_max=Double.parseDouble(properties.getProperty(prefix+"ct_max"));
		if (properties.getProperty(prefix+"ct_step")!=null)              this.ct_step=Double.parseDouble(properties.getProperty(prefix+"ct_step"));
		if (properties.getProperty(prefix+"ct_expand")!=null)            this.ct_expand=Integer.parseInt(properties.getProperty(prefix+"ct_expand"));
		
//		if (properties.getProperty(prefix+"terr_enable")!=null)          this.terr_enable=Boolean.parseBoolean(properties.getProperty(prefix+"terr_enable"));
		if (properties.getProperty(prefix+"terr_rmse_above")!=null)      this.terr_rmse_above=Double.parseDouble(properties.getProperty(prefix+"terr_rmse_above"));
		if (properties.getProperty(prefix+"terr_rmse_below")!=null)      this.terr_rmse_below=Double.parseDouble(properties.getProperty(prefix+"terr_rmse_below"));
		if (properties.getProperty(prefix+"terr_num_refine")!=null)      this.terr_num_refine=Integer.parseInt(properties.getProperty(prefix+"terr_num_refine"));
		if (properties.getProperty(prefix+"terr_frac_above")!=null)      this.terr_frac_above=Double.parseDouble(properties.getProperty(prefix+"terr_frac_above"));
		if (properties.getProperty(prefix+"terr_frac_below")!=null)      this.terr_frac_below=Double.parseDouble(properties.getProperty(prefix+"terr_frac_below"));
		if (properties.getProperty(prefix+"terr_rthrsh_abv")!=null)      this.terr_rthrsh_abv=Double.parseDouble(properties.getProperty(prefix+"terr_rthrsh_abv"));
		if (properties.getProperty(prefix+"terr_flat_brd")!=null)        this.terr_flat_brd=Boolean.parseBoolean(properties.getProperty(prefix+"terr_flat_brd"));
		if (properties.getProperty(prefix+"terr_expand")!=null)          this.terr_expand=Integer.parseInt(properties.getProperty(prefix+"terr_expand"));
		if (properties.getProperty(prefix+"terr_save")!=null)            this.terr_save=Boolean.parseBoolean(properties.getProperty(prefix+"terr_save"));
		
		if (properties.getProperty(prefix+"scene_is_ref_test")!=null)    this.scene_is_ref_test=Boolean.parseBoolean(properties.getProperty(prefix+"scene_is_ref_test"));		
		if (properties.getProperty(prefix+"render_ref")!=null)           this.render_ref=Boolean.parseBoolean(properties.getProperty(prefix+"render_ref"));		
		if (properties.getProperty(prefix+"render_scene")!=null)         this.render_scene=Boolean.parseBoolean(properties.getProperty(prefix+"render_scene"));		
		if (properties.getProperty(prefix+"toRGB")!=null)                this.toRGB=Boolean.parseBoolean(properties.getProperty(prefix+"toRGB"));		
		if (properties.getProperty(prefix+"show_2d_correlations")!=null) this.show_2d_correlations=Boolean.parseBoolean(properties.getProperty(prefix+"show_2d_correlations"));		
		if (properties.getProperty(prefix+"show_motion_vectors")!=null)  this.show_motion_vectors=Boolean.parseBoolean(properties.getProperty(prefix+"show_motion_vectors"));		
		if (properties.getProperty(prefix+"debug_level")!=null)          this.debug_level=Integer.parseInt(properties.getProperty(prefix+"debug_level"));
		if (properties.getProperty(prefix+"test_ers")!=null)             this.test_ers=Boolean.parseBoolean(properties.getProperty(prefix+"test_ers"));		
		if (properties.getProperty(prefix+"test_ers0")!=null)            this.test_ers0=Integer.parseInt(properties.getProperty(prefix+"test_ers0"));
		if (properties.getProperty(prefix+"test_ers1")!=null)            this.test_ers1=Integer.parseInt(properties.getProperty(prefix+"test_ers1"));
		if (properties.getProperty(prefix+"num_bottom")!=null)                    this.num_bottom=Integer.parseInt(properties.getProperty(prefix+"num_bottom"));
		if (properties.getProperty(prefix+"num_passes")!=null)                    this.num_passes=Integer.parseInt(properties.getProperty(prefix+"num_passes"));
		if (properties.getProperty(prefix+"max_change")!=null)                    this.max_change=Double.parseDouble(properties.getProperty(prefix+"max_change"));
		if (properties.getProperty(prefix+"min_disparity")!=null)                 this.min_disparity=Double.parseDouble(properties.getProperty(prefix+"min_disparity"));
		if (properties.getProperty(prefix+"max_sym_disparity")!=null)             this.max_sym_disparity=Double.parseDouble(properties.getProperty(prefix+"max_sym_disparity"));
		if (properties.getProperty(prefix+"min_strength_lma")!=null)              this.min_strength_lma=Double.parseDouble(properties.getProperty(prefix+"min_strength_lma"));
		if (properties.getProperty(prefix+"min_strength_replace")!=null)          this.min_strength_replace=Double.parseDouble(properties.getProperty(prefix+"min_strength_replace"));
		if (properties.getProperty(prefix+"min_strength_blur")!=null)             this.min_strength_blur=Double.parseDouble(properties.getProperty(prefix+"min_strength_blur"));
		if (properties.getProperty(prefix+"sigma")!=null)                         this.sigma=Double.parseDouble(properties.getProperty(prefix+"sigma"));
		if (properties.getProperty(prefix+"num_blur")!=null)                      this.num_blur=Integer.parseInt(properties.getProperty(prefix+"num_blur"));
		if (properties.getProperty(prefix+"disparity_corr")!=null)                this.disparity_corr=Double.parseDouble(properties.getProperty(prefix+"disparity_corr"));
		if (properties.getProperty(prefix+"outliers_nth_fromextrem")!=null)       this.outliers_nth_fromextrem=Integer.parseInt(properties.getProperty(prefix+"outliers_nth_fromextrem"));
		if (properties.getProperty(prefix+"outliers_tolerance_absolute")!=null)   this.outliers_tolerance_absolute=Double.parseDouble(properties.getProperty(prefix+"outliers_tolerance_absolute"));
		if (properties.getProperty(prefix+"outliers_tolerance_relative")!=null)   this.outliers_tolerance_relative=Double.parseDouble(properties.getProperty(prefix+"outliers_tolerance_relative"));
		if (properties.getProperty(prefix+"outliers_max_iter")!=null)             this.outliers_max_iter=Integer.parseInt(properties.getProperty(prefix+"outliers_max_iter"));
		if (properties.getProperty(prefix+"outliers_max_strength2")!=null)        this.outliers_max_strength2=Double.parseDouble(properties.getProperty(prefix+"outliers_max_strength2"));
		if (properties.getProperty(prefix+"outliers_nth_fromextrem2")!=null)      this.outliers_nth_fromextrem2=Integer.parseInt(properties.getProperty(prefix+"outliers_nth_fromextrem2"));
		if (properties.getProperty(prefix+"outliers_tolerance_absolute2")!=null)  this.outliers_tolerance_absolute2=Double.parseDouble(properties.getProperty(prefix+"outliers_tolerance_absolute2"));
		if (properties.getProperty(prefix+"outliers_tolerance_relative2")!=null)  this.outliers_tolerance_relative2=Double.parseDouble(properties.getProperty(prefix+"outliers_tolerance_relative2"));
		if (properties.getProperty(prefix+"outliers_lma_max_strength")!=null)     this.outliers_lma_max_strength=Double.parseDouble(properties.getProperty(prefix+"outliers_lma_max_strength"));
		if (properties.getProperty(prefix+"outliers_max_strength")!=null)         this.outliers_max_strength=Double.parseDouble(properties.getProperty(prefix+"outliers_max_strength"));
		if (properties.getProperty(prefix+"outliers_from_lma_max_strength")!=null)this.outliers_from_lma_max_strength=Double.parseDouble(properties.getProperty(prefix+"outliers_from_lma_max_strength"));
		if (properties.getProperty(prefix+"search_radius")!=null)                 this.search_radius=Integer.parseInt(properties.getProperty(prefix+"search_radius"));
		if (properties.getProperty(prefix+"remove_no_lma_neib")!=null)            this.remove_no_lma_neib=Boolean.parseBoolean(properties.getProperty(prefix+"remove_no_lma_neib"));
		if (properties.getProperty(prefix+"diff_from_lma_pos")!=null)             this.diff_from_lma_pos=Double.parseDouble(properties.getProperty(prefix+"diff_from_lma_pos"));
		if (properties.getProperty(prefix+"diff_from_lma_neg")!=null)             this.diff_from_lma_neg=Double.parseDouble(properties.getProperty(prefix+"diff_from_lma_neg"));
		if (properties.getProperty(prefix+"outliers_lma_nth_fromextrem")!=null)   this.outliers_lma_nth_fromextrem=Integer.parseInt(properties.getProperty(prefix+"outliers_lma_nth_fromextrem"));
		if (properties.getProperty(prefix+"filter_margin")!=null)                 this.filter_margin=Integer.parseInt(properties.getProperty(prefix+"filter_margin"));
		if (properties.getProperty(prefix+"weak_tolerance_absolute")!=null)       this.weak_tolerance_absolute=Double.parseDouble(properties.getProperty(prefix+"weak_tolerance_absolute"));
		if (properties.getProperty(prefix+"weak_tolerance_relative")!=null)       this.weak_tolerance_relative=Double.parseDouble(properties.getProperty(prefix+"weak_tolerance_relative"));
		if (properties.getProperty(prefix+"weak_min_neibs")!=null)                this.weak_min_neibs=Integer.parseInt(properties.getProperty(prefix+"weak_min_neibs"));
		if (properties.getProperty(prefix+"strong_strength")!=null)               this.strong_strength=Double.parseDouble(properties.getProperty(prefix+"strong_strength"));
		if (properties.getProperty(prefix+"weak_strength")!=null)                 this.weak_strength=Double.parseDouble(properties.getProperty(prefix+"weak_strength"));
		
		if (properties.getProperty(prefix+"sky_extract")!=null)          this.sky_extract=Boolean.parseBoolean(properties.getProperty(prefix+"sky_extract"));		
		if (properties.getProperty(prefix+"sky_recalc")!=null)           this.sky_recalc=Boolean.parseBoolean(properties.getProperty(prefix+"sky_recalc"));		
		if (properties.getProperty(prefix+"sky_highest_min")!=null)      this.sky_highest_min=Double.parseDouble(properties.getProperty(prefix+"sky_highest_min"));
		if (properties.getProperty(prefix+"cold_frac")!=null)            this.cold_frac=Double.parseDouble(properties.getProperty(prefix+"cold_frac"));
		if (properties.getProperty(prefix+"hot_frac")!=null)             this.hot_frac=Double.parseDouble(properties.getProperty(prefix+"hot_frac"));
		if (properties.getProperty(prefix+"cold_scale")!=null)           this.cold_scale=Double.parseDouble(properties.getProperty(prefix+"cold_scale"));
		if (properties.getProperty(prefix+"sky_seed")!=null)             this.sky_seed=Double.parseDouble(properties.getProperty(prefix+"sky_seed"));
		if (properties.getProperty(prefix+"lma_seed")!=null)             this.lma_seed=Double.parseDouble(properties.getProperty(prefix+"lma_seed"));
		if (properties.getProperty(prefix+"seed_temp")!=null)            this.seed_temp=Double.parseDouble(properties.getProperty(prefix+"seed_temp"));

		
		if (properties.getProperty(prefix+"sky_shrink")!=null)           this.sky_shrink=Integer.parseInt(properties.getProperty(prefix+"sky_shrink"));
		if (properties.getProperty(prefix+"seed_rows")!=null)            this.seed_rows=Integer.parseInt(properties.getProperty(prefix+"seed_rows"));
		if (properties.getProperty(prefix+"max_disparity")!=null)        this.max_disparity=Double.parseDouble(properties.getProperty(prefix+"max_disparity"));
		if (properties.getProperty(prefix+"max_disparity_strength")!=null)this.max_disparity_strength=Double.parseDouble(properties.getProperty(prefix+"max_disparity_strength"));
		if (properties.getProperty(prefix+"sky_lim")!=null)              this.sky_lim=Double.parseDouble(properties.getProperty(prefix+"sky_lim"));
		if (properties.getProperty(prefix+"lim_temp")!=null)             this.lim_temp=Double.parseDouble(properties.getProperty(prefix+"lim_temp"));
		if (properties.getProperty(prefix+"sky_expand_extra")!=null)     this.sky_expand_extra=Integer.parseInt(properties.getProperty(prefix+"sky_expand_extra"));
		if (properties.getProperty(prefix+"sky_bottleneck")!=null)       this.sky_bottleneck=Integer.parseInt(properties.getProperty(prefix+"sky_bottleneck"));
		if (properties.getProperty(prefix+"sky_reexpand_extra")!=null)   this.sky_reexpand_extra=Integer.parseInt(properties.getProperty(prefix+"sky_reexpand_extra"));
		if (properties.getProperty(prefix+"min_strength")!=null)         this.min_strength=Double.parseDouble(properties.getProperty(prefix+"min_strength"));
		if (properties.getProperty(prefix+"lowest_sky_row")!=null)       this.lowest_sky_row=Integer.parseInt(properties.getProperty(prefix+"lowest_sky_row"));
		if (properties.getProperty(prefix+"sky_bottom_override")!=null)  this.sky_bottom_override=Double.parseDouble(properties.getProperty(prefix+"sky_bottom_override"));
		if (properties.getProperty(prefix+"sky_override_shrink")!=null)  this.sky_override_shrink=Integer.parseInt(properties.getProperty(prefix+"sky_override_shrink"));

		if (properties.getProperty(prefix+"clouds_en")!=null)            this.clouds_en=Boolean.parseBoolean(properties.getProperty(prefix+"clouds_en"));		
		if (properties.getProperty(prefix+"clouds_fom")!=null)           this.clouds_fom=Double.parseDouble(properties.getProperty(prefix+"clouds_fom"));
		if (properties.getProperty(prefix+"clouds_spread")!=null)        this.clouds_spread=Double.parseDouble(properties.getProperty(prefix+"clouds_spread"));
		if (properties.getProperty(prefix+"clouds_disparity")!=null)     this.clouds_disparity=Double.parseDouble(properties.getProperty(prefix+"clouds_disparity"));
		if (properties.getProperty(prefix+"clouds_weak")!=null)          this.clouds_weak=Double.parseDouble(properties.getProperty(prefix+"clouds_weak"));
		if (properties.getProperty(prefix+"clouds_strength")!=null)      this.clouds_strength=Double.parseDouble(properties.getProperty(prefix+"clouds_strength"));
		if (properties.getProperty(prefix+"clouds_not_strength")!=null)  this.clouds_not_strength=Double.parseDouble(properties.getProperty(prefix+"clouds_not_strength"));
		if (properties.getProperty(prefix+"clouds_strong")!=null)        this.clouds_strong=Boolean.parseBoolean(properties.getProperty(prefix+"clouds_strong"));		

		if (properties.getProperty(prefix+"wall_en")!=null)              this.wall_en=Boolean.parseBoolean(properties.getProperty(prefix+"wall_en"));		
		if (properties.getProperty(prefix+"wall_dflt")!=null)            this.indoors_en=Boolean.parseBoolean(properties.getProperty(prefix+"wall_dflt"));		
		if (properties.getProperty(prefix+"wall_str")!=null)             this.wall_str=Double.parseDouble(properties.getProperty(prefix+"wall_str"));
		if (properties.getProperty(prefix+"wall_far")!=null)             this.wall_far=Double.parseDouble(properties.getProperty(prefix+"wall_far"));
		if (properties.getProperty(prefix+"wall_near")!=null)            this.wall_near=Double.parseDouble(properties.getProperty(prefix+"wall_near"));
		
		if (properties.getProperty(prefix+"treeline_en")!=null)          this.treeline_en=Boolean.parseBoolean(properties.getProperty(prefix+"treeline_en"));		
		if (properties.getProperty(prefix+"treeline_wide")!=null)        this.treeline_wide=Boolean.parseBoolean(properties.getProperty(prefix+"treeline_wide"));		
		if (properties.getProperty(prefix+"treeline_height")!=null)      this.treeline_height=Integer.parseInt(properties.getProperty(prefix+"treeline_height"));
		if (properties.getProperty(prefix+"treeline_width")!=null)       this.treeline_width=Integer.parseInt(properties.getProperty(prefix+"treeline_width"));
		if (properties.getProperty(prefix+"treeline_lim_high")!=null)    this.treeline_lim_high=Boolean.parseBoolean(properties.getProperty(prefix+"treeline_lim_high"));		
		if (properties.getProperty(prefix+"treeline_str")!=null)         this.treeline_str=Double.parseDouble(properties.getProperty(prefix+"treeline_str"));
		if (properties.getProperty(prefix+"treeline_far")!=null)         this.treeline_far=Double.parseDouble(properties.getProperty(prefix+"treeline_far"));
		if (properties.getProperty(prefix+"treeline_near")!=null)        this.treeline_near=Double.parseDouble(properties.getProperty(prefix+"treeline_near"));
		if (properties.getProperty(prefix+"treeline_fg_str")!=null)      this.treeline_fg_str=Double.parseDouble(properties.getProperty(prefix+"treeline_fg_str"));
		if (properties.getProperty(prefix+"treeline_fg_far")!=null)      this.treeline_fg_far=Double.parseDouble(properties.getProperty(prefix+"treeline_fg_far"));
		if (properties.getProperty(prefix+"treeline_fg_near")!=null)     this.treeline_fg_near=Double.parseDouble(properties.getProperty(prefix+"treeline_fg_near"));
		
		if (properties.getProperty(prefix+"indoors_en")!=null)           this.indoors_en=Boolean.parseBoolean(properties.getProperty(prefix+"indoors_en"));		
		if (properties.getProperty(prefix+"indoors_str")!=null)          this.indoors_str=Double.parseDouble(properties.getProperty(prefix+"indoors_str"));
		if (properties.getProperty(prefix+"indoors_disp")!=null)         this.indoors_disp=Double.parseDouble(properties.getProperty(prefix+"indoors_disp"));
		if (properties.getProperty(prefix+"indoors_min_out")!=null)      this.indoors_min_out=Integer.parseInt(properties.getProperty(prefix+"indoors_min_out"));
		
		if (properties.getProperty(prefix+"disp_boost_min")!=null)       this.disp_boost_min=Double.parseDouble(properties.getProperty(prefix+"disp_boost_min"));
		if (properties.getProperty(prefix+"disp_boost_diff")!=null)      this.disp_boost_diff=Double.parseDouble(properties.getProperty(prefix+"disp_boost_diff"));
		if (properties.getProperty(prefix+"disp_boost_neibs")!=null)     this.disp_boost_neibs=Integer.parseInt(properties.getProperty(prefix+"disp_boost_neibs"));
		if (properties.getProperty(prefix+"disp_boost_amount")!=null)    this.disp_boost_amount=Double.parseDouble(properties.getProperty(prefix+"disp_boost_amount"));
		if (properties.getProperty(prefix+"scale_combo_strength")!=null) this.scale_combo_strength=Double.parseDouble(properties.getProperty(prefix+"scale_combo_strength"));
		if (properties.getProperty(prefix+"save_debug_images")!=null)    this.save_debug_images=Boolean.parseBoolean(properties.getProperty(prefix+"save_debug_images"));		
		
		if (properties.getProperty(prefix+"margin")!=null)               this.margin=Integer.parseInt(properties.getProperty(prefix+"margin"));
		if (properties.getProperty(prefix+"sensor_mask_inter")!=null)    this.sensor_mask_inter=Integer.parseInt(properties.getProperty(prefix+"sensor_mask_inter"));
		if (properties.getProperty(prefix+"use_partial")!=null)          this.use_partial=Boolean.parseBoolean(properties.getProperty(prefix+"use_partial"));		
		if (properties.getProperty(prefix+"run_poly")!=null)             this.run_poly=Boolean.parseBoolean(properties.getProperty(prefix+"run_poly"));
		if (properties.getProperty(prefix+"centroid_radius")!=null)      this.centroid_radius=Double.parseDouble(properties.getProperty(prefix+"centroid_radius"));
		if (properties.getProperty(prefix+"n_recenter")!=null)           this.n_recenter=Integer.parseInt(properties.getProperty(prefix+"n_recenter"));
		if (properties.getProperty(prefix+"td_weight")!=null)            this.td_weight=Double.parseDouble(properties.getProperty(prefix+"td_weight"));
		if (properties.getProperty(prefix+"td_neib_weight")!=null)       this.td_neib_weight=Double.parseDouble(properties.getProperty(prefix+"td_neib_weight"));
		if (properties.getProperty(prefix+"pd_weight")!=null)            this.pd_weight=Double.parseDouble(properties.getProperty(prefix+"pd_weight"));
		if (properties.getProperty(prefix+"td_nopd_only")!=null)         this.td_nopd_only=Boolean.parseBoolean(properties.getProperty(prefix+"td_nopd_only"));
		if (properties.getProperty(prefix+"neib_notd_only")!=null)       this.neib_notd_only=Boolean.parseBoolean(properties.getProperty(prefix+"neib_notd_only"));

		if (properties.getProperty(prefix+"use_neibs")!=null)            this.use_neibs=Boolean.parseBoolean(properties.getProperty(prefix+       "use_neibs"));
		if (properties.getProperty(prefix+"neibs_nofpn_only")!=null)     this.neibs_nofpn_only=Boolean.parseBoolean(properties.getProperty(prefix+"neibs_nofpn_only"));
		if (properties.getProperty(prefix+"neibs_nofpn_init")!=null)     this.neibs_nofpn_init=Boolean.parseBoolean(properties.getProperty(prefix+"neibs_nofpn_init"));
		if (properties.getProperty(prefix+"redo_both")!=null)            this.redo_both=Boolean.parseBoolean(properties.getProperty(prefix+       "redo_both"));
		if (properties.getProperty(prefix+"min_num_neibs")!=null)        this.min_num_neibs=Integer.parseInt(properties.getProperty(prefix+       "min_num_neibs"));
		if (properties.getProperty(prefix+"scale_neibs_pd")!=null)       this.scale_neibs_pd=Double.parseDouble(properties.getProperty(prefix+    "scale_neibs_pd"));
		if (properties.getProperty(prefix+"scale_neibs_td")!=null)       this.scale_neibs_td=Double.parseDouble(properties.getProperty(prefix+    "scale_neibs_td"));
		if (properties.getProperty(prefix+"scale_avg_weight")!=null)     this.scale_avg_weight =Double.parseDouble(properties.getProperty(prefix+ "scale_avg_weight"));
		if (properties.getProperty(prefix+"min_str")!=null)              this.min_str=Double.parseDouble(properties.getProperty(prefix+"min_str"));
		if (properties.getProperty(prefix+"min_str_fpn")!=null)          this.min_str_fpn=Double.parseDouble(properties.getProperty(prefix+"min_str_fpn"));
		if (properties.getProperty(prefix+"min_str_sum")!=null)          this.min_str_sum=Double.parseDouble(properties.getProperty(prefix+"min_str_sum"));
		if (properties.getProperty(prefix+"min_str_sum_fpn")!=null)      this.min_str_sum_fpn=Double.parseDouble(properties.getProperty(prefix+"min_str_sum_fpn"));
		if (properties.getProperty(prefix+"min_str_neib")!=null)         this.min_str_neib=Double.parseDouble(properties.getProperty(prefix+"min_str_neib"));
		if (properties.getProperty(prefix+"min_str_neib_fpn")!=null)     this.min_str_neib_fpn=Double.parseDouble(properties.getProperty(prefix+"min_str_neib_fpn"));
		
		if (properties.getProperty(prefix+"min_neibs")!=null)            this.min_neibs=Integer.parseInt(properties.getProperty(prefix+"min_neibs"));
		if (properties.getProperty(prefix+"weight_zero_neibs")!=null)    this.weight_zero_neibs=Double.parseDouble(properties.getProperty(prefix+"weight_zero_neibs"));
		if (properties.getProperty(prefix+"half_disparity")!=null)       this.half_disparity=Double.parseDouble(properties.getProperty(prefix+"half_disparity"));
		if (properties.getProperty(prefix+"half_avg_diff")!=null)        this.half_avg_diff=Double.parseDouble(properties.getProperty(prefix+"half_avg_diff"));

		if (properties.getProperty(prefix+"eig_use")!=null)              this.eig_use=Boolean.parseBoolean(properties.getProperty(prefix+"eig_use"));		
		if (properties.getProperty(prefix+"eig_xy_lma")!=null)           this.eig_xy_lma=Boolean.parseBoolean(properties.getProperty(prefix+"eig_xy_lma"));		
		if (properties.getProperty(prefix+"eig_str_sum")!=null)          this.eig_str_sum=Double.parseDouble(properties.getProperty(prefix+"eig_str_sum"));
		if (properties.getProperty(prefix+"eig_str_sum_fpn")!=null)      this.eig_str_sum_fpn=Double.parseDouble(properties.getProperty(prefix+"eig_str_sum_fpn"));
		if (properties.getProperty(prefix+"eig_str_neib")!=null)         this.eig_str_neib=Double.parseDouble(properties.getProperty(prefix+"eig_str_neib"));
		if (properties.getProperty(prefix+"eig_str_neib_fpn")!=null)     this.eig_str_neib_fpn=Double.parseDouble(properties.getProperty(prefix+"eig_str_neib_fpn"));
		if (properties.getProperty(prefix+"eig_min_abs")!=null)          this.eig_min_abs=Double.parseDouble(properties.getProperty(prefix+"eig_min_abs"));
		if (properties.getProperty(prefix+"eig_min_rel")!=null)          this.eig_min_rel=Double.parseDouble(properties.getProperty(prefix+"eig_min_rel"));
		if (properties.getProperty(prefix+"eig_sub_frac")!=null)         this.eig_sub_frac=Double.parseDouble(properties.getProperty(prefix+"eig_sub_frac"));
		if (properties.getProperty(prefix+"eig_max_sqrt")!=null)         this.eig_max_sqrt=Double.parseDouble(properties.getProperty(prefix+"eig_max_sqrt"));
		if (properties.getProperty(prefix+"eig_min_sqrt")!=null)         this.eig_min_sqrt=Double.parseDouble(properties.getProperty(prefix+"eig_min_sqrt"));
		
		if (properties.getProperty(prefix+"eig_recenter")!=null)         this.eig_recenter=Integer.parseInt(properties.getProperty(prefix+"eig_recenter"));
		if (properties.getProperty(prefix+"eig_sub_frac1")!=null)        this.eig_sub_frac1=Double.parseDouble(properties.getProperty(prefix+"eig_sub_frac1"));
		if (properties.getProperty(prefix+"eig_scale_axes")!=null)       this.eig_scale_axes=Double.parseDouble(properties.getProperty(prefix+"eig_scale_axes"));
		if (properties.getProperty(prefix+"eig_inc_axes")!=null)         this.eig_inc_axes=Double.parseDouble(properties.getProperty(prefix+"eig_inc_axes"));
		
        if (properties.getProperty(prefix+"eig_use_neibs")!=null)        this.eig_use_neibs=Boolean.parseBoolean(properties.getProperty(prefix+"eig_use_neibs"));		
		if (properties.getProperty(prefix+"eig_min_weaks")!=null)        this.eig_min_weaks=Integer.parseInt(properties.getProperty(prefix+"eig_min_weaks"));
		if (properties.getProperty(prefix+"eig_min_strongs")!=null)      this.eig_min_strongs=Integer.parseInt(properties.getProperty(prefix+"eig_min_strongs"));
		if (properties.getProperty(prefix+"eig_disp_diff")!=null)        this.eig_disp_diff=Double.parseDouble(properties.getProperty(prefix+"eig_disp_diff"));
        if (properties.getProperty(prefix+"eig_remove_neibs")!=null)     this.eig_remove_neibs=Boolean.parseBoolean(properties.getProperty(prefix+"eig_remove_neibs"));		
		if (properties.getProperty(prefix+"eig_filt_other")!=null)       this.eig_filt_other=Boolean.parseBoolean(properties.getProperty(prefix+"eig_filt_other"));		
		if (properties.getProperty(prefix+"eig_max_rms")!=null)          this.eig_max_rms=Double.parseDouble(properties.getProperty(prefix+"eig_max_rms"));
        if (properties.getProperty(prefix+"eig_fast2x2")!=null)          this.eig_fast2x2=Boolean.parseBoolean(properties.getProperty(prefix+"eig_fast2x2"));		
		
		if (properties.getProperty(prefix+"use_combo_reliable")!=null)   this.use_combo_reliable=Boolean.parseBoolean(properties.getProperty(prefix+"use_combo_reliable"));
		else if (properties.getProperty(prefix+"use_combo_relaible")!=null)   this.use_combo_reliable=Boolean.parseBoolean(properties.getProperty(prefix+"use_combo_relaible"));		
		if (properties.getProperty(prefix+"ref_need_lma")!=null)         this.ref_need_lma=Boolean.parseBoolean(properties.getProperty(prefix+"ref_need_lma"));		
		if (properties.getProperty(prefix+"ref_need_lma_combo")!=null)   this.ref_need_lma_combo=Boolean.parseBoolean(properties.getProperty(prefix+"ref_need_lma_combo"));
		if (properties.getProperty(prefix+"min_ref_str")!=null)          this.min_ref_str=Double.parseDouble(properties.getProperty(prefix+"min_ref_str"));
		if (properties.getProperty(prefix+"min_ref_str_lma")!=null)      this.min_ref_str_lma=Double.parseDouble(properties.getProperty(prefix+"min_ref_str_lma"));
		if (properties.getProperty(prefix+"min_ref_frac")!=null)         this.min_ref_frac=Double.parseDouble(properties.getProperty(prefix+"min_ref_frac"));

		if (properties.getProperty(prefix+"ref_smooth")!=null)           this.ref_smooth=Boolean.parseBoolean(properties.getProperty(prefix+"ref_smooth"));		
		if (properties.getProperty(prefix+"ref_smooth_always")!=null)    this.ref_smooth_always=Boolean.parseBoolean(properties.getProperty(prefix+"ref_smooth_always"));		
		if (properties.getProperty(prefix+"ref_sigma")!=null)            this.ref_sigma=Double.parseDouble(properties.getProperty(prefix+"ref_sigma"));
		if (properties.getProperty(prefix+"ref_smooth_diff")!=null)      this.ref_smooth_diff=Double.parseDouble(properties.getProperty(prefix+"ref_smooth_diff"));

		if (properties.getProperty(prefix+"sfm_filter")!=null)           this.sfm_filter=Boolean.parseBoolean(properties.getProperty(prefix+"sfm_filter"));		
		if (properties.getProperty(prefix+"sfm_minmax")!=null)           this.sfm_minmax=Double.parseDouble(properties.getProperty(prefix+"sfm_minmax"));
		if (properties.getProperty(prefix+"sfm_fracmax")!=null)          this.sfm_fracmax=Double.parseDouble(properties.getProperty(prefix+"sfm_fracmax"));
		if (properties.getProperty(prefix+"sfm_fracall")!=null)          this.sfm_fracall=Double.parseDouble(properties.getProperty(prefix+"sfm_fracall"));
		
		if (properties.getProperty(prefix+"pix_step")!=null)             this.pix_step=Integer.parseInt(properties.getProperty(prefix+"pix_step"));
		if (properties.getProperty(prefix+"search_rad")!=null)           this.search_rad=Integer.parseInt(properties.getProperty(prefix+"search_rad"));
		if (properties.getProperty(prefix+"maybe_sum")!=null)            this.maybe_sum=Double.parseDouble(properties.getProperty(prefix+"maybe_sum"));
		if (properties.getProperty(prefix+"sure_sum")!=null)             this.sure_sum=Double.parseDouble(properties.getProperty(prefix+"sure_sum"));
		if (properties.getProperty(prefix+"maybe_avg")!=null)            this.maybe_avg=Double.parseDouble(properties.getProperty(prefix+"maybe_avg"));
		if (properties.getProperty(prefix+"sure_avg")!=null)             this.sure_avg=Double.parseDouble(properties.getProperty(prefix+"sure_avg"));
		if (properties.getProperty(prefix+"max_search_rms")!=null)       this.max_search_rms=Double.parseDouble(properties.getProperty(prefix+"max_search_rms"));
		if (properties.getProperty(prefix+"maybe_fom")!=null)            this.maybe_fom=Double.parseDouble(properties.getProperty(prefix+"maybe_fom"));
		if (properties.getProperty(prefix+"sure_fom")!=null)             this.sure_fom=Double.parseDouble(properties.getProperty(prefix+"sure_fom"));
		if (properties.getProperty(prefix+"treat_serch_fpn")!=null)      this.treat_serch_fpn=Boolean.parseBoolean(properties.getProperty(prefix+"treat_serch_fpn"));		
		if (properties.getProperty(prefix+"use_combo_dsi")!=null)        this.use_combo_dsi=Boolean.parseBoolean(properties.getProperty(prefix+"use_combo_dsi"));		
		if (properties.getProperty(prefix+"use_lma_dsi")!=null)          this.use_lma_dsi=Boolean.parseBoolean(properties.getProperty(prefix+"use_lma_dsi"));
		if (properties.getProperty(prefix+"fpn_remove")!=null)           this.fpn_remove=Boolean.parseBoolean(properties.getProperty(prefix+"fpn_remove"));
		if (properties.getProperty(prefix+"fpn_max_offset")!=null)       this.fpn_max_offset=Double.parseDouble(properties.getProperty(prefix+"fpn_max_offset"));
		if (properties.getProperty(prefix+"fpn_min_offset")!=null)       this.fpn_min_offset=Double.parseDouble(properties.getProperty(prefix+"fpn_min_offset"));
		if (properties.getProperty(prefix+"fpn_radius")!=null)           this.fpn_radius=Double.parseDouble(properties.getProperty(prefix+"fpn_radius"));
		if (properties.getProperty(prefix+"fpn_ignore_border")!=null)    this.fpn_ignore_border=Boolean.parseBoolean(properties.getProperty(prefix+"fpn_ignore_border"));
		
		if (properties.getProperty(prefix+"min_offset")!=null)           this.min_offset=Double.parseDouble(properties.getProperty(prefix+"min_offset"));
		if (properties.getProperty(prefix+"max_pull_jump")!=null)        this.max_pull_jump=Double.parseDouble(properties.getProperty(prefix+"max_pull_jump"));
		if (properties.getProperty(prefix+"max_rel_offset")!=null)       this.max_rel_offset=Double.parseDouble(properties.getProperty(prefix+"max_rel_offset"));
		if (properties.getProperty(prefix+"max_roll_deg")!=null)         this.max_roll_deg=Double.parseDouble(properties.getProperty(prefix+"max_roll_deg"));
		if (properties.getProperty(prefix+"max_zoom_diff")!=null)        this.max_zoom_diff=Double.parseDouble(properties.getProperty(prefix+"max_zoom_diff"));
		if (properties.getProperty(prefix+"fpn_skip")!=null)             this.fpn_skip=Boolean.parseBoolean(properties.getProperty(prefix+"fpn_skip"));
		if (properties.getProperty(prefix+"fpn_rematch")!=null)          this.fpn_rematch=Boolean.parseBoolean(properties.getProperty(prefix+"fpn_rematch"));
		if (properties.getProperty(prefix+"refine_invert")!=null)        this.refine_invert=Boolean.parseBoolean(properties.getProperty(prefix+"refine_invert"));
		if (properties.getProperty(prefix+"use_precomp")!=null)          this.use_precomp=Boolean.parseBoolean(properties.getProperty(prefix+"use_precomp"));
		
		if (properties.getProperty(prefix+"mov_en")!=null)               this.mov_en=Boolean.parseBoolean(properties.getProperty(prefix+"mov_en"));
		if (properties.getProperty(prefix+"mov_sigma")!=null)            this.mov_sigma=Double.parseDouble(properties.getProperty(prefix+"mov_sigma"));
		if (properties.getProperty(prefix+"mov_max_std")!=null)          this.mov_max_std=Double.parseDouble(properties.getProperty(prefix+"mov_max_std"));
		if (properties.getProperty(prefix+"mov_thresh_rel")!=null)       this.mov_thresh_rel=Double.parseDouble(properties.getProperty(prefix+"mov_thresh_rel"));
		if (properties.getProperty(prefix+"mov_thresh_abs")!=null)       this.mov_thresh_abs=Double.parseDouble(properties.getProperty(prefix+"mov_thresh_abs"));
		if (properties.getProperty(prefix+"mov_clust_max")!=null)        this.mov_clust_max=Double.parseDouble(properties.getProperty(prefix+"mov_clust_max"));
		if (properties.getProperty(prefix+"mov_grow")!=null)             this.mov_grow=Integer.parseInt(properties.getProperty(prefix+"mov_grow"));
		if (properties.getProperty(prefix+"mov_max_len")!=null)          this.mov_max_len=Integer.parseInt(properties.getProperty(prefix+"mov_max_len"));

		if (properties.getProperty(prefix+"mov_show")!=null)             this.mov_show=Boolean.parseBoolean(properties.getProperty(prefix+"mov_show"));
		if (properties.getProperty(prefix+"mov_debug_level")!=null)      this.mov_debug_level=Integer.parseInt(properties.getProperty(prefix+"mov_debug_level"));
		if (properties.getProperty(prefix+"adjust_atr_0")!=null)         this.adjust_atr[0]=Boolean.parseBoolean(properties.getProperty(prefix+"adjust_atr_0"));		
		if (properties.getProperty(prefix+"adjust_atr_1")!=null)         this.adjust_atr[1]=Boolean.parseBoolean(properties.getProperty(prefix+"adjust_atr_1"));		
		if (properties.getProperty(prefix+"adjust_atr_2")!=null)         this.adjust_atr[2]=Boolean.parseBoolean(properties.getProperty(prefix+"adjust_atr_2"));		
		if (properties.getProperty(prefix+"adjust_xyz_0")!=null)         this.adjust_xyz[0]=Boolean.parseBoolean(properties.getProperty(prefix+"adjust_xyz_0"));		
		if (properties.getProperty(prefix+"adjust_xyz_1")!=null)         this.adjust_xyz[1]=Boolean.parseBoolean(properties.getProperty(prefix+"adjust_xyz_1"));		
		if (properties.getProperty(prefix+"adjust_xyz_2")!=null)         this.adjust_xyz[2]=Boolean.parseBoolean(properties.getProperty(prefix+"adjust_xyz_2"));		
		if (properties.getProperty(prefix+"exit_change_atr")!=null)      this.exit_change_atr=Double.parseDouble(properties.getProperty(prefix+"exit_change_atr"));
		if (properties.getProperty(prefix+"exit_change_xyz")!=null)      this.exit_change_xyz=Double.parseDouble(properties.getProperty(prefix+"exit_change_xyz"));
		if (properties.getProperty(prefix+"max_cycles")!=null)           this.max_cycles=Integer.parseInt(properties.getProperty(prefix+"max_cycles"));
		if (properties.getProperty(prefix+"max_LMA")!=null)              this.max_LMA=Integer.parseInt(properties.getProperty(prefix+"max_LMA"));
		if (properties.getProperty(prefix+"max_rms")!=null)              this.max_rms=Double.parseDouble(properties.getProperty(prefix+"max_rms"));
		
		if (properties.getProperty(prefix+"eq_en")!=null)                this.eq_en=Boolean.parseBoolean(properties.getProperty(prefix+"eq_en"));		
		if (properties.getProperty(prefix+"eq_stride_hor")!=null)        this.eq_stride_hor=Integer.parseInt(properties.getProperty(prefix+"eq_stride_hor"));
		if (properties.getProperty(prefix+"eq_stride_vert")!=null)       this.eq_stride_vert=Integer.parseInt(properties.getProperty(prefix+"eq_stride_vert"));
		if (properties.getProperty(prefix+"eq_min_stile_weight")!=null)  this.eq_min_stile_weight=Double.parseDouble(properties.getProperty(prefix+"eq_min_stile_weight"));
		if (properties.getProperty(prefix+"eq_min_stile_number")!=null)  this.eq_min_stile_number=Integer.parseInt(properties.getProperty(prefix+"eq_min_stile_number"));
		if (properties.getProperty(prefix+"eq_min_stile_fraction")!=null)this.eq_min_stile_fraction=Double.parseDouble(properties.getProperty(prefix+"eq_min_stile_fraction"));
		if (properties.getProperty(prefix+"eq_min_disparity")!=null)     this.eq_min_disparity=Double.parseDouble(properties.getProperty(prefix+"eq_min_disparity"));
		if (properties.getProperty(prefix+"eq_max_disparity")!=null)     this.eq_max_disparity=Double.parseDouble(properties.getProperty(prefix+"eq_max_disparity"));
		if (properties.getProperty(prefix+"eq_weight_add")!=null)        this.eq_weight_add=Double.parseDouble(properties.getProperty(prefix+"eq_weight_add"));
		if (properties.getProperty(prefix+"eq_weight_scale")!=null)      this.eq_weight_scale=Double.parseDouble(properties.getProperty(prefix+"eq_weight_scale"));
		if (properties.getProperty(prefix+"eq_level")!=null)             this.eq_level=Double.parseDouble(properties.getProperty(prefix+"eq_level"));
		
		if (properties.getProperty(prefix+"mb_en")!=null)                this.mb_en=Boolean.parseBoolean(properties.getProperty(prefix+"mb_en"));
		if (properties.getProperty(prefix+"mb_tau")!=null)               this.mb_tau=Double.parseDouble(properties.getProperty(prefix+"mb_tau"));
		if (properties.getProperty(prefix+"mb_max_gain")!=null)          this.mb_max_gain=Double.parseDouble(properties.getProperty(prefix+"mb_max_gain"));
		if (properties.getProperty(prefix+"mb_max_gain_inter")!=null)    this.mb_max_gain_inter=Double.parseDouble(properties.getProperty(prefix+"mb_max_gain_inter"));

		if (properties.getProperty(prefix+"cuas_update_existing")!=null) this.cuas_update_existing=Boolean.parseBoolean(properties.getProperty(prefix+"cuas_update_existing"));
		if (properties.getProperty(prefix+"cuas_discard_border")!=null)  this.cuas_discard_border=Integer.parseInt(properties.getProperty(prefix+"cuas_discard_border"));
		if (properties.getProperty(prefix+"cuas_max_fold")!=null)        this.cuas_max_fold=Double.parseDouble(properties.getProperty(prefix+"cuas_max_fold"));
		if (properties.getProperty(prefix+"cuas_min_in_row_col")!=null)  this.cuas_min_in_row_col=Integer.parseInt(properties.getProperty(prefix+"cuas_min_in_row_col"));
		if (properties.getProperty(prefix+"cuas_clt_variant")!=null)     this.cuas_clt_variant=Double.parseDouble(properties.getProperty(prefix+"cuas_clt_variant"));
		if (properties.getProperty(prefix+"cuas_clt_threshold")!=null)   this.cuas_clt_threshold=Double.parseDouble(properties.getProperty(prefix+"cuas_clt_threshold"));
		if (properties.getProperty(prefix+"cuas_clt_decrease")!=null)    this.cuas_clt_decrease=Double.parseDouble(properties.getProperty(prefix+"cuas_clt_decrease"));
		if (properties.getProperty(prefix+"cuas_decay_average")!=null)   this.cuas_decay_average=Double.parseDouble(properties.getProperty(prefix+"cuas_decay_average"));
		if (properties.getProperty(prefix+"cuas_keep_fraction")!=null)   this.cuas_keep_fraction=Double.parseDouble(properties.getProperty(prefix+"cuas_keep_fraction"));
		if (properties.getProperty(prefix+"cuas_step")!=null)            this.cuas_step=Boolean.parseBoolean(properties.getProperty(prefix+"cuas_step"));
		if (properties.getProperty(prefix+"cuas_subtract_fpn")!=null)    this.cuas_subtract_fpn=Boolean.parseBoolean(properties.getProperty(prefix+"cuas_subtract_fpn"));
		if (properties.getProperty(prefix+"cuas_calc_fpn")!=null)        this.cuas_calc_fpn=Boolean.parseBoolean(properties.getProperty(prefix+"cuas_calc_fpn"));
		if (properties.getProperty(prefix+"cuas_rot_period")!=null)      this.cuas_rot_period=Double.parseDouble(properties.getProperty(prefix+"cuas_rot_period"));
		if (properties.getProperty(prefix+"cuas_min_series")!=null)      this.cuas_min_series=Double.parseDouble(properties.getProperty(prefix+"cuas_min_series"));
		
		if (properties.getProperty(prefix+"cuas_subtract_rowcol")!=null) this.cuas_subtract_rowcol=Boolean.parseBoolean(properties.getProperty(prefix+"cuas_subtract_rowcol"));
		if (properties.getProperty(prefix+"cuas_calc_rowcol")!=null)     this.cuas_calc_rowcol=Boolean.parseBoolean(properties.getProperty(prefix+"cuas_calc_rowcol"));
		if (properties.getProperty(prefix+"cuas_um_sigma")!=null)        this.cuas_um_sigma=Double.parseDouble(properties.getProperty(prefix+"cuas_um_sigma"));
		if (properties.getProperty(prefix+"cuas_max_abs_rowcol")!=null)  this.cuas_max_abs_rowcol=Double.parseDouble(properties.getProperty(prefix+"cuas_max_abs_rowcol"));
		if (properties.getProperty(prefix+"cuas_outliers_rowcol")!=null) this.cuas_outliers_rowcol=Double.parseDouble(properties.getProperty(prefix+"cuas_outliers_rowcol"));
		if (properties.getProperty(prefix+"cuas_reset_first")!=null)     this.cuas_reset_first=Boolean.parseBoolean(properties.getProperty(prefix+"cuas_reset_first"));
		
		if (properties.getProperty(prefix+"cuas_invert_margins")!=null)  this.cuas_invert_margins=Integer.parseInt(properties.getProperty(prefix+"cuas_invert_margins"));
		if (properties.getProperty(prefix+"cuas_invert_iters")!=null)    this.cuas_invert_iters=Integer.parseInt(properties.getProperty(prefix+"cuas_invert_iters"));
		if (properties.getProperty(prefix+"cuas_invert_tolerance")!=null)this.cuas_invert_tolerance=Double.parseDouble(properties.getProperty(prefix+"cuas_invert_tolerance"));
		if (properties.getProperty(prefix+"cuas_invert_gap2")!=null)     this.cuas_invert_gap2=Integer.parseInt(properties.getProperty(prefix+"cuas_invert_gap2"));
		
		if (properties.getProperty(prefix+"cuas_w_um_sigma")!=null)      this.cuas_w_um_sigma=Double.parseDouble(properties.getProperty(prefix+"cuas_w_um_sigma"));
		if (properties.getProperty(prefix+"cuas_w_good_margins")!=null)  this.cuas_w_good_margins=Integer.parseInt(properties.getProperty(prefix+"cuas_w_good_margins"));
		if (properties.getProperty(prefix+"cuas_w_threshold")!=null)     this.cuas_w_threshold=Double.parseDouble(properties.getProperty(prefix+"cuas_w_threshold"));
		if (properties.getProperty(prefix+"cuas_w_scale_thresh")!=null)  this.cuas_w_scale_thresh=Double.parseDouble(properties.getProperty(prefix+"cuas_w_scale_thresh"));
		if (properties.getProperty(prefix+"cuas_w_expand_max")!=null)    this.cuas_w_expand_max=Boolean.parseBoolean(properties.getProperty(prefix+"cuas_w_expand_max"));
		if (properties.getProperty(prefix+"cuas_w_outliers")!=null)      this.cuas_w_outliers=Double.parseDouble(properties.getProperty(prefix+"cuas_w_outliers"));
		if (properties.getProperty(prefix+"cuas_w_blur")!=null)          this.cuas_w_blur=Double.parseDouble(properties.getProperty(prefix+"cuas_w_blur"));

		if (properties.getProperty(prefix+"cuas_dsi_parent")!=null)      this.cuas_dsi_parent=Boolean.parseBoolean(properties.getProperty(prefix+"cuas_dsi_parent"));
		if (properties.getProperty(prefix+"cuas_fg_strength")!=null)     this.cuas_fg_strength=Double.parseDouble(properties.getProperty(prefix+"cuas_fg_strength"));
		if (properties.getProperty(prefix+"cuas_disp_add")!=null)        this.cuas_disp_add=Double.parseDouble(properties.getProperty(prefix+"cuas_disp_add"));
		if (properties.getProperty(prefix+"cuas_skip_bottom")!=null)     this.cuas_skip_bottom=Integer.parseInt(properties.getProperty(prefix+"cuas_skip_bottom"));
		if (properties.getProperty(prefix+"cuas_multi_strength")!=null)  this.cuas_multi_strength=Double.parseDouble(properties.getProperty(prefix+"cuas_multi_strength"));
		if (properties.getProperty(prefix+"cuas_reliable_str")!=null)    this.cuas_reliable_str=Double.parseDouble(properties.getProperty(prefix+"cuas_reliable_str"));
		
		if (properties.getProperty(prefix+"cuas_fat_zero")!=null)        this.cuas_fat_zero=Double.parseDouble(properties.getProperty(prefix+"cuas_fat_zero"));
		if (properties.getProperty(prefix+"cuas_cent_radius")!=null)     this.cuas_cent_radius=Double.parseDouble(properties.getProperty(prefix+"cuas_cent_radius"));
		if (properties.getProperty(prefix+"cuas_n_recenter")!=null)      this.cuas_n_recenter=Integer.parseInt(properties.getProperty(prefix+"cuas_n_recenter"));
		if (properties.getProperty(prefix+"cuas_rstr")!=null)            this.cuas_rstr=Double.parseDouble(properties.getProperty(prefix+"cuas_rstr"));
		if (properties.getProperty(prefix+"cuas_smooth")!=null)          this.cuas_smooth=Boolean.parseBoolean(properties.getProperty(prefix+"cuas_smooth"));
		if (properties.getProperty(prefix+"cuas_corr_pairs")!=null)      this.cuas_corr_pairs=Integer.parseInt(properties.getProperty(prefix+"cuas_corr_pairs"));
		if (properties.getProperty(prefix+"cuas_corr_offset")!=null)     this.cuas_corr_offset=Integer.parseInt(properties.getProperty(prefix+"cuas_corr_offset"));
		if (properties.getProperty(prefix+"cuas_gaussian_ra")!=null)     this.cuas_gaussian_ra=Boolean.parseBoolean(properties.getProperty(prefix+"cuas_gaussian_ra"));
		if (properties.getProperty(prefix+"cuas_temporal_um")!=null)     this.cuas_temporal_um=Integer.parseInt(properties.getProperty(prefix+"cuas_temporal_um"));
		if (properties.getProperty(prefix+"cuas_tum_threshold")!=null)   this.cuas_tum_threshold=Double.parseDouble(properties.getProperty(prefix+"cuas_tum_threshold"));
		
		if (properties.getProperty(prefix+"cuas_precorr_ra")!=null)      this.cuas_precorr_ra=Integer.parseInt(properties.getProperty(prefix+"cuas_precorr_ra"));
		if (properties.getProperty(prefix+"cuas_corr_step")!=null)       this.cuas_corr_step=Integer.parseInt(properties.getProperty(prefix+"cuas_corr_step"));
		
		if (properties.getProperty(prefix+"cuas_half_step")!=null)       this.cuas_half_step=Boolean.parseBoolean(properties.getProperty(prefix+"cuas_half_step"));
		if (properties.getProperty(prefix+"cuas_max_range")!=null)       this.cuas_max_range=Integer.parseInt(properties.getProperty(prefix+"cuas_max_range"));
		if (properties.getProperty(prefix+"cuas_num_cycles")!=null)      this.cuas_num_cycles=Integer.parseInt(properties.getProperty(prefix+"cuas_num_cycles"));

		if (properties.getProperty(prefix+"cuas_speed_min")!=null)       this.cuas_speed_min=Double.parseDouble(properties.getProperty(prefix+"cuas_speed_min"));
		if (properties.getProperty(prefix+"cuas_speed_pref")!=null)      this.cuas_speed_pref=Double.parseDouble(properties.getProperty(prefix+"cuas_speed_pref"));
		if (properties.getProperty(prefix+"cuas_speed_boost")!=null)     this.cuas_speed_boost=Double.parseDouble(properties.getProperty(prefix+"cuas_speed_boost"));

		if (properties.getProperty(prefix+"cuas_lmax_flt_hsigma")!=null) this.cuas_lmax_flt_hsigma=Double.parseDouble(properties.getProperty(prefix+"cuas_lmax_flt_hsigma"));
		if (properties.getProperty(prefix+"cuas_lmax_flt_neglim")!=null) this.cuas_lmax_flt_neglim=Double.parseDouble(properties.getProperty(prefix+"cuas_lmax_flt_neglim"));
		if (properties.getProperty(prefix+"cuas_lmax_flt_lsigma")!=null) this.cuas_lmax_flt_lsigma=Double.parseDouble(properties.getProperty(prefix+"cuas_lmax_flt_lsigma"));
		if (properties.getProperty(prefix+"cuas_lmax_flt_scale")!=null)  this.cuas_lmax_flt_scale=Double.parseDouble(properties.getProperty(prefix+"cuas_lmax_flt_scale"));

		if (properties.getProperty(prefix+"cuas_sky_threshold")!=null)   this.cuas_sky_threshold=Double.parseDouble(properties.getProperty(prefix+"cuas_sky_threshold"));
		if (properties.getProperty(prefix+"cuas_sky_offset")!=null)      this.cuas_sky_offset= IntersceneMatchParameters.StringToDoubles(properties.getProperty(prefix+"cuas_sky_offset"),this.cuas_sky_offset);
		
		if (properties.getProperty(prefix+"cuas_lmax_fraction")!=null)   this.cuas_lmax_fraction=Double.parseDouble(properties.getProperty(prefix+"cuas_lmax_fraction"));
		if (properties.getProperty(prefix+"cuas_lmax_hack_ridge")!=null) this.cuas_lmax_hack_ridge=Double.parseDouble(properties.getProperty(prefix+"cuas_lmax_hack_ridge"));
		if (properties.getProperty(prefix+"cuas_lmax_radius")!=null)     this.cuas_lmax_radius=Double.parseDouble(properties.getProperty(prefix+"cuas_lmax_radius"));
		if (properties.getProperty(prefix+"cuas_lmax_zero")!=null)       this.cuas_lmax_zero=Boolean.parseBoolean(properties.getProperty(prefix+"cuas_lmax_zero"));
		
		if (properties.getProperty(prefix+"cuas_centered_radius")!=null) this.cuas_centered_radius=Double.parseDouble(properties.getProperty(prefix+"cuas_centered_radius"));
		if (properties.getProperty(prefix+"cuas_centered_blur")!=null)   this.cuas_centered_blur=Double.parseDouble(properties.getProperty(prefix+"cuas_centered_blur"));
		if (properties.getProperty(prefix+"cuas_duplicate_tolerance")!=null)this.cuas_duplicate_tolerance=Double.parseDouble(properties.getProperty(prefix+"cuas_duplicate_tolerance"));
		
		if (properties.getProperty(prefix+"cuas_target_radius")!=null)   this.cuas_target_radius=Double.parseDouble(properties.getProperty(prefix+"cuas_target_radius"));
		if (properties.getProperty(prefix+"cuas_target_strength")!=null) this.cuas_target_strength=Double.parseDouble(properties.getProperty(prefix+"cuas_target_strength"));
		if (properties.getProperty(prefix+"cuas_target_frac")!=          null) cuas_target_frac=stringToDouble2d((String) properties.getProperty(prefix+"cuas_target_frac"));
		if (properties.getProperty(prefix+"cuas_no_border")!=null)       this.cuas_no_border=Boolean.parseBoolean(properties.getProperty(prefix+"cuas_no_border"));
		
		if (properties.getProperty(prefix+"cuas_lma_sigma")!=null)       this.cuas_lma_sigma=Double.parseDouble(properties.getProperty(prefix+"cuas_lma_sigma"));
		if (properties.getProperty(prefix+"cuas_wnd_pedestal")!=null)    this.cuas_wnd_pedestal=Double.parseDouble(properties.getProperty(prefix+"cuas_wnd_pedestal"));
		if (properties.getProperty(prefix+"cuas_lma_r0")!=null)          this.cuas_lma_r0=Double.parseDouble(properties.getProperty(prefix+"cuas_lma_r0"));
		if (properties.getProperty(prefix+"cuas_lma_ovrsht")!=null)      this.cuas_lma_ovrsht=Double.parseDouble(properties.getProperty(prefix+"cuas_lma_ovrsht"));
		if (properties.getProperty(prefix+"cuas_lma_fit_xy")!=null)      this.cuas_lma_fit_xy=Boolean.parseBoolean(properties.getProperty(prefix+"cuas_lma_fit_xy"));
		if (properties.getProperty(prefix+"cuas_lma_fit_a")!=null)       this.cuas_lma_fit_a=Boolean.parseBoolean(properties.getProperty(prefix+"cuas_lma_fit_a"));
		if (properties.getProperty(prefix+"cuas_lma_fit_c")!=null)       this.cuas_lma_fit_c=Boolean.parseBoolean(properties.getProperty(prefix+"cuas_lma_fit_c"));
		if (properties.getProperty(prefix+"cuas_lma_fit_r")!=null)       this.cuas_lma_fit_r=Boolean.parseBoolean(properties.getProperty(prefix+"cuas_lma_fit_r"));
		if (properties.getProperty(prefix+"cuas_lma_fit_k")!=null)       this.cuas_lma_fit_k=Boolean.parseBoolean(properties.getProperty(prefix+"cuas_lma_fit_k"));
		
		if (properties.getProperty(prefix+"cuas_lambda")!=null)          this.cuas_lambda=Double.parseDouble(properties.getProperty(prefix+"cuas_lambda"));
		if (properties.getProperty(prefix+"cuas_lambda_good")!=null)     this.cuas_lambda_good=Double.parseDouble(properties.getProperty(prefix+"cuas_lambda_good"));
		if (properties.getProperty(prefix+"cuas_lambda_bad")!=null)      this.cuas_lambda_bad=Double.parseDouble(properties.getProperty(prefix+"cuas_lambda_bad"));
		if (properties.getProperty(prefix+"cuas_lambda_max")!=null)      this.cuas_lambda_max=Double.parseDouble(properties.getProperty(prefix+"cuas_lambda_max"));
		if (properties.getProperty(prefix+"cuas_rms_diff")!=null)        this.cuas_rms_diff=Double.parseDouble(properties.getProperty(prefix+"cuas_rms_diff"));
		if (properties.getProperty(prefix+"cuas_num_iter")!=null)        this.cuas_num_iter=Integer.parseInt(properties.getProperty(prefix+  "cuas_num_iter"));
		if (properties.getProperty(prefix+"cuas_lma_rms")!=null)         this.cuas_lma_rms=Double.parseDouble(properties.getProperty(prefix+"cuas_lma_rms"));
		if (properties.getProperty(prefix+"cuas_lma_arms")!=null)        this.cuas_lma_arms=Double.parseDouble(properties.getProperty(prefix+"cuas_lma_arms"));
		if (properties.getProperty(prefix+"cuas_lma_rrms")!=null)        this.cuas_lma_rrms=Double.parseDouble(properties.getProperty(prefix+"cuas_lma_rrms"));
		if (properties.getProperty(prefix+"cuas_lma_mina")!=null)        this.cuas_lma_mina=Double.parseDouble(properties.getProperty(prefix+"cuas_lma_mina"));
		if (properties.getProperty(prefix+"cuas_lma_pre_mina")!=null)    this.cuas_lma_pre_mina=Double.parseDouble(properties.getProperty(prefix+"cuas_lma_pre_mina"));
		if (properties.getProperty(prefix+"cuas_min_keep")!=null)        this.cuas_min_keep=Integer.parseInt(properties.getProperty(prefix+  "cuas_min_keep"));
		
		if (properties.getProperty(prefix+"cuas_lma_maxr")!=null)        this.cuas_lma_maxr=Double.parseDouble(properties.getProperty(prefix+"cuas_lma_maxr"));
		if (properties.getProperty(prefix+"cuas_lma_minr1")!=null)       this.cuas_lma_minr1=Double.parseDouble(properties.getProperty(prefix+"cuas_lma_minr1"));
		if (properties.getProperty(prefix+"cuas_lma_mink")!=null)        this.cuas_lma_mink=Double.parseDouble(properties.getProperty(prefix+"cuas_lma_mink"));
		if (properties.getProperty(prefix+"cuas_lma_maxk")!=null)        this.cuas_lma_maxk=Double.parseDouble(properties.getProperty(prefix+"cuas_lma_maxk"));
		if (properties.getProperty(prefix+"cuas_lma_a2a")!=null)         this.cuas_lma_a2a=Double.parseDouble(properties.getProperty(prefix+"cuas_lma_a2a"));

		if (properties.getProperty(prefix+"cuas_slow_ra")!=null)         this.cuas_slow_ra=Double.parseDouble(properties.getProperty(prefix+"cuas_slow_ra"));
		if (properties.getProperty(prefix+"cuas_slow_score")!=null)      this.cuas_slow_score=Double.parseDouble(properties.getProperty(prefix+"cuas_slow_score"));
		
		if (properties.getProperty(prefix+"cuas_score_mv")!=null)        this.cuas_score_mv=Double.parseDouble(properties.getProperty(prefix+"cuas_score_mv"));
		if (properties.getProperty(prefix+"cuas_score_lma")!=null)       this.cuas_score_lma=Double.parseDouble(properties.getProperty(prefix+"cuas_score_lma"));
		if (properties.getProperty(prefix+"cuas_factor_lim")!=null)      this.cuas_factor_lim=Double.parseDouble(properties.getProperty(prefix+"cuas_factor_lim"));
		if (properties.getProperty(prefix+"cuas_factor_pow")!=null)      this.cuas_factor_pow=Double.parseDouble(properties.getProperty(prefix+"cuas_factor_pow"));
		if (properties.getProperty(prefix+"cuas_score_coeff")!=null)     this.cuas_score_coeff= IntersceneMatchParameters.StringToDoubles(properties.getProperty(prefix+"cuas_score_coeff"),this.cuas_score_coeff);
		if (properties.getProperty(prefix+"cuas_center_targ")!=null)     this.cuas_center_targ=Boolean.parseBoolean(properties.getProperty(prefix+"cuas_center_targ"));
		if (properties.getProperty(prefix+"cuas_multi_targ")!=null)      this.cuas_multi_targ=Boolean.parseBoolean(properties.getProperty(prefix+"cuas_multi_targ"));
		
		if (properties.getProperty(prefix+"cuas_isolated")!=null)        this.cuas_isolated=Boolean.parseBoolean(properties.getProperty(prefix+"cuas_isolated"));
		if (properties.getProperty(prefix+"cuas_min_seq")!=null)         this.cuas_min_seq=Integer.parseInt(properties.getProperty(prefix+"cuas_min_seq"));
		if (properties.getProperty(prefix+"cuas_enough_seq")!=null)      this.cuas_enough_seq=Integer.parseInt(properties.getProperty(prefix+"cuas_enough_seq"));
		if (properties.getProperty(prefix+"cuas_seq_travel")!=null)      this.cuas_seq_travel=Double.parseDouble(properties.getProperty(prefix+"cuas_seq_travel"));

		if (properties.getProperty(prefix+"cuas_min_seq_fl")!=null)      this.cuas_min_seq_fl=Integer.parseInt(properties.getProperty(prefix+"cuas_min_seq_fl"));
		if (properties.getProperty(prefix+"cuas_max_mismatch")!=null)    this.cuas_max_mismatch=Double.parseDouble(properties.getProperty(prefix+"cuas_max_mismatch"));
		if (properties.getProperty(prefix+"cuas_good_mismatch")!=null)   this.cuas_good_mismatch=Double.parseDouble(properties.getProperty(prefix+"cuas_good_mismatch"));
		if (properties.getProperty(prefix+"cuas_slow_fast_mismatch")!=null)this.cuas_slow_fast_mismatch=Double.parseDouble(properties.getProperty(prefix+"cuas_slow_fast_mismatch"));
		if (properties.getProperty(prefix+"cuas_match_len_pwr")!=null)   this.cuas_match_len_pwr=Double.parseDouble(properties.getProperty(prefix+"cuas_match_len_pwr"));
		
		if (properties.getProperty(prefix+"cuas_fail_mismatch")!=null)   this.cuas_fail_mismatch=Boolean.parseBoolean(properties.getProperty(prefix+"cuas_fail_mismatch"));
		if (properties.getProperty(prefix+"cuas_ignore_mismatch")!=null) this.cuas_ignore_mismatch=Boolean.parseBoolean(properties.getProperty(prefix+"cuas_ignore_mismatch"));

		if (properties.getProperty(prefix+"cuas_by_horizon")!=null)      this.cuas_by_horizon=Boolean.parseBoolean(properties.getProperty(prefix+"cuas_by_horizon"));
		if (properties.getProperty(prefix+"cuas_horizon")!=null)         this.cuas_horizon=Double.parseDouble(properties.getProperty(prefix+"cuas_horizon"));
		
		if (properties.getProperty(prefix+"cuas_mask_width")!=null)      this.cuas_mask_width=Double.parseDouble(properties.getProperty(prefix+"cuas_mask_width"));
		if (properties.getProperty(prefix+"cuas_mask_blur")!=null)       this.cuas_mask_blur=Double.parseDouble(properties.getProperty(prefix+"cuas_mask_blur"));
		if (properties.getProperty(prefix+"cuas_mask_round")!=null)      this.cuas_mask_round=Boolean.parseBoolean(properties.getProperty(prefix+"cuas_mask_round"));
		if (properties.getProperty(prefix+"cuas_target_type")!=null)     this.cuas_target_type=Integer.parseInt(properties.getProperty(prefix+"cuas_target_type"));
		if (properties.getProperty(prefix+"cuas_known_type")!=null)      this.cuas_known_type=Integer.parseInt(properties.getProperty(prefix+"cuas_known_type"));
		
		if (properties.getProperty(prefix+"cuas_input_range")!=null)     this.cuas_input_range=Double.parseDouble(properties.getProperty(prefix+"cuas_input_range"));
		if (properties.getProperty(prefix+"cuas_scale2x")!=null)         this.cuas_scale2x=Boolean.parseBoolean(properties.getProperty(prefix+"cuas_scale2x"));
		if (properties.getProperty(prefix+"cuas_ra_background")!=null)   this.cuas_ra_background=Boolean.parseBoolean(properties.getProperty(prefix+"cuas_ra_background"));

		if (properties.getProperty(prefix+"cuas_targets")!=null)         this.cuas_targets=Boolean.parseBoolean(properties.getProperty(prefix+"cuas_targets"));
		if (properties.getProperty(prefix+"cuas_iter_show")!=null)       this.cuas_iter_show=Integer.parseInt(properties.getProperty(prefix+"cuas_iter_show"));
		if (properties.getProperty(prefix+"cuas_2d_save_show")!=null)    this.cuas_2d_save_show=Boolean.parseBoolean(properties.getProperty(prefix+"cuas_2d_save_show"));
		if (properties.getProperty(prefix+"cuas_intermed_low")!=null)    this.cuas_intermed_low=Boolean.parseBoolean(properties.getProperty(prefix+"cuas_intermed_low"));
		if (properties.getProperty(prefix+"cuas_intermed_high")!=null)   this.cuas_intermed_high=Boolean.parseBoolean(properties.getProperty(prefix+"cuas_intermed_high"));
		if (properties.getProperty(prefix+"cuas_intermed_giga")!=null)   this.cuas_intermed_giga=Boolean.parseBoolean(properties.getProperty(prefix+"cuas_intermed_giga"));
		if (properties.getProperty(prefix+"cuas_debug_more")!=null)      this.cuas_debug_more=Boolean.parseBoolean(properties.getProperty(prefix+"cuas_debug_more"));
		
		if (properties.getProperty(prefix+"cuas_generate")!=null)        this.cuas_generate=Boolean.parseBoolean(properties.getProperty(prefix+"cuas_generate"));
		if (properties.getProperty(prefix+"cuas_generate_csv")!=null)    this.cuas_generate_csv=Boolean.parseBoolean(properties.getProperty(prefix+"cuas_generate_csv"));
		if (properties.getProperty(prefix+"cuas_save_mono")!=null)       this.cuas_save_mono=Boolean.parseBoolean(properties.getProperty(prefix+"cuas_save_mono"));
		if (properties.getProperty(prefix+"cuas_save_color")!=null)      this.cuas_save_color=Boolean.parseBoolean(properties.getProperty(prefix+"cuas_save_color"));
		if (properties.getProperty(prefix+"cuas_save_video")!=null)      this.cuas_save_video=Boolean.parseBoolean(properties.getProperty(prefix+"cuas_save_video"));
		if (properties.getProperty(prefix+"cuas_clean_video")!=null)     this.cuas_clean_video=Boolean.parseBoolean(properties.getProperty(prefix+"cuas_clean_video"));
		
		if (properties.getProperty(prefix+"cuas_annotate")!=null)        this.cuas_annotate=Boolean.parseBoolean(properties.getProperty(prefix+"cuas_annotate"));
		if (properties.getProperty(prefix+"cuas_annotate_uas")!=null)    this.cuas_annotate_uas=Boolean.parseBoolean(properties.getProperty(prefix+"cuas_annotate_uas"));
		if  (properties.getProperty(prefix+"cuas_text_color")!=null) {
			long lcolor_annotate = Long.parseLong(properties.getProperty(prefix+"cuas_text_color"));
			if (lcolor_annotate < 0) this.cuas_text_color = setLongColor(0x00FF00);
			else this.cuas_text_color = setLongColor(lcolor_annotate);
		}
		if  (properties.getProperty(prefix+"cuas_text_bg")!=null) {
			long lcolor_annotate = Long.parseLong(properties.getProperty(prefix+"cuas_text_bg"));
			if (lcolor_annotate < 0) this.cuas_text_bg = setLongColor(0x777777);
			else this.cuas_text_bg = setLongColor(lcolor_annotate);
		}
		if  (properties.getProperty(prefix+"cuas_selected_color")!=null) {
			long lcolor_annotate = Long.parseLong(properties.getProperty(prefix+"cuas_selected_color"));
			if (lcolor_annotate < 0) this.cuas_selected_color = setLongColor(0xFF0088);
			else this.cuas_selected_color = setLongColor(lcolor_annotate);
		}
		if (properties.getProperty(prefix+"cuas_transparent")!=null)     this.cuas_transparent=Boolean.parseBoolean(properties.getProperty(prefix+"cuas_transparent"));
		if (properties.getProperty(prefix+"cuas_transparent_uas")!=null) this.cuas_transparent_uas=Boolean.parseBoolean(properties.getProperty(prefix+"cuas_transparent_uas"));
		if (properties.getProperty(prefix+"cuas_font_name")!=            null) cuas_font_name=(String) properties.getProperty(prefix+"cuas_font_name");
		if (properties.getProperty(prefix+"cuas_font_size")!=null)       this.cuas_font_size=Integer.parseInt(properties.getProperty(prefix+"cuas_font_size"));
		if (properties.getProperty(prefix+"cuas_font_type")!=null)       this.cuas_font_type=Integer.parseInt(properties.getProperty(prefix+"cuas_font_type"));
		if (properties.getProperty(prefix+"cuas_font_spacing")!=null)    this.cuas_font_spacing=Double.parseDouble(properties.getProperty(prefix+"cuas_font_spacing"));
		if (properties.getProperty(prefix+"cuas_annot_missing")!=null)   this.cuas_annot_missing=Boolean.parseBoolean(properties.getProperty(prefix+"cuas_annot_missing"));
		if (properties.getProperty(prefix+"cuas_px0")!=null)             this.cuas_px0=Integer.parseInt(properties.getProperty(prefix+"cuas_px0"));
		if (properties.getProperty(prefix+"cuas_py0")!=null)             this.cuas_py0=Integer.parseInt(properties.getProperty(prefix+"cuas_py0"));
		if (properties.getProperty(prefix+"cuas_az0")!=null)             this.cuas_az0=Double.parseDouble(properties.getProperty(prefix+"cuas_az0"));
		if (properties.getProperty(prefix+"cuas_el0")!=null)             this.cuas_el0=Double.parseDouble(properties.getProperty(prefix+"cuas_el0"));
		if (properties.getProperty(prefix+"cuas_show_disp")!=null)       this.cuas_show_disp=Boolean.parseBoolean(properties.getProperty(prefix+"cuas_show_disp"));
		if (properties.getProperty(prefix+"cuas_show_rng")!=null)        this.cuas_show_rng=Boolean.parseBoolean(properties.getProperty(prefix+"cuas_show_rng"));
		if (properties.getProperty(prefix+"cuas_show_inf")!=null)        this.cuas_show_inf=Boolean.parseBoolean(properties.getProperty(prefix+"cuas_show_inf"));
		if (properties.getProperty(prefix+"cuas_show_inf_gt")!=null)     this.cuas_show_inf_gt=Boolean.parseBoolean(properties.getProperty(prefix+"cuas_show_inf_gt"));
		if (properties.getProperty(prefix+"cuas_show_true_rng")!=null)   this.cuas_show_true_rng=Boolean.parseBoolean(properties.getProperty(prefix+"cuas_show_true_rng"));
		
		if (properties.getProperty(prefix+"cuas_smooth_omegas")!=null)   this.cuas_smooth_omegas=Boolean.parseBoolean(properties.getProperty(prefix+"cuas_smooth_omegas"));
		if (properties.getProperty(prefix+"cuas_rng_um")!=null)          this.cuas_rng_um=Boolean.parseBoolean(properties.getProperty(prefix+"cuas_rng_um"));
		if (properties.getProperty(prefix+"cuas_rng_um_all")!=null)      this.cuas_rng_um_all=Boolean.parseBoolean(properties.getProperty(prefix+"cuas_rng_um_all"));
		if (properties.getProperty(prefix+"cuas_rng_um_sigma")!=null)    this.cuas_rng_um_sigma=Double.parseDouble(properties.getProperty(prefix+"cuas_rng_um_sigma"));
		if (properties.getProperty(prefix+"cuas_rng_um2")!=null)         this.cuas_rng_um2=Boolean.parseBoolean(properties.getProperty(prefix+"cuas_rng_um2"));
		if (properties.getProperty(prefix+"cuas_rng_um_weight")!=null)   this.cuas_rng_um_weight=Double.parseDouble(properties.getProperty(prefix+"cuas_rng_um_weight"));
		if (properties.getProperty(prefix+"cuas_rng_coswnd")!=null)      this.cuas_rng_coswnd=Boolean.parseBoolean(properties.getProperty(prefix+"cuas_rng_coswnd"));
		
		if (properties.getProperty(prefix+"cuas_rng_combine")!=null)     this.cuas_rng_combine=Integer.parseInt(properties.getProperty(prefix+"cuas_rng_combine"));
		
		if (properties.getProperty(prefix+"cuas_rng_radius0")!=null)     this.cuas_rng_radius0=Double.parseDouble(properties.getProperty(prefix+"cuas_rng_radius0"));
		if (properties.getProperty(prefix+"cuas_rng_radius")!=null)      this.cuas_rng_radius=Double.parseDouble(properties.getProperty(prefix+"cuas_rng_radius"));
		if (properties.getProperty(prefix+"cuas_rng_blur")!=null)        this.cuas_rng_blur=Double.parseDouble(properties.getProperty(prefix+"cuas_rng_blur"));
		if (properties.getProperty(prefix+"cuas_rng_fz0")!=null)         this.cuas_rng_fz0=Double.parseDouble(properties.getProperty(prefix+"cuas_rng_fz0"));
		if (properties.getProperty(prefix+"cuas_rng_fz")!=null)          this.cuas_rng_fz=Double.parseDouble(properties.getProperty(prefix+"cuas_rng_fz"));
		if (properties.getProperty(prefix+"cuas_rng_scale")!=null)       this.cuas_rng_scale=Double.parseDouble(properties.getProperty(prefix+"cuas_rng_scale"));
		
		if (properties.getProperty(prefix+"cuas_dual_pass")!=null)       this.cuas_dual_pass=Boolean.parseBoolean(properties.getProperty(prefix+"cuas_dual_pass"));
		if (properties.getProperty(prefix+"cuas_disp_only2")!=null)      this.cuas_disp_only2=Boolean.parseBoolean(properties.getProperty(prefix+"cuas_disp_only2"));
		if (properties.getProperty(prefix+"cuas_lma_bypass")!=null)      this.cuas_lma_bypass=Boolean.parseBoolean(properties.getProperty(prefix+"cuas_lma_bypass"));
		if (properties.getProperty(prefix+"cuas_mcorr_sel")!=null)       this.cuas_mcorr_sel=Integer.parseInt(properties.getProperty(prefix+"cuas_mcorr_sel"));
		if (properties.getProperty(prefix+"cuas_mcorr_sel_lma")!=null)   this.cuas_mcorr_sel_lma=Integer.parseInt(properties.getProperty(prefix+"cuas_mcorr_sel_lma"));
		
		if (properties.getProperty(prefix+"cuas_max_rel_rms")!=null)     this.cuas_max_rel_rms=Double.parseDouble(properties.getProperty(prefix+"cuas_max_rel_rms"));
		if (properties.getProperty(prefix+"cuas_min_strength")!=null)    this.cuas_min_strength=Double.parseDouble(properties.getProperty(prefix+"cuas_min_strength"));
		if (properties.getProperty(prefix+"cuas_ac_offset")!=null)       this.cuas_ac_offset=Double.parseDouble(properties.getProperty(prefix+"cuas_ac_offset"));
		if (properties.getProperty(prefix+"cuas_min_max_ac")!=null)      this.cuas_min_max_ac=Double.parseDouble(properties.getProperty(prefix+"cuas_min_max_ac"));
		if (properties.getProperty(prefix+"cuas_min_min_ac")!=null)      this.cuas_min_min_ac=Double.parseDouble(properties.getProperty(prefix+"cuas_min_min_ac"));
		if (properties.getProperty(prefix+"cuas_reset_disparity")!=null) this.cuas_reset_disparity=Boolean.parseBoolean(properties.getProperty(prefix+"cuas_reset_disparity"));
		if (properties.getProperty(prefix+"cuas_initial_disparity")!=null)this.cuas_initial_disparity=Double.parseDouble(properties.getProperty(prefix+"cuas_initial_disparity"));
		if (properties.getProperty(prefix+"cuas_infinity")!=null)        this.cuas_infinity=Double.parseDouble(properties.getProperty(prefix+"cuas_infinity"));
		
		if (properties.getProperty(prefix+"cuas_rng_img")!=null)         this.cuas_rng_img=Boolean.parseBoolean(properties.getProperty(prefix+"cuas_rng_img"));
		if (properties.getProperty(prefix+"cuas_rng_glob")!=null)        this.cuas_rng_glob=Boolean.parseBoolean(properties.getProperty(prefix+"cuas_rng_glob"));
		if (properties.getProperty(prefix+"cuas_glob_ends")!=null)       this.cuas_glob_ends=Integer.parseInt(properties.getProperty(prefix+"cuas_glob_ends"));
		if (properties.getProperty(prefix+"cuas_rng_disp")!=null)        this.cuas_rng_disp=Boolean.parseBoolean(properties.getProperty(prefix+"cuas_rng_disp"));
		if (properties.getProperty(prefix+"cuas_rng_vfy")!=null)         this.cuas_rng_vfy=Boolean.parseBoolean(properties.getProperty(prefix+"cuas_rng_vfy"));
		if (properties.getProperty(prefix+"cuas_rng_niterate")!=null)    this.cuas_rng_niterate=Integer.parseInt(properties.getProperty(prefix+"cuas_rng_niterate"));
		if (properties.getProperty(prefix+"cuas_rng_diff")!=null)        this.cuas_rng_diff=Double.parseDouble(properties.getProperty(prefix+"cuas_rng_diff"));

		if (properties.getProperty(prefix+"cuas_flw_levels")!=null)      this.cuas_flw_levels=Integer.parseInt(properties.getProperty(prefix+"cuas_flw_levels"));
		if (properties.getProperty(prefix+"cuas_flw_len")!=null)         this.cuas_flw_len=Integer.parseInt(properties.getProperty(prefix+"cuas_flw_len"));
		if (properties.getProperty(prefix+"cuas_flw_diff")!=null)        this.cuas_flw_diff=Double.parseDouble(properties.getProperty(prefix+"cuas_flw_diff"));
		
		if (properties.getProperty(prefix+"cuas_known_err")!=null)       this.cuas_known_err=Double.parseDouble(properties.getProperty(prefix+"cuas_known_err"));
		if (properties.getProperty(prefix+"cuas_tmtch_frac")!=null)      this.cuas_tmtch_frac=Double.parseDouble(properties.getProperty(prefix+"cuas_tmtch_frac"));
		if (properties.getProperty(prefix+"cuas_tmtch_ends")!=null)      this.cuas_tmtch_ends=Integer.parseInt(properties.getProperty(prefix+"cuas_tmtch_ends"));
		if (properties.getProperty(prefix+"cuas_tmtch_gaps")!=null)      this.cuas_tmtch_gaps=Integer.parseInt(properties.getProperty(prefix+"cuas_tmtch_gaps"));
		if (properties.getProperty(prefix+"cuas_tmtch_pix")!=null)       this.cuas_tmtch_pix=Double.parseDouble(properties.getProperty(prefix+"cuas_tmtch_pix"));
		if (properties.getProperty(prefix+"cuas_local_apix")!=null)      this.cuas_local_apix=Double.parseDouble(properties.getProperty(prefix+"cuas_local_apix"));
		if (properties.getProperty(prefix+"cuas_tmtch_apix")!=null)      this.cuas_tmtch_apix=Double.parseDouble(properties.getProperty(prefix+"cuas_tmtch_apix"));
		if (properties.getProperty(prefix+"cuas_tmtch_rpix")!=null)      this.cuas_tmtch_rpix=Double.parseDouble(properties.getProperty(prefix+"cuas_tmtch_rpix"));
		if (properties.getProperty(prefix+"cuas_tmtch_axv")!=null)       this.cuas_tmtch_axv=Double.parseDouble(properties.getProperty(prefix+"cuas_tmtch_axv"));
		if (properties.getProperty(prefix+"cuas_tmtch_axv_k")!=null)     this.cuas_tmtch_axv_k=Double.parseDouble(properties.getProperty(prefix+"cuas_tmtch_axv_k"));
		if (properties.getProperty(prefix+"cuas_local_diff")!=null)      this.cuas_local_diff=Double.parseDouble(properties.getProperty(prefix+"cuas_local_diff"));
		if (properties.getProperty(prefix+"cuas_tmtch_diff")!=null)      this.cuas_tmtch_diff=Double.parseDouble(properties.getProperty(prefix+"cuas_tmtch_diff"));
		if (properties.getProperty(prefix+"cuas_tmtch_disp")!=null)      this.cuas_tmtch_disp=Double.parseDouble(properties.getProperty(prefix+"cuas_tmtch_disp"));
		if (properties.getProperty(prefix+"cuas_tmtch_short")!=null)     this.cuas_tmtch_short=Boolean.parseBoolean(properties.getProperty(prefix+"cuas_tmtch_short"));
		
		if (properties.getProperty(prefix+"cuas_radar_range")!=null)     this.cuas_radar_range=Double.parseDouble(properties.getProperty(prefix+"cuas_radar_range"));
		if (properties.getProperty(prefix+"cuas_radar_height")!=null)    this.cuas_radar_height=Integer.parseInt(properties.getProperty(prefix+"cuas_radar_height"));
		if (properties.getProperty(prefix+"cuas_radar_bottom")!=null)    this.cuas_radar_bottom=Integer.parseInt(properties.getProperty(prefix+"cuas_radar_bottom"));
		if (properties.getProperty(prefix+"cuas_radar_top")!=null)       this.cuas_radar_top=Integer.parseInt(properties.getProperty(prefix+"cuas_radar_top"));
		if (properties.getProperty(prefix+"cuas_radar_uas_color")!=null) this.cuas_radar_uas_color = getColorFromHex(
				properties.getProperty(prefix+"cuas_radar_uas_color"), // String hex_color,
				DEFAULT_cuas_radar_uas_color); // Color  default_color) 
		if (properties.getProperty(prefix+"cuas_radar_color")!=null)     this.cuas_radar_color = getColorFromHex(
				properties.getProperty(prefix+"cuas_radar_color"), // String hex_color,
				DEFAULT_cuas_radar_color); // Color  default_color) 
		if (properties.getProperty(prefix+"cuas_radar_radius")!=null)    this.cuas_radar_radius=Double.parseDouble(properties.getProperty(prefix+"cuas_radar_radius"));
		if (properties.getProperty(prefix+"cuas_radar_uas_rad")!=null)   this.cuas_radar_uas_rad=Double.parseDouble(properties.getProperty(prefix+"cuas_radar_uas_rad"));
		if (properties.getProperty(prefix+"cuas_grid_ring")!=null)       this.cuas_grid_ring=Double.parseDouble(properties.getProperty(prefix+"cuas_grid_ring"));
		if (properties.getProperty(prefix+"cuas_grid_line")!=null)       this.cuas_grid_line=Double.parseDouble(properties.getProperty(prefix+"cuas_grid_line"));
		if (properties.getProperty(prefix+"cuas_grid_annot")!=null)      this.cuas_grid_annot=Boolean.parseBoolean(properties.getProperty(prefix+"cuas_grid_annot"));
		if (properties.getProperty(prefix+"cuas_grid_color")!=null)      this.cuas_grid_color = getColorFromHex(
				properties.getProperty(prefix+"cuas_grid_color"), // String hex_color,
				DEFAULT_cuas_grid_color); // Color  default_color) 
		if (properties.getProperty(prefix+"cuas_grid_font")!=null)       this.cuas_grid_font=Integer.parseInt(properties.getProperty(prefix+"cuas_grid_font"));
		if (properties.getProperty(prefix+"cuas_grid_az_top")!=null)     this.cuas_grid_az_top=Integer.parseInt(properties.getProperty(prefix+"cuas_grid_az_top"));
		if (properties.getProperty(prefix+"cuas_grid_sides")!=null)      this.cuas_grid_sides=Integer.parseInt(properties.getProperty(prefix+"cuas_grid_sides"));

		if (properties.getProperty(prefix+"cuas_annot_sel")!=null)      this.cuas_annot_sel= IntersceneMatchParameters.StringToInts(properties.getProperty(prefix+"cuas_annot_sel"),this.cuas_annot_sel);
		
		if (properties.getProperty(prefix+"cuas_debug")!=null)           this.cuas_debug=Boolean.parseBoolean(properties.getProperty(prefix+"cuas_debug"));
		if (properties.getProperty(prefix+"cuas_dbg_rng_seq")!=null)     this.cuas_dbg_rng_seq=Integer.parseInt(properties.getProperty(prefix+"cuas_dbg_rng_seq"));
		if (properties.getProperty(prefix+"cuas_dbg_rng_tgt")!=null)     this.cuas_dbg_rng_tgt=Integer.parseInt(properties.getProperty(prefix+"cuas_dbg_rng_tgt"));
		if (properties.getProperty(prefix+"cuas_dbg_show_mode")!=null)   this.cuas_dbg_show_mode=Integer.parseInt(properties.getProperty(prefix+"cuas_dbg_show_mode"));
		if (properties.getProperty(prefix+"cuas_step_debug")!=null)      this.cuas_step_debug=Boolean.parseBoolean(properties.getProperty(prefix+"cuas_step_debug"));
		if (properties.getProperty(prefix+"cuas_save_stats")!=null)      this.cuas_save_stats=Boolean.parseBoolean(properties.getProperty(prefix+"cuas_save_stats"));
		if (properties.getProperty(prefix+"cuas_target_debug")!=null)    this.cuas_target_debug=Boolean.parseBoolean(properties.getProperty(prefix+"cuas_target_debug"));
		if (properties.getProperty(prefix+"cuas_target_score")!=null)    this.cuas_target_score=Boolean.parseBoolean(properties.getProperty(prefix+"cuas_target_score"));
		if (properties.getProperty(prefix+"cuas_target_color")!=null)    this.cuas_target_color=Boolean.parseBoolean(properties.getProperty(prefix+"cuas_target_color"));
		if (properties.getProperty(prefix+"cuas_log_ranging")!=null)     this.cuas_log_ranging=Boolean.parseBoolean(properties.getProperty(prefix+"cuas_log_ranging"));
		
		if (properties.getProperty(prefix+"cuas_overwrite")!=null)       this.cuas_overwrite=Boolean.parseBoolean(properties.getProperty(prefix+"cuas_overwrite"));
		if (properties.getProperty(prefix+"cuas_num_orient")!=null)      this.cuas_num_orient=Integer.parseInt(properties.getProperty(prefix+"cuas_num_orient"));
		if (properties.getProperty(prefix+"cuas_num_accum")!=null)       this.cuas_num_accum=Integer.parseInt(properties.getProperty(prefix+"cuas_num_accum"));
		
		if (properties.getProperty(prefix+"cuas_reuse_targets")!=null)   this.cuas_reuse_targets=Boolean.parseBoolean(properties.getProperty(prefix+"cuas_reuse_targets"));
		if (properties.getProperty(prefix+"cuas_reuse_path")!=null)      this.cuas_reuse_path=(String) properties.getProperty(prefix+"cuas_reuse_path");
		
		if (properties.getProperty(prefix+"cuas_reuse_disparity")!=null) this.cuas_reuse_disparity=Boolean.parseBoolean(properties.getProperty(prefix+"cuas_reuse_disparity"));
		if (properties.getProperty(prefix+"cuas_reuse_globals")!=null)   this.cuas_reuse_globals=Boolean.parseBoolean(properties.getProperty(prefix+"cuas_reuse_globals"));
		if (properties.getProperty(prefix+"cuas_max_disp_diff")!=null)   this.cuas_max_disp_diff=Double.parseDouble(properties.getProperty(prefix+"cuas_max_disp_diff"));
		if (properties.getProperty(prefix+"cuas_min_disp_str")!=null)    this.cuas_min_disp_str=Double.parseDouble(properties.getProperty(prefix+"cuas_min_disp_str"));
		if (properties.getProperty(prefix+"cuas_rng_limit")!=null)       this.cuas_rng_limit=Double.parseDouble(properties.getProperty(prefix+"cuas_rng_limit"));
		
		if (properties.getProperty(prefix+"mb_gain_index_pose")!=null)   this.mb_gain_index_pose=Integer.parseInt(properties.getProperty(prefix+"mb_gain_index_pose"));
		if (properties.getProperty(prefix+"mb_ers_index")!=null)         this.mb_ers_index=Integer.parseInt(properties.getProperty(prefix+"mb_ers_index"));
		if (properties.getProperty(prefix+"mb_ers_y_index")!=null)       this.mb_ers_y_index=Integer.parseInt(properties.getProperty(prefix+"mb_ers_y_index"));
		if (properties.getProperty(prefix+"mb_ers_r_index")!=null)       this.mb_ers_r_index=Integer.parseInt(properties.getProperty(prefix+"mb_ers_r_index"));
		if (properties.getProperty(prefix+"mb_all_index")!=null)         this.mb_all_index=Integer.parseInt(properties.getProperty(prefix+"mb_all_index"));
		if (properties.getProperty(prefix+"mb_pref_orient")!=null)       this.mb_pref_orient=Boolean.parseBoolean(properties.getProperty(prefix+"mb_pref_orient"));
		if (properties.getProperty(prefix+"lma_use_Z")!=null)            this.lma_use_Z=Boolean.parseBoolean(properties.getProperty(prefix+"lma_use_Z"));
		if (properties.getProperty(prefix+"lma_use_R")!=null)            this.lma_use_R=Boolean.parseBoolean(properties.getProperty(prefix+"lma_use_R"));
		
		if (properties.getProperty(prefix+"configured_lma")!=null)       this.configured_lma=Boolean.parseBoolean(properties.getProperty(prefix+"configured_lma"));
		if (properties.getProperty(prefix+"freeze_xy_pull")!=null)       this.freeze_xy_pull=Boolean.parseBoolean(properties.getProperty(prefix+"freeze_xy_pull"));
		if (properties.getProperty(prefix+"copy_pull_current")!=null)    this.copy_pull_current=Boolean.parseBoolean(properties.getProperty(prefix+"copy_pull_current"));
		if (properties.getProperty(prefix+"restore_imu")!=null)          this.restore_imu=Boolean.parseBoolean(properties.getProperty(prefix+"restore_imu"));
		if (properties.getProperty(prefix+"lpf_xy")!=null)               this.lpf_xy=Boolean.parseBoolean(properties.getProperty(prefix+"lpf_xy"));
		if (properties.getProperty(prefix+"readjust_xy_ims")!=null)      this.readjust_xy_ims=Boolean.parseBoolean(properties.getProperty(prefix+"readjust_xy_ims"));
		if (properties.getProperty(prefix+"reg_weight_xy")!=null)        this.reg_weight_xy=Double.parseDouble(properties.getProperty(prefix+"reg_weight_xy"));
		
		if (properties.getProperty(prefix+"quat_lambda")!=null)          this.quat_lambda=Double.parseDouble(properties.getProperty(prefix+"quat_lambda"));
		if (properties.getProperty(prefix+"quat_lambda_scale_good")!=null)this.quat_lambda_scale_good=Double.parseDouble(properties.getProperty(prefix+"quat_lambda_scale_good"));
		if (properties.getProperty(prefix+"quat_lambda_scale_bad")!=null)this.quat_lambda_scale_bad=Double.parseDouble(properties.getProperty(prefix+"quat_lambda_scale_bad"));
		if (properties.getProperty(prefix+"quat_lambda_max")!=null)      this.quat_lambda_max=Double.parseDouble(properties.getProperty(prefix+"quat_lambda_max"));
		if (properties.getProperty(prefix+"quat_rms_diff")!=null)        this.quat_rms_diff=Double.parseDouble(properties.getProperty(prefix+"quat_rms_diff"));
		if (properties.getProperty(prefix+"reg_weight_xy")!=null)        this.reg_weight_xy=Double.parseDouble(properties.getProperty(prefix+"reg_weight_xy"));
		if (properties.getProperty(prefix+"quat_num_iter")!=null)        this.quat_num_iter=Integer.parseInt(properties.getProperty(prefix+"quat_num_iter"));
		if (properties.getProperty(prefix+"quat_reg_w")!=null)           this.quat_reg_w=Double.parseDouble(properties.getProperty(prefix+"quat_reg_w"));
		
		if (properties.getProperty(prefix+"quat_max_ratio")!=null)       this.quat_max_ratio=Double.parseDouble(properties.getProperty(prefix+"quat_max_ratio"));
		if (properties.getProperty(prefix+"quat_max_change")!=null)      this.quat_max_change=Double.parseDouble(properties.getProperty(prefix+"quat_max_change"));
		if (properties.getProperty(prefix+"quat_min_transl")!=null)      this.quat_min_transl=Double.parseDouble(properties.getProperty(prefix+"quat_min_transl"));
		if (properties.getProperty(prefix+"quat_min_rot")!=null)         this.quat_min_rot=Double.parseDouble(properties.getProperty(prefix+"quat_min_rot"));
		if (properties.getProperty(prefix+"quat_min_lin")!=null)         this.quat_min_lin=Double.parseDouble(properties.getProperty(prefix+"quat_min_lin"));
		
		if (properties.getProperty(prefix+"stereo_merge")!=null)         this.stereo_merge=Boolean.parseBoolean(properties.getProperty(prefix+"stereo_merge"));
		if (properties.getProperty(prefix+"stereo_gap")!=null)           this.stereo_gap=Integer.parseInt(properties.getProperty(prefix+"stereo_gap"));
		if (properties.getProperty(prefix+"stereo_intereye")!=null)      this.stereo_intereye=Double.parseDouble(properties.getProperty(prefix+"stereo_intereye"));
		if (properties.getProperty(prefix+"stereo_phone_width")!=null)   this.stereo_phone_width=Double.parseDouble(properties.getProperty(prefix+"stereo_phone_width"));
		
		if (properties.getProperty(prefix+"extra_hor_tile")!=null)       this.extra_hor_tile=Integer.parseInt(properties.getProperty(prefix+"extra_hor_tile"));
		if (properties.getProperty(prefix+"extra_vert_tile")!=null)      this.extra_vert_tile=Integer.parseInt(properties.getProperty(prefix+"extra_vert_tile"));
		if (properties.getProperty(prefix+"crop_3d")!=null)              this.crop_3d=Boolean.parseBoolean(properties.getProperty(prefix+"crop_3d"));		
		if (properties.getProperty(prefix+"sensor_mask")!=null)          this.sensor_mask=Integer.parseInt(properties.getProperty(prefix+"sensor_mask"));
		if (properties.getProperty(prefix+"merge_all")!=null)            this.merge_all=Boolean.parseBoolean(properties.getProperty(prefix+"merge_all"));		
		
		if (properties.getProperty(prefix+"video_fps")!=null)            this.video_fps=Double.parseDouble(properties.getProperty(prefix+"video_fps"));
		if (properties.getProperty(prefix+"sensor_fps")!=null)           this.sensor_fps=Double.parseDouble(properties.getProperty(prefix+"sensor_fps"));
		if (properties.getProperty(prefix+"mode_avi")!=null)             this.mode_avi=Integer.parseInt(properties.getProperty(prefix+"mode_avi"));
		if (properties.getProperty(prefix+"avi_JPEG_quality")!=null)     this.avi_JPEG_quality=Integer.parseInt(properties.getProperty(prefix+"avi_JPEG_quality"));
		if (properties.getProperty(prefix+"run_ffmpeg")!=null)           this.run_ffmpeg=Boolean.parseBoolean(properties.getProperty(prefix+"run_ffmpeg"));
		if (properties.getProperty(prefix+"video_ext")!=null)            this.video_ext=(String) properties.getProperty(prefix+"video_ext");
		if (properties.getProperty(prefix+"video_codec")!=null)          this.video_codec=(String) properties.getProperty(prefix+"video_codec");
		if (properties.getProperty(prefix+"video_crf")!=null)            this.video_crf=Integer.parseInt(properties.getProperty(prefix+"video_crf"));
		if (properties.getProperty(prefix+"video_bitrate_m")!=null)      this.video_bitrate_m=Double.parseDouble(properties.getProperty(prefix+"video_bitrate_m"));
		
		if (properties.getProperty(prefix+"remove_avi")!=null)           this.remove_avi=Boolean.parseBoolean(properties.getProperty(prefix+"remove_avi"));
		if (properties.getProperty(prefix+"video_codec_combo")!=null)    this.video_codec_combo=(String) properties.getProperty(prefix+"video_codec_combo");
		if (properties.getProperty(prefix+"video_crf_combo")!=null)      this.video_crf_combo=Integer.parseInt(properties.getProperty(prefix+"video_crf_combo"));
		if (properties.getProperty(prefix+"add_average")!=null)          this.add_average=Boolean.parseBoolean(properties.getProperty(prefix+"add_average"));
		if (properties.getProperty(prefix+"calculate_average")!=null)    this.calculate_average=Boolean.parseBoolean(properties.getProperty(prefix+"calculate_average"));
//		
		if (properties.getProperty(prefix+"subtract_average")!=null)     this.subtract_average=Boolean.parseBoolean(properties.getProperty(prefix+"subtract_average"));
		if (properties.getProperty(prefix+"running_average")!=null)      this.running_average=Integer.parseInt(properties.getProperty(prefix+"running_average"));
		if (properties.getProperty(prefix+"extract_center_orientation")!=null)this.extract_center_orientation=Boolean.parseBoolean(properties.getProperty(prefix+"extract_center_orientation"));
		if (properties.getProperty(prefix+"um_mono")!=null)              this.um_mono=Boolean.parseBoolean(properties.getProperty(prefix+"um_mono"));
		if (properties.getProperty(prefix+"um_sigma")!=null)             this.um_sigma=Double.parseDouble(properties.getProperty(prefix+"um_sigma"));
		if (properties.getProperty(prefix+"um_weight")!=null)            this.um_weight=Double.parseDouble(properties.getProperty(prefix+"um_weight"));
		if (properties.getProperty(prefix+"um_mono_linear")!=null)       this.um_mono_linear=Boolean.parseBoolean(properties.getProperty(prefix+"um_mono_linear"));
		if (properties.getProperty(prefix+"mono_fixed")!=null)           this.mono_fixed=Boolean.parseBoolean(properties.getProperty(prefix+"mono_fixed"));
		if (properties.getProperty(prefix+"mono_range")!=null)           this.mono_range=Double.parseDouble(properties.getProperty(prefix+"mono_range"));
		
		if (properties.getProperty(prefix+"anaglyth_en")!=null)           this.anaglyth_en=Boolean.parseBoolean(properties.getProperty(prefix+"anaglyth_en"));
		if  (properties.getProperty(prefix+"anaglyph_left") != null) {
			try {
				this.anaglyph_left = setLongColor(Long.parseLong(properties.getProperty(prefix+"anaglyph_left")));
			} catch(NumberFormatException e){
				this.anaglyph_left = anaglyph_left_default;
			}
		}
		if  (properties.getProperty(prefix+"anaglyph_right") != null) {
			try {
				this.anaglyph_right = setLongColor(Long.parseLong(properties.getProperty(prefix+"anaglyph_right")));
			} catch(NumberFormatException e){
				this.anaglyph_right = anaglyph_right_default;
			}
		}
		
		if (properties.getProperty(prefix+"annotate_color")!=null)       this.annotate_color=Boolean.parseBoolean(properties.getProperty(prefix+"annotate_color"));
		if (properties.getProperty(prefix+"annotate_mono")!=null)        this.annotate_mono=Boolean.parseBoolean(properties.getProperty(prefix+"annotate_mono"));

		if  (properties.getProperty(prefix+"annotate_color_color")!=null) {
			long lcolor_annotate = Long.parseLong(properties.getProperty(prefix+"annotate_color_color"));
			if (lcolor_annotate < 0) this.annotate_color_color = null;
			else this.annotate_color_color = setLongColor(lcolor_annotate);
		}
		if  (properties.getProperty(prefix+"annotate_color_mono")!=null) {
			long lcolor_annotate = Long.parseLong(properties.getProperty(prefix+"annotate_color_mono"));
			if (lcolor_annotate < 0) this.annotate_color_mono = null;
			else this.annotate_color_mono = setLongColor(lcolor_annotate);
		}
		if (properties.getProperty(prefix+"annotate_transparent_mono")!=null) this.annotate_transparent_mono=Boolean.parseBoolean(properties.getProperty(prefix+"annotate_transparent_mono"));
		
		// vegetation
		if (properties.getProperty(prefix+"terr_model_path")!=         null) terr_model_path=(String) properties.getProperty(prefix+"terr_model_path");
		if (properties.getProperty(prefix+"terr_model_state")!=        null) terr_model_state=(String) properties.getProperty(prefix+"terr_model_state");
		if (properties.getProperty(prefix+"terr_segments_dir")!=       null) terr_segments_dir=(String) properties.getProperty(prefix+"terr_segments_dir");
		if (properties.getProperty(prefix+"terr_segments_suffix")!=    null) terr_segments_suffix=(String) properties.getProperty(prefix+"terr_segments_suffix");

		if (properties.getProperty(prefix+"terr_par_dir")!=            null) terr_par_dir=(String) properties.getProperty(prefix+"terr_par_dir");
		if (properties.getProperty(prefix+"terr_par_file")!=           null) terr_par_file=(String) properties.getProperty(prefix+"terr_par_file");
		if (properties.getProperty(prefix+"terr_par_restore")!=        null) terr_par_restore=Boolean.parseBoolean(properties.getProperty(prefix+"terr_par_restore"));
		if (properties.getProperty(prefix+"terr_step_restore")!=       null) terr_step_restore=Integer.parseInt(properties.getProperty(prefix+"terr_step_restore"));
		
		if (properties.getProperty(prefix+"terr_um_en")!=              null) terr_um_en=Boolean.parseBoolean(properties.getProperty(prefix+"terr_um_en"));
		if (properties.getProperty(prefix+"terr_um_sigma")!=           null) terr_um_sigma=Double.parseDouble(properties.getProperty(prefix+"terr_um_sigma"));
		if (properties.getProperty(prefix+"terr_um_weight")!=          null) terr_um_weight=Double.parseDouble(properties.getProperty(prefix+"terr_um_weight"));
		if (properties.getProperty(prefix+"terr_nan_tolerance")!=      null) terr_nan_tolerance=Double.parseDouble(properties.getProperty(prefix+"terr_nan_tolerance"));
		if (properties.getProperty(prefix+"terr_nan_grow")!=           null) terr_nan_grow=Integer.parseInt(properties.getProperty(prefix+"terr_nan_grow"));
		if (properties.getProperty(prefix+"terr_shrink_veget")!=       null) terr_shrink_veget=Integer.parseInt(properties.getProperty(prefix+"terr_shrink_veget"));
		if (properties.getProperty(prefix+"terr_shrink_terrain")!=     null) terr_shrink_terrain=Integer.parseInt(properties.getProperty(prefix+"terr_shrink_terrain"));
		if (properties.getProperty(prefix+"terr_vegetation_over")!=    null) terr_vegetation_over=Double.parseDouble(properties.getProperty(prefix+"terr_vegetation_over"));
		if (properties.getProperty(prefix+"terr_filter_veget")!=       null) terr_filter_veget=Integer.parseInt(properties.getProperty(prefix+"terr_filter_veget"));
		if (properties.getProperty(prefix+"terr_veget_grow")!=         null) terr_veget_grow=Integer.parseInt(properties.getProperty(prefix+"terr_veget_grow"));
		if (properties.getProperty(prefix+"terr_tile_woi")!=           null) terr_tile_woi=Boolean.parseBoolean(properties.getProperty(prefix+"terr_tile_woi"));
		if (properties.getProperty(prefix+"terr_continue")!=           null) terr_continue=Boolean.parseBoolean(properties.getProperty(prefix+"terr_continue"));
		if (properties.getProperty(prefix+"terr_woi_enclos")!=         null) terr_woi_enclos=stringToRectangle((String) properties.getProperty(prefix+"terr_woi_enclos"));
		if (properties.getProperty(prefix+"terr_woi_step")!=           null) terr_woi_step=stringToRectangle((String) properties.getProperty(prefix+"terr_woi_step"));
		if (properties.getProperty(prefix+"terr_woi_last")!=           null) terr_woi_last=stringToRectangle((String) properties.getProperty(prefix+"terr_woi_last"));
		if (properties.getProperty(prefix+"terr_skip_exist")!=         null) terr_skip_exist=Boolean.parseBoolean(properties.getProperty(prefix+"terr_skip_exist"));
		
		if (properties.getProperty(prefix+"terr_fit_terr")!=           null) terr_fit_terr=Boolean.parseBoolean(properties.getProperty(prefix+"terr_fit_terr"));
		if (properties.getProperty(prefix+"terr_fit_veget")!=          null) terr_fit_veget=Boolean.parseBoolean(properties.getProperty(prefix+"terr_fit_veget"));
		if (properties.getProperty(prefix+"terr_fit_alpha")!=          null) terr_fit_alpha=Boolean.parseBoolean(properties.getProperty(prefix+"terr_fit_alpha"));
		if (properties.getProperty(prefix+"terr_fit_elevations")!=     null) terr_fit_elevations=Boolean.parseBoolean(properties.getProperty(prefix+"terr_fit_elevations"));
		if (properties.getProperty(prefix+"terr_fit_terr_elev")!=      null) terr_fit_terr_elev=Boolean.parseBoolean(properties.getProperty(prefix+"terr_fit_terr_elev"));
		if (properties.getProperty(prefix+"terr_fit_terr_elev_pix")!=  null) terr_fit_terr_elev_pix=Boolean.parseBoolean(properties.getProperty(prefix+"terr_fit_terr_elev_pix"));
		if (properties.getProperty(prefix+"terr_fit_scenes")!=         null) terr_fit_scenes=Boolean.parseBoolean(properties.getProperty(prefix+"terr_fit_scenes"));
		for (int i = 0; i < terr_fits_disable.length; i++) {
			String prop_name_old = prefix+"terr_fit_disable_"+VegetationLMA.TVAO_NAMES[i];
			if (properties.getProperty(prop_name_old)!=null) terr_fits_disable[i]= StringToBooleans(properties.getProperty(prop_name_old));// booleans
			String prop_name = prefix+"terr_fits_disable_"+VegetationLMA.TVAO_NAMES[i];
			if (properties.getProperty(prop_name)!=null)     terr_fits_disable[i]= StringToBooleans(properties.getProperty(prop_name));// booleans
		}

		for (int i = 0; i < terr_only_fits_disable.length; i++) {
			String prop_name = prefix+"terr_only_fits_disable_"+VegetationLMA.TVAO_NAMES[i];
			if (properties.getProperty(prop_name)!=null)     terr_only_fits_disable[i]= StringToBooleans(properties.getProperty(prop_name));// booleans
		}
		
		if (properties.getProperty(prefix+"terr_max_warp")!=           null) terr_max_warp=Double.parseDouble(properties.getProperty(prefix+"terr_max_warp"));
		if (properties.getProperty(prefix+"terr_max_elevation")!=      null) terr_max_elevation=Integer.parseInt(properties.getProperty(prefix+"terr_max_elevation"));
		if (properties.getProperty(prefix+"terr_max_elev_terr")!=      null) terr_max_elev_terr=Integer.parseInt(properties.getProperty(prefix+"terr_max_elev_terr"));
		if (properties.getProperty(prefix+"terr_max_elev_chng")!=      null) terr_max_elev_chng=Double.parseDouble(properties.getProperty(prefix+"terr_max_elev_chng"));
		if (properties.getProperty(prefix+"terr_min_scenes")!=         null) terr_min_scenes=Integer.parseInt(properties.getProperty(prefix+"terr_min_scenes"));
		if (properties.getProperty(prefix+"terr_min_samples_scene")!=  null) terr_min_samples_scene=Integer.parseInt(properties.getProperty(prefix+"terr_min_samples_scene"));
		if (properties.getProperty(prefix+"terr_min_total_scenes")!=   null) terr_min_total_scenes=Integer.parseInt(properties.getProperty(prefix+"terr_min_total_scenes"));
		if (properties.getProperty(prefix+"terr_min_pixels")!=         null) terr_min_pixels=Integer.parseInt(properties.getProperty(prefix+"terr_min_pixels"));
		if (properties.getProperty(prefix+"terr_warm_veget")!=         null) terr_warm_veget=Boolean.parseBoolean(properties.getProperty(prefix+"terr_warm_veget"));
		if (properties.getProperty(prefix+"terr_warmest")!=            null) terr_warmest=Double.parseDouble(properties.getProperty(prefix+"terr_warmest"));
		if (properties.getProperty(prefix+"terr_initial_split")!=      null) terr_initial_split=Double.parseDouble(properties.getProperty(prefix+"terr_initial_split"));
		if (properties.getProperty(prefix+"terr_min_split_frac")!=     null) terr_min_split_frac=Double.parseDouble(properties.getProperty(prefix+"terr_min_split_frac"));
		
		if (properties.getProperty(prefix+"terr_difference")!=         null) terr_difference=Double.parseDouble(properties.getProperty(prefix+"terr_difference"));
		if (properties.getProperty(prefix+"terr_pull_cold")!=          null) terr_pull_cold=Double.parseDouble(properties.getProperty(prefix+"terr_pull_cold"));
		if (properties.getProperty(prefix+"terr_elevation_radius")!=   null) terr_elevation_radius=Double.parseDouble(properties.getProperty(prefix+"terr_elevation_radius"));
		if (properties.getProperty(prefix+"terr_terr_elev_radius")!=   null) terr_terr_elev_radius=Double.parseDouble(properties.getProperty(prefix+"terr_terr_elev_radius"));
		if (properties.getProperty(prefix+"terr_elev_radius_extra")!=  null) terr_elev_radius_extra=Double.parseDouble(properties.getProperty(prefix+"terr_elev_radius_extra"));
		
		if (properties.getProperty(prefix+"terr_alpha_contrast")!=     null) terr_alpha_contrast=Double.parseDouble(properties.getProperty(prefix+"terr_alpha_contrast"));
		if (properties.getProperty(prefix+"terr_alpha_dflt")!=         null) terr_alpha_dflt=Double.parseDouble(properties.getProperty(prefix+"terr_alpha_dflt"));
		
		if (properties.getProperty(prefix+"terr_alpha_sigma")!=        null) terr_alpha_sigma=Double.parseDouble(properties.getProperty(prefix+"terr_alpha_sigma"));
		if (properties.getProperty(prefix+"terr_alpha_init_min")!=     null) terr_alpha_init_min=Double.parseDouble(properties.getProperty(prefix+"terr_alpha_init_min")); 
		if (properties.getProperty(prefix+"terr_alpha_init_max")!=     null) terr_alpha_init_max=Double.parseDouble(properties.getProperty(prefix+"terr_alpha_init_max")); 
		if (properties.getProperty(prefix+"terr_alpha_init_offs")!=    null) terr_alpha_init_offs=Double.parseDouble(properties.getProperty(prefix+"terr_alpha_init_offs")); 
		
		if (properties.getProperty(prefix+"terr_alpha_loss")!=         null) terr_alpha_loss=Double.parseDouble(properties.getProperty(prefix+"terr_alpha_loss"));
		if (properties.getProperty(prefix+"terr_alpha_loss_lin")!=     null) terr_alpha_loss_lin=Double.parseDouble(properties.getProperty(prefix+"terr_alpha_loss_lin"));
		if (properties.getProperty(prefix+"terr_alpha_offset")!=       null) terr_alpha_offset=Double.parseDouble(properties.getProperty(prefix+"terr_alpha_offset"));
		if (properties.getProperty(prefix+"terr_alpha_0offset")!=      null) terr_alpha_0offset=Double.parseDouble(properties.getProperty(prefix+"terr_alpha_0offset"));
		if (properties.getProperty(prefix+"terr_alpha_min_veg")!=      null) terr_alpha_min_veg=Double.parseDouble(properties.getProperty(prefix+"terr_alpha_min_veg"));
		if (properties.getProperty(prefix+"terr_alpha_max_terrain")!=  null) terr_alpha_max_terrain=Double.parseDouble(properties.getProperty(prefix+"terr_alpha_max_terrain"));
		if (properties.getProperty(prefix+"terr_alpha_pull_pwr")!=     null) terr_alpha_pull_pwr=Double.parseDouble(properties.getProperty(prefix+"terr_alpha_pull_pwr"));
		if (properties.getProperty(prefix+"terr_alpha_lpf")!=          null) terr_alpha_lpf=Double.parseDouble(properties.getProperty(prefix+"terr_alpha_lpf"));
		if (properties.getProperty(prefix+"terr_alpha_lpf_border")!=   null) terr_alpha_lpf_border=Double.parseDouble(properties.getProperty(prefix+"terr_alpha_lpf_border"));
		if (properties.getProperty(prefix+"terr_alpha_piece_linear")!= null) terr_alpha_piece_linear=Boolean.parseBoolean(properties.getProperty(prefix+"terr_alpha_piece_linear"));
		if (properties.getProperty(prefix+"terr_alpha_scale_avg")!=    null) terr_alpha_scale_avg=Double.parseDouble(properties.getProperty(prefix+"terr_alpha_scale_avg"));
		if (properties.getProperty(prefix+"terr_alpha_push")!=         null) terr_alpha_push=Double.parseDouble(properties.getProperty(prefix+"terr_alpha_push"));
		if (properties.getProperty(prefix+"terr_alpha_push_neutral")!= null) terr_alpha_push_neutral=Double.parseDouble(properties.getProperty(prefix+"terr_alpha_push_neutral"));
		if (properties.getProperty(prefix+"terr_alpha_weight_center")!=null) terr_alpha_weight_center=Double.parseDouble(properties.getProperty(prefix+"terr_alpha_weight_center"));
		if (properties.getProperty(prefix+"terr_en_holes")!=           null) terr_en_holes=Boolean.parseBoolean(properties.getProperty(prefix+"terr_en_holes"));
		if (properties.getProperty(prefix+"terr_alpha_mm_hole")!=      null) terr_alpha_mm_hole=Double.parseDouble(properties.getProperty(prefix+"terr_alpha_mm_hole"));
		if (properties.getProperty(prefix+"terr_alpha_diff_hole")!=    null) terr_alpha_diff_hole=Double.parseDouble(properties.getProperty(prefix+"terr_alpha_diff_hole"));

		if (properties.getProperty(prefix+"terr_terr_lpf")!=           null) terr_terr_lpf=Double.parseDouble(properties.getProperty(prefix+"terr_terr_lpf"));
		if (properties.getProperty(prefix+"terr_veget_lpf")!=          null) terr_veget_lpf=Double.parseDouble(properties.getProperty(prefix+"terr_veget_lpf"));
		if (properties.getProperty(prefix+"terr_elev_lpf")!=           null) terr_elev_lpf=Double.parseDouble(properties.getProperty(prefix+"terr_elev_lpf"));
		if (properties.getProperty(prefix+"terr_terr_elev_lpf")!=      null) terr_terr_elev_lpf=Double.parseDouble(properties.getProperty(prefix+"terr_terr_elev_lpf"));
		if (properties.getProperty(prefix+"terr_terr_pull0")!=         null) terr_terr_pull0=Double.parseDouble(properties.getProperty(prefix+"terr_terr_pull0"));
		if (properties.getProperty(prefix+"terr_terr_pull_up")!=       null) terr_terr_pull_up=Double.parseDouble(properties.getProperty(prefix+"terr_terr_pull_up"));
		if (properties.getProperty(prefix+"terr_terr_pull_avg")!=      null) terr_terr_pull_avg=Double.parseDouble(properties.getProperty(prefix+"terr_terr_pull_avg"));
		if (properties.getProperty(prefix+"terr_veget_pull0")!=        null) terr_veget_pull0=Double.parseDouble(properties.getProperty(prefix+"terr_veget_pull0"));
		if (properties.getProperty(prefix+"terr_veget_pull_low_alpha")!= null)terr_veget_pull_low_alpha=Double.parseDouble(properties.getProperty(prefix+"terr_veget_pull_low_alpha"));
		if (properties.getProperty(prefix+"terr_elev_pull0")!=         null) terr_elev_pull0=Double.parseDouble(properties.getProperty(prefix+"terr_elev_pull0"));
		if (properties.getProperty(prefix+"terr_terr_elev_pull0")!=    null) terr_terr_elev_pull0=Double.parseDouble(properties.getProperty(prefix+"terr_terr_elev_pull0"));
		
		if (properties.getProperty(prefix+"terr_elev_alpha_en")!=      null) terr_elev_alpha_en=Boolean.parseBoolean(properties.getProperty(prefix+"terr_elev_alpha_en"));
		if (properties.getProperty(prefix+"terr_elev_alpha")!=         null) terr_elev_alpha=Double.parseDouble(properties.getProperty(prefix+"terr_elev_alpha"));
		if (properties.getProperty(prefix+"terr_elev_alpha_pwr")!=     null) terr_elev_alpha_pwr=Double.parseDouble(properties.getProperty(prefix+"terr_elev_alpha_pwr"));
		if (properties.getProperty(prefix+"terr_low_veget")!=          null) terr_low_veget=Double.parseDouble(properties.getProperty(prefix+"terr_low_veget"));
		if (properties.getProperty(prefix+"terr_scenes_pull0")!=       null) terr_scenes_pull0=Double.parseDouble(properties.getProperty(prefix+"terr_scenes_pull0"));
		// scaling elevation losses for high elevations (decrease pull and/or lpf for high elevations)
		if (properties.getProperty(prefix+"terr_elev_scale_thresh")!=  null) terr_elev_scale_thresh=Double.parseDouble(properties.getProperty(prefix+"terr_elev_scale_thresh"));
		if (properties.getProperty(prefix+"terr_elev_scale_pull")!=    null) terr_elev_scale_pull=Boolean.parseBoolean(properties.getProperty(prefix+"terr_elev_scale_pull"));
		if (properties.getProperty(prefix+"terr_elev_scale_lpf")!=     null) terr_elev_scale_lpf=Boolean.parseBoolean(properties.getProperty(prefix+"terr_elev_scale_lpf"));
		// tree-top removal
		if (properties.getProperty(prefix+"terr_ttop_en")!=            null) terr_ttop_en=Boolean.parseBoolean(properties.getProperty(prefix+"terr_ttop_en"));
		if (properties.getProperty(prefix+"terr_ttop_gb")!=            null) terr_ttop_gb=Double.parseDouble(properties.getProperty(prefix+"terr_ttop_gb"));
		if (properties.getProperty(prefix+"terr_ttop_min")!=           null) terr_ttop_min=Double.parseDouble(properties.getProperty(prefix+"terr_ttop_min"));
		if (properties.getProperty(prefix+"terr_ttop_rel_lev")!=       null) terr_ttop_rel_lev=Double.parseDouble(properties.getProperty(prefix+"terr_ttop_rel_lev"));
		if (properties.getProperty(prefix+"terr_ttop_rel_rad")!=       null) terr_ttop_rel_rad=Double.parseDouble(properties.getProperty(prefix+"terr_ttop_rel_rad"));
		if (properties.getProperty(prefix+"terr_ttop_frac")!=          null) terr_ttop_frac=Double.parseDouble(properties.getProperty(prefix+"terr_ttop_frac"));
		if (properties.getProperty(prefix+"terr_ttop_rem_rad")!=       null) terr_ttop_rem_rad=Double.parseDouble(properties.getProperty(prefix+"terr_ttop_rem_rad"));
		
		if (properties.getProperty(prefix+"terr_boost_parallax")!=     null) terr_boost_parallax=Double.parseDouble(properties.getProperty(prefix+"terr_boost_parallax"));
		if (properties.getProperty(prefix+"terr_max_parallax")!=       null) terr_max_parallax=Double.parseDouble(properties.getProperty(prefix+"terr_max_parallax"));
		if (properties.getProperty(prefix+"terr_hifreq_weight")!=      null) terr_hifreq_weight=Double.parseDouble(properties.getProperty(prefix+"terr_hifreq_weight"));
		if (properties.getProperty(prefix+"terr_terr_corr")!=          null) terr_terr_corr=Double.parseDouble(properties.getProperty(prefix+"terr_terr_corr"));
		if (properties.getProperty(prefix+"terr_reg_weights")!=        null) terr_reg_weights=Double.parseDouble(properties.getProperty(prefix+"terr_reg_weights"));
		if (properties.getProperty(prefix+"terr_lambda")!=             null) terr_lambda=Double.parseDouble(properties.getProperty(prefix+"terr_lambda"));
		if (properties.getProperty(prefix+"terr_lambda_scale_good")!=  null) terr_lambda_scale_good=Double.parseDouble(properties.getProperty(prefix+"terr_lambda_scale_good"));
		if (properties.getProperty(prefix+"terr_lambda_scale_bad")!=   null) terr_lambda_scale_bad=Double.parseDouble(properties.getProperty(prefix+"terr_lambda_scale_bad"));
		if (properties.getProperty(prefix+"terr_lambda_max")!=         null) terr_lambda_max=Double.parseDouble(properties.getProperty(prefix+"terr_lambda_max"));
		if (properties.getProperty(prefix+"terr_rms_diff")!=           null) terr_rms_diff=Double.parseDouble(properties.getProperty(prefix+"terr_rms_diff"));
		if (properties.getProperty(prefix+"terr_last_series")!=        null) terr_last_series=Integer.parseInt(properties.getProperty(prefix+"terr_last_series"));
		if (properties.getProperty(prefix+"terr_num_iter")!=           null) terr_num_iters=new int[] {Integer.parseInt(properties.getProperty(prefix+"terr_num_iter"))};
		if (properties.getProperty(prefix+"terr_num_iters")!=          null) terr_num_iters=StringToInts((String) properties.getProperty(prefix+"terr_num_iters"));
		
///		if (properties.getProperty(prefix+"terr_only_special")!=       null) terr_only_special=Boolean.parseBoolean(properties.getProperty(prefix+"terr_only_special"));
///		if (properties.getProperty(prefix+"terr_only_pix")!=           null) terr_only_pix=Boolean.parseBoolean(properties.getProperty(prefix+"terr_only_pix"));
		if (properties.getProperty(prefix+"terr_only_series")!=        null) terr_only_series=Integer.parseInt(properties.getProperty(prefix+"terr_only_series"));
		if (properties.getProperty(prefix+"terr_only_num_iters")!=     null) terr_only_num_iters=StringToInts((String) properties.getProperty(prefix+"terr_only_num_iters"));
		if (properties.getProperty(prefix+"terr_cholesky")!=           null) terr_cholesky=Integer.parseInt(properties.getProperty(prefix+"terr_cholesky"));
		
		if (properties.getProperty(prefix+"terr_recalc_weights")!=     null) {
			terr_recalc_weights= StringToBooleans(properties.getProperty(prefix+"terr_recalc_weights"));// booleans
		}
		
		if (properties.getProperty(prefix+"terr_recalc_opaque")!=      null) terr_recalc_opaque=Double.parseDouble(properties.getProperty(prefix+"terr_recalc_opaque"));
		if (properties.getProperty(prefix+"terr_recalc_pedestal")!=    null) terr_recalc_pedestal=Double.parseDouble(properties.getProperty(prefix+"terr_recalc_pedestal"));
		if (properties.getProperty(prefix+"terr_recalc_frac")!=        null) terr_recalc_frac=Double.parseDouble(properties.getProperty(prefix+"terr_recalc_frac"));
		if (properties.getProperty(prefix+"terr_recalc_dist")!=        null) terr_recalc_dist=Double.parseDouble(properties.getProperty(prefix+"terr_recalc_dist"));
		if (properties.getProperty(prefix+"terr_recalc_pwr")!=         null) terr_recalc_pwr=Double.parseDouble(properties.getProperty(prefix+"terr_recalc_pwr"));
		if (properties.getProperty(prefix+"terr_recalc_gb")!=          null) terr_recalc_gb=Double.parseDouble(properties.getProperty(prefix+"terr_recalc_gb"));
		if (properties.getProperty(prefix+"terr_recalc_boost")!=       null) terr_recalc_boost=Double.parseDouble(properties.getProperty(prefix+"terr_recalc_boost"));
		if (properties.getProperty(prefix+"terr_recalc_average")!=     null) terr_recalc_average=Boolean.parseBoolean(properties.getProperty(prefix+"terr_recalc_average"));
		
		if (properties.getProperty(prefix+"terr_crop")!=               null) terr_crop=Boolean.parseBoolean(properties.getProperty(prefix+"terr_crop"));
		if (properties.getProperty(prefix+"terr_keep_partial")!=       null) terr_keep_partial=Boolean.parseBoolean(properties.getProperty(prefix+"terr_keep_partial"));
		if (properties.getProperty(prefix+"terr_border_width")!=       null) terr_border_width=Integer.parseInt(properties.getProperty(prefix+"terr_border_width"));
		if (properties.getProperty(prefix+"terr_render_open")!=        null) terr_render_open=Boolean.parseBoolean(properties.getProperty(prefix+"terr_render_open"));
		if (properties.getProperty(prefix+"terr_render_no_alpha")!=    null) terr_render_no_alpha=Boolean.parseBoolean(properties.getProperty(prefix+"terr_render_no_alpha"));
		if (properties.getProperty(prefix+"terr_alpha_min")!=          null) terr_alpha_min=Double.parseDouble(properties.getProperty(prefix+"terr_alpha_min"));
		if (properties.getProperty(prefix+"terr_alpha_max")!=          null) terr_alpha_max=Double.parseDouble(properties.getProperty(prefix+"terr_alpha_max"));
		if (properties.getProperty(prefix+"terr_weight_opaque")!=      null) terr_weight_opaque=Double.parseDouble(properties.getProperty(prefix+"terr_weight_opaque"));
		if (properties.getProperty(prefix+"terr_boost_render")!=       null) terr_boost_render=Double.parseDouble(properties.getProperty(prefix+"terr_boost_render"));
		if (properties.getProperty(prefix+"terr_max_render")!=         null) terr_max_render=Double.parseDouble(properties.getProperty(prefix+"terr_max_render"));
		if (properties.getProperty(prefix+"terr_num_exaggerate")!=     null) terr_num_exaggerate=Integer.parseInt(properties.getProperty(prefix+"terr_num_exaggerate"));
		if (properties.getProperty(prefix+"terr_rebuild_elev")!=       null) terr_rebuild_elev=Boolean.parseBoolean(properties.getProperty(prefix+"terr_rebuild_elev"));
		if (properties.getProperty(prefix+"terr_elev_grow")!=          null) terr_elev_grow=Integer.parseInt(properties.getProperty(prefix+"terr_elev_grow"));
		if (properties.getProperty(prefix+"terr_threshold_terrain")!=  null) terr_threshold_terrain=Double.parseDouble(properties.getProperty(prefix+"terr_threshold_terrain"));
		if (properties.getProperty(prefix+"terr_min_max_terrain")!=    null) terr_min_max_terrain=Double.parseDouble(properties.getProperty(prefix+"terr_min_max_terrain"));
		if (properties.getProperty(prefix+"terr_min_terrain")!=        null) terr_min_terrain=Double.parseDouble(properties.getProperty(prefix+"terr_min_terrain"));
		if (properties.getProperty(prefix+"terr_min_vegetation")!=     null) terr_min_vegetation=Double.parseDouble(properties.getProperty(prefix+"terr_min_vegetation"));
		if (properties.getProperty(prefix+"terr_debug")!=              null) terr_debug=Integer.parseInt(properties.getProperty(prefix+"terr_debug"));
		if (properties.getProperty(prefix+"terr_debug_path")!=         null) terr_debug_path=(String) properties.getProperty(prefix+"terr_debug_path");
		if (properties.getProperty(prefix+"terr_debug_length")!=       null) terr_debug_length=Integer.parseInt(properties.getProperty(prefix+"terr_debug_length"));
		if (properties.getProperty(prefix+"terr_debug_improved")!=     null) terr_debug_improved=Boolean.parseBoolean(properties.getProperty(prefix+"terr_debug_improved"));
		if (properties.getProperty(prefix+"terr_debug_worsened")!=     null) terr_debug_worsened=Boolean.parseBoolean(properties.getProperty(prefix+"terr_debug_worsened"));
		// end of vegetation
		// Synthesis		
		if (properties.getProperty(prefix+"synth_directory")!=         null) synth_directory=(String) properties.getProperty(prefix+"synth_directory");
		if (properties.getProperty(prefix+"synth_scene")!=             null) synth_scene=(String) properties.getProperty(prefix+"synth_scene");
		if (properties.getProperty(prefix+"synth_model")!=             null) synth_model=(String) properties.getProperty(prefix+"synth_model");
		if (properties.getProperty(prefix+"synth_segments")!=          null) synth_segments=(String) properties.getProperty(prefix+"synth_segments");
		if (properties.getProperty(prefix+"synth_render")!=            null) synth_render=(String) properties.getProperty(prefix+"synth_render");
		if (properties.getProperty(prefix+"synth_crop_real")!=         null) synth_crop_real=Boolean.parseBoolean(properties.getProperty(prefix+"synth_crop_real"));
		if (properties.getProperty(prefix+"synth_scene_offs")!=        null) synth_scene_offs=(String) properties.getProperty(prefix+"synth_scene_offs");
		if (properties.getProperty(prefix+"synth_add_offs")!=          null) synth_add_offs=Boolean.parseBoolean(properties.getProperty(prefix+"synth_add_offs"));
	}
	
	@Override
	public IntersceneMatchParameters clone() throws CloneNotSupportedException {
		IntersceneMatchParameters imp =     new IntersceneMatchParameters();
 		imp.ims_use                       = this.ims_use;
 		imp.ims_rebuild                   = this.ims_rebuild;
 		imp.ims_offset                    = this.ims_offset;
 		imp.gmt_plus                      = this.gmt_plus;
 		imp.ims_ortho                     = this.ims_ortho.clone();
 		imp.ims_mount_atr                 = this.ims_mount_atr.clone();
 		imp.ims_mount_xyz                 = this.ims_mount_xyz.clone();
 		imp.pimu_gyr_offs                 = this.pimu_gyr_offs.clone();
 		imp.pimu_acc_offs                 = this.pimu_acc_offs.clone();
 		imp.ims_scale_xyz                 = this.ims_scale_xyz.clone();
 		imp.ims_scale_atr                 = this.ims_scale_atr.clone();
 		imp.fmg_initial_en                = this.fmg_initial_en;
 		imp.fmg_reorient_en               = this.fmg_reorient_en;
 		imp.fmg_distance                  = this.fmg_distance;
 		imp.fmg_max_quad                  = this.fmg_max_quad;
 		imp.fmg_rectilinear               = this.fmg_rectilinear;
 		imp.sfm_use                       = this.sfm_use;
 		imp.sfm_min_base                  = this.sfm_min_base;
 		imp.sfm_min_gain                  = this.sfm_min_gain;
 		imp.sfm_min_frac                  = this.sfm_min_frac;
 		imp.sfm_num_pairs 	              = this.sfm_num_pairs;
 		imp.sfp_tolerance 	              = this.sfp_tolerance;
 		imp.sfm_readjust                  = this.sfm_readjust;
 		imp.sfm_prev_frac 	              = this.sfm_prev_frac;
 		imp.sfm_same_weight               = this.sfm_same_weight;
 		imp.sfm_shrink                    = this.sfm_shrink;
 		imp.sfm_fade_sigma	              = this.sfm_fade_sigma;
 		imp.sfm_min_str1   	              = this.sfm_min_str1;
 		imp.sfm_min_str16   	          = this.sfm_min_str16;
 		imp.sfm_use_neibs 	              = this.sfm_use_neibs;
 		imp.sfm_neib_too_str1              = this.sfm_neib_too_str1;
 		imp.sfm_neib_too_str16            = this.sfm_neib_too_str16;
 		imp.sfm_neib_str1   	          = this.sfm_neib_str1;
 		imp.sfm_neib_str16                = this.sfm_neib_str16;
		imp.sfm_extrap_steps              = this.sfm_extrap_steps;
		imp.sfm_extrap_radius             = this.sfm_extrap_radius;
		imp.sfm_average_neibs             = this.sfm_average_neibs;
		imp.sfm_fill_weak                 = this.sfm_fill_weak;
		imp.sfm_extrapolate               = this.sfm_extrapolate;

		imp.sfm_max_dip                   = this.sfm_max_dip;
		imp.sfm_max_bump                  = this.sfm_max_bump;
		imp.sfm_filter_nan                = this.sfm_filter_nan;
		imp.sfm_only                      = this.sfm_only;
		
		imp.sfm_save_seq                  = this.sfm_save_seq;
		imp.sfm_show_seq                  = this.sfm_show_seq;
		imp.sfm_show_corr_ind             = this.sfm_show_corr_ind;
		imp.sfm_show_corr                 = this.sfm_show_corr;
		imp.omtch_img_set                 = this.omtch_img_set;
		
		imp.rln_gpu_width                 = this.rln_gpu_width;
		imp.rln_gpu_height                = this.rln_gpu_height;
		
		imp.rln_fat_zero                  = this.rln_fat_zero;
		imp.rln_use_neibs                 = this.rln_use_neibs;
		imp.rln_neibs_fill                = this.rln_neibs_fill;
		imp.rln_neib_radius               = this.rln_neib_radius;
		imp.rln_radius_frac               = this.rln_radius_frac;
		imp.rln_cent_radius               = this.rln_cent_radius;
		imp.rln_n_recenter                = this.rln_n_recenter;
		imp.rln_sngl_rstr                 = this.rln_sngl_rstr;
		imp.rln_neib_rstr	              = this.rln_neib_rstr;

		imp.pmtch_use_affine =              this.pmtch_use_affine;
        imp.pmtch_frac_remove             = this.pmtch_frac_remove;
		imp.pmtch_metric_err              = this.pmtch_metric_err;
		imp.pmtch_max_std                 = this.pmtch_max_std;
		imp.pmtch_min_std_rad             = this.pmtch_min_std_rad;
		imp.pmtch_ignore_rms              = this.pmtch_ignore_rms;
		imp.pmtch_num_iter                = this.pmtch_num_iter;
		imp.pmtch_pull_skew               = this.pmtch_pull_skew;
		imp.pmtch_pull_tilt               = this.pmtch_pull_tilt;
		imp.pmtch_pull_scale              = this.pmtch_pull_scale;
		imp.pmtch_cent_rad                = this.pmtch_cent_rad;
		imp.pmtch_max_cent_rad            = this.pmtch_max_cent_rad;
		imp.pmtch_cent_fill               = this.pmtch_cent_fill;
		imp.pmtch_cent_final              = this.pmtch_cent_final;
		imp.pmtch_ease_nosfm              = this.pmtch_ease_nosfm;
		
		imp.ospir_existing                = this.ospir_existing;
        imp.ospir_step                    = this.ospir_step;
        imp.ospir_range                   = this.ospir_range;
        imp.ospir_double                  = this.ospir_double;
        imp.ospir_good_rms                = this.ospir_good_rms;
        imp.ospir_max_rms                 = this.ospir_max_rms;
        imp.ospir_overlap                 = this.ospir_overlap;
        imp.ospir_num_iter                = this.ospir_num_iter;
        imp.ospir_rms_iter                = this.ospir_rms_iter.clone();
        
        imp.ospir_ignore_rms              = this.ospir_ignore_rms;
        imp.ospir_debug                   = this.ospir_debug;
        imp.ospir_augment                 = this.ospir_augment;

        imp.pwise_zoom                    = this.pwise_zoom;
        imp.pwise_overlap                 = this.pwise_overlap;
        imp.pwise_max_rms                 = this.pwise_max_rms;
        imp.pwise_skip_exist              = this.pwise_skip_exist;
        imp.pwise_refine_exist            = this.pwise_refine_exist;
        imp.pwise_delete_fail             = this.pwise_delete_fail;
        imp.pwise_gen_inverse             = this.pwise_gen_inverse;
        imp.pwise_save_each               = this.pwise_save_each;
        imp.pwise_log_append              = this.pwise_log_append;
        imp.pwise_log_path                = this.pwise_log_path;
        imp.pwise_debug                   = this.pwise_debug;

        imp.alt_overwrite                 = this.alt_overwrite;
        imp.alt_pairwise                  = this.alt_pairwise;
        imp.alt_sigma                     = this.alt_sigma;
        imp.alt_abs_outliers              = this.alt_abs_outliers;
        imp.alt_outliers                  = this.alt_outliers;
        imp.alt_refine                    = this.alt_refine;
        
        imp.alt_weight_rot                = this.alt_weight_rot;
        imp.alt_weight_tilt               = this.alt_weight_tilt;
        imp.alt_weight_scale              = this.alt_weight_scale;
        imp.alt_pull                      = this.alt_pull;
        imp.alt_pull_rots                 = this.alt_pull_rots;
        imp.alt_pull_tilts                = this.alt_pull_tilts  ;
        imp.alt_pull_scales               = this.alt_pull_scales;
        
        imp.pmap_move_only                = this.pmap_move_only;
        imp.pmap_ignore_affines           = this.pmap_ignore_affines;
        imp.pmap_use_inv                  = this.pmap_use_inv;
        imp.pmap_skew_pull                = this.pmap_skew_pull;
        imp.pmap_tilt_pull                = this.pmap_tilt_pull;
        imp.pmap_scale_pull               = this.pmap_scale_pull;
        imp.pmap_position_pull            = this.pmap_position_pull;
        imp.pmap_overlap_pow              = this.pmap_overlap_pow;
        imp.pmap_rms_diff                 = this.pmap_rms_diff;
        imp.pmap_num_iter                 = this.pmap_num_iter;
        imp.pmap_show_result              = this.pmap_show_result;
        imp.pmap_debugLevel               = this.pmap_debugLevel;

        imp.pequ_ignore_equalize          = this.pequ_ignore_equalize;
        imp.pequ_use_inv                  = this.pequ_use_inv;
        imp.pequ_scale_weight             = this.pequ_scale_weight;
        imp.pequ_pull_weight              = this.pequ_pull_weight;
        imp.pequ_half_weight_sec          = this.pequ_half_weight_sec;
        imp.pequ_min_weight_sec           = this.pequ_min_weight_sec ;
        imp.pequ_overlap_pow              = this.pequ_overlap_pow;
        imp.pequ_rms_diff                 = this.pequ_rms_diff;
        imp.pequ_num_iter                 = this.pequ_num_iter;
        imp.pequ_debugLevel               = this.pequ_debugLevel;
        
        imp.flt_list                      = this.flt_list;
        imp.flt_undef_only                = this.flt_undef_only;
        imp.flt_min_overlap               = this.flt_min_overlap;  
        imp.flt_max_overlap               = this.flt_max_overlap;  
        imp.flt_min_rms                   = this.flt_min_rms;
        imp.flt_max_rms                   = this.flt_max_rms;
        imp.flt_nan_rms                   = this.flt_nan_rms;
        imp.flt_filt_zoom                 = this.flt_filt_zoom;
        imp.flt_min_zoom                  = this.flt_min_zoom;
        imp.flt_max_zoom                  = this.flt_max_zoom;
        imp.flt_min_sfm                   = this.flt_min_sfm;
        imp.flt_max_sfm                   = this.flt_max_sfm;
        
        imp.flt_alt                       = this.flt_alt;
        imp.flt_orient                    = this.flt_orient;
        imp.flt_show_names                = this.flt_show_names;
        imp.flt_show_overlaps             = this.flt_show_overlaps;
        imp.flt_show_rms                  = this.flt_show_rms;
        imp.flt_show_zoom                 = this.flt_show_zoom;
        imp.flt_show_alt                  = this.flt_show_alt;
        imp.patt_save_top                 = this.patt_save_top;
        imp.patt_save_subdir              = this.patt_save_subdir;
        
		imp.center_reference              = this.center_reference;
		imp.lock_position                 = this.lock_position || this.cuas_rotation;
		imp.cuas_rotation                 = this.cuas_rotation;
		imp.manual_correction             = this.manual_correction;
		imp.overlap_sequences             = this.overlap_sequences;
		imp.reset_photometric             = this.reset_photometric;
		imp.force_ref_dsi                 = this.force_ref_dsi;
		imp.force_orientations            = this.force_orientations;
		imp.run_ly                        = this.run_ly;
		imp.run_ly_mode                   = this.run_ly_mode;
		imp.run_ly_lma                    = this.run_ly_lma;
		imp.min_num_orient                = this.min_num_orient;
		imp.min_num_interscene            = this.min_num_interscene;
		imp.adjust_imu_orient             = this.adjust_imu_orient;
		imp.apply_imu_orient              = this.apply_imu_orient;
		imp.orient_by_move                = this.orient_by_move;
		imp.orient_by_rot                 = this.orient_by_rot;
		imp.orient_combo                  = this.orient_combo;
		imp.adjust_gyro                   = this.adjust_gyro;
		imp.apply_gyro                    = this.apply_gyro;
		imp.adjust_accl                   = this.adjust_accl;
		imp.apply_accl                    = this.apply_accl;
		imp.calc_quat_corr                = this.calc_quat_corr;
		imp.apply_quat_corr               = this.apply_quat_corr;
		imp.use_quat_corr                 = this.use_quat_corr;
		imp.inertial_only                 = this.inertial_only;
		imp.generate_egomotion            = this.generate_egomotion;
		imp.generate_mapped               = this.generate_mapped;
		imp.reuse_video                   = this.reuse_video;
		imp.save_mapped_color             = this.save_mapped_color;
		imp.save_mapped_mono              = this.save_mapped_mono;
		imp.gen_avi_color                 = this.gen_avi_color;
		imp.gen_avi_mono                  = this.gen_avi_mono;
		imp.show_mapped_color             = this.show_mapped_color;
		imp.show_mapped_mono              = this.show_mapped_mono;
		
		imp.generate_raw                  = this.generate_raw;
		imp.generate_inf                  = this.generate_inf;
		imp.generate_fg                   = this.generate_fg;
		imp.generate_bg                   = this.generate_bg;
		
		imp.generate_stereo               = this.generate_stereo;
		
//		imp.stereo_bases                  = this.stereo_bases.clone();
		imp.stereo_views                  = this.stereo_views.clone();
		for (int i = 0; i < this.stereo_views.length; i++) {
			imp.stereo_views[i]           = this.stereo_views[i].clone();
		}
		imp.generate_stereo_var           = this.generate_stereo_var.clone();
		
		imp.export_images                 = this.export_images;
		imp.show_images                   = this.show_images;
		imp.show_images_bgfg              = this.show_images_bgfg;
		imp.show_images_mono              = this.show_images_mono;
		imp.export_ranges                 = this.export_ranges;
		imp.debug_ranges                  = this.debug_ranges;		
		imp.show_ranges                   = this.show_ranges;
		imp.export_ml_files               = this.export_ml_files;
		imp.show_color_nan                = this.show_color_nan;
		imp.show_mono_nan                 = this.show_mono_nan;
		imp.scale_extrap_atr              = this.scale_extrap_atr;
		imp.scale_extrap_xyz              = this.scale_extrap_xyz;
		imp.avg_len                       = this.avg_len;
		imp.min_num_scenes                = this.min_num_scenes;
		imp.max_num_scenes                = this.max_num_scenes;
		imp.blur_egomotion                = this.blur_egomotion;
		imp.range_disparity_offset        = this.range_disparity_offset;
		imp.range_min_strength            = this.range_min_strength;
		imp.range_max                     = this.range_max;
		imp.export3d                      = this.export3d;
		imp.export3dterrain               = this.export3dterrain;
		
		imp.export_CT                     = this.export_CT;
		imp.ct_min                        = this.ct_min;
		imp.ct_max                        = this.ct_max;
		imp.ct_step                       = this.ct_step;
		imp.ct_expand                     = this.ct_expand;
		
//		imp.terr_enable                   = this.terr_enable;
		imp.terr_rmse_above               = this.terr_rmse_above;
		imp.terr_rmse_below               = this.terr_rmse_below;
		imp.terr_num_refine               = this.terr_num_refine;
		imp.terr_frac_above               = this.terr_frac_above;
		imp.terr_frac_below               = this.terr_frac_below;
		imp.terr_rthrsh_abv               = this.terr_rthrsh_abv;
		imp.terr_flat_brd                 = this.terr_flat_brd;
		imp.terr_expand                   = this.terr_expand;
		imp.terr_save                     = this.terr_save;
		
		imp.scene_is_ref_test     = this.scene_is_ref_test;
		imp.render_ref            = this.render_ref;
		imp.render_scene          = this.render_scene;
		imp.toRGB                 = this.toRGB;
		imp.show_2d_correlations  = this.show_2d_correlations;
		imp.show_motion_vectors   = this.show_motion_vectors ;
		imp.debug_level           = this.debug_level;

		imp.test_ers =              this.test_ers;         
		imp.test_ers0 =             this.test_ers0;
		imp.test_ers1 =             this.test_ers1;

		imp.num_bottom                    = this.num_bottom;
		imp.num_passes                    = this.num_passes;
		imp.max_change                    = this.max_change;
		imp.min_disparity                 = this.min_disparity;
		imp.max_sym_disparity             = this.max_sym_disparity;
		imp.min_strength_lma              = this.min_strength_lma;
		imp.min_strength_replace          = this.min_strength_replace;
		imp.min_strength_blur             = this.min_strength_blur;
		imp.sigma                         = this.sigma;
		imp.num_blur                      = this.num_blur;
		imp.disparity_corr                = this.disparity_corr;
		imp.outliers_nth_fromextrem       = this.outliers_nth_fromextrem;
		imp.outliers_tolerance_absolute   = this.outliers_tolerance_absolute;
		imp.outliers_tolerance_relative   = this.outliers_tolerance_relative;
		imp.outliers_max_iter             = this.outliers_max_iter;
		imp.outliers_max_strength2        = this.outliers_max_strength2;
		imp.outliers_nth_fromextrem2      = this.outliers_nth_fromextrem2;
		imp.outliers_tolerance_absolute2  = this.outliers_tolerance_absolute2;
		imp.outliers_tolerance_relative2  = this.outliers_tolerance_relative2;
		imp.outliers_lma_max_strength     = this.outliers_lma_max_strength;
		imp.outliers_max_strength         = this.outliers_max_strength;
		imp.outliers_from_lma_max_strength= this.outliers_from_lma_max_strength;
		imp.search_radius                 = this.search_radius;
		imp.remove_no_lma_neib            = this.remove_no_lma_neib;
		imp.diff_from_lma_pos             = this.diff_from_lma_pos;
		imp.diff_from_lma_neg             = this.diff_from_lma_neg;
		imp.outliers_lma_nth_fromextrem   = this.outliers_lma_nth_fromextrem;
		imp.filter_margin                 = this.filter_margin;
		imp.weak_tolerance_absolute       = this.weak_tolerance_absolute;
		imp.weak_tolerance_relative       = this.weak_tolerance_relative;
		imp.weak_min_neibs                = this.weak_min_neibs;
		imp.strong_strength               = this.strong_strength;
		imp.weak_strength                 = this.weak_strength;
		
		imp.sky_extract =           this.sky_extract;
		imp.sky_recalc =            this.sky_recalc;
		imp.sky_highest_min =       this.sky_highest_min;
		imp.cold_frac =             this.cold_frac;
		imp.hot_frac =              this.hot_frac;
		imp.cold_scale =            this.cold_scale;
		imp.sky_seed =              this.sky_seed;
		imp.lma_seed =              this.lma_seed;
		imp.seed_temp =             this.seed_temp;
		imp.sky_shrink =            this.sky_shrink;
		imp.seed_rows =             this.seed_rows;
		imp.max_disparity =         this.max_disparity;
		imp.max_disparity_strength= this.max_disparity_strength;
		imp.sky_lim =               this.sky_lim;
		imp.lim_temp =              this.lim_temp;
		imp.sky_expand_extra =      this.sky_expand_extra;
		imp.sky_bottleneck =        this.sky_bottleneck;
		imp.sky_reexpand_extra =    this.sky_reexpand_extra;
		imp.min_strength =          this.min_strength;
		imp.lowest_sky_row =        this.lowest_sky_row;
		imp.sky_bottom_override =   this.sky_bottom_override;
		imp.sky_override_shrink =   this.sky_override_shrink;

		imp.clouds_en =             this.clouds_en;
		imp.clouds_fom =            this.clouds_fom;
		imp.clouds_spread =         this.clouds_spread;
		imp.clouds_disparity =      this.clouds_disparity;
		imp.clouds_weak =           this.clouds_weak;
		imp.clouds_strength =       this.clouds_strength;
		imp.clouds_not_strength =   this.clouds_not_strength;
		imp.clouds_strong =         this.clouds_strong;
		
		imp.wall_en =               this.wall_en;
		imp.wall_dflt =             this.wall_dflt;
		imp.wall_str =              this.wall_str;
		imp.wall_far =              this.wall_far;
		imp.wall_near =             this.wall_near;
		
		imp.treeline_en =     	    this.treeline_en;
		imp.treeline_wide =   	    this.treeline_wide;
		imp.treeline_height =       this.treeline_height;
		imp.treeline_width =        this.treeline_width;  
		imp.treeline_lim_high =     this.treeline_lim_high;
		imp.treeline_str =          this.treeline_str;    
		imp.treeline_far =          this.treeline_far;    
		imp.treeline_near =         this.treeline_near;  
		imp.treeline_fg_str =       this.treeline_fg_str; 
		imp.treeline_fg_far = 	    this.treeline_fg_far; 
		imp.treeline_fg_near =      this.treeline_fg_near;		
		
		imp.indoors_en =            this.indoors_en;
		imp.indoors_str =           this.indoors_str;
		imp.indoors_disp =          this.indoors_disp;
		imp.indoors_min_out =       this.indoors_min_out;
		
		imp.disp_boost_min =        this.disp_boost_min;
		imp.disp_boost_diff =       this.disp_boost_diff;
		imp.disp_boost_neibs =      this.disp_boost_neibs;
		imp.disp_boost_amount =     this.disp_boost_amount;
		imp.scale_combo_strength =  this.scale_combo_strength;
		imp.save_debug_images =     this.save_debug_images;
		
		imp.margin                = this.margin;
		imp.sensor_mask_inter     = this.sensor_mask_inter;
		imp.use_partial           = this.use_partial;
		imp.run_poly              = this.run_poly;
		imp.centroid_radius       = this.centroid_radius;
		imp.n_recenter            = this.n_recenter;
				
		imp.td_weight             = this.td_weight;
		imp.td_neib_weight        = this.td_neib_weight;
		imp.pd_weight             = this.pd_weight;
		imp.td_nopd_only          = this.td_nopd_only;
		imp.neib_notd_only        = this.neib_notd_only;
		
		imp.use_neibs             = this.use_neibs;                                                                
		imp.neibs_nofpn_only      = this.neibs_nofpn_only;
		imp.neibs_nofpn_init      = this.neibs_nofpn_init;
		imp.redo_both             = this.redo_both;
		imp.min_num_neibs         = this.min_num_neibs;
		imp.scale_neibs_pd        = this.scale_neibs_pd;
		imp.scale_neibs_td        = this.scale_neibs_td;
		imp.scale_avg_weight      = this.scale_avg_weight;
		
		imp.min_str               = this.min_str;
		imp.min_str_fpn           = this.min_str_fpn;
		imp.min_str_sum           = this.min_str_sum;
		imp.min_str_sum_fpn       = this.min_str_sum_fpn;
		imp.min_str_neib          = this.min_str_neib;
		imp.min_str_neib_fpn      = this.min_str_neib_fpn;
		
		imp.min_neibs             = this.min_neibs;
		imp.weight_zero_neibs     = this.weight_zero_neibs;
		imp.half_disparity        = this.half_disparity;
		imp.half_avg_diff         = this.half_avg_diff;

        imp.eig_use =               this.eig_use;
        imp.eig_xy_lma =            this.eig_xy_lma;
		imp.eig_str_sum =           this.eig_str_sum;
		imp.eig_str_sum_fpn =       this.eig_str_sum_fpn;
		imp.eig_str_neib =          this.eig_str_neib;
		imp.eig_str_neib_fpn =      this.eig_str_neib_fpn;
		imp.eig_min_abs =           this.eig_min_abs;
		imp.eig_min_rel =           this.eig_min_rel;
		imp.eig_sub_frac =          this.eig_sub_frac;
		imp.eig_max_sqrt =          this.eig_max_sqrt;
		imp.eig_min_sqrt =          this.eig_min_sqrt;
		
		imp.eig_recenter =          this.eig_recenter;
		imp.eig_sub_frac1 =         this.eig_sub_frac1;
		imp.eig_scale_axes =        this.eig_scale_axes;
		imp.eig_inc_axes =          this.eig_inc_axes;
		
		imp.eig_use_neibs =         this.eig_use_neibs;
		imp.eig_min_weaks =         this.eig_min_weaks;
		imp.eig_min_strongs =       this.eig_min_strongs;
		imp.eig_disp_diff =         this.eig_disp_diff;
		
		imp.eig_remove_neibs =      this.eig_remove_neibs;
		imp.eig_filt_other =        this.eig_filt_other;
		imp.eig_max_rms =           this.eig_max_rms;
		imp.eig_fast2x2 =           this.eig_fast2x2;
		
		imp.use_combo_reliable    = this.use_combo_reliable;
		imp.ref_need_lma          = this.ref_need_lma;
		imp.ref_need_lma_combo    = this.ref_need_lma_combo;
		imp.min_ref_str           = this.min_ref_str;
		imp.min_ref_str_lma       = this.min_ref_str_lma;
		imp.min_ref_frac          = this.min_ref_frac;
		
		imp.ref_smooth =            this.ref_smooth;
		imp.ref_smooth_always =     this.ref_smooth_always;
		imp.ref_sigma =             this.ref_sigma;
		imp.ref_smooth_diff =       this.ref_smooth_diff;
		
		imp.sfm_filter =            this.sfm_filter;
		imp.sfm_minmax =            this.sfm_minmax;
		imp.sfm_fracmax =           this.sfm_fracmax;
		imp.sfm_fracall =           this.sfm_fracall;
		
		imp.pix_step =              this.pix_step;
		imp.search_rad =            this.search_rad;
		imp.maybe_sum =             this.maybe_sum;
		imp.sure_sum =              this.sure_sum;
		imp.maybe_avg =             this.maybe_avg;
		imp.sure_avg =              this.sure_avg;
		imp.max_search_rms =        this.max_search_rms;
		imp.maybe_fom =             this.maybe_fom;
		imp.sure_fom =              this.sure_fom;
		imp.treat_serch_fpn       = this.treat_serch_fpn;
		imp.use_combo_dsi         = this.use_combo_dsi;
		imp.use_lma_dsi           = this.use_lma_dsi;
		imp.fpn_remove            = this.fpn_remove;
		imp.fpn_max_offset        = this.fpn_max_offset;
		imp.fpn_min_offset        = this.fpn_min_offset;
		imp.fpn_radius            = this.fpn_radius;
		imp.fpn_ignore_border     = this.fpn_ignore_border;
		
		imp.min_offset            = this.min_offset;
		imp.max_pull_jump         = this.max_pull_jump;
		imp.max_rel_offset        = this.max_rel_offset;
		imp.max_roll_deg          = this.max_roll_deg;
		imp.max_zoom_diff         = this.max_zoom_diff;
		imp.fpn_skip              = this.fpn_skip;
		imp.fpn_rematch           = this.fpn_rematch;
		imp.refine_invert         = this.refine_invert;
		imp.use_precomp           = this.use_precomp;
		
		imp.mov_en                = this.mov_en;
		imp.mov_sigma             = this.mov_sigma;
		imp.mov_max_std           = this.mov_max_std;
		imp.mov_thresh_rel        = this.mov_thresh_rel;
		imp.mov_thresh_abs        = this.mov_thresh_abs;
		imp.mov_clust_max         = this.mov_clust_max;
		imp.mov_grow              = this.mov_grow;
		imp.mov_max_len           = this.mov_max_len;
		imp.mov_show              = this.mov_show;
		imp.mov_debug_level       = this.mov_debug_level;
		imp.adjust_atr[0]         = this.adjust_atr[0];
		imp.adjust_atr[1]         = this.adjust_atr[1];
		imp.adjust_atr[2]         = this.adjust_atr[2];
		imp.adjust_xyz[0]         = this.adjust_xyz[0];
		imp.adjust_xyz[1]         = this.adjust_xyz[1];
		imp.adjust_xyz[2]         = this.adjust_xyz[2];
		imp.exit_change_atr       = this.exit_change_atr;
		imp.exit_change_xyz       = this.exit_change_xyz;
		imp.max_cycles            = this.max_cycles;
		imp.max_LMA               = this.max_LMA;
		imp.max_rms               = this.max_rms;
		
		imp.eq_en =                 this.eq_en;
		imp.eq_stride_hor =         this.eq_stride_hor;
		imp.eq_stride_vert =        this.eq_stride_vert;
		imp.eq_min_stile_weight =   this.eq_min_stile_weight;
		imp.eq_min_stile_number =   this.eq_min_stile_number;
		imp.eq_min_stile_fraction = this.eq_min_stile_fraction;
		imp.eq_min_disparity =      this.eq_min_disparity;
		imp.eq_max_disparity =      this.eq_max_disparity;
		imp.eq_weight_add =         this.eq_weight_add;
		imp.eq_weight_scale =       this.eq_weight_scale;
		imp.eq_level =              this.eq_level;
		
		imp.mb_en =                 this.mb_en;
		imp.mb_tau =                this.mb_tau;
		imp.mb_max_gain =           this.mb_max_gain;
		imp.mb_max_gain_inter =     this.mb_max_gain_inter;

		imp.cuas_update_existing =  this.cuas_update_existing;
		imp.cuas_discard_border =   this.cuas_discard_border;
		imp.cuas_max_fold =         this.cuas_max_fold;
		imp.cuas_min_in_row_col =   this.cuas_min_in_row_col;
		imp.cuas_clt_variant =      this.cuas_clt_variant;
		imp.cuas_clt_threshold =    this.cuas_clt_threshold;
		imp.cuas_clt_decrease =     this.cuas_clt_decrease;
		imp.cuas_decay_average =    this.cuas_decay_average;
		imp.cuas_keep_fraction =    this.cuas_keep_fraction;
		imp.cuas_step =             this.cuas_step;
		imp.cuas_subtract_fpn =     this.cuas_subtract_fpn;
		imp.cuas_calc_fpn =         this.cuas_calc_fpn;
		imp.cuas_rot_period =       this.cuas_rot_period;
		imp.cuas_min_series =       this.cuas_min_series;
		
		imp.cuas_subtract_rowcol =  this.cuas_subtract_rowcol;
		imp.cuas_calc_rowcol =      this.cuas_calc_rowcol;
		imp.cuas_um_sigma =         this.cuas_um_sigma;
		imp.cuas_max_abs_rowcol =   this.cuas_max_abs_rowcol;
		imp.cuas_outliers_rowcol =  this.cuas_outliers_rowcol;
		imp.cuas_reset_first =      this.cuas_reset_first;
		
		imp.cuas_invert_margins =   this.cuas_invert_margins;
		imp.cuas_invert_iters =     this.cuas_invert_iters;
		imp.cuas_invert_tolerance = this.cuas_invert_tolerance;
		imp.cuas_invert_gap2 =      this.cuas_invert_gap2;
		
		imp.cuas_w_um_sigma =  		this.cuas_w_um_sigma;
		imp.cuas_w_good_margins=	this.cuas_w_good_margins;
		imp.cuas_w_threshold = 		this.cuas_w_threshold;
		imp.cuas_w_scale_thresh =	this.cuas_w_scale_thresh;		
		imp.cuas_w_expand_max =		this.cuas_w_expand_max;
		imp.cuas_w_outliers =  		this.cuas_w_outliers;
		imp.cuas_w_blur =      		this.cuas_w_blur;
		
		imp.cuas_dsi_parent =    	this.cuas_dsi_parent;
		imp.cuas_fg_strength =   	this.cuas_fg_strength;
		imp.cuas_disp_add =      	this.cuas_disp_add;
		imp.cuas_skip_bottom =   	this.cuas_skip_bottom;
		imp.cuas_multi_strength =	this.cuas_multi_strength;
		imp.cuas_reliable_str =	    this.cuas_reliable_str;

		imp.cuas_fat_zero =         this.cuas_fat_zero;
		imp.cuas_cent_radius =      this.cuas_cent_radius;
		imp.cuas_n_recenter =       this.cuas_n_recenter;
		imp.cuas_rstr =             this.cuas_rstr;
		imp.cuas_smooth =           this.cuas_smooth;
		imp.cuas_corr_pairs =       this.cuas_corr_pairs;
		imp.cuas_corr_offset =      this.cuas_corr_offset;
		imp.cuas_gaussian_ra =      this.cuas_gaussian_ra;
		imp.cuas_temporal_um =      this.cuas_temporal_um;
		imp.cuas_tum_threshold =    this.cuas_tum_threshold;
		
		imp.cuas_precorr_ra =       this.cuas_precorr_ra;
		imp.cuas_corr_step =        this.cuas_corr_step;
		imp.cuas_half_step =        this.cuas_half_step;
		imp.cuas_max_range =        this.cuas_max_range;
		imp.cuas_num_cycles =       this.cuas_num_cycles;
		imp.cuas_speed_min =        this.cuas_speed_min;
		imp.cuas_speed_pref =       this.cuas_speed_pref;
		imp.cuas_speed_boost =      this.cuas_speed_boost;
		imp.cuas_lmax_flt_neglim =  this.cuas_lmax_flt_neglim;
		imp.cuas_lmax_flt_hsigma =  this.cuas_lmax_flt_hsigma;
		imp.cuas_lmax_flt_lsigma =  this.cuas_lmax_flt_lsigma;
		imp.cuas_lmax_flt_scale =   this.cuas_lmax_flt_scale;
		imp.cuas_sky_threshold =    this.cuas_sky_threshold;
		imp.cuas_sky_offset =       this.cuas_sky_offset.clone();
		
		imp.cuas_lmax_fraction =    this.cuas_lmax_fraction;
		imp.cuas_lmax_hack_ridge =  this.cuas_lmax_hack_ridge;
		imp.cuas_lmax_radius =      this.cuas_lmax_radius;
		imp.cuas_lmax_zero =        this.cuas_lmax_zero;
		
		imp.cuas_centered_radius =  this.cuas_centered_radius;		
		imp.cuas_centered_blur =    this.cuas_centered_blur;		
		imp.cuas_duplicate_tolerance = this.cuas_duplicate_tolerance;	
		
		imp.cuas_target_radius =    this.cuas_target_radius;		
		imp.cuas_target_strength =  this.cuas_target_strength;		
		
		imp.cuas_target_frac =      new double [this.cuas_target_frac.length][];
		for (int i = 0; i < this.cuas_target_frac.length; i++) {
			imp.cuas_target_frac[i] = this.cuas_target_frac[i].clone();
		}
		imp.cuas_no_border =        this.cuas_no_border;
		imp.cuas_lma_sigma =        this.cuas_lma_sigma;
		imp.cuas_wnd_pedestal =     this.cuas_wnd_pedestal;
		imp.cuas_lma_r0 =           this.cuas_lma_r0;
		imp.cuas_lma_ovrsht =       this.cuas_lma_ovrsht;
		imp.cuas_lma_fit_xy =       this.cuas_lma_fit_xy;
		imp.cuas_lma_fit_a =        this.cuas_lma_fit_a;
		imp.cuas_lma_fit_c =        this.cuas_lma_fit_c;
		imp.cuas_lma_fit_r =        this.cuas_lma_fit_r;
		imp.cuas_lma_fit_k =        this.cuas_lma_fit_k;
		
		imp.cuas_lambda =           this.cuas_lambda;
		imp.cuas_lambda_good =      this.cuas_lambda_good;
		imp.cuas_lambda_bad =       this.cuas_lambda_bad;
		imp.cuas_lambda_max =       this.cuas_lambda_max;
		imp.cuas_rms_diff =         this.cuas_rms_diff;
		imp.cuas_num_iter =         this.cuas_num_iter;
		imp.cuas_lma_rms =          this.cuas_lma_rms;
		imp.cuas_lma_arms =         this.cuas_lma_arms;
		imp.cuas_lma_rrms =         this.cuas_lma_rrms;
		imp.cuas_lma_mina =         this.cuas_lma_mina;
		imp.cuas_lma_pre_mina =     this.cuas_lma_pre_mina;
		imp.cuas_min_keep =         this.cuas_min_keep;
		
		imp.cuas_lma_maxr =         this.cuas_lma_maxr;
		imp.cuas_lma_minr1 =        this.cuas_lma_minr1;
		imp.cuas_lma_mink =         this.cuas_lma_mink;
		imp.cuas_lma_maxk =         this.cuas_lma_maxk;
		imp.cuas_lma_a2a =          this.cuas_lma_a2a;
		
		imp.cuas_slow_ra =          this.cuas_slow_ra;
		imp.cuas_slow_score =       this.cuas_slow_score;

		imp.cuas_score_mv =         this.cuas_score_mv;
		imp.cuas_score_lma =        this.cuas_score_lma;
		imp.cuas_factor_lim =       this.cuas_factor_lim;
		imp.cuas_factor_pow =       this.cuas_factor_pow;
		imp.cuas_score_coeff =      this.cuas_score_coeff.clone();
		imp.cuas_center_targ =      this.cuas_center_targ; 
		imp.cuas_multi_targ =       this.cuas_multi_targ; 
		
		imp.cuas_isolated =         this.cuas_isolated; 
		imp.cuas_min_seq =          this.cuas_min_seq;
		imp.cuas_enough_seq =       this.cuas_enough_seq;
		imp.cuas_seq_travel=        this.cuas_seq_travel;
		imp.cuas_min_seq_fl =       this.cuas_min_seq_fl; 
		
		imp.cuas_max_mismatch =     this.cuas_max_mismatch; 
		imp.cuas_good_mismatch =    this.cuas_good_mismatch;
		imp.cuas_slow_fast_mismatch = this.cuas_slow_fast_mismatch;
		imp.cuas_match_len_pwr =    this.cuas_match_len_pwr;
		
		imp.cuas_fail_mismatch =    this.cuas_fail_mismatch; 
		imp.cuas_ignore_mismatch =  this.cuas_ignore_mismatch; 
		imp.cuas_by_horizon =       this.cuas_by_horizon; 
		imp.cuas_horizon =          this.cuas_horizon; 
		
		imp.cuas_mask_width =       this.cuas_mask_width; 
		imp.cuas_mask_blur =        this.cuas_mask_blur;  
		imp.cuas_mask_round =       this.cuas_mask_round; 
		imp.cuas_target_type=       this.cuas_target_type;
		imp.cuas_known_type=        this.cuas_known_type;
		
		imp.cuas_input_range =		this.cuas_input_range;
		imp.cuas_scale2x =    		this.cuas_scale2x;    
		imp.cuas_ra_background =    this.cuas_ra_background;

		imp.cuas_targets =          this.cuas_targets;
		imp.cuas_iter_show=         this.cuas_iter_show;
		imp.cuas_2d_save_show =     this.cuas_2d_save_show;
		imp.cuas_intermed_low =     this.cuas_intermed_low;
		imp.cuas_intermed_high =    this.cuas_intermed_high;
		imp.cuas_intermed_giga =    this.cuas_intermed_giga;
		imp.cuas_debug_more =       this.cuas_debug_more;
		
		imp.cuas_generate =         this.cuas_generate;
		imp.cuas_generate_csv =     this.cuas_generate_csv;
		imp.cuas_save_mono =        this.cuas_save_mono;
		imp.cuas_save_color =       this.cuas_save_color;
		imp.cuas_save_video =       this.cuas_save_video;
		imp.cuas_clean_video =      this.cuas_clean_video;

		imp.cuas_annotate =         this.cuas_annotate;
		imp.cuas_annotate_uas =     this.cuas_annotate_uas;
		imp.cuas_text_color=        this.cuas_text_color; // Color is immutable, no need to clone
		imp.cuas_text_bg=           this.cuas_text_bg;
		imp.cuas_selected_color=    this.cuas_selected_color;
		
		imp.cuas_transparent =      this.cuas_transparent;
		imp.cuas_transparent_uas =  this.cuas_transparent_uas;
		imp.cuas_font_name =        this.cuas_font_name;
		imp.cuas_font_size =        this.cuas_font_size;
		imp.cuas_font_type =        this.cuas_font_type;
		imp.cuas_font_spacing =     this.cuas_font_spacing;
		imp.cuas_annot_missing =    this.cuas_annot_missing;
		imp.cuas_px0 =              this.cuas_px0;
		imp.cuas_py0 =              this.cuas_py0;
		imp.cuas_az0 =              this.cuas_az0;
		imp.cuas_el0 =              this.cuas_el0;
		imp.cuas_show_disp =        this.cuas_show_disp;
		imp.cuas_show_rng =         this.cuas_show_rng;
		imp.cuas_show_inf =         this.cuas_show_inf;
		imp.cuas_show_inf_gt =      this.cuas_show_inf_gt;
		imp.cuas_show_true_rng =    this.cuas_show_true_rng;

		imp.cuas_smooth_omegas =    this.cuas_smooth_omegas;
		imp.cuas_rng_um =           this.cuas_rng_um;
		imp.cuas_rng_um_all =       this.cuas_rng_um_all;
		imp.cuas_rng_um_sigma =     this.cuas_rng_um_sigma;
		imp.cuas_rng_um2 =          this.cuas_rng_um2;
		imp.cuas_rng_um_weight =    this.cuas_rng_um_weight;
		imp.cuas_rng_coswnd =       this.cuas_rng_coswnd;
		imp.cuas_rng_combine =      this.cuas_rng_combine;
		
		imp.cuas_rng_radius0 =      this.cuas_rng_radius0;
		imp.cuas_rng_radius =       this.cuas_rng_radius;
		imp.cuas_rng_blur =         this.cuas_rng_blur;
		imp.cuas_rng_fz0 =          this.cuas_rng_fz0;
		imp.cuas_rng_fz =           this.cuas_rng_fz;
		imp.cuas_rng_scale =        this.cuas_rng_scale;
		
		imp.cuas_dual_pass =        this.cuas_dual_pass;
		imp.cuas_disp_only2 =       this.cuas_disp_only2;
		imp.cuas_lma_bypass =       this.cuas_lma_bypass;
		imp.cuas_mcorr_sel =        this.cuas_mcorr_sel;
		imp.cuas_mcorr_sel_lma =    this.cuas_mcorr_sel_lma;
		
		imp.cuas_max_rel_rms =      this.cuas_max_rel_rms;
		imp.cuas_min_strength =     this.cuas_min_strength;
		imp.cuas_ac_offset =        this.cuas_ac_offset;
		imp.cuas_min_max_ac =       this.cuas_min_max_ac;
		imp.cuas_min_min_ac =       this.cuas_min_min_ac;
		imp.cuas_reset_disparity =  this.cuas_reset_disparity;
		imp.cuas_initial_disparity= this.cuas_initial_disparity;
		imp.cuas_infinity =         this.cuas_infinity;
		
		imp.cuas_rng_img =          this.cuas_rng_img;
		imp.cuas_rng_glob =         this.cuas_rng_glob;
		imp.cuas_rng_niterate =     this.cuas_rng_niterate;
		
		imp.cuas_rng_disp =         this.cuas_rng_disp;
		imp.cuas_rng_vfy =          this.cuas_rng_vfy;
		imp.cuas_glob_ends =        this.cuas_glob_ends;
		imp.cuas_rng_diff =         this.cuas_rng_diff;
		
		imp.cuas_flw_levels =       this.cuas_flw_levels;
		imp.cuas_flw_len =          this.cuas_flw_len;
		imp.cuas_flw_diff =         this.cuas_flw_diff;
		
		imp.cuas_known_err=         this.cuas_known_err;
        imp.cuas_tmtch_frac =       this.cuas_tmtch_frac;
        imp.cuas_tmtch_ends =       this.cuas_tmtch_ends;
        imp.cuas_tmtch_gaps =       this.cuas_tmtch_gaps;
        imp.cuas_tmtch_pix =        this.cuas_tmtch_pix;
        imp.cuas_local_apix =       this.cuas_local_apix;
        imp.cuas_tmtch_apix =       this.cuas_tmtch_apix;
        imp.cuas_tmtch_rpix =       this.cuas_tmtch_rpix;
        imp.cuas_tmtch_axv =        this.cuas_tmtch_axv;
        imp.cuas_tmtch_axv_k =      this.cuas_tmtch_axv_k;
        imp.cuas_tmtch_diff =       this.cuas_tmtch_diff;
        imp.cuas_local_diff =       this.cuas_local_diff;
        imp.cuas_tmtch_disp =       this.cuas_tmtch_disp;
		imp.cuas_tmtch_short =      this.cuas_tmtch_short;
		
        imp.cuas_radar_range =      this.cuas_radar_range;
        imp.cuas_radar_height =     this.cuas_radar_height;
        imp.cuas_radar_bottom =     this.cuas_radar_bottom;
        imp.cuas_radar_top =        this.cuas_radar_top;
        imp.cuas_radar_uas_color =  this.cuas_radar_uas_color;
        imp.cuas_radar_color =      this.cuas_radar_color;
        imp.cuas_radar_radius =     this.cuas_radar_radius;
        imp.cuas_radar_uas_rad =    this.cuas_radar_uas_rad;
        imp.cuas_grid_ring =        this.cuas_grid_ring;
        imp.cuas_grid_line =        this.cuas_grid_line;
        imp.cuas_grid_annot =       this.cuas_grid_annot;
        imp.cuas_grid_color =       this.cuas_grid_color;
        imp.cuas_grid_font =        this.cuas_grid_font;
        imp.cuas_grid_az_top =      this.cuas_grid_az_top;
        imp.cuas_grid_sides =       this.cuas_grid_sides;

        imp.cuas_debug =            this.cuas_debug;
		imp.cuas_dbg_rng_seq =      this.cuas_dbg_rng_seq;
		imp.cuas_dbg_rng_tgt =      this.cuas_dbg_rng_tgt;
		imp.cuas_dbg_show_mode =    this.cuas_dbg_show_mode;
		imp.cuas_step_debug =       this.cuas_step_debug;
		imp.cuas_save_stats =       this.cuas_save_stats;
		imp.cuas_target_debug =     this.cuas_target_debug;
		imp.cuas_target_score =     this.cuas_target_score;
		imp.cuas_target_color =     this.cuas_target_color;
		imp.cuas_log_ranging =      this.cuas_log_ranging;
		
		imp.cuas_overwrite =        this.cuas_overwrite;
		imp.cuas_num_orient =       this.cuas_num_orient;
		imp.cuas_num_accum =        this.cuas_num_accum;
		
		imp.cuas_reuse_targets =    this.cuas_reuse_targets;
		imp.cuas_reuse_path=        this.cuas_reuse_path;
		imp.cuas_reuse_disparity =  this.cuas_reuse_disparity;
		imp.cuas_reuse_globals =    this.cuas_reuse_globals;
		imp.cuas_max_disp_diff =    this.cuas_max_disp_diff;
		imp.cuas_min_disp_str =     this.cuas_min_disp_str;
		imp.cuas_rng_limit =        this.cuas_rng_limit;

		imp.mb_gain_index_pose    = this.mb_gain_index_pose;
		imp.mb_gain_index_depth   = this.mb_gain_index_depth;

		imp.mb_ers_index          = this.mb_ers_index;
		imp.mb_ers_y_index        = this.mb_ers_y_index;
		imp.mb_ers_r_index        = this.mb_ers_r_index;
		imp.mb_all_index          = this.mb_all_index;
		imp.mb_pref_orient        = this.mb_pref_orient;
		imp.lma_use_Z             = this.lma_use_Z;
		imp.lma_use_R             = this.lma_use_R;
		
		imp.configured_lma        = this.configured_lma;
		imp.freeze_xy_pull        = this.freeze_xy_pull;
		imp.copy_pull_current     = this.copy_pull_current;
		imp.restore_imu           = this.restore_imu;
		imp.lpf_xy                = this.lpf_xy;
		imp.readjust_xy_ims       = this.readjust_xy_ims;
		imp.reg_weight_xy         = this.reg_weight_xy;
		
		imp.quat_lambda           = this.quat_lambda;
		imp.quat_lambda_scale_good= this.quat_lambda_scale_good;
		imp.quat_lambda_scale_bad = this.quat_lambda_scale_bad;
		imp.quat_lambda_max       = this.quat_lambda_max;
		imp.quat_rms_diff         = this.quat_rms_diff;
		imp.quat_num_iter         = this.quat_num_iter;
		imp.quat_reg_w            = this.quat_reg_w;
		
		imp.quat_max_ratio        = this.quat_max_ratio;
		imp.quat_max_change       = this.quat_max_change;
		imp.quat_min_transl       = this.quat_min_transl;
		imp.quat_min_rot          = this.quat_min_rot;
		imp.quat_min_lin          = this.quat_min_lin;

		imp.stereo_merge          = this.stereo_merge;
		imp.stereo_gap            = this.stereo_gap;
		imp.stereo_intereye       = this.stereo_intereye;
		imp.stereo_phone_width    = this.stereo_phone_width;
		
		imp.extra_hor_tile        = this.extra_hor_tile;
		imp.extra_vert_tile       = this.extra_vert_tile;
		imp.crop_3d               = this.crop_3d;
		imp.sensor_mask           = this.sensor_mask;
		imp.merge_all             = this.merge_all;

		imp.video_fps =             this.video_fps;
		imp.sensor_fps =            this.sensor_fps;
		imp.mode_avi =              this.mode_avi;
		imp.avi_JPEG_quality =      this.avi_JPEG_quality;
		imp.run_ffmpeg =            this.run_ffmpeg;
		imp.video_ext =             this.video_ext;
		imp.video_codec =           this.video_codec;
		imp.video_crf =             this.video_crf;
		imp.video_bitrate_m =       this.video_bitrate_m;
		imp.remove_avi =            this.remove_avi;
		imp.video_codec_combo =     this.video_codec_combo;
		imp.video_crf_combo =       this.video_crf_combo;
		imp.add_average =           this.add_average;
		imp.calculate_average =     this.calculate_average;
		imp.subtract_average =      this.subtract_average;
		imp.running_average =       this.running_average;
		imp.extract_center_orientation=this.extract_center_orientation;
		imp.um_mono =               this.um_mono;
		imp.um_sigma =              this.um_sigma;
		imp.um_weight =             this.um_weight;
		imp.um_mono_linear =        this.um_mono_linear;
		imp.mono_fixed =            this.mono_fixed;
		imp.mono_range =            this.mono_range;
		
		imp.anaglyth_en =           this.anaglyth_en;
		imp.anaglyph_left =         this.anaglyph_left;
		imp.anaglyph_right =        this.anaglyph_right;
		
		imp.annotate_color =        this.annotate_color;
		imp.annotate_mono =         this.annotate_mono;
		imp.annotate_color_color =  this.annotate_color_color;
		imp.annotate_color_mono =   this.annotate_color_mono;
		imp.annotate_transparent_mono = this. annotate_transparent_mono;
		
		// vegetation
		imp.terr_model_path =                    this. terr_model_path;
		imp.terr_model_state =                   this. terr_model_state;
		imp.terr_segments_dir =                  this. terr_segments_dir;
		imp.terr_segments_suffix =               this.terr_segments_suffix;
		
		imp.terr_par_dir =                       this. terr_par_dir;
		imp.terr_par_file =                      this. terr_par_file;
		imp.terr_par_restore =                   this.terr_par_restore;
		imp.terr_step_restore =                  this.terr_step_restore;
		
		imp.terr_um_en =                         this.terr_um_en;    
		imp.terr_um_sigma =                      this.terr_um_sigma; 
		imp.terr_um_weight =                     this.terr_um_weight;
		imp.terr_nan_tolerance =                 this.terr_nan_tolerance;
		imp.terr_nan_grow =                      this.terr_nan_grow;
		imp.terr_shrink_veget =                  this.terr_shrink_veget;
		imp.terr_shrink_terrain =                this.terr_shrink_terrain;
		imp.terr_vegetation_over =               this.terr_vegetation_over;
		imp.terr_filter_veget =                  this.terr_filter_veget;

		imp.terr_veget_grow =                    this.terr_veget_grow;

		imp.terr_tile_woi =                      this.terr_tile_woi;                           
		imp.terr_continue =                      this.terr_continue;                           
		imp.terr_woi_enclos =                    new Rectangle(this.terr_woi_enclos);                           
		imp.terr_woi_step =                      new Rectangle(this.terr_woi_step);                           
		imp.terr_woi_last =                      new Rectangle(this.terr_woi_last);                           
		imp.terr_skip_exist =                    this.terr_skip_exist;                           

		imp.terr_fit_terr =                      this.terr_fit_terr;                           
		imp.terr_fit_veget =                     this.terr_fit_veget;                           
		imp.terr_fit_alpha =                     this.terr_fit_alpha;                           
		imp.terr_fit_elevations =                this.terr_fit_elevations;
		imp.terr_fit_terr_elev =                 this.terr_fit_terr_elev;
		imp.terr_fit_terr_elev_pix =             this.terr_fit_terr_elev_pix;
		imp.terr_fit_scenes =                    this.terr_fit_scenes;                           
		for (int i = 0; i < terr_fits_disable.length; i++) {
			imp.terr_fits_disable[i] =           this.terr_fits_disable[i].clone();
		}
		for (int i = 0; i < terr_only_fits_disable.length; i++) {
			imp.terr_only_fits_disable[i] =      this.terr_only_fits_disable[i].clone();
		}
		imp.terr_max_warp =                      this.terr_max_warp;
		imp.terr_max_elevation =                 this.terr_max_elevation;
		imp.terr_max_elev_terr =                 this.terr_max_elev_terr;
		imp.terr_max_elev_chng =                 this.terr_max_elev_chng;
		imp.terr_min_scenes =                    this.terr_min_scenes;
		imp.terr_min_samples_scene =             this.terr_min_samples_scene;
		imp.terr_min_total_scenes =              this.terr_min_total_scenes;
		imp.terr_min_pixels =                    this.terr_min_pixels;
		imp.terr_warm_veget =                    this.terr_warm_veget;
		imp.terr_warmest =                       this.terr_warmest;                           
		imp.terr_initial_split =                 this.terr_initial_split; 
		imp.terr_min_split_frac =                this.terr_min_split_frac;
		imp.terr_difference =                    this.terr_difference;		
		imp.terr_pull_cold =                     this.terr_pull_cold; 
		imp.terr_elevation_radius =              this.terr_elevation_radius;
		imp.terr_terr_elev_radius =              this.terr_terr_elev_radius;
		imp.terr_elev_radius_extra =             this.terr_elev_radius_extra;
		
		
		imp.terr_alpha_contrast =                this.terr_alpha_contrast; 
		imp.terr_alpha_dflt =                    this.terr_alpha_dflt;
		
		imp.terr_alpha_sigma =                   this.terr_alpha_sigma;
		imp.terr_alpha_init_min =                this.terr_alpha_init_min;
		imp.terr_alpha_init_max =                this.terr_alpha_init_max;
		imp.terr_alpha_init_offs =               this.terr_alpha_init_offs;
		
		imp.terr_alpha_loss =                    this.terr_alpha_loss;                           
		imp.terr_alpha_loss_lin =                this.terr_alpha_loss_lin;                           
		imp.terr_alpha_offset =                  this.terr_alpha_offset;
		imp.terr_alpha_0offset =                 this.terr_alpha_0offset;
		imp.terr_alpha_min_veg =                 this.terr_alpha_min_veg;
		imp.terr_alpha_max_terrain =             this.terr_alpha_max_terrain;
		imp.terr_alpha_pull_pwr =                this.terr_alpha_pull_pwr;
		
		imp.terr_alpha_lpf =                     this.terr_alpha_lpf;                           
		imp.terr_alpha_lpf_border =              this.terr_alpha_lpf_border;                           
		imp.terr_alpha_piece_linear =            this.terr_alpha_piece_linear;                           
		imp.terr_alpha_scale_avg =               this.terr_alpha_scale_avg;                           
		imp.terr_alpha_push =                    this.terr_alpha_push;                           
		imp.terr_alpha_push_neutral =            this.terr_alpha_push_neutral;                           
		imp.terr_alpha_weight_center =           this.terr_alpha_weight_center;
		imp.terr_en_holes =                      this.terr_en_holes;
		imp.terr_alpha_mm_hole =                 this.terr_alpha_mm_hole; 
		imp.terr_alpha_diff_hole =               this.terr_alpha_diff_hole;
		imp.terr_terr_lpf =                      this.terr_terr_lpf;                           
		imp.terr_veget_lpf =                     this.terr_veget_lpf;
		imp.terr_elev_lpf =                      this.terr_elev_lpf;
		imp.terr_terr_elev_lpf =                 this.terr_terr_elev_lpf;
		imp.terr_terr_pull0 =                    this.terr_terr_pull0;                           
		imp.terr_terr_pull_up =                  this.terr_terr_pull_up;                           
		imp.terr_terr_pull_avg =                 this.terr_terr_pull_avg;                           
		imp.terr_veget_pull0 =                   this.terr_veget_pull0; 
		imp.terr_veget_pull_low_alpha =          this.terr_veget_pull_low_alpha; 
		imp.terr_elev_pull0 =                    this.terr_elev_pull0;
		imp.terr_terr_elev_pull0 =               this.terr_terr_elev_pull0;
		imp.terr_elev_alpha_en =                 this.terr_elev_alpha_en;
		imp.terr_elev_alpha =                    this.terr_elev_alpha;
		imp.terr_elev_alpha_pwr =                this.terr_elev_alpha_pwr;
		imp.terr_low_veget =                     this.terr_low_veget;
		imp.terr_scenes_pull0 =                  this.terr_scenes_pull0;
		// scaling elevation losses for high elevations (decrease pull and/or lpf for high elevations)
		imp.terr_elev_scale_thresh =             this.terr_elev_scale_thresh;
		imp.terr_elev_scale_pull =               this.terr_elev_scale_pull;
		imp.terr_elev_scale_lpf =                this.terr_elev_scale_lpf;
		// tree-top removal
		imp.terr_ttop_en =                       this.terr_ttop_en;
		imp.terr_ttop_gb =                       this.terr_ttop_gb;
		imp.terr_ttop_min =                      this.terr_ttop_min;
		imp.terr_ttop_rel_lev =                  this.terr_ttop_rel_lev;
		imp.terr_ttop_rel_rad =                  this.terr_ttop_rel_rad;
		imp.terr_ttop_frac =                     this.terr_ttop_frac;
		imp.terr_ttop_rem_rad =                  this.terr_ttop_rem_rad;
		
		imp.terr_boost_parallax =                this.terr_boost_parallax;                           
		imp.terr_max_parallax =                  this.terr_max_parallax;   
		imp.terr_hifreq_weight =                 this.terr_hifreq_weight;
		imp.terr_terr_corr =                     this.terr_terr_corr;
		imp.terr_reg_weights =                   this.terr_reg_weights;                           
		imp.terr_lambda =                        this.terr_lambda;                           
		imp.terr_lambda_scale_good =             this.terr_lambda_scale_good;                           
		imp.terr_lambda_scale_bad =              this.terr_lambda_scale_bad;                           
		imp.terr_lambda_max =                    this.terr_lambda_max;                           
		imp.terr_rms_diff =                      this.terr_rms_diff;                           
		imp.terr_last_series =                   this.terr_last_series;                           
		imp.terr_num_iters =                     this.terr_num_iters.clone();
		
///		imp.terr_only_special =                  this.terr_only_special;                           
///		imp.terr_only_pix =                      this.terr_only_pix;                           
		imp.terr_only_series =                   this.terr_only_series;                           
		imp.terr_only_num_iters =                this.terr_only_num_iters.clone();
		imp.terr_cholesky =                      this.terr_cholesky;
		imp.terr_recalc_weights =                this.terr_recalc_weights.clone();                           
		imp.terr_recalc_opaque =                 this.terr_recalc_opaque;                           
		imp.terr_recalc_pedestal =               this.terr_recalc_pedestal;
		imp.terr_recalc_frac =                   this.terr_recalc_frac;
		imp.terr_recalc_dist =                   this.terr_recalc_dist;
		imp.terr_recalc_pwr =                    this.terr_recalc_pwr;
		imp.terr_recalc_gb =                     this.terr_recalc_gb;
		imp.terr_recalc_boost =                  this.terr_recalc_boost;
		imp.terr_recalc_average =                this.terr_recalc_average;
		
		imp.terr_crop =                          this.terr_crop;
		imp.terr_keep_partial =                  this.terr_keep_partial;
		imp.terr_border_width =                  this.terr_border_width;                           
		imp.terr_render_open =                   this.terr_render_open;                           
		imp.terr_render_no_alpha =               this.terr_render_no_alpha;                           
		imp.terr_alpha_min =                     this.terr_alpha_min;                           
		imp.terr_alpha_max =                     this.terr_alpha_max;                           
		imp.terr_weight_opaque =                 this.terr_weight_opaque;
		imp.terr_boost_render =                  this.terr_boost_render;
		imp.terr_max_render =                    this.terr_max_render;
		imp.terr_num_exaggerate =                this.terr_num_exaggerate;                           
		imp.terr_rebuild_elev =                  this.terr_rebuild_elev;                           
		imp.terr_elev_grow =                     this.terr_elev_grow;                           
		imp.terr_threshold_terrain =             this.terr_threshold_terrain;                           
		imp.terr_min_max_terrain =               this.terr_min_max_terrain;                           
		imp.terr_min_terrain =                   this.terr_min_terrain;                           
		imp.terr_min_vegetation =                this.terr_min_vegetation;  
		imp.terr_debug =                         this.terr_debug;
		imp.terr_debug_path =                    this.terr_debug_path;
		imp.terr_debug_length =                  this.terr_debug_length;
		imp.terr_debug_improved =                this.terr_debug_improved;
		imp.terr_debug_worsened =                this.terr_debug_worsened;
		// end of vegetation
		// Synthesis		
		imp.synth_directory =                    this.synth_directory;
		imp.synth_scene =                        this.synth_scene;
		imp.synth_model =                        this.synth_model;
		imp.synth_segments =                     this.synth_segments;
		imp.synth_render =                       this.synth_render;
		imp.synth_crop_real =                    this.synth_crop_real;
		imp.synth_scene_offs =                   this.synth_scene_offs;
		imp.synth_add_offs =                     this.synth_add_offs;
		return imp;
	}
	
	public static Color getColorFromHex(
			String hex_color,
			Color  default_color) {
		long lcolor = -1;
		Color color = default_color;
		try {
			lcolor = Long.parseLong(hex_color,16);
			color = setLongColor(lcolor);
		} catch(NumberFormatException e){
			color = default_color;
		}
		return color;
	}
	
	public static String getStringColor(
			Color color,
			Color default_color) {
		if (color == null) {
			color = default_color;
		}
		return getStringColor(color);
	}
	
	public static String getStringColor(Color color) {
		return String.format("%08x", getLongColor(color));
	}
	
	public static long getLongColor(
			Color color,
			Color default_color) {
		if (color == null) {
			color = default_color;
		}
		return getLongColor(color);
	}

	public static long getLongColor(Color color) {
		return ((long) color.getRGB()) & 0xffffffffL;
	}
	
	public static Color setLongColor(long lcolor) {
		if (lcolor < (1 << 24)) { // no alpha
			return new Color((int) lcolor);
		} else { // has alpha, may or may not fit into int
			if (lcolor > Integer.MAX_VALUE) {
				lcolor -= (1L << 32);
			}
			return new Color((int) lcolor, true);
		}
	}
	
	public void orderStereoViews(){
		boolean ordered;
		do {
			ordered=true;
			for (int i = 0; i < (stereo_views.length - 1); i++) {
				if (stereo_views[i+1][0] > stereo_views[i][0]) {
					continue;
				}
				if (    (stereo_views[i+1][0] == stereo_views[i][0]) &&
						(stereo_views[i+1][1] >  stereo_views[i][1])) {
					continue;
				}
				if (    (stereo_views[i+1][0] == stereo_views[i][0]) &&
						(stereo_views[i+1][1] == stereo_views[i][1]) &&
						(stereo_views[i+1][2] >  stereo_views[i][2])) {
					continue;
				}
				if (    (stereo_views[i+1][0] == stereo_views[i][0]) &&
						(stereo_views[i+1][1] == stereo_views[i][1]) &&
						(stereo_views[i+1][2] == stereo_views[i][2])) {
					// all same values - remove extra
					generate_stereo_var[i] |= generate_stereo_var[i+1];
					for (int j = i+1; j < (stereo_views.length - 1); j++) {
						generate_stereo_var[j] = generate_stereo_var[j+1];
						stereo_views[j] = stereo_views[j + 1]; 
					}
					ordered = false;
					break; // next while
				}
				boolean en = generate_stereo_var[i+1];
				generate_stereo_var[i+1] = generate_stereo_var[i];
				generate_stereo_var[i] = en;
				double [] view =  stereo_views[i+1];
				stereo_views[i+1] = stereo_views[i];
				stereo_views[i] = view;
				ordered = false;
			}

		} while (!ordered);
		return;
	}

	public void addStereoView(String stereo_view_string, boolean en) {
		double[] stereo_view = StringToDoubles(stereo_view_string,3);
		if (stereo_view != null) {
			addStereoView(stereo_view, en);
		}
	}
	
	public void addStereoView(double[] stereo_view, boolean en) {
		double [][]  views = new double [stereo_views.length + 1][];
		boolean [] ens = new boolean [stereo_views.length + 1];
		views[0] = stereo_view;
		ens[0] = en;
		System.arraycopy(stereo_views, 0, views, 1,      stereo_views.length);
		System.arraycopy(generate_stereo_var, 0, ens, 1, stereo_views.length);
		stereo_views = views;
		generate_stereo_var = ens;
		orderStereoViews();
	}
	
	public void removeStereoView(int indx) {
		if ((indx >=0) && (indx <stereo_views.length)) {
			double [][] views = new double [stereo_views.length - 1][];
			boolean [] ens = new boolean [stereo_views.length - 1];
			if (indx > 0) {
				System.arraycopy(stereo_views,        0, views, 0, indx);
				System.arraycopy(generate_stereo_var, 0, ens,   0, indx);
			}
			if (indx < (stereo_views.length - 1)) {
				System.arraycopy(stereo_views,        indx+1, views, indx, stereo_views.length - indx - 1);
				System.arraycopy(generate_stereo_var, indx+1, ens,   indx, stereo_views.length - indx - 1);
			}
			stereo_views = views;
			generate_stereo_var = ens;
		}
	}

	public static String booleansToString (boolean [] data, int mode) { // 0 : true/false, 1: 0/1, 2: +/-, negative - skip separators (", ")
		boolean use_sep = mode >= 0;
		mode = Math.abs(mode);
		String s = "";
		for (int i = 0; i < data.length; i++) {
			switch (mode) {
			case 0: s += data[i];           break;// true/false
			case 1: s += data[i] ? 1:0;     break;
			case 2: s += data[i] ? "+":"-"; break;
			default: s += data[i];// true/false
			}
			if (use_sep && (i < (data.length - 1))) {
				s+= ", ";
			}
		}
		return s;
	}
	
	public static boolean [] StringToBooleans(String s) {
		return StringToBooleans(s, -1);
	}
	
	public static boolean [] StringToBooleans(String s, int len) {
		StringTokenizer st = new StringTokenizer(s, " \t\n\r\f,");
		if (st.countTokens() == 0) {
			return null;
		}
		if (len <= 0) {
			len = st.countTokens();
		}
		boolean [] data = new boolean [len]; 
		int i = 0;
		while (st.hasMoreTokens() && (i < len)) {
			String tok = st.nextToken();
			data[i++] = (tok.equals("true") || tok.equals("1") || tok.equals("+") || tok.equals("=true")); //"+true" - mitigating a bug
		}
		return data;
	}
	
	
	public static String double2dToString(double [][] d) {
		String s = "";
		for (int i = 0; i < d.length; i++) {
			for (int j = 0; j < d[i].length; j++) {
				Double D = d[i][j];
				s+= D.toString();
				if (j <(d[i].length - 1)) {
					s +=",";
				}
			}
			if (i < (d.length - 1)) {
				s +=":";
			}
		}
		return s;
	}
	
	public static double [][] stringToDouble2d(String s) {
		String [] srows = s.split(":");
		double [][] d = new double [srows.length][];
		for (int i = 0; i < d.length; i++) {
			String [] scols = srows[i].split(",");
			d[i] = new double [scols.length];
			for (int j = 0; j < d[i].length; j++) {
				try {
					d[i][j]	= Double.valueOf(scols[j]);
				} catch(NumberFormatException e){
					d[i][j] = 0.0;
				}
			}
		}
		return d;
	}
	
	public static String rectangleToString(Rectangle r) {
		return intsToString(new int []{r.x,r.y,r.width, r.height});
	}
	
	public static Rectangle stringToRectangle(String s) {
		int [] data = StringToInts(s, 4);
		return new Rectangle(data[0], data[1], data[2], data[3]);
	}
	
	
	public static String intsToString(int [] data) {
		return intsToString(data, null);
	}
	
	public static String intsToString(int [] data, String fmt) {
		String s = "";
		for (int i = 0; i < data.length; i++) {
			if (fmt==null) {
				s += data[i]; // unformatted
			} else { 
				s+=String.format(fmt,data[i]);
			}
			if (i < (data.length - 1)) {
				s+= ", ";
			}
		}
		return s;
	}

	public static int [] StringToInts(String s) {
		return StringToInts(s, -1);
	}

	
	public static int [] StringToInts(String s, int len) {
		StringTokenizer st = new StringTokenizer(s, " \t\n\r\f,");
		if (st.countTokens() == 0) {
			return null;
		}
		if (len <= 0) {
			len = st.countTokens();
		}
		int [] data = new int [len]; 
		int i = 0;
		while (st.hasMoreTokens() && (i < len)) {
			int d = 0;
			try {
				d = Integer.parseInt(st.nextToken());
			} catch(NumberFormatException e){
				d = 0;
			}

			data[i++] = d;
		}
		return data;
	}

	public static int [] StringToInts(String s, int [] default_data) {
		int len = default_data.length;
		StringTokenizer st = new StringTokenizer(s, " \t\n\r\f,");
		if (st.countTokens() == 0) {
			return null;
		}
		if (len <= 0) {
			len = st.countTokens();
		}
		int [] data = new int [len]; 
		int i = 0;
		while (st.hasMoreTokens() && (i < len)) {
			int d = 0;
			try {
				d = Integer.parseInt(st.nextToken());
			} catch(NumberFormatException e){
				d = default_data[i];
			}

			data[i++] = d;
		}
		return data;
	}
	
	
	public static String doublesToString(double [] data) {
		return doublesToString(data, null);
	}
	public static String doublesToString(double [] data, String fmt) {
		String s = "";
		for (int i = 0; i < data.length; i++) {
			if (fmt==null) {
				s += data[i]; // unformatted
			} else { 
				s+=String.format(fmt,data[i]);
			}
			if (i < (data.length - 1)) {
				s+= ", ";
			}
		}
		return s;
	}
	
	public static double [] StringToDoubles(String s, double [] default_data) {
		int len = default_data.length;
		StringTokenizer st = new StringTokenizer(s, " \t\n\r\f,");
		if (st.countTokens() == 0) {
			return null;
		}
		if (len <= 0) {
			len = st.countTokens();
		}
		double [] data = new double [len]; 
		int i = 0;
		while (st.hasMoreTokens() && (i < len)) {
			double d = 0;
			try {
				d = Double.parseDouble(st.nextToken());
			} catch(NumberFormatException e){
				d = default_data[i];
			}
			data[i++] = d;
		}
		return data;
	}
	
	
	
	public static double [] StringToDoubles(String s, int len) {
		StringTokenizer st = new StringTokenizer(s, " \t\n\r\f,");
		if (st.countTokens() == 0) {
			return null;
		}
		if (len <= 0) {
			len = st.countTokens();
		}
		double [] data = new double [len]; 
		int i = 0;
		while (st.hasMoreTokens() && (i < len)) {
			double d = 0;
			try {
				d = Double.parseDouble(st.nextToken());
			} catch(NumberFormatException e){
				d = 0;
			}
			data[i++] = d;
		}
		return data;
	}
	
	public static float [] StringToFloats(String s, int len) {
		StringTokenizer st = new StringTokenizer(s, " \t\n\r\f,");
		if (st.countTokens() == 0) {
			return null;
		}
		if (len <= 0) {
			len = st.countTokens();
		}
		float [] data = new float [len]; 
		int i = 0;
		while (st.hasMoreTokens() && (i < len)) {
			double d = 0;
			try {
				d = Double.parseDouble(st.nextToken());
			} catch(NumberFormatException e){
				d = 0;
			}

			data[i++] = (float) d;
		}
		return data;
	}
		
	public static String floatsToString(float [] data) {
		return floatsToString(data, null);
	}
	
	public static String floatsToString(float [] data, String fmt) {
//		if ((fmt == null) || (fmt.trim().length()==0)) {
//			fmt = "%.0f";
//		}
		String s = "";
		for (int i = 0; i < data.length; i++) {
			if (fmt==null) {
				s += data[i]; // unformatted
			} else { 
				s+=String.format(fmt,data[i]);
			}
			if (i < (data.length - 1)) {
				s+= ", ";
			}
		}
		return s;
	}
	
	
	
}

