#!/usr/bin/env bash
set -euo pipefail

SCRIPT_DIR="$(cd "$(dirname "${BASH_SOURCE[0]}")" && pwd)"
REPO_ROOT="$(cd "${SCRIPT_DIR}/.." && pwd)"
WORKSPACE="${REPO_ROOT}/workspace"
LOG_DIR="${REPO_ROOT}/logs"
TMP_WORK="${WORKSPACE}/poky/build/tmp/work"

mkdir -p "${LOG_DIR}"

REPORT_TS="$(date +%Y%m%d_%H%M%S)"
REPORT_FILE="${LOG_DIR}/warning_report_${REPORT_TS}.md"
LATEST_LINK="${LOG_DIR}/warning_report_latest.md"

TMP_DIR="$(mktemp -d)"
trap 'rm -rf "${TMP_DIR}"' EXIT

DEMOTE_RAW="${TMP_DIR}/demote.raw"
CHECKSUM_RAW="${TMP_DIR}/checksum.raw"
FUZZ_RAW="${TMP_DIR}/fuzz.raw"
HOST_CONTAM_RAW="${TMP_DIR}/host_contam.raw"

: > "${DEMOTE_RAW}"
: > "${CHECKSUM_RAW}"
: > "${FUZZ_RAW}"
: > "${HOST_CONTAM_RAW}"

if [[ -d "${TMP_WORK}" ]]; then
  rg -n "Tried to access '.*' as root, demoting to a normal user" "${TMP_WORK}" -S > "${DEMOTE_RAW}" || true
fi

shopt -s nullglob
stage_logs=("${LOG_DIR}"/build_stage1_*.log)
shopt -u nullglob

if [[ ${#stage_logs[@]} -gt 0 ]]; then
  rg -n "Checksum mismatch for local file" "${stage_logs[@]}" -S > "${CHECKSUM_RAW}" || true
  rg -n "patch-fuzz|Fuzz detected|Hunk #[0-9]+ succeeded at .* fuzz" "${stage_logs[@]}" -S > "${FUZZ_RAW}" || true
  rg -n "host-user-contaminated|is owned by uid [0-9]+, which is the same as the user running bitbake" "${stage_logs[@]}" -S > "${HOST_CONTAM_RAW}" || true
fi

demote_total="$(wc -l < "${DEMOTE_RAW}" | tr -d ' ')"
checksum_total="$(wc -l < "${CHECKSUM_RAW}" | tr -d ' ')"
fuzz_total="$(wc -l < "${FUZZ_RAW}" | tr -d ' ')"
host_contam_total="$(wc -l < "${HOST_CONTAM_RAW}" | tr -d ' ')"

demote_paths="${TMP_DIR}/demote_paths.txt"
demote_recipe_task="${TMP_DIR}/demote_recipe_task.txt"
checksum_recipe="${TMP_DIR}/checksum_recipe.txt"

if [[ "${demote_total}" -gt 0 ]]; then
  sed -n "s#.*Tried to access '\\([^']*\\)'.*#\\1#p" "${DEMOTE_RAW}" \
    | sort | uniq -c | sort -nr > "${demote_paths}"

  sed -E -n "s#^(.*/tmp/work/[^/]+/([^/]+)/([^/]+)/temp/log\\.(do_[^.]+)\\.[^:]+):.*#\\2\\t\\3\\t\\4#p" "${DEMOTE_RAW}" \
    | sort | uniq -c | sort -nr > "${demote_recipe_task}"
fi

if [[ "${checksum_total}" -gt 0 ]]; then
  sed -E -n "s#.*WARNING: ([^ ]+) do_fetch: Checksum mismatch.*#\\1#p" "${CHECKSUM_RAW}" \
    | sort | uniq -c | sort -nr > "${checksum_recipe}"
fi

{
  echo "# Warning Report"
  echo
  echo "- Generated: $(date -Is)"
  echo "- Workspace: ${WORKSPACE}"
  echo "- Stage logs scanned: ${#stage_logs[@]}"
  echo
  echo "## Summary"
  echo
  echo "- Root demotion warnings: ${demote_total}"
  echo "- Checksum mismatch warnings: ${checksum_total}"
  echo "- Patch fuzz warnings: ${fuzz_total}"
  echo "- Host contamination warnings: ${host_contam_total}"
  echo

  if [[ "${demote_total}" -gt 0 ]]; then
    echo "## Root Demotion by External Path (top 25)"
    echo
    echo '```text'
    sed -n '1,25p' "${demote_paths}"
    echo '```'
    echo

    echo "## Root Demotion by Recipe / Version / Task (top 40)"
    echo
    echo '```text'
    sed -n '1,40p' "${demote_recipe_task}"
    echo '```'
    echo

    echo "## Root Demotion Sample Source Lines"
    echo
    echo '```text'
    sed -n '1,40p' "${DEMOTE_RAW}"
    echo '```'
    echo
  fi

  if [[ "${checksum_total}" -gt 0 ]]; then
    echo "## Checksum Mismatch by Recipe"
    echo
    echo '```text'
    cat "${checksum_recipe}"
    echo '```'
    echo
    echo "## Checksum Mismatch Sample Lines"
    echo
    echo '```text'
    sed -n '1,40p' "${CHECKSUM_RAW}"
    echo '```'
    echo
  fi

  if [[ "${fuzz_total}" -gt 0 ]]; then
    echo "## Patch Fuzz Sample Lines"
    echo
    echo '```text'
    sed -n '1,40p' "${FUZZ_RAW}"
    echo '```'
    echo
  fi

  if [[ "${host_contam_total}" -gt 0 ]]; then
    echo "## Host Contamination Sample Lines"
    echo
    echo '```text'
    sed -n '1,40p' "${HOST_CONTAM_RAW}"
    echo '```'
    echo
  fi
} > "${REPORT_FILE}"

ln -sfn "$(basename "${REPORT_FILE}")" "${LATEST_LINK}"

echo "Report: ${REPORT_FILE}"
echo "Latest: ${LATEST_LINK}"
