#!/usr/bin/env bash
set -euo pipefail

SCRIPT_DIR="$(cd "$(dirname "${BASH_SOURCE[0]}")" && pwd)"
REPO_ROOT="$(cd "${SCRIPT_DIR}/.." && pwd)"
WORKSPACE="${REPO_ROOT}/workspace"

REMOTE_URL_DEFAULT="git@git.elphel.com:Elphel/elphel393.git"
REMOTE_URL="${REMOTE_URL:-$REMOTE_URL_DEFAULT}"
BRANCH="${BRANCH:-warrior}"
RUN_SETUP=0

usage() {
  cat <<'EOF'
Prepare local workspace/elphel393 clone using SSH remote.

Usage:
  bootstrap_workspace.sh [--run-setup]

Environment overrides:
  REMOTE_URL   default: git@git.elphel.com:Elphel/elphel393.git
  BRANCH       default: warrior

Notes:
  - This script enforces developer mode (SSH remotes only).
  - --run-setup runs ./setup.py after clone/update.
EOF
}

while [[ $# -gt 0 ]]; do
  case "$1" in
    --run-setup)
      RUN_SETUP=1
      shift
      ;;
    -h|--help)
      usage
      exit 0
      ;;
    *)
      echo "ERROR: unknown argument: $1" >&2
      usage
      exit 1
      ;;
  esac
done

if [[ "${REMOTE_URL}" != git@git.elphel.com:* ]]; then
  echo "ERROR: developer mode only. REMOTE_URL must be ssh (git@git.elphel.com:...)" >&2
  exit 2
fi

mkdir -p "${WORKSPACE}"

if [[ ! -d "${WORKSPACE}/.git" ]]; then
  git clone -b "${BRANCH}" "${REMOTE_URL}" "${WORKSPACE}"
else
  (
    cd "${WORKSPACE}"
    git remote set-url origin "${REMOTE_URL}"
    git fetch origin
    git checkout "${BRANCH}"
    git pull --ff-only origin "${BRANCH}"
  )
fi

(
  cd "${WORKSPACE}"
  git remote set-url origin "${REMOTE_URL}"
)

if [[ "${RUN_SETUP}" -eq 1 ]]; then
  (
    cd "${WORKSPACE}"
    ./setup.py
  )
fi

echo "Workspace ready: ${WORKSPACE}"
echo "Origin: $(cd "${WORKSPACE}" && git remote get-url origin)"
