#!/usr/bin/env bash
set -euo pipefail

SCRIPT_DIR="$(cd "$(dirname "${BASH_SOURCE[0]}")" && pwd)"
REPO_ROOT="$(cd "${SCRIPT_DIR}/.." && pwd)"

if command -v docker >/dev/null 2>&1; then
  :
else
  echo "ERROR: docker is not installed" >&2
  exit 1
fi

has_docker_group() {
  id -nG | tr ' ' '\n' | grep -qx docker
}

can_access_docker_daemon() {
  docker info >/dev/null 2>&1
}

if ! can_access_docker_daemon; then
  if [[ "${RUN_DOCKER_SG_FALLBACK:-0}" != "1" ]] \
    && command -v sg >/dev/null 2>&1 \
    && has_docker_group; then
    printf -v REEXEC_CMD '%q ' "$0" "$@"
    exec sg docker -c "RUN_DOCKER_SG_FALLBACK=1 ${REEXEC_CMD}"
  fi

  if has_docker_group; then
    echo "ERROR: cannot access Docker daemon from this shell." >&2
    echo "Hint: run 'newgrp docker' in this shell, or rerun with:" >&2
    echo "  sg docker -c '$0 ${*:-}'" >&2
  else
    echo "ERROR: user '$(id -un)' is not in docker group." >&2
    echo "Run once as sudo, then re-login:" >&2
    echo "  sudo usermod -aG docker $(id -un)" >&2
  fi
  exit 1
fi

if docker compose version >/dev/null 2>&1; then
  COMPOSE=(docker compose)
elif command -v docker-compose >/dev/null 2>&1; then
  COMPOSE=(docker-compose)
else
  echo "ERROR: docker compose is not available" >&2
  exit 1
fi

mkdir -p "${REPO_ROOT}/workspace" "${REPO_ROOT}/cache/downloads" "${REPO_ROOT}/cache/sstate-cache" "${REPO_ROOT}/cache/ccache"

export USER_ID="${USER_ID:-$(id -u)}"
export GROUP_ID="${GROUP_ID:-$(id -g)}"
export USER_NAME="${USER_NAME:-$(id -un)}"

if [[ $# -eq 0 ]]; then
  CMD=(bash)
else
  CMD=("$@")
fi

"${COMPOSE[@]}" -f "${REPO_ROOT}/docker/docker-compose.yml" run --rm warrior "${CMD[@]}"
