#!/usr/bin/env bash
set -euo pipefail

# Open a new Konsole window in a 2x3 split layout and run SSH sessions.
# Default targets:
#   root@192.168.0.41 ... root@192.168.0.46

usage() {
  cat <<'EOF'
Usage:
  konsole_ssh_2x3.sh [user@host ...]
  konsole_ssh_2x3.sh [--user USER] [--prefix A.B.C] [--from N] [--dmesg]

Options:
  --user USER      SSH user for generated host list (default: root)
  --prefix A.B.C   First 3 octets for generated host list (default: 192.168.0)
  --from N         Starting host number, creates 6 hosts N..N+5 (default: 41)
  --dmesg          Run 'dmesg' on remote after SSH login, then keep interactive shell
  -h, --help       Show this help

Examples:
  konsole_ssh_2x3.sh
  konsole_ssh_2x3.sh --dmesg
  konsole_ssh_2x3.sh --user root --prefix 192.168.0 --from 41
  konsole_ssh_2x3.sh root@192.168.0.41 root@192.168.0.42 root@192.168.0.43 \
                     root@192.168.0.44 root@192.168.0.45 root@192.168.0.46
EOF
}

log() {
  printf '[%s] %s\n' "$(date '+%F %T')" "$*"
}

need_cmd() {
  command -v "$1" >/dev/null 2>&1 || {
    echo "ERROR: required command not found: $1" >&2
    exit 1
  }
}

need_cmd konsole
need_cmd qdbus
need_cmd rg

ssh_user="root"
host_prefix="192.168.0"
start_octet=41
run_dmesg=0
declare -a targets=()

while [[ $# -gt 0 ]]; do
  case "$1" in
    --user)
      ssh_user="$2"
      shift 2
      ;;
    --prefix)
      host_prefix="$2"
      shift 2
      ;;
    --from)
      start_octet="$2"
      shift 2
      ;;
    --dmesg)
      run_dmesg=1
      shift
      ;;
    -h|--help)
      usage
      exit 0
      ;;
    --)
      shift
      while [[ $# -gt 0 ]]; do
        targets+=("$1")
        shift
      done
      ;;
    -*)
      echo "ERROR: unknown option: $1" >&2
      usage
      exit 1
      ;;
    *)
      targets+=("$1")
      shift
      ;;
  esac
done

if [[ "${#targets[@]}" -eq 0 ]]; then
  for i in 0 1 2 3 4 5; do
    n=$((start_octet + i))
    targets+=("${ssh_user}@${host_prefix}.${n}")
  done
fi

if [[ "${#targets[@]}" -ne 6 ]]; then
  echo "ERROR: expected exactly 6 SSH targets, got ${#targets[@]}" >&2
  exit 1
fi

layout_file="$(mktemp --tmpdir konsole_2x3_layout_XXXXXX.json)"
trap 'rm -f "$layout_file"' EXIT

cat >"$layout_file" <<'EOF'
{
  "Orientation": "Vertical",
  "Widgets": [
    {
      "Orientation": "Horizontal",
      "Widgets": [ { "SessionRestoreId": 0 }, { "SessionRestoreId": 0 } ]
    },
    {
      "Orientation": "Horizontal",
      "Widgets": [ { "SessionRestoreId": 0 }, { "SessionRestoreId": 0 } ]
    },
    {
      "Orientation": "Horizontal",
      "Widgets": [ { "SessionRestoreId": 0 }, { "SessionRestoreId": 0 } ]
    }
  ]
}
EOF

log "Opening Konsole window with 2x3 layout"
konsole --separate --layout "$layout_file" >/dev/null 2>&1 &
konsole_pid="$!"
service="org.kde.konsole-${konsole_pid}"

# Wait for Konsole DBus service.
for _ in $(seq 1 100); do
  if qdbus "$service" >/dev/null 2>&1; then
    break
  fi
  sleep 0.05
done

if ! qdbus "$service" >/dev/null 2>&1; then
  echo "ERROR: failed to connect to $service" >&2
  exit 1
fi

window_path=""
for _ in $(seq 1 100); do
  window_path="$(qdbus "$service" 2>/dev/null | rg '^/Windows/[0-9]+$' | head -n1 || true)"
  if [[ -n "$window_path" ]]; then
    break
  fi
  sleep 0.05
done

if [[ -z "$window_path" ]]; then
  echo "ERROR: could not find Konsole window DBus path for $service" >&2
  exit 1
fi

# Wait until all split sessions are created.
for _ in $(seq 1 100); do
  count="$(qdbus "$service" "$window_path" org.kde.konsole.Window.sessionCount 2>/dev/null || echo 0)"
  if [[ "$count" =~ ^[0-9]+$ ]] && [[ "$count" -ge 6 ]]; then
    break
  fi
  sleep 0.05
done

if ! [[ "$count" =~ ^[0-9]+$ ]] || [[ "$count" -lt 6 ]]; then
  echo "ERROR: expected >=6 sessions, got: ${count:-unknown}" >&2
  exit 1
fi

mapfile -t sessions < <(
  qdbus "$service" "$window_path" org.kde.konsole.Window.sessionList \
    | tr -d '"' \
    | sed '/^$/d' \
    | head -n 6 \
    | awk '{a[NR]=$0} END{for(i=NR;i>=1;i--) print a[i]}'
)

if [[ "${#sessions[@]}" -ne 6 ]]; then
  echo "ERROR: failed to collect 6 session IDs (got ${#sessions[@]})" >&2
  exit 1
fi

for i in 0 1 2 3 4 5; do
  session_path="${sessions[$i]}"
  [[ "$session_path" == /Sessions/* ]] || session_path="/Sessions/$session_path"
  if [[ "$run_dmesg" -eq 1 ]]; then
    cmd="ssh -t -o ConnectTimeout=5 ${targets[$i]} 'dmesg; exec \${SHELL:-/bin/sh} -i'"
  else
    cmd="ssh -o ConnectTimeout=5 ${targets[$i]}"
  fi
  log "Session $((i + 1)): $cmd"
  qdbus "$service" "$session_path" org.kde.konsole.Session.runCommand "$cmd" >/dev/null
done

log "Done."
