#!/usr/bin/env bash
set -euo pipefail

SCRIPT_DIR="$(cd "$(dirname "${BASH_SOURCE[0]}")" && pwd)"
REPO_ROOT="$(cd "${SCRIPT_DIR}/.." && pwd)"
WORKSPACE="${REPO_ROOT}/workspace"
REF_DIR="${REPO_ROOT}/docs/reference/107"
USE_107_LOCALCONF="${USE_107_LOCALCONF:-1}"
LOG_DIR="${REPO_ROOT}/logs"
LOG_TS="$(date +%Y%m%d_%H%M%S)"
LOG_FILE="${LOG_DIR}/build_stage1_${LOG_TS}.log"
LATEST_LOG="${LOG_DIR}/build_stage1_latest.log"

mkdir -p "${LOG_DIR}"
ln -sfn "$(basename "${LOG_FILE}")" "${LATEST_LOG}"
exec > >(tee -a "${LOG_FILE}") 2>&1
echo "Stage1 log: ${LOG_FILE}"
echo "Latest log symlink: ${LATEST_LOG}"

copy_localconf_107() {
  local src="${REF_DIR}/local.conf"
  local dst="${WORKSPACE}/poky/build/conf/local.conf"

  if [[ ! -f "${src}" ]]; then
    echo "WARN: ${src} missing, skipping .107 local.conf override" >&2
    return 0
  fi
  if [[ ! -f "${dst}" ]]; then
    echo "WARN: ${dst} missing, cannot apply .107 local.conf yet" >&2
    return 0
  fi

  cp -f "${dst}" "${dst}.before_107" || true
  cp -f "${src}" "${dst}"
  echo "Applied .107 local.conf -> ${dst}"
}

if [[ ! -x "${WORKSPACE}/setup.py" ]]; then
  echo "ERROR: ${WORKSPACE}/setup.py not found or not executable." >&2
  echo "Run ./scripts/bootstrap_workspace.sh first." >&2
  exit 1
fi

if [[ $# -eq 0 ]]; then
  TARGETS=(u-boot device-tree linux-xlnx core-image-elphel393)
else
  TARGETS=("$@")
fi

TARGETS_STR="${TARGETS[*]}"

# Step 1: setup and generate config in workspace.
"${SCRIPT_DIR}/run_docker.sh" bash -lc "
set -e
cd /work/elphel393
mkdir -p \"\${HOME}/.ssh\"
chmod 700 \"\${HOME}/.ssh\"
touch \"\${HOME}/.ssh/known_hosts\"
chmod 600 \"\${HOME}/.ssh/known_hosts\"
ssh-keyscan -H git.elphel.com >> \"\${HOME}/.ssh/known_hosts\" 2>/dev/null || true
./setup.py
"

# Step 2: enforce known-good local.conf from .107 when requested.
if [[ "${USE_107_LOCALCONF}" == "1" ]]; then
  copy_localconf_107
fi

# Step 3: build targets.
"${SCRIPT_DIR}/run_docker.sh" bash -lc "
set -e
cd /work/elphel393/poky
set +u
. ./oe-init-build-env build
set -u
export DL_DIR=/cache/downloads
export SSTATE_DIR=/cache/sstate-cache
bitbake ${TARGETS_STR}
"
