#!/usr/bin/env python3

# Download footage from ssd's raw partition:
# * standalone SSDs connected using a docking station or enclosure or else, raw partition, camera can be powered off
#   ** All information is in the Exif headers, so no need to match SSDs to subcameras
# * camera's internal SSD, raw partition, camera is powered on.
#

__author__ = "Elphel"
__copyright__ = "Copyright 2016, Elphel, Inc."
__license__ = "GPL"
__version__ = "3.0+"
__maintainer__ = "Oleg K Dzhimiev"
__email__ = "oleg@elphel.com"
__status__ = "Development"

import argparse
import os
import sys
import time
import math

import x393
from x393 import bcolors

# TODO: Download imu logs

ssd_detect_timeout = 20

usage = """

keys:
  To access camera without a password run:
  
  $ ssh-copy-id root@192.168.0.9
  
  To create keys, run:
  
  $ ssh-keygen

examples:\n

I.
* internal SSD, camera is powered on
* The data is on the raw partition /dev/sdb2
* The file with camogm pointers is on /dev/sdb1
* Destination path: /data/footage/test
  Download:
  * Whole footage based on the pointers in (/dev/sdb1) camogm.disk file in 10GB chunks:
    {1}${0} -c root@192.168.0.9 /data/footage/test{2}
  * Whole footage (based on camogm.disk) in 10GB chunks but skipping the first 2 chunks:
    {1}${0} -c root@192.168.0.9 -s 2 /data/footage/test{2}
  * Whole footage (based on camogm.disk) in 1GB chunks:
    {1}${0} -c root@192.168.0.9 -bc 50 /data/footage/test{2}
  * 5GB in 1GB chunks, skip the first 3GBs:
    {1}${0} -c root@192.168.0.9 -bc 50 -s 3 -n 5 /data/footage/test{2}
    
II. multiple cameras modes:
  * "eyesis4pi"
      * Whole footage (based on camogm.disk) in 10GB chunks:
        {1}${0} -m eyesis4pi /data/footage/test{2}  

""".format(sys.argv[0],bcolors.OKGREEN,bcolors.ENDC)

parser = argparse.ArgumentParser(description=usage,formatter_class=argparse.RawDescriptionHelpFormatter)
parser.add_argument("-c","--camera", default="", help="provide camera ip address if downloading data from internal SSD, default = root@192.168.0.9")
parser.add_argument("-m","--mode",default="",help="preset for multiple cameras, available modes: 'eyesis4pi'")
parser.add_argument("-s","--skip",type=int,default=0, help="Number of blocks to skip from the beginning") # chunks to skip from the beginning")
parser.add_argument("-n","--n",type=int,default=0,help="Number of blocks to download) # chunks to download") 
parser.add_argument("-bs",type=int,default=20,help="block size in MB, default = 20")
parser.add_argument("-bc",type=int,default=512,help="Number of blocks of size [bs] in a single chunk, default = 512, so the default chunk size is 10GB")
parser.add_argument("-fs","--file_start",default="",help=".disk file name with start LBA pointer, will calculate 'skip' value")
parser.add_argument("-fe","--file_end",default="camogm.disk",help=".disk file name with end LBA pointer, default is 'camogm.disk'")
parser.add_argument("dest",help="desitnation directory: /data/footage/test")
parser.add_argument("-l","--lan", action='store_true', help="Transfer data over LAN instead of eSATA")

args = parser.parse_args()
print (args)

cams = []
dirs = []

if args.camera!="":
  tmp = args.camera.split("@")
  if len(tmp)==2:
    cams = [{"user":tmp[0],"ip":tmp[1],"imu_log":0}]
elif args.mode=="eyesis4pi":
  cams = [
    {"user":"root","ip":"192.168.0.161","imu_log":0},
    {"user":"root","ip":"192.168.0.162","imu_log":0},
    {"user":"root","ip":"192.168.0.163","imu_log":0}
    ]
else:
  cams = [{"user":"root","ip":"192.168.0.9","imu_log":0}]
  print("running default: "+cams[0]['user']+"@"+cams[0]['ip'])

# init checks all connections
for cam in cams:
  cam['obj'] = x393.Camera(cam['user'],cam['ip'])
  if cam['obj'].disable:
    print(bcolors.WARNING+"Will skip "+cam['user']+"@"+cam['ip']+bcolors.ENDC)

cams[:] = [tmp for tmp in cams if not tmp['obj'].disable]

if len(cams)==0:
  print(usage)

pc = x393.PC()



if (args.lan):
  print ("Using data transfer over LAN")
else:  
  print ("Using data transfer over eSATA, multiplexing camera SSD to eSATA external port")
# ssd to camera for all
  for cam in cams:
    cam['obj'].ssd_to_camera()
  # get raw partitions
for cam in cams:
  d = cam['obj'].first_found_raw_partition_name() # ssh
  if d!="undefined":
    dirs.append(d)
    print(cam['user']+"@"+cam['ip']+": raw partition name: "+d)
  else:
    cam['disable']=1
    print(bcolors.FAIL+cam['user']+"@"+cam['ip']+" : error: already switched or raw partition not found"+bcolors.ENDC)
      #raise Exception(cam['user']+"@"+cam['ip']+" : error: already switched or raw partition not found")
  # no need
  cams[:] = [tmp for tmp in cams if tmp.get('disable')!=1]
  
if (not args.lan):
  # switch ssd to pc for all (not caring which cable is in)
  for cam in cams:
    cam['obj'].ssd_to_pc()

# download

plist = []
all_downloaded = False

#for i in range(len(cams)):
for i,cam in enumerate(cams):
#  raw_input(bcolors.OKGREEN+"Connect camera (eSATA) to PC (eSATA/SATA). Press Enter to continue..."+bcolors.ENDC)
  if (args.lan):
    print(bcolors.OKGREEN+"Initiating data transfer over LAN ..."+bcolors.ENDC)
  else:    
    input(bcolors.OKGREEN+"Connect camera (eSATA) to PC (eSATA/SATA). Press Enter to continue..."+bcolors.ENDC)
  proceed_to_next = False
  t = 0
  while not all_downloaded:
    if (args.lan):
      plist = cam['obj'].list_partitions_as_host() #  Camera partitions, same format as for host (label, path)
      #[['NT-1TB_2242_0007988000104', 'sda'], ['NT-1TB_2242_0007988000104-part1', 'sda1'], ['NT-1TB_2242_0007988000104-part2', 'sda2']]
    else:
      plist = pc.list_partitions() #  HOST(!) partitions, such as /dev/sdc1
      #[[u'NT-1TB_2242_0007988000104', u'sdc'], [u'NT-1TB_2242_0007988000104-part1', u'sdc1'], [u'NT-1TB_2242_0007988000104-part2', u'sdc2']]
    for d in dirs:
      for p in plist:
        if d==p[0]:
          # p[1] == sdb2
          # hardcoded /dev/sd?1
          if args.n==0:
            if (args.lan):
#              data_size_blocks = pc.read_camogm_disk_file_blocks_lan(cams[i].ip, p[1][0:-1]+"1", args.file_end)
#              data_skip_blocks = pc.read_camogm_disk_file_blocks_lan(cams[i].ip, p[1][0:-1]+"1", args.file_start)
              data_size_blocks = cam['obj'].read_camogm_disk_file_blocks(p[1][0:-1]+"1", args.file_end)
              data_skip_blocks = cam['obj'].read_camogm_disk_file_blocks(p[1][0:-1]+"1", args.file_start)
              #read_camogm_disk_file_blocks
            else:
              data_size_blocks = pc.read_camogm_disk_file_blocks("/dev/"+p[1][0:-1]+"1", args.file_end)
              data_skip_blocks = pc.read_camogm_disk_file_blocks("/dev/"+p[1][0:-1]+"1", args.file_start)
            data_size_blocks -= data_skip_blocks # before it included skipped !
          else:
            data_size_blocks = args.n
            data_skip_blocks = args.skip
          if (data_size_blocks < 0):
              data_size_blocks = 0 # if camogm.disk could not be read?  
          print ("data_size_blocks=%d, data_skip_blocks=%d"%(data_size_blocks,data_skip_blocks))  
          chunk_blocks=32768 # 4096
          block_size=    512 # 4096
          data_size_gb = (data_size_blocks * block_size) / (1024 * 1024 * 1024)
          file_gb = args.bs*args.bc // 1024 # just for compatibility, change to parameter
          print("Data size: %d %d-byte blocks (%f GB)"%(data_size_blocks, block_size, data_size_gb))
          if (args.lan):
            print(cam)  
            pc.download_blocks_lan(
                               cam,
                               args.dest,
                               "/dev/"+p[1], 
                               blocks_load=data_size_blocks,
                               blocks_skip= data_skip_blocks,
                               file_gb=file_gb,
                               chunk_blocks=chunk_blocks,
                               block_size=block_size)
          else:
            pc.download_blocks( # after testing use download_blocks_lan (rename/delete download_blocks) with cam=None
                               args.dest,
                               "/dev/"+p[1], 
                               blocks_load=data_size_blocks,
                               blocks_skip= data_skip_blocks,
                               file_gb=file_gb,
                               chunk_blocks=chunk_blocks,
                               block_size=block_size)
                
          dirs.remove(d)
          proceed_to_next = True
          if len(dirs) != 0:
            print("wait for the next ssd")
          else:
            all_downloaded = True
          break
    if proceed_to_next:
      break
    print("Waiting for disks to show up:")
    print(dirs)
    t = t + 1
    time.sleep(1)
    if t>ssd_detect_timeout:
      print(bcolors.FAIL+"PC: Error: Disks were not found:")
      print(dirs)
      print(bcolors.ENDC)
      break
    #cam['rp_name'].ssd_to_pc()

# ssd to camera for all
if (not args.lan):
  for cam in cams:
    cam['obj'].ssd_to_camera()

print("Done")
